/* histmatch -- string matching function for history field matches */

/* m.a.huckvale */

/* version 1.0 - june 1987 */

/*--------------------------------------------------------------------------*/
/**MAN
.TH HISTMATCH SFS3 UCL SFS
.SH NAME
histmatch - string matching on speech file histories
.SH SYNOPSIS
.nf

int     \fBhistmatch\fR(exp,targ,ret0,ret1,...)
char    *exp;             /\* match expression *\/
char    *targ;            /\* target string *\/
char    *ret0,*ret1,...   /\* returned sub-strings *\/

.fi
.SH DESCRIPTION
.I histmatch
is a function to perform string matching on speech file histories as
stored in item headers and processed using the
.I label
routines.
It returns a code expressing a match between a match expression
.I exp,
and a target string (item history)
.I targ.
The code is "1" for success, "0" for failure, and "-1" for an
error on the string expression.
.PP
The following string match characters are currently supported in
.I exp:
.TP 11
.B "?"
Match
.I any
single character (except line start and line end).
.TP 11
.B "*"
Match zero or more characters (including line start and line end).
.TP 11
.BI "%" digit, "%%"
Match a string of characters up to ";", ",", ")" and line end, and
return the string in a function argument given after the target string.
Thus "%0" will return a substring into variable
.I ret0,
etc, "%%" performs matching without returning a string.
Matched substrings are further constrained to include equal numbers
of open and close parentheses.
.TP 11
.B any
Any other characters must match exactly.  Escape special characters
with "\\".
.PP
The parameter "exp" may contain a single match expression or a sequence of
expressions separated by "\\033" (ESCAPE) characters.
.I histmatch
returns at the first matching expression.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
.SH SEE ALSO
regex(3)
.SH BUGS
"exp" must match
.I whole
string.
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <malloc.h>
#include "sfs.h"

/* special characters */
#define VALUE 	'%'
#define MANY	'*'
#define SINGLE	'?'
#define BSLASH	'\\'
#define ENDSTR	'\0'

#define ISTERM(c)	((c=='\0') || (c==';') || (c==',') || (c==')'))

#define MATCH 	(1)
#define FAIL 	(0)
#define ERROR 	(-1)

/* returned strings */
static char	*ret[11];

static int hmatch();

int histmatch(exp,targ,ret0,ret1,ret2,ret3,ret4,ret5,ret6,ret7,ret8,ret9)
char	*exp;		/* match expression */
char	*targ;		/* target string */
char	*ret0,*ret1,*ret2,*ret3,*ret4,*ret5,*ret6,*ret7,*ret8,*ret9;
			/* returned matches */
{
	char	*e,*f;
	int	code;

	/* set up return addresses in array */
	ret[0] = ret0;
	ret[1] = ret1;
	ret[2] = ret2;
	ret[3] = ret3;
	ret[4] = ret4;
	ret[5] = ret5;
	ret[6] = ret6;
	ret[7] = ret7;
	ret[8] = ret8;
	ret[9] = ret9;
	ret[10] = NULL;

	e = malloc(strlen(exp)+1);
	strcpy(e,exp);
	f = strtok(e,"\033");
	while (f) {
		if ((code=hmatch(f,targ))) {
			free(e);
			return(code);
		}
		f = strtok(0,"\033");
	}
	free(e);
	return(FAIL);
}

static int hmatch(exp,targ)
char	*exp;		/* match expression */
char	*targ;		/* target string */
{
	register char	e,f,t;
	int		code,i,parcnt;

top:	e = *exp;
	if (e) f = *(exp+1); else f='\0';
	t = *targ;
/*
	printf("e='%c' t='%c'\n",e,t);
*/
	/* check for end of target */
	if (t==ENDSTR) {
		if (e==ENDSTR)
			return(MATCH);
		else if (e==MANY)
			return(hmatch(exp+1,targ));
		else
			return(FAIL);
	}
	/* check for end of expression */
	if (e==ENDSTR)
		return(FAIL);
	/* do single character match */
	if (e==SINGLE)
		return(hmatch(exp+1,targ+1));
	/* do many character matching (with backtracking) */
	if (e==MANY) {
		if ((code = hmatch(exp+1,targ))) return(code);
		return(hmatch(exp,targ+1));
	}
	/* return values from history string */
	if (e==VALUE) {
		if (f=='%')
			f=10;
		else if (isdigit(f))
			f-='0';
		else
			return(ERROR);
		i=0;
		parcnt=0;
		do {
			while (!ISTERM(*targ)) {
				if (*targ=='(') parcnt++;
				if (ret[(int)f]) ret[(int)f][i++] = *targ;
				targ++;
			}
			if (ret[(int)f]) ret[(int)f][i]=ENDSTR;
			if (parcnt==0) {
				if ((code=hmatch(exp+2,targ))) return(code);
			}
			if (*targ) {
				if (*targ==')') parcnt--;
				if (ret[(int)f]) ret[(int)f][i++] = *targ;
				targ++;
			}
		} while (*targ);
		return(FAIL);
	}
	/* do "protected" characters */
	if (e==BSLASH) {
		e=f;
		exp++;
	}
	/* fail if not good character match */
	if (e!=t)
		return(FAIL);
	/* current match ok, do some more */
	exp++;
	targ++;
	goto top;	/* tail recursion */
}

#ifdef IAG
main()
{
	char	targ[80];
	char	exp[80];
	char	ret[10][80];
	int	i;

	printf("Target : ");
	fflush(stdout);
	gets(targ);

	printf("Expr   : ");
	fflush(stdout);
	while (strcmp("",gets(exp)) != 0) {
		for (i=0;i<10;i++) strcpy(ret[i],"");
		i=histmatch(exp,targ,ret[0],ret[1],ret[2],ret[3],ret[4],ret[5],ret[6],ret[7],ret[8],ret[9]);
		if (i == -1)
			printf("Error\n");
		else if (i == 0)
			printf("No match\n");
		else {
			printf("Match: ");
			for (i=0;i<10;i++)
				if (ret[i][0]!='\0')
					printf("%d '%s',",i,ret[i]);
			printf("\n");
		}
		printf("Expr   : ");
		fflush(stdout);
	}
}
#endif
