/* sfsadc -- SFS routines for support of Analogue-to-Digital conversion */

/* M.A.Huckvale - University College London */

/* version 1.0 - October 1995
	- DOS/SoundBlaster16 only
*/
/* version 1.1 - July 1996
	- add Sun-16 code from sunin.c
*/
/* version 2.0 - September 1996
	- add automatic endpointing
*/
/* version 2.1 - June 1998
	- WIN32 record support
*/

/* #define VERBOSE2 */

/*-------------------------------------------------------------------------*/
/**MAN
.TH SFSADC 3 SFS UCL
.SH NAME
sfsadc -- SFS support for analogue to digital converters
.SH SYNOPSIS
.nf
int adc_open(name)
char *name;	/\* device name - supply as NULL for auto-detection *\/

int adc_record(buf,numf,srate,nchan,flags)
short *buf;	/\* sample buffer *\/
int numf;	/\* number of samples to record *\/
double srate;	/\* sampling rate *\/
int nchan;	/\* number of channels of data to record *\/
int flags;	/\* 0=fixed time, 1=automatic endpointing *\/

void adc_close(rapid)
int rapid;	/\* flag to shut down device immediately *\/

extern int adc_available_channels;	/\* # available channels *\/
extern double adc_selected_rate;	/\* selected sampling rate *\/
.fi
.SH DESCRIPTION
Primitive support for Analogue-to-Digital conversion for a number
of devices is supported through these routines.  Many machines
have their own specific acquisition systems to supplement these
routines.  The routines only support acquisition of a maximum sized
block into memory.
.PP
ADC devices are given types which are selected by name.  These
names are coded into sfsadc.c and which types are available
depend on the machine configuration specified in SFSCONFG.h.
The name of the ADC is discovered at open time from a supplied
name, or from the ADC environment variable, or using the terminal name from the file
$(SFSBASE)/data/adctable.
.SS ADC_OPEN
The open routine attempts to find a suitable ADC type and device
and opens it to give the user unique access.  The mnemonic name
for the device may be given in the routine call, or if NULL is
found from the ADC environment variable, or if ADC is NULL from
the file $(SFSBASE)/data/adctable - which identifies ADC types with
specific terminal names.
adc_open() returns the ADC type if successful, otherwise -1.  The external
variable adc_available_channels is set to indicate the maximum number
of channels of simultaneous acquisition available.
.SS ADC_CLOSE
This routine closes the ADC device and frees any allocated memory.
.SS ADC_RECORD
This routine takes a buffer and a maximum size of waveform, and fills it
from the current input channel according to the given sample rate and number
of channels.  Data is always signed 16-bit samples with linear quantisation.
Note that many ADCs have a limited range of sampling frequencies
and that the required rate may not be available.  In this case, adc_record()
silently selects the nearest frequency.  The selected frequency may be
recovered from the global variable adc_selected_rate.
A buffer count of zero will only set the sampling rate.
.PP
The flags parameter control whether the buffer should be completely
filled or whether automatic end-pointing should take place.
For details of the end-pointing algorithm see endpoint(SFS1).
.PP
For stereo recording, double the number of samples requested.
.SH FILES
.IP SFSBASE/data/adcmap
Table of mappings from ttynames to ADC name.
.SH SUPPORTED TYPES
.IP sun16 11
Sun SPARC DBRI mic input.
.IP sun16-line 11
Sun SPARC DBRI line input.
.IP sb16 11
SoundBlaster-16 on DOS machine.
.IP pclx 11
Laryngograph PCLX card on DOS machine.
.IP win32 11
Windows multi-media support
.SH VERSION/AUTHOR
.IP 2.1
Mark Huckvale
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"		/* contains ADC selection flags */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#ifndef WIN32
#ifdef DOS
#include <pc.h>
#endif
#endif
#include "sfs.h"
#include "sfsadc.h"

#define MIN(x,y)	(((x)<(y))?(x):(y))

/* standard size acquire buffer */
#define REPBUFSIZE	4096		/* bytes */

/* device-specific data */
#ifdef ADC_SUN16
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/signal.h>
#include <stropts.h>
#include <sys/ioctl.h>
#include <multimedia/libaudio.h>
#include <multimedia/audio_device.h>

Audio_hdr	audev;		/* audio device params */
#endif

#ifdef ADC_DOS_PCLX
/* base address of board */
unsigned short int pclxbase=0x2c0;

/* addresses of control registers */
#define PCLX_DATA_LO	(2)
#define PCLX_DATA_HI	(3)
#define PCLX_ADDR_LO	(4)
#define PCLX_ADDR_HI	(5)
#define PCLX_SAMPLE	(6)
#define PCLX_READY	0x10

/* store a value in the board memory */
#define PCLX_POKE(addr,data) { \
	outportb(pclxbase+PCLX_ADDR_LO,(addr & 0xFF)); \
	outportb(pclxbase+PCLX_ADDR_HI,(addr >> 8)); \
	outportb(pclxbase+PCLX_DATA_LO,(data & 0xFF)); \
	outportb(pclxbase+PCLX_DATA_HI,(data >> 8)); \
	}

/* sampling frequency selection table */
static struct pclx_rate_rec {
	int	minf;		/* transition frequency */
	int	code;		/* code to send to baord */
	int	freq;		/* actual frequency of board */
} pclx_rate_tab[]={
	30000,	0,	40000,
	15000,	4,	20000,
	7500,	8,	10000,
	3750,	12,	5000,
	0,	0,	40000	/* guard value */
};

#endif

#ifdef ADC_DOS_BLAST16
/* routines for Protected Mode recording of 16-bit data */
int	SBR16Init(void);
int	SBR16Record(short *buff,int numf,int srate,int nchan,int flags);
void	SBR16Close(void);
#endif

/* WINDOWS API */
#ifdef ADC_WIN32
extern int win32record(unsigned char *buf,int len,int smr,int nbs,int nch,int flag);
#endif

/* supported ADCs - major types */
enum ADCTYPE { SUN8, SUN16, XRECORD, SB16, PCLX, WINDAC };

/* environment variable look-up table */
struct adc_tab_rec {
	char		*name;		/* name supplied in adc_open */
					/*  or found in ADC environment */
					/*  or in $(SFSBASE)/data/adctable */
	enum ADCTYPE	type;		/* major device type */
	int		subtype;	/* minor device type */
};
static struct adc_tab_rec adc_tab[]={
{ "sun",	SUN8,		0 },	/* SPARC-2 with 8-bit ulaw */
{ "sparc2",	SUN8,		0 },	/* SPARC-2 with 8-bit ulaw */
{ "sun8",	SUN8,		0 },	/* SPARC-2 with 8-bit ulaw */
{ "sun8-mic",	SUN8,		0 },	/* SPARC-2 with 8-bit ulaw */
{ "sun8-line",	SUN8,		1 },	/* SPARC-2 with 8-bit ulaw */
{ "sparc10",	SUN16,		0 },	/* SPARC-10 with dbri */
{ "sun16",	SUN16,		0 },	/* SPARC-10 with dbri */
{ "dbri",	SUN16,		0 },	/* SPARC-10 with dbri */
{ "sun16-mic",	SUN16,		0 },	/* SPARC-10 with dbri */
{ "sun16-line",	SUN16,		1 },	/* SPARC-10 with dbri */
{ "pclx",	PCLX,		0 },	/* PC: Laryngograph PC/LX board */
{ "sb16",	SB16,		0 },	/* PC: SoundBlaster 16 bit */
{ "extend",	XRECORD,	0 },	/* Vista Extend, (X-Server/PC) */
{ "windows",	WINDAC,		0 },	/* WIN32 API */
{ "win32",	WINDAC,		0 },	/* WIN32 API */
};
#define NUMADC (sizeof(adc_tab)/sizeof(struct adc_tab_rec))

/* only one ADC selected at any one time */
static int	adctype= -1;	/* current ADC type */
static int	adcsubtype=0;	/* current ADC subtype */
static int	afd;		/* audio handle */
static short	*lastbuff;	/* last known buffer */
static int	lastnumf;	/* last known # samples */
static double	lastsrate;	/* last known sampling rate */
static int	lastnchannels;	/* last known # channels */
#ifdef ADC_SUN16
static int	doabort=0;	/* ctrl/c flag */
#endif
/* external variables for communication with user */
int		adc_available_channels;	/* # available channels */
double		adc_selected_rate;	/* to hold set sample rate */

/*--------------------- endpointing code ------------------------------*/
/* This endpointing code is based on the endpoint C++ class
   developed by Bruce Lowerre. */

/* endpointing definitions */

typedef enum { isFALSE, isTRUE } BOOLEAN;
typedef enum { NOSILENCE, INSILENCE, START, INSIGNAL, END } EPSTATE;

/* control */
#define DEFSTEPTIME	0.050	/* window step time */
#define DEFWINTIME	0.100	/* window size */
#define DEFENDTIME	0.700	/* end-of-utterance time */
#define DEFMINTIME	0.100	/* minimum utterance time */
#define DEFZCTHRESH	600	/* zero-cross threshold (Hz) */
#define DEFBEGFACT	3.0	/* begin factor */
#define DEFENDFACT	3.0	/* end factor */
#define DEFENERGYFACT	20.0	/* energy factor */
#define DEFSTARTSIL	2000.0	/* start silence */
#define DEFTRIGFACT	2.5	/* trigger factor */
#define DEFDPNOISE	6	/* num dp noise */
#define DEFMINFRCLENG	0.050	/* min fric length */
#define DEFMAXPAUSE	0.150	/* max pause length */
#define DEFSTARTBLIP	0.150	/* start blip length */
#define DEFENDBLIP	0.020	/* end blip length */
#define DEFMINVOICE	0.060	/* min voice length */
#define DEFMINRISE	0.050	/* min rise length */
static double	steptime = DEFSTEPTIME;
static double	wintime = DEFWINTIME;
static double	endtime = DEFENDTIME;
static double 	mintime = DEFMINTIME;

/* parameters */
static struct endpoint_params {
	int	samprate;
	int	maxipause;
	int	minuttlng;
	int	zcthresh;
	double	begfact;
	double	endfact;
	double	energyfact;
	double	minstartsilence;
	double	triggerfact;
	int	numdpnoise;
	int	minfriclng;
	int	maxpause;
	int	startblip;
	int	endblip;
	int	minvoicelng;
	int	minrise;
	
	/* endpoint state */
	EPSTATE	epstate;
	double	ave;
	double	noise;
	double	begthresh;
	double	energy;
	double	maxpeak;
	double	endthresh;
	double	mnbe;
	double	peakreturn;
	double	dpnoise;
	int	scnt;
	int	avescnt;
	int	vcnt;
	int	evcnt;
	int	voicecount;
	int	bscnt;
	int	zccnt;
	int	startframe;
	int	endframe;
	int	ncount;
	int	zc;
	BOOLEAN	startsilenceok;
	BOOLEAN	low;
	double	lastdpnoise[DEFDPNOISE];
} ep;

/* initialise endpoint parameters */
void	initparams(double srate)
{
	int	i;
	
	ep.samprate   = (int)srate;
	ep_windowsize = (int)(0.5+wintime*srate);
	ep_stepsize   = (int)(0.5+steptime*srate);
	ep.maxipause  = (int)(0.5+endtime/steptime);
	ep.minuttlng  = (int)(0.5+mintime/steptime);
	ep.zcthresh   = (int)(0.5+DEFZCTHRESH*wintime);
	ep.begfact    = DEFBEGFACT;
	ep.endfact    = DEFENDFACT;
	ep.energyfact = DEFENERGYFACT;
	ep.minstartsilence = DEFSTARTSIL;
	ep.numdpnoise = DEFDPNOISE;
	ep.triggerfact = DEFTRIGFACT;
	ep.minfriclng = (int)(0.5+DEFMINFRCLENG/steptime);
	ep.maxpause   = (int)(0.5+DEFMAXPAUSE/steptime);
	ep.startblip  = (int)(0.5+DEFSTARTBLIP/steptime);
	ep.endblip    = (int)(0.5+DEFENDBLIP/steptime);
	ep.minvoicelng = (int)(0.5+DEFMINVOICE/steptime);
	ep.minrise     = (int)(0.5+DEFMINRISE/steptime);

	/* clear state */
	for (i=0;i<ep.numdpnoise;i++)
		ep.lastdpnoise[i]=0;

	/* initialise state */
	ep.epstate = NOSILENCE;
	ep.noise = 0.0;
	ep.ave = 0.0;
	ep.begthresh = 0.0;
	ep.endthresh = 0.0;
	ep.energy = 0.0;
	ep.maxpeak = 0.0;
	ep.scnt = 0;
	ep.vcnt = 0;
	ep.evcnt = 0;
	ep.voicecount = 0;
	ep.zccnt = 0;
	ep.bscnt = 0;
	ep.startframe = 0;
	ep.endframe = 0;
	ep.avescnt = 0;
	ep.startsilenceok = isFALSE;
	ep.ncount = 0;
	ep.low = isTRUE;
}

/* initialise noise record */
void setnoise ()
{
	ep.dpnoise = ep.lastdpnoise[1] = ep.lastdpnoise[0];
	ep.ncount = 2;
}

/* keep record of background noise */
void averagenoise ()
{
	int	i;

	for (ep.dpnoise = 0.0, i = ep.ncount - 1; i > 0; i--) {
	        ep.dpnoise += ep.lastdpnoise[i];
	        ep.lastdpnoise[i] = ep.lastdpnoise[i - 1];
	}
	ep.dpnoise = (ep.dpnoise + ep.lastdpnoise[0]) / ep.ncount;
	if (ep.ncount < ep.numdpnoise)
	        ep.ncount++;
}


/* get the zero cross count and average energy */
void zcpeakpick(samples)
short	*samples;
{
	register int	i;
	double	sum,trigger;
	register short	*smp;

	for (sum = 0.0, i = 0, smp = samples; i < ep_windowsize; i++, smp++)
        	sum += *smp * *smp;
	ep.peakreturn = (sqrt (sum / ep_windowsize));
	ep.lastdpnoise[0] = ep.peakreturn;

	if (ep.ncount == 0)
        	ep.dpnoise = ep.peakreturn;		/* initial value */
	trigger = ep.dpnoise * ep.triggerfact;	/* schmidt trigger band */

    	for (i = 0, ep.zc = 0, smp = samples; i < ep_windowsize; i++, smp++) {
		if (ep.low) {
			if (*smp > trigger) {
	                	ep.zc++;
				ep.low = isFALSE;
			}
		}
		else {
			if (*smp < -trigger) {
				ep.zc++;
				ep.low = isTRUE;
			}
		}
	}
}

/* endpoint algorithm */
EPTAG getendpoint(samples)
short	*samples;
{
	double	tmp;

	/* get zc count and peak energy */
	zcpeakpick(samples);

	if (ep.peakreturn > ep.maxpeak) {
		ep.maxpeak = ep.peakreturn;
		if ((tmp = ep.maxpeak / ep.endfact) > ep.endthresh)
			ep.endthresh = tmp;
	}

	switch (ep.epstate) {
	case NOSILENCE:		/* start, get background silence */
		ep.ave += ep.peakreturn;
		if (++ep.scnt <= 3) {	/* average 3 frame's worth */
			if (ep.scnt == 1)
				setnoise();
			else
				averagenoise();
			
			if (ep.dpnoise < ep.minstartsilence) {
				ep.startsilenceok = isTRUE;
				ep.ave += ep.peakreturn;
				ep.avescnt++;
			}
#ifdef VERBOSE
fprintf(stderr,"SILENCE\n");
#endif
			return (EP_SILENCE);
		}
		if (!ep.startsilenceok) {
#ifdef VERBOSE
fprintf(stderr,"NOSTARTSILENCE\n");
#endif
			return (EP_NOSTARTSILENCE);
		}
		ep.ave /= ep.avescnt;
		ep.noise = ep.ave;
		ep.begthresh = ep.noise * ep.begfact;
		ep.endthresh = ep.begthresh;
		ep.mnbe = ep.noise * ep.energyfact;
		ep.epstate = INSILENCE;
#ifdef VERBOSE
fprintf(stderr,"SILENCE\n");
#endif
		return (EP_SILENCE);

	case INSILENCE:
		ep.ave = ((3.0 * ep.ave) + ep.peakreturn) / 4.0;
		if ((ep.peakreturn > ep.begthresh) || (ep.zc > ep.zcthresh)) {
			/* looks like start of signal */
			ep.energy += ep.peakreturn - ep.noise;
			if (ep.zc > ep.zcthresh)
				ep.zccnt++;
			if (ep.peakreturn > ep.begthresh)
				ep.voicecount++;
			if (++ep.vcnt > ep.minrise) {
				ep.scnt = 0;
				ep.epstate = START;	/* definitely start of signal */
			}
#ifdef VERBOSE2
fprintf(stderr,"SIGNAL %g>%g %d>%d\n",ep.peakreturn,ep.begthresh,ep.zc,ep.zcthresh);
#endif
			return (EP_SIGNAL);
		}
		else {
			/* still in silence */
			ep.energy = 0.0;
			if (ep.ave < ep.noise) {
				ep.noise = ep.ave;
				ep.begthresh = ep.noise * ep.begfact;
				ep.endthresh = ep.begthresh;
				ep.mnbe = ep.noise * ep.energyfact;
			}
			if (ep.vcnt > 0) {
				/* previous frame was signal */
				if ((++ep.bscnt > ep.startblip) || (ep.zccnt == ep.vcnt)) {
					/* Oops, no longer in the signal */
					ep.noise = ep.ave;
					ep.begthresh = ep.noise * ep.begfact;
					ep.endthresh = ep.begthresh;
					ep.mnbe = ep.noise * ep.energyfact;
					ep.vcnt = 0;
					ep.zccnt = 0;
					ep.bscnt = 0;
					ep.voicecount = 0;
					ep.startframe = 0;
#ifdef VERBOSE
fprintf(stderr,"RESET\n");
#endif
					return (EP_RESET); /* not in the signal, ignore previous */
				}
#ifdef VERBOSE
fprintf(stderr,"SIGNAL\n");
#endif
				return (EP_SIGNAL);
			}
			ep.zccnt = 0;
#ifdef VERBOSE
fprintf(stderr,"SILENCE\n");
#endif
			return (EP_SILENCE);
		}

	 case START:
		 if ((ep.peakreturn > ep.begthresh) || (ep.zc > ep.zcthresh)) {
			/* possible start of signal */
			 ep.energy += ep.peakreturn - ep.noise;
			 if (ep.zc > ep.zcthresh)
				 ep.zccnt++;
			 if (ep.peakreturn > ep.begthresh)
				 ep.voicecount++;
			 ep.vcnt += ep.scnt + 1;
			 ep.scnt = 0;
			 if ((ep.energy > ep.mnbe) || (ep.zccnt > ep.minfriclng)) {
				 ep.epstate = INSIGNAL;
#ifdef VERBOSE2
fprintf(stderr,"INSIGNAL %g>%g %d>%d\n",ep.energy,ep.mnbe,ep.zccnt,ep.minfriclng);
#endif
			 }
			 return (EP_SIGNAL);
		 }
		 else if (++ep.scnt > ep.maxpause) {
			/* signal went low again, false start */
			 ep.vcnt = ep.zccnt = ep.voicecount = 0;
			 ep.energy = 0.0;
			 ep.epstate = INSILENCE;
			 ep.ave = ((3.0 * ep.ave) + ep.peakreturn) / 4.0;
			 if (ep.ave < ep.noise * ep.begfact) {
				/* lower noise level */
				 ep.noise = ep.ave;
				 ep.begthresh = ep.noise * ep.begfact;
				 ep.endthresh = ep.begthresh;
				 ep.mnbe = ep.noise * ep.energyfact;
			 }
#ifdef VERBOSE2
fprintf(stderr,"RESET\n");
#endif
			 return (EP_RESET);
		 }
		 else {
#ifdef VERBOSE
fprintf(stderr,"SIGNAL\n");
#endif	
			 return (EP_SIGNAL);
		 }

	case INSIGNAL:
		if ((ep.peakreturn > ep.endthresh) || (ep.zc > ep.zcthresh)) {
			/* still in signal */
			if (ep.peakreturn > ep.endthresh)
				ep.voicecount++;
			ep.vcnt++;
			ep.scnt = 0;
#ifdef VERBOSE
fprintf(stderr,"SIGNAL\n");
#endif
			return (EP_SIGNAL);
		}
		else {
			/* below end threshold, may be end */
			ep.scnt++;
			ep.epstate = END;
#ifdef VERBOSE2
fprintf(stderr,"MAYBEEND %g>%g %d>%d\n",ep.peakreturn,ep.begthresh,ep.zc,ep.zcthresh);
#endif
			return (EP_MAYBEEND);
		}

	case END:
		if ((ep.peakreturn > ep.endthresh) || (ep.zc > ep.zcthresh)) {
			/* signal went up again, may not be end */
			if (ep.peakreturn > ep.endthresh)
				ep.voicecount++;
			if (++ep.evcnt > ep.endblip) {
				/* back in signal again */
				ep.vcnt += ep.scnt + 1;
				ep.evcnt = 0;
				ep.scnt = 0;
				ep.epstate = INSIGNAL;
#ifdef VERBOSE2
fprintf(stderr,"NOTEND\n");
#endif
				return (EP_NOTEND);
			}
			else {
#ifdef VERBOSE
fprintf(stderr,"SIGNAL\n");
#endif
				return (EP_SIGNAL);
			}
		}
		else if (++ep.scnt > ep.maxipause) {
			/* silence exceeds inter-word pause */
			if ((ep.vcnt > ep.minuttlng) && (ep.voicecount > ep.minvoicelng)) {
				/* end of utterance */
#ifdef VERBOSE2
fprintf(stderr,"ENDOFUTT\n");
#endif
				return (EP_ENDOFUTT);
			}
			else {
#ifdef VERBOSE2
fprintf(stderr,"DORESET %d>%d %d>%d\n",ep.vcnt,ep.minuttlng,ep.voicecount,ep.minvoicelng);
#endif
				/* signal is too short */
				ep.scnt = ep.vcnt = ep.voicecount = 0;
				ep.epstate = INSILENCE;
				/* false utterance, keep looking */
#ifdef VERBOSE2
fprintf(stderr,"RESET\n");
#endif
				return (EP_RESET);
			}
		}
		else {
			/* may be an inter-word pause */
			if (ep.peakreturn == 0) {
				/* zero filler frame */
#ifdef VERBOSE2
fprintf(stderr,"ENDOFUTT\n");
#endif
				return (EP_ENDOFUTT);
			}
			ep.evcnt = 0;
			/* assume still in signal */
#ifdef VERBOSE
fprintf(stderr,"SIGNAL\n");
#endif
			return (EP_SIGNAL);
		}
	}
#ifdef VERBOSE
fprintf(stderr,"NONE\n");
#endif
	return(EP_NONE);
}

/*---------------------- recording code -------------------------------*/
void adc_close();

/* open ADC */
int	adc_open(device)
char	*device;
{
	int	i;
#ifdef ADC_SUN16
	struct stat st;
#endif	
	/* if ADC already open, close it */
	if (adctype >= 0) adc_close(0);

	/* if device is NULL, look for ADC environment variable */
	if (device==NULL) device = getenv("ADC");

	/* if device is NULL, look for terminal in adctable */
#ifndef DOS
	if (device==NULL) {
		char	adcfilename[SFSMAXFILENAME];
		char	*ttynm,*ttyname(),tname[80];
		char	dname[80];
		FILE	*ip;
		
		strcpy(adcfilename,sfsbase());
		strcat(adcfilename,"/data/adcmap");
		if ((ip=fopen(adcfilename,"r"))==NULL) {
			/* no file - abort */
			return(-1);
		}
		/* scan for tty name */
		ttynm=ttyname(0);
		while (fscanf(ip,"%s %s\n",tname,dname)==2) {
			if (strcmp(tname,ttynm)==0) {
				device = dname;
				break;
			}
		}
		fclose(ip);
	}
#endif

#ifdef ADC_DEFAULT_DEVICE
	/* default may be defined in SFSCONFG.h */
	if (device==NULL) device=ADC_DEFAULT_DEVICE;
#endif
	if (!device)
		/* not found - abort */
		return(-1);

	/* by now should have ADC name */
	adctype = -1;
	adcsubtype = 0;
	for (i=0;i<NUMADC;i++) {
		if (strcmp(device,adc_tab[i].name)==0) {
			adctype = adc_tab[i].type;
			adcsubtype = adc_tab[i].subtype;
			break;
		}
	}
#ifdef IAG
printf("device='%s' adctype=%d adcsubtype=%d\n",device,adctype,adcsubtype);
#endif
	if (adctype < 0)
		/* not found - abort */
		return(-1);

	/* set up default number of available channels */
	adc_available_channels = 1;

	/* OK now open device */
	switch (adctype) {
#ifdef ADC_SUN16
	case SUN16:
		/* Validate and open the audio device */
		if (stat(ADC_SUN16_PATH, &st) < 0)
			error("could not open '%s'",ADC_SUN16_PATH);
		if (!S_ISCHR(st.st_mode))
			error("'%s' is not an audio device",ADC_SUN16_PATH);

		if ((afd=open(ADC_SUN16_PATH, O_RDONLY)) < 0) {
			if (errno == EBUSY)
				error("'%s' is busy", ADC_SUN16_PATH);
			else
				error("could not open '%s'",ADC_SUN16_PATH);
		}
		if (audio_get_record_config(afd, &audev) != AUDIO_SUCCESS)
			error("could not get audio configuration");
		break;
#endif

#ifdef ADC_DOS_BLAST16
	case SB16:
		/* ADC attached directly to PC so no need to open */
		adc_available_channels=2;
		break;
#endif
#ifdef ADC_DOS_PCLX
	case PCLX:
		/* Laryngograph PC/LX board */
		if (p=getenv("PCLXBASE")) pclxbase=atoi(p);
		adc_available_channels=2;
		break;
#endif

#ifdef ADC_WIN32
	case WINDAC:
		break;
#endif

	default:
		/* unknown, or unsupported */
		adctype = -1;
		return(-1);
	}
	return(adctype);
}

/* close ADC */
void adc_close(rapid)
int	rapid;		/* set to 1 for rapid close */
{
#ifdef ADC_SUN16
	int	err;
#endif	
	/* device specific rapid close */
	if (rapid) switch (adctype) {

	}

	/* do standard close */
	switch (adctype) {
#ifdef ADC_SUN16
	case SUN16:
		audio_pause_record(afd);
		/* Check for error during record */
		if (audio_get_record_error(afd, (unsigned *)&err) != AUDIO_SUCCESS)
			error("error reading ADC device status");
		else if (err)
			fprintf(stderr,"WARNING: SFSADC data buffer overflow occurred\n");
		close(afd);
		afd = -1;
		break;		
#endif
#ifdef ADC_DOS_BLAST16
	case SB16:
		/* ADC connected directly to PC, so no need to close */
		break;
#endif
#ifdef ADC_DOS_PCLX
	case PCLX:
		/* ADC connected directly to PC, so no need to close */
		break;
#endif
#ifdef ADC_WIN32
	case WINDAC:
		break;
#endif
	}

	/* clear out static area */
	adctype = -1;
	adcsubtype = 0;
	lastbuff = NULL;
	lastnumf = 0;
	adc_selected_rate = 0;
	adc_available_channels = 0;
	afd = -1;
}

#ifdef ADC_SUN16
/* Sun SPARC-10 dbri playback frequency table */
static int	sunfrq[]={8000,9600,11025,16000,18900,
			22050,32000,37800,44100,48000,0};

/* get nearest sample rate */
static int	sunnearfreq(srate)
int	srate;
{
	int	i;
	int	min,idx;

	idx=0;
	min=abs(sunfrq[0]-srate);
	for (i=0;sunfrq[i];i++) {
		if (abs(sunfrq[i]-srate)<min) {
			min=abs(sunfrq[i]-srate);
			idx=i;
		}
	}
#ifdef IAG
printf("Selected %d Hz\n",sunfrq[idx]);
#endif
	return(sunfrq[idx]);
}

void intrupt()
{
	fprintf(stderr,"Interrupted\n");
	doabort=1;
}

/* kbhit -- test if RETURN has been pressed on keyboard input */
int	kbhit()
{
	char	c;
	
	fcntl(0,F_SETFL,O_NDELAY);

	if (read(0,&c,1)!=1) c = 0;
	
	fcntl(0,F_SETFL,0);
#ifdef IAG
if (c) printf("c=%d\n",c);
#endif
	return(c=='\n');
}

#endif

/* record sample buffer */
int32	adc_record(buff,numf,srate,nchannels,flags)
short	*buff;		/* sample buffer */
int32	numf;		/* number of samples */
double	srate;		/* sample rate */
int	nchannels;	/* # channels to record, NOTE: if 2 channels required,
			   - should 1 channel devices record mono ?
			 */
int	flags;		/* 0=fill buffer, 1=endpoint */
{
	int32	count=0;
#ifdef ADC_SUN16
	int32	len;
	int	err,port;
#endif
#ifdef ADC_DOS_PCLX
	int32	i;
	short	sample,sample2;
#endif

	/* keep record of this */
	lastbuff = buff;
	lastnumf = numf;
	lastsrate = srate;
	lastnchannels = nchannels;

	/* initialise endpointing */
	if (flags)
		initparams(srate);

	/* device specific processing */
	switch (adctype) {
#ifdef ADC_SUN16
	case SUN16:
		/* set up audio characteristics */
		audev.sample_rate = sunnearfreq((int)srate);
		audev.samples_per_unit = 1;
		audev.bytes_per_unit = 2;
		audev.channels = nchannels;
		audev.encoding = AUDIO_ENCODING_LINEAR;
		audev.data_size = numf*nchannels;

		/* set record characteristics */
		if (audio_set_record_config(afd,&audev)!=AUDIO_SUCCESS) {
			fprintf(stderr,"SFSADC couldn't set audio param (%d,%d)\n",
					audev.sample_rate,audev.channels);
			error("abandoned");
		}
		adc_selected_rate = audev.sample_rate;
		port = (adcsubtype==0)?AUDIO_MICROPHONE:AUDIO_LINE_IN;
		audio_set_record_port(afd,&port);

		/* get ready to go */
		audio_flush_record(afd);
		err=0;
		audio_set_record_error(afd,&err);
		
		/* set up exit */
		signal(SIGINT,intrupt);
		doabort = 0;
		count = 0;
		while (!doabort && !kbhit() && (numf > 0)) {
			len = (numf < REPBUFSIZE) ? numf : REPBUFSIZE;
			len = read(afd,buff,2*len);
			numf -= len/2;
			buff += len/2;
			count += len/2;
		}
		audio_pause_record(afd);
		
		break;
#endif
#ifdef ADC_DOS_BLAST16
	case SB16:
		if (SBR16Init()) {
			count = SBR16Record(buff,numf,(int)srate,nchannels,flags);
			adc_selected_rate = srate;
			SBR16Close();
		}
		break;
#endif
#ifdef ADC_DOS_PCLX
	case PCLX:
		/* reset card */
		outportb(pclxbase,0);
		PCLX_POKE(0x400,33);	/* 2 chan record mode */

		/* select sampling rate */
		for (i=0;(int)srate < pclx_rate_tab[i].minf;i++) /* loop */;
		PCLX_POKE(0x401,pclx_rate_tab[i].code);
		adc_selected_rate = pclx_rate_tab[i].freq;

		/* set gain to max */
		PCLX_POKE(0x410,0x7FFF);
		PCLX_POKE(0x411,0x7FFF);

		/* set DSP running */
		outportb(pclxbase,0x11);

		for (i=0;i<numf;i++) {

			/* wait for timer */
			while ((inportb(pclxbase) & PCLX_READY)==0) {
				if (kbhit() && (getkey()==0x1B)) goto pclxdone;
			}

			/* get lo byte channel 0 */
			sample = inportb(pclxbase+PCLX_SAMPLE);

			/* wait for timer */
			while ((inportb(pclxbase) & PCLX_READY)==0) /* wait */;

			/* get hi byte channel 0 */
			sample = sample | ((short)inportb(pclxbase+PCLX_SAMPLE)<<8);

			/* wait for timer */
			while ((inportb(pclxbase) & PCLX_READY)==0) /* wait */;

			/* get lo byte channel 1 */
			sample2 = inportb(pclxbase+PCLX_SAMPLE);

			/* wait for timer */
			while ((inportb(pclxbase) & PCLX_READY)==0) /* wait */;

			/* get hi byte channel 1 */
			sample2 = sample2 | ((short)inportb(pclxbase+PCLX_SAMPLE)<<8);

			buff[i]=sample;
			count++;
			if (nchannels==2) {
				buff[++i] = sample2;
				count++;
			}
			if (kbhit() && (getkey()==0x1B)) break;
		}
pclxdone:
		/* turn off DSP */
		outportb(pclxbase,0);

		break;
#endif
#ifdef ADC_WIN32
	case WINDAC:
		count = win32record((unsigned char *)buff,2*numf,(int)srate,2,nchannels,flags)/2;
		adc_selected_rate = srate;	/* not actually true .. */
		break;
#endif
	default:
		/* unknown, or unsupported */
		return(-1);
	}
	/* 0 is OK */
	return(count);
}	

#ifdef IAG
char *progname="sfsadc";
main(argc,argv)
int	argc;
char	*argv[];
{
	short	*sp;
	struct item_header item;
	int	code;
	int	count;

	if (argc!=2)
		error("usage: sfsadc sfsfile");

	if ((sp=(short *)calloc(100000,sizeof(short)))==NULL)
		error("could not get memory buffer");

	code = adc_open(NULL);
	printf("adc_open returns %d\n",code);

	printf("recording up to 5 seconds at 20,000Hz ...\n");
	count = adc_record(sp,100000,20000.0,1,1);
	printf("adc_record delivered %d samples\n",count);
	printf("sampling rate was %gHz\n",adc_selected_rate);
	
	sfsheader(&item,SP_TYPE,0,2,1,1.0/adc_selected_rate,0.0,1,0,0);
	sprintf(item.history,"sfsadc(freq=20000)");
	putitem(argv[1],&item,count,sp);

	adc_close(0);
	free(sp);
	exit(0);
}	
#endif
