/* windac -- Digital <-> Analogue conversion routines for WIN32 API */

#include "SFSCONFG.h"
#ifdef DAC_WIN32

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include <windows.h>
#include <mmsystem.h>
#include <sys/types.h>
#include <sys/stat.h>

/* replay done flag */
volatile int replaydone;

/* replay callback function */
static void CALLBACK waveOutProc(HWAVEOUT hwo,UINT uMsg,DWORD dwInstance,DWORD dwParam1,DWORD dwParam2)
{
	if (uMsg==WOM_DONE) replaydone=1;
}

/* replay function */
int win32replay(unsigned char *buf,int len,int smr,int nbs,int nch)
{
	HGLOBAL  	hWaveOut;	/* waveform device handle */
	PCMWAVEFORMAT 	pcm;	/* waveform device configuration */
	HGLOBAL		whdr;
	LPWAVEHDR	lpwhdr;

	/* set up device */
	pcm.wf.wFormatTag	= WAVE_FORMAT_PCM;
	pcm.wf.nChannels	= nch;
	pcm.wf.nSamplesPerSec	= smr;
	pcm.wf.nAvgBytesPerSec	= smr * nbs * nch;
	pcm.wf.nBlockAlign	= nbs * nch;
	pcm.wBitsPerSample	= 8 * nbs;

	/* query the replay devices */
	if (waveOutOpen(NULL,(UINT)WAVE_MAPPER,(LPCWAVEFORMATEX)&pcm,0,0,WAVE_FORMAT_QUERY)) {
		/* not supported */
		fprintf(stderr,"Unsupported waveform format/rate");
		return(-1);
	}
		
	/* open device for real */
	if (waveOutOpen(&hWaveOut,(UINT)WAVE_MAPPER,(LPCWAVEFORMATEX)&pcm,(DWORD)waveOutProc,0,CALLBACK_FUNCTION)) {
		/* some kind of error on open - in use? */
		fprintf(stderr,"Unable to open waveform output device\n");
		return(-1);
	}

	/* reset everything */
	waveOutReset(hWaveOut);

	/* malloc up a wave header */
	whdr = GlobalAlloc(GHND|GMEM_SHARE,sizeof(WAVEHDR));
	lpwhdr = (LPWAVEHDR)GlobalLock(whdr);

	/* point it to the data to replay */
	lpwhdr->lpData			= buf;
	lpwhdr->dwBufferLength  = len;
	lpwhdr->dwBytesRecorded = 0L;
	lpwhdr->dwUser			= 0L;
	lpwhdr->dwFlags			= 0L;
	lpwhdr->dwLoops			= 1L;
	lpwhdr->lpNext			= NULL;
	lpwhdr->reserved		= 0L;

	/* prepare it */
	waveOutPrepareHeader(hWaveOut,lpwhdr,sizeof(WAVEHDR));
	replaydone=0;

	/* write it */
	if (waveOutWrite(hWaveOut,lpwhdr,sizeof(WAVEHDR))!=0) {
		/* argh! */
		waveOutUnprepareHeader(hWaveOut,lpwhdr,sizeof(WAVEHDR));
		fprintf(stderr,"Error on write to wave out device\n");
		waveOutClose(hWaveOut);
		GlobalUnlock(whdr);
		GlobalFree(whdr);
		return(-1);
	}

	/* wait for done */
	while (!replaydone) /* loop */;

	/* close down */
	waveOutClose(hWaveOut);
	GlobalUnlock(whdr);
	GlobalFree(whdr);
	return(0);

}

#endif

#ifdef ADC_WIN32

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include <windows.h>
#include <mmsystem.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "sfsadc.h"

#define NUMBUF	4

/* replay done flag */
volatile int recorddone;
static HGLOBAL		whdr[NUMBUF];
static LPWAVEHDR	lpwhdr[NUMBUF];
static HGLOBAL		wmem[NUMBUF];
static unsigned char 	*lpwmem[NUMBUF];
static int		queued[NUMBUF];
static int		bufsiz;
static int		curbuf;
static unsigned char	*obuf;
static int		count;		/* length of buffer */
static int		opos;		/* end of recording */
static int		owin;		/* last part checked */
static int		active;		/* voice activated */
static int		startpos;
static int		endpos;

/* replay callback function */
static void CALLBACK waveInProc(HWAVEIN hwi,UINT uMsg,DWORD dwInstance,DWORD dwParam1,DWORD dwParam2)
{
	int	len,i;
	unsigned char *buf;

	if (recorddone!=0) return;

	if (uMsg==WIM_DATA) {
		/* copy the data on to the end of the buffer */
		len=bufsiz;
		buf=lpwmem[curbuf];
		while ((len > 0)&&(opos < count)) {
			obuf[opos++] = *buf++;
			len--;
		}

		if (active) {
			while ((owin+2*ep_windowsize) < opos) {
				switch (i=getendpoint((short *)&obuf[owin])) {
				case EP_RESET:
				case EP_SILENCE:
					/* HACK by MAH 961028 */
					startpos = owin /* +ep_stepsize */;
					break;
				case EP_MAYBEEND:
					/* HACK by MAH 961028 */
					endpos = owin + 4*ep_windowsize;
				case EP_SIGNAL:
					break;
				case EP_NOTEND:
					endpos = -1;
					break;
				case EP_ENDOFUTT:
					recorddone=1;
					return;
				case EP_NOSTARTSILENCE:
				case EP_NONE:
					break;
				}
				owin += 2*ep_stepsize;
			}
			if (startpos!=0) {
				/* reset buffer to start */
				memcpy(&obuf[0],&obuf[startpos],(opos-startpos));
				opos -= startpos;
				owin -= startpos;
				endpos -= startpos;
				startpos = 0;
			}
		}

		if (opos < count) {
			queued[curbuf] = 0;
			curbuf = (curbuf+1)%NUMBUF;
		}
		else
			recorddone=1;
	}
}

/* kbhit for record */
static int kbhit()
{
	INPUT_RECORD	ir;
	DWORD		nr=0;

	PeekConsoleInput(GetStdHandle(STD_INPUT_HANDLE),&ir,1,&nr);
	if (nr > 0) {
		ReadConsoleInput(GetStdHandle(STD_INPUT_HANDLE),&ir,1,&nr);
		if ((ir.EventType==KEY_EVENT) && ir.Event.KeyEvent.bKeyDown) {
			return(ir.Event.KeyEvent.uChar.AsciiChar);
		}
	}
	return(0);
}

/* record function */
int win32record(unsigned char *buf,int len,int smr,int nbs,int nch,int flag)
{
	HGLOBAL  	hWaveIn;	/* waveform device handle */
	PCMWAVEFORMAT 	pcm;	/* waveform device configuration */
	MMTIME		mr;
	int		i;

	/* set up device */
	pcm.wf.wFormatTag	= WAVE_FORMAT_PCM;
	pcm.wf.nChannels	= nch;
	pcm.wf.nSamplesPerSec	= smr;
	pcm.wf.nAvgBytesPerSec	= smr * nbs * nch;
	pcm.wf.nBlockAlign	= nbs * nch;
	pcm.wBitsPerSample	= 8 * nbs;

	/* query the replay devices */
	if (waveInOpen(NULL,(UINT)WAVE_MAPPER,(LPCWAVEFORMATEX)&pcm,0,0,WAVE_FORMAT_QUERY)) {
		/* not supported */
		fprintf(stderr,"Unsupported waveform format/rate");
		return(-1);
	}
		
	/* open device for real */
	if (waveInOpen(&hWaveIn,(UINT)WAVE_MAPPER,(LPCWAVEFORMATEX)&pcm,(DWORD)waveInProc,0,CALLBACK_FUNCTION /* |WAVE_ALLOWSYNC */)) {
		/* some kind of error on open - in use? */
		fprintf(stderr,"Unable to open waveform input device\n");
		return(-1);
	}

	/* reset everything */
	waveInReset(hWaveIn);

	/* get suitable buffer size */
	bufsiz = 4*(pcm.wf.nAvgBytesPerSec/16);

	/* set up global parameters for record */
	obuf = buf;
	count = len;
	opos = owin = 0;
	startpos = 0;
	endpos = -1;
	recorddone = 0;
	curbuf = 0;
	active = flag;
	
	/* malloc up a wave headers */
	for (i=0;i<NUMBUF;i++) {
		/* get header */
		whdr[i] = GlobalAlloc(GHND|GMEM_SHARE,sizeof(WAVEHDR));
		lpwhdr[i] = (LPWAVEHDR)GlobalLock(whdr[i]);

		/* get buffer */
		wmem[i] = GlobalAlloc(GHND|GMEM_SHARE,bufsiz);
		lpwmem[i] = (unsigned char *)GlobalLock(wmem[i]);

		/* point header at buffer */
		lpwhdr[i]->lpData		= lpwmem[i];
		lpwhdr[i]->dwBufferLength  	= bufsiz;
		lpwhdr[i]->dwBytesRecorded 	= 0L;
		lpwhdr[i]->dwUser		= 0L;
		lpwhdr[i]->dwFlags		= 0L;
		lpwhdr[i]->dwLoops		= 1L;
		lpwhdr[i]->lpNext		= NULL;
		lpwhdr[i]->reserved		= 0L;

		/* prepare it */
		waveInPrepareHeader(hWaveIn,lpwhdr[i],sizeof(WAVEHDR));
		waveInAddBuffer(hWaveIn,lpwhdr[i],sizeof(WAVEHDR));
		queued[i] = 1;
	}

	/* start input */
	waveInStart(hWaveIn);

	/* wait for done */
	while (!recorddone) {
		if (kbhit()==0x1) {
			/* user hit ESCAPE - get current position */
			mr.wType = TIME_BYTES;
			waveInGetPosition(hWaveIn,&mr,sizeof(MMTIME));
			len = mr.u.cb;
			/* then stop */
			recorddone = 1;
			break;
		}
		for (i=0;i<NUMBUF;i++) if (queued[i]==0) {
			waveInPrepareHeader(hWaveIn,lpwhdr[i],sizeof(WAVEHDR));
			waveInAddBuffer(hWaveIn,lpwhdr[i],sizeof(WAVEHDR));
			queued[i]=1;
		}
		
	}

	/* stop recording */
	waveInReset(hWaveIn);

	/* close down */
	for (i=0;i<NUMBUF;i++) {
		waveInUnprepareHeader(hWaveIn,lpwhdr[i],sizeof(WAVEHDR));
		GlobalUnlock(whdr[i]);
		GlobalFree(whdr[i]);
		GlobalUnlock(wmem[i]);
		GlobalFree(wmem[i]);
	}
	waveInClose(hWaveIn);

	if (flag) {
		if (endpos <= 0)
			return(opos);
		else {
			if ((endpos + 2*ep_windowsize) < opos)
				return(endpos+2*ep_windowsize);
			else
				return(opos);
		}
	}	
	else
		return(opos);
}
#endif

#ifdef EMO
/* global parameters */
int	srate;			/* sampling rate */
int	nbytesamp;		/* num bytes per samp */
int	nchan;			/* number of channels */
char	fname[256];		/* filename */
unsigned char	*sbuf;			/* sample buffer */

#define DEFTIME	10		/* 10 seconds */

/* main program */
void main(int argc,char **argv)
{
	struct stat st;
	FILE	*ip;
	int	len;
	int	code;

	if (argc<4) {
		fprintf(stderr,"Usage: windac rate nbytesamp nchan (file)\n");
		exit(1);
	}
	srate = atoi(argv[1]);
	nbytesamp = atoi(argv[2]);
	nchan = atoi(argv[3]);

	if (argc==5) {
		strcpy(fname,argv[4]);

		/* get file size */
		if (stat(fname,&st)!=0) {
			fprintf(stderr,"Could not stat '%s'\n",fname);
			exit(1);
		}
		len = st.st_size;

		/* malloc up a buffer and load data into memory */
		if ((sbuf=(unsigned char *)malloc(len))==NULL) {
			fprintf(stderr,"Could not get buffer\n");
			exit(1);
		}

		ip = fopen(fname,"rb");
		if (fread(sbuf,1,len,ip)!=len) {
			fprintf(stderr,"Read error on '%s'\n",fname);
			exit(1);
		}
		fclose(ip);
	}
	else {
		/* record something first */
		len = nbytesamp * srate * DEFTIME;

		/* report on progress so far */
		printf("Recording %d bytes at %d samp/sec, %d bytes/samp, %d channels\n",
				len,srate,nbytesamp,nchan);

		/* get buffer */
		if ((sbuf=(unsigned char *)malloc(len))==NULL) {
			fprintf(stderr,"Could not get buffer\n");
			exit(1);
		}

		/* initialise endpointing */
		initparams(srate);

		/* do recording */
		if ((len=win32record(sbuf,len,srate,nbytesamp,nchan,1))<0) {
			fprintf(stderr,"win32record returns code %d\n",len);
			exit(1);
		}
	}

	/* report on progress so far */
	printf("Replaying %d bytes at %d samp/sec, %d bytes/samp, %d channels\n",
			len,srate,nbytesamp,nchan);

	/* call replay function */
	win32replay(sbuf,len,srate,nbytesamp,nchan);

	/* that's all folks */
	free(sbuf);
	exit(0);
}
#endif
