/* fmanal -- speech waveform to formant estimates by LPC and root-solving */

/* M.A.Huckvale - March 1988 */

/* version 1.0
	- standard version
   version 1.2
	Correct bug in selecting voiceless window size in excitation
		synchronous analysis
	Add possibility of fixed window size in excitation analysis
*/
/* version 1.3 - December 1993
	- adapt number of poles to sampling rate
	- add flag to specify number of poles
*/
/* version 1.4 - November 1994
	- fix bug in Tx window positioning for sample rate mismatch
	- fix bug in bandwidth calculation
*/
/* version 1.5 - March 1996
	- FORTRAN-free version
*/

/* version 1.6 - November 2001
	- fix buffer overflow at 441000Hz
*/

#define PROGNAME "fmanal"
#define PROGVERS "1.6"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH FMANAL SFS1 UCL
.SH NAME
fmanal - formant estimates from speech waveforms
.SH SYNOPSIS
.B fmanal
(-w window) (-s step) (-v|-t offset) (-f fixtxsize) (-p numpoles) (-i item) file
.SH DESCRIPTION
.I fmanal
performs a windowed LPC analysis on sections of a speech waveform and solves
the predictor polynomial to obtain spectral peak (formant) estimates.
Analysis windows may be of fixed length with a selected overlap,
or may be positioned by means of a TX item (in which case the unvoiced periods
are analysed according to the fixed length parameters).  Pre-emphasis, a Hamming
window and the autocorrelation method is used for all analyses.
If a Tx item used to position analysis windows, the window
size defaults to the pitch period. However, an explicitly given fixed
window size (-f option) will over ride the default.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -w window
Select analysis window size in milliseconds.  Default: 20ms.
.TP 11
.BI -s step
Select analysis window step size in milliseconds. Default: 10ms.
.TP 11
.BI -f fixtxsize
Select fixed analysis window size in milliseconds for excitation
synchronous analysis.
.TP 11
.B -v
Treat all frames as voiced.  Default is to use autocorrelation pitch
detector and adapt analysis to voicing decision. Voiced frames are
pre-emphasised, unvoiced frames use half as many poles.
.TP 11
.BI -t offset
Select excitation synchronous analysis with supplied offset in
micro-seconds between tx markers and start point of analysis window.
Negative is earlier, positive later.  Windows less than 20ms are treated
as voiced.
.TP 11
.BI -p numpoles
Specify number of poles to use in LPC analysis.  Default is (2+sample_rate/1000).
.SH INPUT ITEMS
.IP 1.xx 11
Any speech waveform (10kHz recommended)
.IP 3.xx 11
(Optional) Excitation markers.
.SH OUTPUT ITEMS
.IP 12 11
Formant estimates.
.SH VERSION/AUTHOR
.nf
1.0s - Mark Huckvale
1.2s - Stuart Rosen (modifications)
1.5 - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "complex.h"
#include "lsys.h"
#include "lpca.h"
#include "root.h"
#include "sfs.h"

#define MAXCOEFF	50	/* max # LPC coefficients */
#define NFORMANT	5	/* # formants output */
#define	PREEMP		0.95	/* pre-emphasis coeff */
#define MAXWINDOW	2205	/* maximum analysis window size = 50ms @ 44.1kHz */
#define FREQLIMIT	100	/* Hz limit for a formant freq */
#define BANDLIMIT	3000	/* Hz limit for a formant bandwidth */
#define CLIPVAL		0.5	/* clip at 0.5 of secondary peak */
#define ARATIO1	0.35	/* size of autocorrelation peak for voicing start */
#define ARATIO2 0.25	/* size of autocorrelation peak for voicing continue */

/* global data */
struct item_header	spitem;		/* speech item header */
short			*sp;		/* speech buffer */
struct item_header	txitem;		/* tx item header */
int			*tx;		/* tx buffer */
double			txoffset = 0.0;	/* analysis window offset from tx data */
struct item_header	pcitem;		/* LPC item header */
struct item_header	fmitem;		/* formant item header */
int			lxsync = 0;	/* excitation synchronous = false */
int			vdecide = 1;	/* make voicing decision ? */
double			wisize = -1;	/* analysis window size */
double			ovsize = -1;	/* analysis window overlap */
double			stsize = -1;	/* analysis window stepsize */
double			lxwisize = -1;	/* fixed window size for Tx synchronous analysis */
int			NCOEFF;		/* max # LPC coefficients */
int			ncoeff;		/* # LPC coefficients for window */
struct wm_rec {
	int	posn;		/* window location in speech samples */
	int	size;		/* window size in speech samples */
	int	flag;		/* voicing flag */
};				/* window marker structure */
struct wm_rec	*wmb;		/* window marker buffer */
int	lastvoice = 0;		/* record of previous frame voicing */

#ifdef __STDC__
int calcwmsize(struct item_header *txitem,int *tx);
void initwm(struct item_header *txitem,int *tx,struct wm_rec *wm,double spdur,double offset,int nframes);
void lpc(short *sp,struct wm_rec *wm,struct co_rec *lpcrec,int ncoeff,int lastvoice);
float logenergy(struct co_rec *lpcrec,short *sp);
void fmamp(struct fm_rec *formrec,struct co_rec *lpcrec,int ncoeff,double spdur);
float calcamplitude(struct co_rec *lpcrec,int ncoeff,double tau,double freq);
int voice(short *sp,int len,double sdur);
void autoc(float *sp,int len,float *acoeff,int l1,int l2);
int findpeak(float *acoeff,int len,int l1,int l2);
void roots(struct co_rec *lpcrec,int degree,struct fm_rec *formrec,double sdur);
#else
int calcwmsize();
void initwm();
void lpc();
float logenergy();
void fmamp();
float calcamplitude();
int voice();
void autoc();
int findpeak();
void roots();
#endif

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selections */
	char		*ty;
	char		*sptype="0";	/* default sub-type = last */
	char		*txtype="0";	/* default sub-type = last */
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	int		fid,ofid;
	/* data variables */
	int		wmsize;		/* no. of analysis windows */
	int		wmax;		/* largest window */
	struct co_rec	*lpcrec;	/* buffer for LPC data */
	struct fm_rec	*fmrec;		/* buffer for FM data */
	double		totime;		/* window time (s) */
	int		i;
	char		polemsg[32];
	char		htmp[80];

	strcpy(polemsg,"");
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:w:s:f:t:vp:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Formant analysis V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else if (it == TX_TYPE) {
					txtype = ty;
					lxsync++;
					vdecide=0;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'w' :	/* window size */
			wisize = atof(optarg)/1E3;
			break;
		case 'f' :	/* fixed window size for excitation synchronous analysis */
			lxwisize = atof(optarg)/1E3;
			lxsync++;
			vdecide=0;
			break;
		case 's' :	/* window step */
			stsize = atof(optarg)/1E3;
			break;
		case 't' :	/* tx offset */
			txoffset = atof(optarg)/1E6;
			lxsync++;
			vdecide=0;
			break;
		case 'v' :	/* voicing decision */
			vdecide=0;
			break;
		case 'p' :	/* specify # poles */
			NCOEFF = atoi(optarg);
			if (NCOEFF < 2)
				error("too few poles: '%s'",optarg);
			if (NCOEFF > MAXCOEFF) NCOEFF=MAXCOEFF;
			sprintf(polemsg,",predcoeffs=%d",NCOEFF);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-w winsize[ms]) (-s stepsize[ms]) (-f fixtxsize[ms]) (-v|-t offset[us]) (-p numpoles) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* check file ok for writing */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on %s",filename);

	/* load original tx */
	if (lxsync) getitem(filename,TX_TYPE,txtype,&txitem,(void **)&tx);

	/* locate speech data */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("unable to find input speech item in %s",filename);

	/* default analysis window sizes */
	if (wisize < 0) wisize = 0.02;	/* 20ms */
	if (stsize < 0) stsize = 0.01;	/* 10ms overlap */
	ovsize = wisize-stsize;

	/* calculate/estimate size of window marker buffer */
	if (lxsync)
		wmsize = calcwmsize(&txitem,tx);
	else {
		if (ovsize >= wisize)
			error("analysis window specification error",NULL);
		wmsize = (int)(1.0 + (spitem.numframes*spitem.frameduration)/
				(wisize - ovsize));
	}
	if (wmsize <= 0)
		error("analysis window specification error",NULL);

	/* allocate window marker buffer */
	wmb = (struct wm_rec *) calloc(wmsize+1,sizeof(struct wm_rec));

	/* initialise window marker buffer */
	if (lxsync)		/* using tx markers */
		initwm(&txitem,tx,wmb,spitem.frameduration,txoffset,spitem.numframes);
	else {			/* using fixed specs */
		wmb[0].posn = 0;
		wmb[0].size = (int)(0.5 + wisize/spitem.frameduration);
		wmb[0].flag = 0;
		totime = (wisize - ovsize);
		for (wmsize=1;(int)((totime + wisize)/spitem.frameduration) <= spitem.numframes;wmsize++) {
			wmb[wmsize].posn = (int)(0.5 + totime/spitem.frameduration);
			wmb[wmsize].size = (int)(0.5 + wisize/spitem.frameduration);
			wmb[wmsize].flag = 0;
			totime += (wisize - ovsize);
		}
		wmb[wmsize-1].size = spitem.numframes - wmb[wmsize-1].posn;
	}

	/* find largest analysis window */
	for (wmax=0,i=0;i<wmsize;i++) wmax = (wmax > wmb[i].size) ? wmax : wmb[i].size;
	if (wmax > MAXWINDOW)
		error("input data analysis window too large",NULL);

	/* create buffer for LPC coefficients */
	if (NCOEFF==0)
		NCOEFF = (int)(2 + 0.001/spitem.frameduration);
	if (NCOEFF > MAXCOEFF) NCOEFF=MAXCOEFF;
	sfsheader(&pcitem,PC_TYPE,-1,4,sfsstruct[PC_TYPE]/4 + NCOEFF,
			spitem.frameduration,spitem.offset,0,0,1);
	if ((lpcrec=(struct co_rec *)sfsbuffer(&pcitem,1))==NULL)
		error("cannot get LPC buffer",NULL);

	/* create new item header for spectral coefficients */
	if (lxsync) {
		sfsheader(&fmitem,FM_TYPE,-1,4,(sfsstruct[FM_TYPE]+NFORMANT*sizeof(struct fm_rec_array))/4,
			spitem.frameduration,spitem.offset,0,0,1);
		sprintf(fmitem.history,"%s(%d.%02d,%d.%02d;window=%g,overlap=%g,offset=%g%s,txwindow=",
			PROGNAME,spitem.datatype,spitem.subtype,
			txitem.datatype,txitem.subtype,
			(wisize*1000),
			(ovsize*1000),
			(txoffset*1E6),polemsg);
		if (lxwisize <= 0)
			sprintf(htmp,"TX)");
		else
			sprintf(htmp,"%g)",(lxwisize*1000));
		strcat(fmitem.history, htmp);
	}
	else {
		sfsheader(&fmitem,FM_TYPE,-1,4,(sfsstruct[FM_TYPE]+NFORMANT*sizeof(struct fm_rec_array))/4,
			spitem.frameduration,spitem.offset+wisize/2,
			(int)(wisize/spitem.frameduration),
			(int)(ovsize/spitem.frameduration),0);
		sprintf(fmitem.history,"%s(%d.%02d;window=%g,overlap=%g%s)",
			PROGNAME,spitem.datatype,spitem.subtype,
			(wisize*1000),
			(ovsize*1000),polemsg);
	}

	/* open output channel */
	if ((ofid=sfschannel(filename,&fmitem))<0)
		error("could not open temporary file",NULL);

	/* get data buffers */
	fmrec = (struct fm_rec *)sfsbuffer(&fmitem,1);
	sp = (short *)sfsbuffer(&spitem,wmax);

	/* process speech frame by frame */
	for (i=0;i<wmsize;i++) {

		/* get speech */
		if (sfsread(fid,wmb[i].posn,wmb[i].size,sp) != wmb[i].size)
			error("read error on input file",NULL);

		/* get voicing, if not provided */
		if (vdecide)
			wmb[i].flag=voice(sp,wmb[i].size,spitem.frameduration);
		else if (!lxsync)
			wmb[i].flag=1;

		/* choose # coefficients */
		if (wmb[i].flag)
			ncoeff=NCOEFF;
		else
			ncoeff=NCOEFF/2;

		/* perform LPC analysis on this window */
		lpc(sp,&wmb[i],lpcrec,ncoeff,lastvoice);

		/* perform root solving */
		roots(lpcrec,ncoeff,fmrec,spitem.frameduration);

		/* calculate frame energy from original speech */
		fmrec->gain = logenergy(lpcrec,sp);

		/* calculate formant amplitudes */
		fmamp(fmrec,lpcrec,ncoeff,fmitem.frameduration);

		/* write formant record */
		if (sfswrite(ofid,1,fmrec) != 1)
			error("write error on output file",NULL);

		/* print progress */
		if (ttytest()) {
			printf("\rFrame %d/%d",i+1,wmsize);
			fflush(stdout);
		}

		/* keep record of last frames voicing */
		lastvoice = wmb[i].flag;
	}
	if (ttytest()) printf("\r                     \r");

	/* update */
	if (!sfsupdate(filename) != 0)
		error("update error on %s",filename);

	/* that's all folks ... */
	exit(0);
}

/* calculate number of analysis windows required */
/* for excitation synchronous analysis           */
int calcwmsize(txitem,tx)
struct item_header	*txitem;	/* input tx data */
int			*tx;		/* tx buffer */
{
	int	size=0;		/* number of windows found */
	int	ms20;		/* number of samples in 50 Hz */
	int	wlen;		/* number of samples in window gap */
	int	i,txval;

	/* calculate 20ms and window gap in samples, rounding */
	ms20 = (int)(0.5 + 0.02/txitem->frameduration);
	wlen = (int)(0.5 + stsize/txitem->frameduration);

	/* investigate # windows required for each unvoiced period */
	for (i=0;i<txitem->numframes;i++) {
		txval = tx[i];
		if (txval >= ms20) while (txval >= 2*wlen) {
			txval -= wlen;
			size++;
		}
		size++;
	}

	/* check size of last window */
	if ((txitem->numframes > 1) && (tx[txitem->numframes-1] < wlen)) {
		tx[txitem->numframes-2] += tx[txitem->numframes-1];
		txitem->numframes--;
		size--;
	}

	return(size);
}

/* initialise window marker buffer */
void initwm(txitem,tx,wm,spdur,offset,nframes)
struct item_header	*txitem;	/* tx item header */
int			*tx;		/* tx data */
struct wm_rec		*wm;		/* window marker buffer */
double			spdur;		/* speech signal sample duration */
double			offset;		/* tx offset from speech */
int			nframes;	/* # speech samples to cover */
{
	int	ms20;		/* number of TX samples in 50 Hz */
	int	wlen;		/* number of TX samples in window gap */
	int	lxwlen=0;	/* number of samples in fixed window */
	int	vcless_size;	/* number of samples in window for voiceless sounds */
	int	i,txval;
	double	totime;		/* total time */
	double	txtime;		/* tx sample end time */

	/* calculate 20ms and stepsize in TX units */
	ms20 = (int)(0.5 + 0.02/txitem->frameduration);
	wlen = (int)(0.5 + stsize/txitem->frameduration);

	/* get voiceless window size in SP units */
	vcless_size = (int)(0.5 + wisize/spdur);

	/* Also, length in samples of fixed window, if desired */
	if (lxwisize > 0)
		lxwlen = (int)(0.5 + lxwisize/spdur);

	/* investigate # windows required for each unvoiced period */
	totime = offset;	/* location of zeroth tx w.r.t speech */
	for (i=0;i<txitem->numframes;i++) {
		txval = tx[i];
		if (txval >= ms20) {	/* unvoiced section */
			while (txval >= 2*wlen) {
				/* initialise "stsize" window marker */
				txval -= wlen;
				txtime = totime + wlen*txitem->frameduration;
				wm->posn = (int)((totime > 0.0) ? 0.5 + (totime/spdur) : 0.0);
				wm->size = vcless_size;
				wm->flag = 0;
				wm++;
				totime = txtime;
			}
			wm->flag = 0;
		}
		else			/* voiced section */
			wm->flag = 1;
		/* initialise window marker data from tx */
		txtime = totime + txval*txitem->frameduration;
		wm->posn = (int)((totime > 0.0) ? 0.5 + (totime/spdur) : 0.0);
		if (lxwisize > 0)
			wm->size = lxwlen;
		else
			wm->size = (int)(0.5 + txtime/spdur - wm->posn);
		wm++;
		totime = txtime;
	}

	/* fix-up last tx sample to match waveform length */
	wm--;
	wm->size = nframes - wm->posn;
}

/* lpc analysis subroutine */
void lpc(sp,wm,lpcrec,ncoeff,lastvoice)
short			*sp;		/* speech signal */
struct wm_rec		*wm;		/* window markers */
struct co_rec		*lpcrec; 	/* output lpc record */
int			ncoeff;		/* # lpc coefficients */
int			lastvoice;	/* voicing of previous frame */
{
	register int	i;
	register float	*xp,*hp;
	float		x[MAXWINDOW];	/* speech data for analysis */
static	float		hamm[MAXWINDOW];	/* hamming window coefficient */
static  int		hammsize=0;	/* size of last hamming window */
	float		hc;
	LTIState	ltis;
	double		pe;

	/* put speech data into floating point buffer */
	xp = x;
	for (i=0;i<wm->size;i++) {
		/* start with low-level random noise - to help stabilise analysis */
		*xp = (float)(rand()&0x7FFF) / 32768.0 - 0.5;
		/* add in signal */
		*xp++ += *sp++;
	}

	/* Hamming window */
	if (hammsize != wm->size) {
		hammsize = wm->size;
		hc = 8.0*atan(1.0)/(hammsize-1);
		xp = hamm;
		for (i=0;i<hammsize;i++,xp++)
			*xp = (0.54 - (0.46 * cos(hc * i)));
	}

	/* pre-emphasise and window */
	xp = x+wm->size-1;
	hp = hamm+wm->size-1;
	for (i=1;i<wm->size;i++,xp--,hp--)
		*xp = *hp * (*xp - PREEMP * *(xp-1));
	*xp *= *hp;

	/* do autocorrelation analysis */
	LPCAutocorrelation(x,wm->size,ncoeff,&ltis,&pe);

	/* set up lpc record */
	lpcrec->posn = wm->posn;
	lpcrec->size = wm->size;
	lpcrec->flag = wm->flag;
	lpcrec->mix  = (float)wm->flag;
	lpcrec->gain = (pe > 0.0) ? sqrt(pe) : 0.0;
	for (i=0;i<ncoeff;i++) lpcrec->data[i] = ltis.b[i+1];
	for (i=ncoeff;i<NCOEFF;i++) lpcrec->data[i] = 0.0;

}

/* calculate energy in speech waveform */
float logenergy(lpcrec,sp)
struct co_rec		*lpcrec;	/* LPC frame */
short			*sp;		/* orginal speech waveform */
{
	int		i,len;
	short		*s;
	float		mean=0.0,sum=0.0001,val;

	s = sp;
	len = (int)(0.9 * lpcrec->size);	/* over first 90% of window only */
	/* find mean value */
	for (i=0;i<len;i++,s++) mean += (float)*s;
	mean /= len;
	/* find sum squared values */
	s = sp;
	for (i=0;i<len;i++,s++) {
		val = (float)*s - mean;
		sum += val*val;
	}
	return((float)(10.0*log10(sum)));
}

/* calculate formant amplitudes */
void fmamp(formrec,lpcrec,ncoeff,spdur)
struct fm_rec		*formrec;	/* formant data */
struct co_rec		*lpcrec;	/* LPC coefficients */
int			ncoeff;		/* # coefficients */
double			spdur;		/* sample duration */
{
	float		amp;
	int		i;

	for (i=0;i<formrec->npeaks;i++) {
		amp = calcamplitude(lpcrec,ncoeff,spdur,formrec->formant[i].freq);
#if 0
		formrec->formant[i].amp = 20.0*log10(lpcrec->gain/amp);
		/* correct for pre-emphasis */
		formrec->formant[i].amp -= 6.0*log10(formrec->formant[i].freq/100.0)/log10(2.0);
#else
		formrec->formant[i].amp = 20.0*log10(lpcrec->gain/amp);
		if (formrec->flag)
			formrec->formant[i].amp -= 6.0*log10(formrec->formant[i].freq/500.0)/log10(2.0);
#endif
	}
}

/* calculate amplitude at given frequency in LPC frame */
float calcamplitude(lpcrec,ncoeff,tau,freq)
struct co_rec		*lpcrec;	/* LPC coeffs for this frame */
int			ncoeff;		/* # coefficients */
double			tau;		/* sampling interval */
double			freq;		/* specified frequency */
{
	float		sumsin;		/* sum of sine products */
	float		sumcos;		/* sum of cosine products */
	int		i;
	float		omega;		/* 2.pi.f.t */
	float		amp;		/* amplitude */

	/* initialise omega */
	omega = 8.0 * atan(1.0) * freq * tau;

	/* initialise sine and cosine products */
	sumsin = 0.0;
	sumcos = 1.0;

	/* sum sine and cosine products */
	for (i=0;i<ncoeff;i++) {
		sumsin += lpcrec->data[i] * sin(omega * (i+1));
		sumcos += lpcrec->data[i] * cos(omega * (i+1));
	}

	/* return amplitude */
	amp = sqrt(sumsin*sumsin + sumcos*sumcos);
	return(amp);
}

int	voice(sp,len,sdur)	/* return 0=unvoiced, 1=voiced */
short	*sp;			/* input speech window */
int	len;			/* # samples */
double	sdur;			/* sample duration */
{
	int	ms10;		/* # samples in 10 ms */
	float	fsp[MAXWINDOW];	/* float speech (cubed) */
	short	fxval;		/* estimated fx */
	float	acoeff[MAXWINDOW];	/* autocorrelation */
	int	lofx,hifx;	/* autocorrelation limits */
	int	peak;		/* location of peak */
	static short lastfx=0;	/* previous FX value */
	int	i;

	/* get # samples in 10ms */
	ms10 = (int)(0.5 + 0.01/sdur);

	/* get autocorrelation coefficients */
	hifx = ms10/4;	/* 2.5 ms */
	lofx = (int)(1.25*ms10);	/* 12.5 ms */
	for (i=0;i<len;i++) fsp[i] = (float)(sp[i])*(float)(sp[i])*(float)(sp[i]);
	autoc(fsp,len,acoeff,hifx,lofx);

	/* find autocorrelation peak */
	peak = findpeak(acoeff,len,hifx,lofx);

	/* use autocorrelation values as voicing determiner */
	if (acoeff[0] < 100)
		fxval=0;		/* no speech */
	else if (peak==0)
		fxval=0;		/* no peak */
	else {
		fxval = 100*ms10/peak;
		if ((lastfx==0) || (abs(fxval-lastfx) > 50)) {
			/* voicing start or rapid change */
			if ((acoeff[peak]/acoeff[0]) < ARATIO1)
				fxval=0;	/* peak not high enough */
		}
		/* voicing continuation */
		else if ((acoeff[peak]/acoeff[0]) < ARATIO2)
			fxval=0;	/* peak not high enough */
	}

	/* return voicing decision */
	lastfx = fxval;
	return((fxval==0) ? 0 : 1);
}

/* autocorrelation */
void autoc(sp,len,acoeff,l1,l2)
float	*sp;
int	len;
float	*acoeff;
int	l1,l2;
{
	register int	i,j;
	int		num;
	float		sum;
	float		*s1,*s2;

	/* zero autocorrelation vector */
	for (i=0;i<len;i++) acoeff[i]=0.0;

	/* for zero delay */
	sum=0.0;
	num=len;
	s1 = sp;
	for (j=0;j<num;j++,s1++) sum += *s1 * *s1;
	acoeff[0] = sum/len;

	/* for each delay in expected freq. range */
	for (i=l1;i<l2;i++) {	/* from 2.5ms to 12.5ms */
		sum=0.0;
		num=len-i;
		s1 = sp;
		s2 = sp + i;
		for (j=0;j<num;j++) sum += *s1++ * *s2++;
		acoeff[i] = sum/num;
	}

}

/* find peak */
int findpeak(acoeff,len,l1,l2)
float	*acoeff;
int	len;
int	l1,l2;
{
	register int	i,pos;
	float		max;

	/* find largest peak > max freq allowed */
	pos = 0;
	max = acoeff[l1];
	for (i=l1+1;i<l2;i++) {
		if (acoeff[i] > max) {
			max = acoeff[i];
			pos=i;
		}
	}

	return(pos);
}

/* LPC coefficient root-solving */
void roots(lpcrec,degree,formrec,sdur)
struct co_rec		*lpcrec;	/* LPC coefficients */
int			degree;		/* # lpc coefficients */
struct fm_rec		*formrec;	/* formant data structure */
double			sdur;		/* sample duration */
{
	COMPLEX		poly[MAXWINDOW];	/* polynomial to be solved */
	COMPLEX		root[MAXWINDOW];	/* roots */
	double		twopi;		/* real constant */
	double		omega,rsq;
	struct fm_rec_array fspec[MAXWINDOW],ftemp;
					/* formant freq, band, amp specification */
	int		i,j,npeak=0;

	/* copy LPC coefficients into polynomial */
	poly[0].re = 1.0; poly[0].im = 0.0;
	for (i=0;i<degree;i++) {
		poly[i+1].re = lpcrec->data[i];
		poly[i+1].im = 0.0;
	}

	/* do root-solving using DSP library */
	memset(root,0,sizeof(root));
	AllRoots(poly,degree,root);

	/* convert roots to pole positions */
	twopi = 8.0*atan(1.0);
	for (i=0;i<=degree;i++) if (root[i].im > 1.0E-10) {
		omega = atan2(root[i].im,root[i].re);
 		fspec[npeak].freq = fabs(omega/(twopi*sdur));
 		rsq = root[i].re*root[i].re+root[i].im*root[i].im;
		fspec[npeak].band = fabs(log(rsq)/(2*twopi*sdur));
		if ((fspec[npeak].freq > FREQLIMIT) && (fspec[npeak].band < BANDLIMIT)) npeak++;
	}

	/* sort pole positions in terms of bandwidth */
	for (i=1;i<npeak;i++) {
		j = i;
		ftemp = fspec[i];
		while ((j>0) && (ftemp.band < fspec[j-1].band)) {
			fspec[j]=fspec[j-1];
			j--;
		}
		fspec[j]=ftemp;
	}

	/* sort first NFORMANT pole positions in terms of frequency */
	for (i=1;(i<NFORMANT) && (i<npeak);i++) {
		j = i;
		ftemp = fspec[i];
		while ((j>0) && (ftemp.freq < fspec[j-1].freq)) {
			fspec[j]=fspec[j-1];
			j--;
		}
		fspec[j]=ftemp;
	}

	/* initialise formant record */
	formrec->posn = lpcrec->posn;
	formrec->size = lpcrec->size;
	formrec->flag = lpcrec->flag;
	formrec->gain = lpcrec->gain;
	formrec->npeaks = 0;
	for (i=0;i<NFORMANT;i++) {
		formrec->formant[i].freq=0.0;
		formrec->formant[i].band=0.0;
		formrec->formant[i].amp=0.0;
	}

	/* put upto NFORMANT peaks into formant structure */
	for (i=0;(i<NFORMANT) && (i<npeak);i++) {
		formrec->formant[formrec->npeaks].freq = fspec[i].freq;
		formrec->formant[formrec->npeaks].band = fspec[i].band;
		formrec->npeaks++;
	}

}
