/* testsig - generate some test signals */

/* M.A.Huckvale - February 1991 */

/* version 1.0 */
/* version 1.1 - May 1993
	- add pulse option
	- fix -S switch
*/
/* version 1.2 - January 2002
	- add pink noise option
*/
/* version 1.3 - December 2002
	- add "silence"
*/

#define PROGNAME "testsig"
#define PROGVERS "1.3"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH TESTSIG UCL1 UCL SFS
.SH NAME
testsig - generate some test signals
.SH SYNOPSIS
.B testsig
(-s|-n|-p|-P|-Q|-T type) (-f freq) (-e stopfreq) (-t time) (-r time|-R time) (-d) (-S sampfreq) file
.SH DESCRIPTION
.I testsig
is a program to generate some test signals directly into a speech file.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.B -s
Generate a sine wave. Default.
.TP 11
.B -n
Generate white noise.
.TP 11
.B -P
Generate pink noise.
.TP 11
.B -p
Generate pulse train.
.TP 11
.B -Q
Generate silence (quiet).
.TP 11
.BI -f freq
Specify frequency of sine wave.  Default 1000Hz.
.TP 11
.BI -e endfreq
Specify end frequency for chirp.
.TP 11
.BI -t time
Specify duration of signal.  Default 0.5 seconds.
.TP 11
.BI -r time
Specify time to linearly ramp up amplitude to maximum at start and end.
Default 0 seconds.
.TP 11
.BI -R time
Specify time to logarithmically ramp up amplitude to maximum at start and end.
Default 0 seconds.
.TP 11
.B -d
Differentiate output.
.TP 11
.BI -S sampling_rate
Specify the signal sampling rate.  Default 20000Hz.
.TP 11
.BI -T type
Specify type of waveform: available: sine, pulse, pulse-train, white-noise,
sawtooth, chirp, pink-noise, silence.
.SH OUTPUT ITEMS
.IP SP
Test signal.
.SH HISTORY
.IP type= 11
shows sine or noise waveform
.IP freq= 11
shows sine wave frequency
.IP time= 11
shows duration of signal
.IP ramp= 11
shows ramp time of signal
.IP differenced 11
shows signal has been differentiated.
.IP samprate= 11
shows sampling rate.
.SH VERSION/AUTHOR
.IP 1.1
Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"

/* global variables */
struct item_header	spitem;
short			*sp;

/* parameters */
#define SIG_SINE	0
#define SIG_WHITE_NOISE	1
#define SIG_PULSE	2
#define SIG_PULSE_TRAIN	3
#define SIG_SAWTOOTH	4
#define SIG_CHIRP	5
#define SIG_PINK_NOISE	6
#define SIG_SILENCE	7
char *signames[]={"sine","white-noise","pulse","pulse-train","sawtooth","chirp","pink-noise","silence"};
#define NUMSIGNAMES sizeof(signames)/sizeof(char *)

int	sigtype=SIG_SINE;
double	sigfreq=1000;
double	sigendfreq=1000;
double	sigtime=0.5;
double	sigramp=0.0;
double	samprate=20000.0;
int	sigdif=0;
int	sigramplog=0;

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid,ofid;		/* file descriptors */
	int		numframes;
	int		numramp;
	int		i,j,val,period;
	double		t,sampdur,omega,omegadiff;
	char		messg[80];
	double		dval,slope;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Isnpdf:t:r:R:S:T:e:PQ")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Test Signals V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 's' :	/* sine wave */
			sigtype = SIG_SINE;
			break;
		case 'n' :	/* noise wave */
			sigtype = SIG_WHITE_NOISE;
			break;
		case 'P' :	/* pink wave */
			sigtype = SIG_PINK_NOISE;
			break;
		case 'p':	/* pulse train */
			sigtype = SIG_PULSE_TRAIN;
			break;
		case 'Q':	/* pulse train */
			sigtype = SIG_SILENCE;
			break;
		case 'T':
			sigtype=-1;
			for (i=0;i<NUMSIGNAMES;i++)
				if (stricmp(optarg,signames[i])==0)
					sigtype = i;
			if (sigtype < 0) {
				fprintf(stderr,"Available types: ");
				for (i=0;i<NUMSIGNAMES;i++)
					fprintf(stderr,"%s,",signames[i]);
				fprintf(stderr,"\n");
				error("unknown signal type: '%s'",optarg);
			}
			break;
		case 'f' :	/* sine frequency */
			sigfreq = atof(optarg);
			break;
		case 'e' :	/* end freq */
			sigendfreq = atof(optarg);
			break;
		case 't' :	/* signal duration */
			sigtime = atof(optarg);
			break;
		case 'r' :	/* linear ramp time */
			sigramp = atof(optarg);
			break;
		case 'R' :	/* log ramp time */
			sigramp = atof(optarg);
			sigramplog++;
			break;
		case 'S' :	/* sampling rate */
			samprate = atof(optarg);
			break;
		case 'd' :	/* differentiate */
			sigdif++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-s|-n|-p|-P|-T type) (-f freq) (-e stopfreq) (-t time) (-r time|-R time) (-S samprate) (-d) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* check file exists */
	if ((fid=sfsopen(filename,"w",NULL))<0)
		error("access error on '%s'",filename);
	sfsclose(fid);

	/* get parameters in numbers of samples */
	numframes = (int)(sigtime * samprate);
	numramp = (int)(sigramp * samprate);
	if (numramp > numframes/2) {
		numramp = numframes/2;
		sigramp = sigtime/2;
	}

	/* set up output header */
	sfsheader(&spitem,SP_TYPE,0,2,1,1.0/samprate,0.0,1,0,0);
	if ((sigtype==SIG_SINE)||(sigtype==SIG_PULSE_TRAIN)||(sigtype==SIG_SAWTOOTH))
		sprintf(spitem.history,"%s(type=%s,freq=%g",PROGNAME,signames[sigtype],sigfreq);
	else if (sigtype==SIG_CHIRP)
		sprintf(spitem.history,"%s(type=%s,startfreq=%g,stopfreq=%g",
			PROGNAME,signames[sigtype],sigfreq,sigendfreq);
	else
		sprintf(spitem.history,"%s(type=%s",PROGNAME,signames[sigtype]);
	if (sigtime != 0.5) {
		sprintf(messg,",time=%g",sigtime);
		strcat(spitem.history,messg);
	}
	if (sigramp != 0) {
		sprintf(messg,",%sramp=%g",(sigramplog)?"log":"",sigramp);
		strcat(spitem.history,messg);
	}
	if (samprate != 20000) {
		sprintf(messg,",samprate=%g",samprate);
		strcat(spitem.history,messg);
	}
	if (sigdif)
		strcat(spitem.history,",differenced");
	strcat(spitem.history,")");

	/* get buffer */
	if ((sp=(short *)sfsbuffer(&spitem,numframes))==NULL)
		error("could not get memory for signal");

	/* open output channel */
	if ((ofid=sfschannel(filename,&spitem)) < 0)
		error("could not open channel to '%s'",filename);

	/* OK go make the signal */
	if (sigtype==SIG_SINE) {
		t = 0.0;
		sampdur = 1.0/samprate;
		omega = 8.0 * atan(1.0) * sigfreq;
		for (i=0;i<numframes;i++,t+=sampdur)
			sp[i] = (short)(15000.0 * sin(omega*t));
	}
	else if (sigtype==SIG_CHIRP) {
		t = 0.0;
		sampdur = 1.0/samprate;
		omega = 8.0 * atan(1.0) * sigfreq * sampdur;
		omegadiff = 8.0*atan(1.0)*(sigendfreq-sigfreq)*sampdur/numframes;
		for (i=0;i<numframes;i++) {
			sp[i] = (short)(15000.0 * sin(t));
			t += omega;
			omega += omegadiff;
		}
	}
	else if (sigtype==SIG_PULSE_TRAIN) {
		i=0;
		period = (int)(samprate/sigfreq);
		while (i < numframes) {
			sp[i++] = 15000;
			for (j=1;(j<period) && (i<numframes);j++,i++)
				sp[i] = 0;
		}
	}
	else if (sigtype==SIG_SAWTOOTH) {
		i=0;
		period = (int)(samprate/sigfreq);
		slope = 30000.0/period;
		dval = 0;
		while (i < numframes) {
			sp[i++] = (short)dval;
			dval += slope;
			if (dval > 15000) dval = -15000;
		}
	}
	else if (sigtype==SIG_PULSE) {
		i=0;
		period = (int)(samprate/sigfreq);
		while (i < numframes) {
			if (i==period)
				sp[i++] = 15000;
			else
				sp[i++] = 0;
			for (j=1;(j<period) && (i<numframes);j++,i++)
				sp[i] = 0;
		}
	}
	else if (sigtype==SIG_PINK_NOISE) {
		/* algorithm from
			http://shoko.calarts.edu/~glmrboy/musicdsp/sourcecode/pink.txt
		*/
		double b0=0,b1=0,b2=0,b3=0,b4=0,b5=0,b6=0;
		double	white,pink;
		for (i=0;i<numframes;i++) {
			white = ((rand()&0x00007FFF)-0x4000)/10.0;
			b0 = 0.99886 * b0 + white * 0.0555179;
			b1 = 0.99332 * b1 + white * 0.0750759;
			b2 = 0.96900 * b2 + white * 0.1538520;
			b3 = 0.86650 * b3 + white * 0.3104856;
			b4 = 0.55000 * b4 + white * 0.5329522;
			b5 = -0.7616 * b5 - white * 0.0168980;
			pink = b0 + b1 + b2 + b3 + b4 + b5 + b6 + white * 0.5362;
			b6 = white * 0.115926;

			sp[i] = (short)pink;
		}
	}
	else if (sigtype==SIG_WHITE_NOISE) {
		for (i=0;i<numframes;i++) {
			val = 0;
			for (j=0;j<6;j++)
				val += rand()%30000 - 15000;
			sp[i] = val/6;
		}
	}
	else {
		/* silence */
		for (i=0;i<numframes;i++) sp[i]=0;
	}

	/* now do the differencing */
	if (sigdif) {
		for (i=0;i<numframes-1;i++)
			sp[i] = sp[i+1] - sp[i];
		numframes--;
	}

	/* put in the ramping */
	if (sigramplog==0) {
		for (i=0;i<numramp;i++)
			sp[i] = (short)(sp[i]*(double)(i+1)/(double)numramp);
		for (i=numframes-1;i>numframes-numramp;i--)
			sp[i] = (short)(sp[i]*(double)(numframes-i)/(double)numramp);
	}
	else {
		for (i=0;i<numramp;i++)
			sp[i] = (short)(sp[i]*pow(10.0,(-3.0 + 3.0*(double)(i+1)/(double)numramp)));
		for (i=numframes-1;i>numframes-numramp;i--)
			sp[i] = (short)(sp[i]*pow(10.0,(-3.0 + 3.0*(double)(numframes-i)/(double)numramp)));
	}

	/* save to file */
	if (sfswrite(ofid,numframes,sp) != numframes)
		error("write error on output file");

	/* update file */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);

	/* that's all folks! */
	exit(0);
}
