/* coload -- load CO item from coefficients listing produced by colist */

/* M.A.Huckvale */

/* version 1.0 - March 1990 */

/* version 1.1 - March 1996
	- fix deletion of second frame!
	- add offset switch
*/
/* version 1.2 - June 2004
	- add -h switch
*/

#define PROGNAME "coload"
#define PROGVERS "1.2"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH COLOAD SFS1 UCL
.SH NAME
coload - load spectral coefficients item from file
.SH SYNOPSIS
.B coload
(-O offset) (-h) datafile sfsfile
.SH DESCRIPTION
.I coload
creates a new CO item in the file
.I sfsfile
from spectral coefficients listed in the file
.I listfile.
Default input format is that written by
.I colist,
namely 1 line per frame, with a leading time specified in milliseconds.
.PP
To load parameters into the coefficient item header, use a line before the first
valid data frame with the format:
.br
	"! params=<parameters to load>"
.br
The
.I colist
program may be used to demonstrate the format.
An option allows the loading of HTK format data files.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version.
.TP 11
.BI -O offset
Give the loaded item an offset in seconds.
.TP 11
.B -h
Input file is in HTK parameter file format.  Only uncompressed formats
are supported.  The HTK type is saved in the item parameter string.
.SH OUTPUT ITEMS
.IP 11 11
Edited CO item.
.SH VERSION/AUTHOR
1.2 - Mark Huckvale
.SH SEE ALSO
colist
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header coitem;	/* filing system item header for new item */

#define MAXLINE	10000
char	iline[MAXLINE];

int	dohtk;
char	htkname[SFSMAXFILENAME];
struct htk_header {
	long	nsamp;
	long	nperiod;
	short	ssize;
	short	skind;
} hhead;

/* macros to reverse ints and floats */
#define REVERSE2(x)	p = (unsigned char *)(&x); \
			t = *p; *p = *(p+1); *(p+1) = t;
#define REVERSE4(x)	p = (unsigned char *)(&x); \
			t = *p; *p = *(p+3); *(p+3) = t; \
			t = *++p; *p = *(p+1); *(p+1) = t;

/* get number of numbers on a line */
int getnumbernum(str)
char	*str;
{
	int	cnt=0;
	char	*p;

	p=strtok(str," \t\n");
	while (p && *p && (isdigit(*p) || (*p=='-'))) {
		cnt++;
		p = strtok(NULL," \t\n");
	}
	return(cnt);
}

/* get number of coefficients */
int getnumcoeff(ip)
FILE	*ip;
{
	int	nc,ncoeff=0;

	while (fgets(iline,MAXLINE,ip)) if (isdigit(iline[0]) || (iline[0]==' ')) {
		nc = getnumbernum(iline)-1;
		if (nc > ncoeff)
			ncoeff = nc;
	}
	rewind(ip);
	return(ncoeff);
}

/* get coefficient data from text file */
int getframe(corec,num,ip)
struct co_rec		*corec;		/* returned data */
int			num;		/* number of coefficients */
FILE			*ip;		/* input file */
{
	char		*p;
	int		i;

	/* get next data line */
	do {
		if (fgets(iline,MAXLINE,ip) == NULL) return(1);

		if (strncmp(iline,"! params=",9)==0)
			strncpy(coitem.params,strtok(iline+9,"\n"),sizeof(coitem.params));

	} while (!isdigit(iline[0]) && (iline[0] != ' '));

	/* get time */
	p = strtok(iline," \t\n");
	corec->posn = (int)(0.5+(atof(p)*0.001)/coitem.frameduration);
	corec->flag = 0;
	corec->gain = (float)1.0;
	for (i=0;i<num;i++)
		corec->data[i]=(float)0.0;

	/* get fixed # coefficients */
	for (i=0;i<num;i++) {
		p = strtok(NULL," \t\n");
		if (p && *p && (isdigit(*p) || (*p=='-')))
			corec->data[i] = (float)atof(p);
		else
			break;
	}

	return(0);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	char		textname[SFSMAXFILENAME];	/* text file name */
	FILE		*ip;		/* text file variable */
	int		ofid;
	/* processing variables */
	struct co_rec	*corec,*lcorec;		/* input coefficient record */
	int		ncoeff;
	int		cnt=0;
	int		i,j;
	double		offset=0;
	unsigned char *p,t;
	int		dorev=0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"IO:h")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Load spectral coefficients V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'O' :	/* offset */
			offset = atof(optarg);
			break;
		case 'h' :	/* HTK format input */
			dohtk=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}

	/* check for option decoding error */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-O offset) (-h) textfile sfsfile",PROGNAME);

	/* get text filename */
	if (optind < argc)
		strcpy(textname,argv[optind]);
	else
		error("no listing file specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	if (dohtk) {
		/* open file */
		if ((ip=fopen(textname,"rb")) == NULL)
			error("unable to open file %s",textname);

		/* read header */
		if (fread(&hhead,sizeof(struct htk_header),1,ip)!=1)
			error("read error on '%s'",textname);

		if ((hhead.nsamp & 0xC0000000)||(hhead.nperiod & 0xFF000000)||(hhead.ssize < 0))
			dorev=1;

		if (dorev) {
			REVERSE4(hhead.nsamp);
			REVERSE4(hhead.nperiod);
			REVERSE2(hhead.ssize);
			REVERSE2(hhead.skind);
		}

		printf("File:%s nsamp=%d nperiod=%d ssize=%d skind=%X\n",
			textname,
			hhead.nsamp,hhead.nperiod,hhead.ssize,hhead.skind);
		ncoeff = hhead.ssize/sizeof(float);

		if ((hhead.skind&0x0400)!=0)
			error("coload does not support compressed HTK data files");
		if ((hhead.skind&0x003F)==0)
			error("coload does not support waveform HTK data files");

		/* create CO item header */
		sfsheader(&coitem,CO_TYPE,1,4,
			(sfsstruct[CO_TYPE]+ncoeff*sizeof(float))/4,
			hhead.nperiod*0.0000001,offset,1,0,0);
		sprintf(coitem.history,"%s(file=%s)",PROGNAME,textname);
		sprintf(coitem.params,"htktype=%X",hhead.skind & 0xEFFF);	/* strip CRC bit */

		/* open output channel */
		if ((ofid=sfschannel(filename,&coitem)) < 0)
			error("unable to create temporary file",NULL);

		/* get input buffer */
		if ((corec=(struct co_rec *)sfsbuffer(&coitem,1))==NULL)
			error("could not get memory buffer",NULL);

		/* copy frames */
		cnt=0;
		for (i=0;i<hhead.nsamp;i++) {
			if (fread(corec->data,4,ncoeff,ip)!=ncoeff)
				error("read error on '%s'",textname);
			if (dorev) {
				for (j=0;j<ncoeff;j++) {
					REVERSE4(corec->data[j]);
				}
			}
			corec->posn=i;
			corec->size=1;
			corec->flag=0;
			corec->mix=0;
			corec->gain=0;
			if (sfswrite(ofid,1,corec) != 1)
				error("write error on temporary file",NULL);
			cnt++;
		}

		fclose(ip);
	}
	else {
		/* open text file */
		if ((ip=fopen(textname,"r")) == NULL)
			error("unable to open text file %s",textname);

		/* get number of coefficients per frame */
		ncoeff = getnumcoeff(ip);

		/* create CO item header */
		sfsheader(&coitem,CO_TYPE,1,4,
			(sfsstruct[CO_TYPE]+ncoeff*sizeof(float))/4,
			0.0001,offset,0,0,1);
		sprintf(coitem.history,"%s(file=%s)",PROGNAME,textname);

		/* get input buffer */
		if ((lcorec=(struct co_rec *)sfsbuffer(&coitem,1))==NULL)
			error("could not get memory buffer",NULL);
		if ((corec=(struct co_rec *)sfsbuffer(&coitem,1))==NULL)
			error("could not get memory buffer",NULL);

		/* get first frame */
		if (getframe(lcorec,ncoeff,ip)==0) {

			/* open output channel */
			if ((ofid=sfschannel(filename,&coitem)) < 0)
				error("unable to create temporary file",NULL);

			/* load frames from text file */
			while (getframe(corec,ncoeff,ip) == 0) {
				lcorec->size = corec->posn - lcorec->posn;
				if (sfswrite(ofid,1,lcorec) != 1)
					error("write error on temporary file",NULL);
				lcorec->posn = corec->posn;
				lcorec->size = corec->size;
				lcorec->flag = corec->flag;
				lcorec->mix = corec->mix;
				lcorec->gain = corec->gain;
				for (i=0;i<ncoeff;i++)
					lcorec->data[i] = corec->data[i];
				cnt++;
			}

			/* do last frame */
			lcorec->size = 0;
			if (sfswrite(ofid,1,lcorec) != 1)
				error("write error on temporary file",NULL);
			cnt++;
		}
	}

	/* update data file */
	if (cnt == 0)
		error("no legal frames found in %s",textname);
	else if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks */
	exit(0);
}


