/* hed -- interactive and command line editor for main headers of speech files */

/* m.a. huckvale - june 1987 */

/*--------------------------------------------------------------------------*/
/**MAN
.TH HED SFS1 UCL SFS
.SH NAME
hed - interactive and command-line editor for speech file main headers
.SH SYNOPSIS
.B hed
[ options ] file
.SH DESCRIPTION
.I hed
is a program to edit the fields of the main header of a speech
file.  It can operate in an "interactive" mode, prompting for inputs,
or in a command-line mode, wherby replacement values may be entered using
program switches.  Default operation is interactive; the command line
mode is used when any command line switch for header fields is used.
.I hed
is also a useful utility for creating new (empty) speech files.
.PP
General options and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.B -n
Interactive mode but no questions - just create a new file with a default
header.  If file already exists reports 'file exists', otherwise 'new file'.
.SH COMMAND LINE EDITING
Each field in the main header
is allocated one option. To
.I insert
text specify the text after the option. To
.I delete
text
specify the null string ("") after the option.  Strings are automatically truncated
to fit fields if necessary.
.TP 11
.B -u username
Owner of file.
.TP 11
.B -s source
Source Institute of recording.
.TP 11
.B -d dbase
Name of database to which recording belongs.
.TP 11
.B -S speaker
Speaker identification.
.TP 11
.B -C session
Recording session identification.
.TP 11
.B -D date
Recording session date.
.TP 11
.B -t token
Name of token (transcription/title).
.TP 11
.B -r rep
Repetition code of token.
.TP 11
.B -e environ
Recording environment.
.TP 11
.B -a archive
Archiving details.
.TP 11
.B -c comment
Comments.
.SH INTERACTIVE MODE
In interactive mode, each of the above fields are presented in turn, and
the user can enter the replacement text, or a single space character to
delete the current entry, or a [RETURN] to leave entry unchanged.  In
addition the user can jump to a particular question by typing "!<num>" and
jump to the end by typing "!!".  When all the questions have been answered
the program prompts you to save (y), quit (q) or re-edit the fields (n).
.SH VERSION/AUTHOR
1.1 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/
#define PROGNAME "hed"
#define PROGVERS "1.2"
char *progname=PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "sfs.h"		/* header structures */

/* global data */
struct main_header head;		/* main header data */
struct main_header new;			/* new main header data */
int	change[11];			/* field changes */
int	interactive=1;			/* default interactive */
int	questions=1;			/* default ask questions */

#ifdef __STDC__
int 	edithead(void);
int 	getfield(int,char *,char *,int);
#else
int 	edithead();
int 	getfield();
#endif

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	char		c;		/* option char */
	char		filename[SFSMAXFILENAME];
					/* database file name */
	int		code;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Inu:s:d:S:C:D:t:r:e:a:c:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: SFS file header editor V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'n' :	/* no questions */
			questions = 0;
			break;
		case 'u' :	/* username */
			strncpy(new.username,optarg,20);
			new.username[19]='\0';
			change[0]++;
			interactive=0;
			break;
		case 's' :	/* source */
			strncpy(new.source,optarg,20);
			new.source[19]='\0';
			change[1]++;
			interactive=0;
			break;
		case 'd' :	/* database */
			strncpy(new.dbase,optarg,20);
			new.dbase[19]='\0';
			change[2]++;
			interactive=0;
			break;
		case 'S' :	/* Speaker */
			strncpy(new.speaker,optarg,20);
			new.speaker[19]='\0';
			change[3]++;
			interactive=0;
			break;
		case 'C' :	/* session */
			strncpy(new.session,optarg,20);
			new.session[19]='\0';
			change[4]++;
			interactive=0;
			break;
		case 'D' :	/* session date */
			strncpy(new.sessdate,optarg,20);
			new.sessdate[19]='\0';
			change[5]++;
			interactive=0;
			break;
		case 't' :	/* token */
			strncpy(new.token,optarg,160);
			new.token[159]='\0';
			change[6]++;
			interactive=0;
			break;
		case 'r' :	/* repetition */
			strncpy(new.rep,optarg,8);
			new.rep[7]='\0';
			change[7]++;
			interactive=0;
			break;
		case 'e' :	/* environment */
#ifdef WIN32
			strncpy(new.environment,optarg,8);
			new.environment[7]='\0';
#else
			strncpy(new.environ,optarg,8);
			new.environ[7]='\0';
#endif
			change[8]++;
			interactive=0;
			break;
		case 'a' :	/* archive */
			strncpy(new.archive,optarg,20);
			new.archive[19]='\0';
			change[9]++;
			interactive=0;
			break;
		case 'c' :	/* comment */
			strncpy(new.comment,optarg,80);
			new.comment[79]='\0';
			change[10]++;
			interactive=0;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-n) (-u username) (-s source) (-d database) (-S speaker) (-C session) (-D sessdate) (-t token) (-r rep) (-e environ) (-a archive) (-c comment) dbase_file\n",NULL);

	/* get filename */
	if (optind < argc)
		strcpy(filename,argv[optind]);
	else
		error("no database file specified",NULL);

	/* see if file exists */
	code = sfsopen(filename,"w",&head);
	if (code == -1) {
		/* file does not exist */
		if ((code = sfsopen(filename,"c",&head)) < 0)
			error("unable to create %s",filename);
		printf("new file\n");
	}
	else if (code == -2)
		error("error on access to %s",filename);
	else if (!questions)
		printf("file exists\n");

	/* perform non-interactive stuff */
	if (!interactive) {
		/* stuff header */
		if (change[0]) strcpy(head.username,new.username);
		if (change[1]) strcpy(head.source,new.source);
		if (change[2]) strcpy(head.dbase,new.dbase);
		if (change[3]) strcpy(head.speaker,new.speaker);
		if (change[4]) strcpy(head.session,new.session);
		if (change[5]) strcpy(head.sessdate,new.sessdate);
		if (change[6]) strcpy(head.token,new.token);
		if (change[7]) strcpy(head.rep,new.rep);
#ifdef WIN32
		if (change[8]) strcpy(head.environment,new.environment);
#else
		if (change[8]) strcpy(head.environ,new.environ);
#endif
		if (change[9]) strcpy(head.archive,new.archive);
		if (change[10]) strcpy(head.comment,new.comment);

		/* now matches machine */
		head.machine = SFSMACHINE;

		/* write header back */
		if (sfsopen(filename,"h",&head) != 0)
			error("write error on %s",filename);
	}
	else if (questions) {
		/* edit header */
		if (edithead())
			error("edit abandoned",NULL);
		else {
			/* now matches machine */
			head.machine = SFSMACHINE;

			/* write header back */
			if (sfsopen(filename,"h",&head) != 0)
				error("write error on %s",filename);
		}
	}

	exit(0);
}

/* edit header data */
int edithead()
{
	int	current,next;
	char	ans[10];

	printf("SPEECH FILE HEADER EDITOR\n\n");
	printf("Fill in the details below.\n");
	printf("\tType <details><RETURN> to enter details.\n");
	printf("\tType <RETURN> to skip question.\n");
	printf("\tType !<number> to jump to question.\n");
	printf("\tType <SPACE><RETURN> to delete an answer.\n");

	/* get in speaker and datafile details */
	do {
		current = 1;
		next = 1;
		while ((current > 0) && (current <= 9)) {
			printf("\n");
			switch (current) {
			case 1:	if ((next = getfield(current,"source of recording",head.source,20)) != (current+1) ) break;
				current++;
			case 2:	if ((next = getfield(current,"name of database",head.dbase,20)) != (current+1) ) break;
				current++;
			case 3:	if ((next = getfield(current,"name of speaker",head.speaker,20)) != (current+1) ) break;
				current++;
			case 4:	if ((next = getfield(current,"session ref",head.session,20)) != (current+1) ) break;
				current++;
			case 5:	if ((next = getfield(current,"session date",head.sessdate,20)) != (current+1) ) break;
				current++;
			case 6:	if ((next = getfield(current,"token",head.token,160)) != (current+1) ) break;
				current++;
			case 7:	if ((next = getfield(current,"repetition",head.rep,8)) != (current+1) ) break;
				current++;
#ifdef WIN32
			case 8:	if ((next = getfield(current,"recording conditions",head.environment,8)) != (current+1) ) break;
#else
			case 8:	if ((next = getfield(current,"recording conditions",head.environ,8)) != (current+1) ) break;
#endif
				current++;
			case 9:	if ((next = getfield(current,"comment",head.comment,80)) != (current+1) ) break;
				current++;
			}
			current = next;
		}

		printf("\n");
		do {
			printf("Is the data correct (y/n/q) ? ");
			if (getline(ans,5)==NULL) ans[0]='q';
		} while ( (ans[0] != 'y') && (ans[0] != 'n') && (ans[0] != 'q') );

	} while ((ans[0] != 'y') && (ans[0] != 'q'));

	if (ans[0] == 'q') return(1); else return(0);
}

/* get field */
int getfield(num,prompt,s,len)
int num,len;
char *prompt,*s;
{
	char ans[100];

	printf("%d) %s [%s] : ",num,prompt,s);
	if (getline(ans,len)==NULL) return(0);

	if (ans[0] == '\0') return(num+1);
	if (ans[0] == '!') return(atoi(&ans[1]));

	if ((ans[0] == ' ') && (ans[1] == '\0'))
		strcpy(s,"");
	else
		strcpy(s,ans);
	return(num+1);
}

