/* sdump -- dump contents of database file */

/*
 m.a. huckvale - may 1985

 version 2
 version 2.1 - Release 1, new format SY, CO, FM & PC items
 version 2.2 - improved dump format
 version 2.3 - new WD item
 version 2.4 - new history access
 version 2.5 - December 1986
	- zap leading digits in TERMCHAR string
	- use SPARPATH

 version 2.6  -  EN and VU items added
 version 2.7  - TR item added
 version 3.0 - SFS routines
 version 3.1 - fix -f flag
 version 3.2 - new di_rec
 version 3.3 - RR and RP items added
 version 3.4 - new fields in link_header
 version 3.5 - print main header only with -h flag
 version 3.6 - add UG and XM items
 version 3.7 - add CA, GE, AE, IP and SC items
 version 3.8 - add AR item
 version 4.0 - better formatting of bold/italic, add first XML output
 version 4.1 - add support for lastposn
*/
/*--------------------------------------------------------------------------*/
/**MAN
.TH SDUMP SFS1 UCL SFS
.SH NAME
sdump - dump contents of database file
.SH SYNOPSIS
.B sdump
(-I) (-i item) (-a item) (-f) (-h|-H) (-T) (-v) (-F) file
.SH DESCRIPTION
.I sdump
is a program to give a listing of the contents of database files in a fairly
unprocessed form.  Default operation is to list all item headers and to
summarise data sets using the first few frames.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select first item matching item specification.
.TP 11
.BI -a type
Select all items matching item specification.
.TP 11
.B -f
dump selected items in full, rather than the first few frames.
.TP 11
.B -h
dump main header only.
.TP 11
.B -H
dump main header only in XML format.
.TP 11
.B -T
Force terminal output video enhancement (default is use video enhancement
only when talking directly to a terminal).
.TP 11
.B -v
Follow links to virtual items, rather than simply reporting them.
.TP 11
.B -F
Report filename at head of report.
.SH VERSION/AUTHOR
4.0 - Mark Huckvale
.SH BUGS
Assumes preferred format in some cases.
*/
/*--------------------------------------------------------------------------*/

/* program version */
#define PROGNAME "sdump"
#define PROGVERS "4.1"
char *progname=PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
#ifdef DOS
#include <io.h>
#endif
#include "sfs.h"
#include "sfsdata.h"

/* global data */
int 	full=0;			/* everything in full */
int 	mainonly=0;		/* main header only */
int 	pagemode=0;		/* "page" mode */
int	follow=0;		/* follow links */
int	lineno=0;		/* line no on page */
int	colno=0;		/* column no on line */

/* item selection data */
#define MAXSELECT 20
struct {
	int	it;
	int	ty;
	char	*match;
	int	all;
} itab[MAXSELECT];
int itcnt=0;

/* item format control */
struct dumpstruct{
	int	numf;
	int	width;
	char	*form;
} dumpformat[]={
	{ 0,	0,	"" },
	{ 50,	10,	"%6d " },	/* SP */
	{ 50,	10,	"%6d " },	/* LX */
	{ 25,	5,	"%8d " },	/* TX */
	{ 50,	10,	"%6d " },	/* FX */
	{ 5,	1,	"" },		/* AN */
	{ 5,	1,	"" },		/* LP */
	{ 5,	1,	"%5d " },	/* SY */
	{ 5,	1,	"" },		/* WD */
	{ 5,	1,	"" },		/* DI */
	{ 5,	1,	"" },		/* EN */
	{ 5,	1,	"" },		/* CO */
	{ 5,	1,	"" },		/* FM */
	{ 5,	1,	"" },		/* VU */
	{ 5,	1,	"" },		/* PC */
	{ 5,	1,	"" },		/* MM */
	{ 25,	5,	"%13g " },	/* TR */
	{ 5,	5,	"" },		/* 17 */
	{ 3,	5,	"" },		/* GE */
	{ 5,	5,	"" },		/* AE */
	{ 5,	5,	"" },		/* IP */
	{ 50,	10,	"%7.3f" },	/* SC */
	{ 50,	10,	"%6d " },	/* PH */
	{ 5,	1,	"" },		/* RP */
	{ 5,	1,	"" },		/* RR */
	{ 50,	10,	"%6d" },	/* UG */
	{ 50,	10,	"%13g " },	/* XM */
	{ 5,	1,	"" },		/* NA */
	{ 5,	1,	"" },		/* CA */
	{ 50, 10,	"%6d" }		/* AR */
};
#define NDUMPFORMAT (sizeof(dumpformat)/sizeof(struct dumpstruct))

#if defined(SVR4) || defined(SOLARIS)
#define getdate sfs_getdate
#endif

#ifdef __STDC__
void reportmain(char *,struct main_header *);
void reportmainXML(char *,struct main_header *);
void reportitem(int,struct item_header *,int,char *);
void reportlink(int);
char *getdate(int32);
int ttyin(void);
void dmpframe(struct item_header *,unsigned char *);
int itrequired(struct item_header *,char *,int);
#else
void reportmain();
void reportmainXML();
void reportitem();
void reportlink();
char *getdate();
int ttyin();
void dmpframe();
int itrequired();
#endif

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* opion decoding variables */
	extern int	optind;
	extern char	*optarg;
	int		errflg=0;
	int		c;
	/* file variables */
	int		dbfil;
	char		filename[SFSMAXFILENAME];
	/* processing variables */
        int             cnt=0;
	int		pass1=0;
	char		*exphist;
	struct main_header head;
	struct item_header item;
	int		dofname=0;
	time_t	tim;

	initvideo(0);

	/* decode switches */
	while ( (c = getopt(argc,argv,"IfhHpi:a:TvF")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Dump SFS file contents V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'f' :	/* full dump */
			full++;
			break;
		case 'h' :	/* main header only */
			mainonly=1;
			break;
		case 'H' :	/* main header onl in XML */
			mainonly=2;
			break;
		case 'p' :	/* page mode */
			pagemode++;
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&itab[itcnt].it,&itab[itcnt].match) != 0)
				error("bad 'i' option : %s",optarg);
			if (strcmp(itab[itcnt].match,"0")==0)
				pass1++;
			itcnt++;
			break;
		case 'a' :	/* generic item */
			if (itspec(optarg,&itab[itcnt].it,&itab[itcnt].match) != 0)
				error("bad 'a' option : %s",optarg);
			if (strcmp(itab[itcnt].match,"0")==0)
				itab[itcnt].match = "*";
			itab[itcnt].all=1;
			itcnt++;
			break;
		case 'T' :	/* force terminal output */
			initvideo(1);
			break;
		case 'v' :	/* follow links */
			follow++;
			break;
		case 'F':
			dofname++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s [-I] | [-h|-H] | [(-f) (-v) (-i item) (-a item)] (-T) (-F) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* check local file */
	if (strcmp(filename,argv[optind]) != 0) {
		videoprint("dump of %s:\n",filename);
		lineno++;
	}

	/* open datafile */
	if ((dbfil = sfsopen(filename,"r",&head)) < 0) {
		if (dbfil==-1)
			error("cannot open %s",filename);
		else
			error("access error on %s",filename);
	}

	/* report filename */
	if (dofname) {
		tim = time((time_t *)0);
		printf("Dump of \"%s\" on %s",filename,ctime(&tim));
	}

	/* report main header */
	if (mainonly) {
		if (mainonly==1)
			reportmain(filename,&head);
		else
			reportmainXML(filename,&head);
		exit(0);
	}

	/* first pass for specially selected items */
	if (pass1) {
	        while (sfsnextitem(dbfil,&item)) {
			exphist = lab_store(&item);
			itrequired(&item,exphist,0);
		}
		close(dbfil);
	        dbfil = sfsopen(filename,"r",NULL);
	}

	/* report selected items */
        while (sfsnextitem(dbfil,&item)) {
		exphist = lab_store(&item);
		cnt++;
		if ((itcnt==0) || itrequired(&item,exphist,1))
	                /* report item */
        	        reportitem(dbfil,&item,cnt,exphist);
	}
	close(dbfil);

	/* that's all folks */
        exit(0);
}

/* print bold text */
void boldfield(s,n)
char	*s;	/* string to print */
int	n;	/* field width */
{
	boldon();
	videoprint("%s",s);
	boldoff();
	if (n) videoprint("%*s",(int)(n-strlen(s)),"");
}

/* print italic text */
void italicfield(s,n)
char	*s;	/* string to print */
int	n;	/* field width */
{
	italicon();
	videoprint("%s",s);
	italicoff();
	if (n) videoprint("%*s",(int)(n-strlen(s)),"");
}

/* print newline checking for end of page */
void newline()
{
	char	ans[80];
	videoprint("\n");
	colno=0;
	lineno++;
	if (pagemode && (lineno==24) && ttytest()) {
		italicon();
		videoprint("--More--");
		italicoff();
		fflush(stdout);
		fgets(ans,80,stdin);
		uparrow();
		videoprint("\r        \r");
		lineno=0;
	}
}

/* print a single value checking for 80 columns */
#ifdef __STDC__
#include <stdarg.h>
void datumprint(char *fmt, ...)
{
	char	buffer[1024];
	va_list	params;
	int	len;

	/* print message */
	va_start(params, fmt);
	len = vsprintf(buffer, fmt, params);
	va_end(params);
	if ((colno+len) >= 80) newline();
	videoprint(buffer);
	colno += len;
}
#else
#define datumprint printf
#endif

/* report main header */
void reportmain(fname,head)
char *fname;
struct main_header *head;
{
	char	field[80];

	boldfield("Main",0);
	newline();
	videoprint("File id        : ");
	italicfield(head->fid.site,0);
	videoprint("-");
	sprintf(field,"%d",head->fid.num);
	italicfield(field,0);
	videoprint(" created ");
	italicfield(getdate(head->creatdate),0);
	videoprint(" by ");
	italicfield(head->username,0);
	newline();
	videoprint("Database       : ");
	italicfield(head->dbase,20);
	videoprint("Speaker        : ");
	italicfield(head->speaker,0);
	newline();
	videoprint("Session        : ");
	italicfield(head->session,20);
	videoprint("Session Date   : ");
	italicfield(head->sessdate,0);
	newline();
	videoprint("Repetition     : ");
	italicfield(head->rep,20);
	videoprint("Environment    : ");
#ifdef WIN32
	italicfield(head->environment,0);
#else
	italicfield(head->environ,0);
#endif
	newline();
	videoprint("Token          : ");
	italicfield(head->token,0);
	newline();
	videoprint("Comment        : ");
	italicfield(head->comment,0);
	newline();
}

/* report main header in XML */
void reportmainXML(fname,head)
char *fname;
struct main_header *head;
{
	char	basename[SFSMAXFILENAME];
	char	*p;

	printf("<?xml version=\"1.0\"?>\n");
	strcpy(basename,fname);
	p = strrchr(basename,'/');
	if (p) strcpy(basename,p+1);
	p = strrchr(basename,'\\');
	if (p) strcpy(basename,p+1);
	p = strrchr(basename,'.');
	if (p) *p='\0';
	for (p = basename;*p;p++) *p = toupper(*p);
	printf("<SFSFILE ID=\"%s\">\n",basename);
	printf("<FILE HREF=\"%s\"/>\n",fname);
	if (head->source[0])
		printf("<SOURCE>%s</SOURCE>\n",head->source);
	if (head->dbase[0])
		printf("<DBASE>%s</DBASE>\n",head->dbase);
	if (head->speaker[0])
		printf("<SPEAKER>%s</SPEAKER>\n",head->speaker);
	if (head->session[0])
		printf("<SESSION>%s</SESSION>\n",head->session);
	if (head->sessdate[0])
		printf("<DATE>%s</DATE>\n",head->sessdate);
	if (head->rep[0])
		printf("<REP>%s</REP>\n",head->rep);
#ifdef WIN32
	if (head->environment[0])
		printf("<ENVIRON>%s</ENVIRON>\n",head->environment);
#else
	if (head->environ[0])
		printf("<ENVIRON>%s</ENVIRON>\n",head->environ);
#endif
	if (head->token[0])
		printf("<TOKEN>%s</TOKEN>\n",head->token);
	printf("</SFSFILE>\n");
}

/* report item header */
void reportitem(fid,item,num,exphist)
int		fid;
struct item_header *item;
int		num;
char		*exphist;
{
	extern	char	*sfsname[];
	char	*label;
	char	field[80];
	int	i;
	char	*buff;

	newline();
	sprintf(field,"Item %d",num);
	boldfield(field,0);
	newline();
	videoprint("Data Type      : ");
	sprintf(field,"%d.%02d %s",
		abs(item->datatype),item->subtype,
		sfsname[abs(item->datatype)]);
	italicfield(field,0);
	if ((label=lab_gettext(exphist)) == NULL)
		newline();
	else {
		videoprint(" (%s)",label);
		newline();
	}

	videoprint("History        : ");
	italicfield(item->history,0);
	newline();
	videoprint("Parameters     : ");
	italicfield(item->params,0);
	newline();
	videoprint("Process Date   : ");
	italicfield(getdate(item->processdate),0);
	newline();
	videoprint("Format         : ");
	if (item->floating > 0)
		sprintf(field,"%d byte floating",item->datasize);
	else if (item->floating == 0)
		sprintf(field,"%d byte integer",item->datasize);
	else if (item->floating < 0)
		sprintf(field,"%d byte structured",item->datasize);
	italicfield(field,0);
	newline();
	videoprint("Frame size     : ");
	sprintf(field,"%d",item->framesize);
	italicfield(field,20);
	videoprint("Frame count    : ");
	sprintf(field,"%d",item->numframes);
	italicfield(field,0);
	newline();
	videoprint("Total Length   : ");
	sprintf(field,"%d",item->length);
	italicfield(field,20);
	videoprint("Frame Duration : ");
	if (item->frameduration==0.0)
		italicfield("undefined",0);
	else {
		sprintf(field,"%g (%d Hz)",
			item->frameduration,
			(int) (0.5+1.0/item->frameduration));
		italicfield(field,0);
	}
	newline();
	if (item->lxsync) {
		videoprint("Window size    : ");
		italicfield("synchronous",20);
	}
	else {
		videoprint("Window size    : ");
		sprintf(field,"%d",item->windowsize);
		italicfield(field,20);
		videoprint("Overlap        : ");
		sprintf(field,"%d",item->overlap);
		italicfield(field,0);
	}
	newline();
	videoprint("Offset         : ");
	sprintf(field,"%g",item->offset);
	italicfield(field,20);
	videoprint("Last Position  : ");
	sprintf(field,"%d",item->lastposn);
	italicfield(field,0);
	newline();
	item->comment[19] = '\0';
	videoprint("Comment        : ");
	italicfield(item->comment,0);
	newline();
	if (item->datatype<0) {
		videoprint("Data           : ");
		italicfield(" * DELETED * ",0);
		newline();
		return;
	}
	else if (item->length <= 0) {
		videoprint("No Data");
		newline();
		return;
	}
	if (item->datapresent==2) {
		if (!follow) {
			reportlink(fid);
			return;
		}
		else
			videoprint("Linked ");
	}
	if (full) dumpformat[item->datatype].numf=item->numframes;
	if (item->numframes > dumpformat[item->datatype].numf)
		videoprint("Data Starts:");
	else
		videoprint("Data:");
	newline();
	if ((buff=sfsbuffer(item,1))==NULL)
		fprintf(stderr,"could not get data buffer\n");
	else {
		i=0;
		while ((i<dumpformat[item->datatype].numf) &&
			sfsread(fid,i++,1,buff)) {
			dmpframe(item,buff);
			if ((item->framesize > 1) ||
			    ((i % dumpformat[item->datatype].width)==0))
				newline();
		}
		free(buff);
	}
	newline();
}

/* report link */
void reportlink(fid)
int	fid;
{
	struct link_header link;
	char	field[80];

	/* read link header */
	lseek(fid,sfsdata[fid]->datastart,0);
	read(fid,&link,sizeof(struct link_header));
	sfsdata[fid]->currpos = lseek(fid,0L,1);

	/* report contents */
	videoprint("Linked item    :"); newline();
	videoprint("\t\tFilename  : "); italicfield(link.filename,0); newline();
	videoprint("\t\tFilepath  : "); italicfield(link.filepath,0); newline();
	sprintf(field,"%d",link.filetype);
	videoprint("\t\tFiletype  : "); italicfield(field,0); newline();
	sprintf(field,"%d.%02d",link.datatype,link.subtype);
	videoprint("\t\tItem no.  : "); italicfield(field,0); newline();
	sprintf(field,"%d",link.offset);
	videoprint("\t\tOffset    : "); italicfield(field,0); newline();
	sprintf(field,"%d",link.multiplex);
	videoprint("\t\tMultiplex : "); italicfield(field,0); newline();
	videoprint("\t\tByte swap : "); italicfield((link.swab)?"yes":"no",0); newline();
	sprintf(field,"%d",link.dcoffset);
	videoprint("\t\tDC offset : "); italicfield(field,0); newline();
	sprintf(field,"%d",link.shift);
	videoprint("\t\tBit shift : "); italicfield(field,0); newline();
	videoprint("\t\tDate      : "); italicfield(getdate(link.linkdate),0);
	newline();
	newline();
}

/* return processed date */
char *getdate(tim)
int32 tim;
{
	static char buf[26];
	time_t	t=(time_t)tim;

	strncpy(buf,ctime(&t),24);

	return(buf);
}

void dmpframe(item,buff)
struct item_header *item;
unsigned char	*buff;
{
	int		i,fsize,dsize,ncoeff,npole;
	char		*cp;
	short		*sp;
	int32		*ip;
	float		*fp;
	struct lp_rec	*lp;
	struct an_rec	*an;
	struct wd_rec	*wd;
	struct co_rec	*co;
	struct fm_rec	*fm;
	struct di_rec	*di;
	struct rp_rec	*rp;
	struct rr_rec	*rr;
	struct xm_rec	*xm;
	GEO_ITEM		*geo;
	AERO_ITEM		*aero;
	INTER_ITEM		*inter;
	CAL_ITEM		*calib;

	fsize = item->datasize * item->framesize;
	dsize = item->datasize;

	if (item->framesize==1) {
		switch (item->datasize) {
		case 1:
			cp = (char *) buff;
			datumprint(dumpformat[item->datatype].form,*cp);
			break;
		case 2:
			sp = (short *) buff;
			datumprint(dumpformat[item->datatype].form,*sp);
			break;
		case 4:
			switch (item->floating) {
			case 0:
				ip = (int32 *) buff;
				datumprint(dumpformat[item->datatype].form,*ip);
				break;
			case 1:
				fp = (float *) buff;
				datumprint(dumpformat[item->datatype].form,*fp);
				break;
			}
			break;
		}
	}
	else {
		ncoeff = (fsize-sfsstruct[item->datatype])/dsize;
		switch (item->datatype) {
		case TX_TYPE:
			ip = (int32 *)buff;
			for (i=0;i<item->framesize;i++)
				datumprint(dumpformat[TX_TYPE].form,*ip++);
			break;
		case AN_TYPE:
			an = (struct an_rec *) buff;
			datumprint("posn=%d size=%d label=%s",
				an->posn,
				an->size,
				an->label);
			break;
		case LP_TYPE:
			lp = (struct lp_rec *) buff;
			datumprint("sname=%s length1=%d pitch1=%d length2=%d pitch2=%d alist=",
				lp->sname,
				lp->length1,
				lp->pitch1,
				lp->length2,
				lp->pitch2);
			for (i=0;i<ncoeff;i++)
				datumprint("%d ",lp->alist[i]);
			break;
		case SY_TYPE:
			sp = (short *) buff;
			for (i=0;i<item->framesize;i++) datumprint("%d ",sp[i]);
			break;
		case WD_TYPE:
			wd = (struct wd_rec *) buff;
			datumprint("start=%d end=%d score=%g symbol='%s' attr='%s'",
				wd->start,
				wd->end,
				wd->score,
				wd->symbol,
				wd->attr);
			break;
		case DI_TYPE:
			di = (struct di_rec *) buff;
			datumprint("posn=%d size=%d pixel=",di->posn,di->size);
			for (i=0;i<ncoeff;i++)
				datumprint("%d ",di->pixel[i]);
			break;
		case VU_TYPE:
		case EN_TYPE:
		case CO_TYPE:
		case PC_TYPE:
			co = (struct co_rec *) buff;
			datumprint("posn=%d size=%d flag=%d mix=%g gain=%g data=",
				co->posn,
				co->size,
				co->flag,
				co->mix,
				co->gain);
			for (i=0;i<ncoeff;i++)
				datumprint("%g ",co->data[i]);
			break;
		case FM_TYPE:
			fm = (struct fm_rec *) buff;
			datumprint("posn=%d size=%d flag=%d gain=%g npeaks=%d ",
				fm->posn,
				fm->size,
				fm->flag,
				fm->gain,
				fm->npeaks);
			for (i=0;i<ncoeff/3;i++)
				datumprint("(%g %g %g) ",
					fm->formant[i].freq,
					fm->formant[i].amp,
					fm->formant[i].band);
			break;
		case RP_TYPE:
			rp = (struct rp_rec *) buff;
			datumprint("posn=%d size=%d gain=%g delay=%d : 1",
			    rp->posn,rp->size,
			    rp->gain,rp->delay);
			fp = rp->data;
			for (i=rp->nzero; i>0; i--) datumprint(" %g",*fp++);
			i = ncoeff - rp->nzero;
			if (i) {
			    datumprint(" / 1");
			    for (; i>0; i--)
				datumprint(" %g",*fp++);
			    }
			break;
		case GE_TYPE:
			geo = ( GEO_ITEM * ) buff;
			datumprint("ag=%d av=%d q=%6.2f plu=%6.2f vc=%6.2f ac=%5.2f",
				geo->ag,
				geo->av,
				geo->q,
				geo->plu,
				geo->vc,
				geo->ac );
			newline();
			for ( i = 0 ; i < ALT_RECL ; i++ )
				datumprint( "%d ", geo->areas[ i ] );
			break;
		case AE_TYPE:
			aero = ( AERO_ITEM * ) buff;
			datumprint("psg=%f pc=%f pdiff=%f ug=%f uc=%f uc=%f vlu=%4.0f",
				aero->psg,
				aero->pc,
				aero->pdiff,
				aero->ug,
				aero->uc,
				aero->uc,
				aero->vlu );
			break;
		case IP_TYPE:
			inter = ( INTER_ITEM * ) buff;
			datumprint("voia=%f voif=%f frica=%f frcvpa=%f aspa=%f tcr=%f td=%f k=%f",
				inter->voia,
				inter->voif,
				inter->frica,
				inter->frcvpa,
				inter->aspa,
				inter->tcr,
				inter->td,
				inter->k );
			break;
		case RR_TYPE:
			rr = (struct rr_rec *) buff;
			datumprint("posn=%d size=%d gain=%g delay=%d",
			    rr->posn,rr->size,
			    rr->gain,rr->delay);
			fp = rr->data;
			if (rr->nzero) {
			    datumprint(" Zeros:");
			    for (i=rr->nzpair; i>0; i--) {
				datumprint(" (%g,+-%g)",fp[0],fp[1]);
				fp += 2;
				}
			    for (i=rr->nzero-2*rr->nzpair; i>0; i--) {
				datumprint(" %g",*fp++);
				}
			    }
			npole = ncoeff - rr->nzero;
			if (npole) {
			    datumprint(" Poles:");
			    for (i=rr->nppair; i>0; i--) {
				datumprint(" (%g,+-%g)",fp[0],fp[1]);
				fp += 2;
				}
			    for (i=npole-2*rr->nppair; i>0; i--) {
				datumprint(" %g",*fp++);
				}
			    }
			break;

		case XM_TYPE:
			xm = (struct xm_rec *) buff;
			datumprint("start=%g,length=%g,amp=%g,mix=%g,params=", xm->start,xm->length,xm->amp,xm->mix);
			fp = xm->data;
			for (i=ncoeff; i>0; i--) datumprint("%g ",*fp++);
			break;

		case CA_TYPE :
			calib = (CAL_ITEM *) buff;
			datumprint( "Units: %s", calib->units ); newline();
			datumprint( "Simple scale: %f, residual %f", calib->slope1, calib->resid1 );
			newline();
			datumprint( "Straight line: const %f, slope %f, residual %f",
				calib->const2, calib->slope2, calib->resid2 );
			newline();
			datumprint( "Quadratic fit: %f, %f, %f,  residual %f",
				calib->coeff3[ 0 ], calib->coeff3[ 1 ] , calib->coeff3[ 2 ], calib->resid3 );
			newline();
			datumprint( "Calibration range from %f to %f", calib->min, calib->max );
			newline();
			break;
		default:
			datumprint("unknown framed data");
		}
	}
}

/* check if item required */
int itrequired(item,hist,flag)
struct item_header 	*item;
char			*hist;
int			flag;		/* 0=first pass; 1=second pass */
{
	int	i,ty,req=0;
	char	junk[16];
	for (i=0;i<itcnt;i++) if ((itab[i].it==0) || (itab[i].it==item->datatype)) {
		if (itab[i].ty==item->subtype)
			req++;
		else if (isdigit(itab[i].match[0]) || (itab[i].match[0]=='-')) {
			ty=atoi(itab[i].match);
			if ((ty==-1) || (ty==item->subtype)) {
				if (itab[i].all==0) itab[i].ty=item->subtype;
				req++;
			}
			else if ((ty==0) && (flag==0)) {
				itab[i].it=abs(item->datatype);
				itab[i].ty=item->subtype;
			}
		}
		else if (itab[i].ty==0) {
			if (histmatch(itab[i].match,hist,junk,junk,junk,junk,junk,junk,junk,junk,junk,junk)==1) {
				if (itab[i].all==0) itab[i].ty=item->subtype;
				req++;
			}
		}
	}
	return(req);
}

#if 0

/* old page mode code */

#include <fcntl.h>
int ttyin()
{
	char	c;
	int	i;
	static int tty = -1;
	if (tty == -1) tty = open("/dev/tty",O_RDONLY);
	read(tty,&c,1);
	i = c;
	while (c != '\n') read(tty,&c,1);
	return(i);
}

		if (req) {
			/* report item header */
			reportitem(cnt);

			/* stop at item boundaries in page mode */
			pflag=0;
			if (pagemode) {
				datumprint("%s--More--%s",so,se);
				fflush(stdout);
				c = ttyin();
				datumprint("%s\r                          \r",up);
				fflush(stdout);
				switch (c) {
				case 'q':
					fclose(dbfil);
					exit(0);
					break;
				case 's':
					pflag = -1;
					break;
				case 'f':
					pflag = 1;
					break;
				}
			}

			/* dump data accordingly */
			if (pflag==1) {
				oldfull=full;
				full=1;
			}
			if ((pflag>=0) && (item.length > 0)) {
				if ((buff=sfsbuffer(&item,1))==NULL)
					fprintf(stderr,"could not get data buffer\n");
				else {
					i=0;
					while ((i<dumpformat[abs(item.datatype)].numf) &&
						sfsread(fid,i++,1,buff))
						dmpframe(item,buff);
				}
			}
			if (pflag==1) full=oldfull;
#endif
