/* sfs2wav -- convert an SFS waveform to .WAV (RIFF) format */

/* M.A.Huckvale - University College London */

/* version 1.0 - July 1994 */

/* version 1.1 - March 1995
	- add time sections
	- add stereo output
*/
	
#define PROGNAME "sfs2wav"
#define PROGVERS "1.1"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH SFS2WAV 1 UCL
.SH NAME
sfs2wav - convert waveforms to Microsoft .WAV format (RIFF)
.SH SYNOPSIS
.B sfs2wav
(-i item) (-l item) (-r item) (-2) (-8) (-12|-a) (-s starttime) (-e endtime) (-o wavfile) file
.SH DESCRIPTION
.I sfs2wav
is a program to copy data from speech, and lx items into Microsft .WAV format files.
The byte order is selected for operation on PCs.  If no output filename is given
the output name is constructed from the source name by substituting an extension
of '.WAV'.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item. (monophonic)
.TP 11
.BI -l item
Select left channel (stereophonic).
.TP 11
.BI -r item
Select right channel (stereophonic)
.TP 11
.B -2
Make stereo from single input channel.
.TP 11
.BI -o wavfile
name the file to create in .WAV format.  Default is to use the input file
name with a .WAV extension.
.TP 11
.B -a
Automatic gain control.  Scale the output waveform so that the maximum waveform
amplitude is set to +/- 24576.
.TP 11
.B -8
Create an 8 bit waveform from the top 8 bits of the input waveform.
Default is to create a 16-bit waveform.
.TP 11
.B -12
Specify that the input waveform is only 12 bits.  Scale up by 4 bits
before using.
.TP 11
.BI -s starttime
Specify start time to extract.  Default: 0.0.
.TP 11
.BI -e endtime
Specify end time for extraction.  Default: end of input.
.SH INPUT ITEMS
.IP 1.xx 11
Any speech item.
.IP 2.xx 11
Any excitation item.
.SH VERSION/AUTHOR
.IP 1.0 11
Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include "sfs.h"

#define BUFSIZE		16384

/* global data */
struct item_header	item;
short			*sig;
struct item_header	item2;
short			*sig2;

/* program options */
int		do8bit=0;
int		doscale12=0;
int		doagc=0;
int		dostereo=0;
int		domakestereo=0;
double		starttime = -1;
double		endtime = -1;

char		wavname[SFSMAXFILENAME];

struct riff_file_header {
	char		riffid[4];	/* "RIFF" */
	unsigned long	size;		/* length of RIFF block */
} riffhead;

struct riff_wave_header {
	char		waveid[4];	/* "WAVE" */
	char		fmtid[4];	/* "fmt " */
	unsigned long	size;		/* 16 */
	unsigned short	format;		/* 1 */
	unsigned short	nchan;		/* 1 or 2 */
	unsigned long	srate;		/* srate */
	unsigned long	brate;		/* srate * #bytes/samp * nchan */
	unsigned short	balign;		/* 1 or 2 or 4 */
	unsigned short	nbits;		/* 8 or 16 */
} riffwavehead;

struct riff_data_header {
	char		dataid[4];	/* "data" */
	unsigned long	size;		/* length of signal in bytes */
} riffdatahead;

#define REVERSESHORT(s)	s = ((s & 0xFF00) >> 8) | ((s & 0x00FF) << 8);
#define REVERSELONG(l)	l = ((l & 0xFF000000L) >> 24) | ((l & 0x00FF0000L) >> 8) | ((l & 0x0000FF00L) << 8) | ((l & 0x000000FFL) << 24);

void RIFFheader(op,nsample,nchan)
FILE	*op;
int	nsample;
int	nchan;
{
	strcpy(riffhead.riffid,"RIFF");
	if (do8bit)
		riffhead.size = sizeof(struct riff_wave_header) +
				sizeof(struct riff_data_header) +
				nsample * nchan;
	else
		riffhead.size = sizeof(struct riff_wave_header) +
				sizeof(struct riff_data_header) +
				nsample * 2 * nchan;
#if SFSMACHINE==0
	REVERSELONG(riffhead.size);
#endif
	if (fwrite((char *)&riffhead,sizeof(struct riff_file_header),1,op)!=1)
		error("write error on '%s'",wavname);
}

void RIFFwaveheader(op,nsample,srate,nchan)
FILE	*op;
int	nsample;
double	srate;
int	nchan;
{
	strcpy(riffwavehead.waveid,"WAVE");
	strcpy(riffwavehead.fmtid,"fmt ");
	riffwavehead.size = 16;
	riffwavehead.format = 1;
	riffwavehead.nchan = nchan;
	riffwavehead.srate = (long)srate;
	riffwavehead.brate = (long)srate * ((do8bit)?1:2) * nchan;
	riffwavehead.balign = ((do8bit) ? 1 : 2)*nchan;
	riffwavehead.nbits = (do8bit) ? 8 : 16;

#if SFSMACHINE==0
	REVERSELONG(riffwavehead.size);
	REVERSESHORT(riffwavehead.format);
	REVERSESHORT(riffwavehead.nchan);
	REVERSELONG(riffwavehead.srate);
	REVERSELONG(riffwavehead.brate);
	REVERSESHORT(riffwavehead.balign);
	REVERSESHORT(riffwavehead.nbits);
#endif

	if (fwrite((char *)&riffwavehead,sizeof(struct riff_wave_header),1,op)!=1)
		error("write error on '%s'",wavname);

}

void RIFFdataheader(op,nsample,nchan)
FILE	*op;
int	nsample;
int	nchan;
{
	strcpy(riffdatahead.dataid,"data");
	if (do8bit)
		riffdatahead.size = nsample * nchan;
	else
		riffdatahead.size = nsample * 2 * nchan;
#if SFSMACHINE==0
	REVERSELONG(riffdatahead.size);
#endif
	if (fwrite((char *)&riffdatahead,sizeof(struct riff_data_header),1,op)!=1)
		error("write error on '%s'",wavname);
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* option decoding */
	extern int	optind;
	extern char	*optarg;
	int		errflg=0;
	int		c;
	int		it;		/* selected item type */
	char		*ty;		/* item specification */
	int		initem=SP_TYPE;
	char		*intype="0";
	int		initem2=SP_TYPE;
	char		*intype2="*";
	int		nitem=0;
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* data file */
	int		fid,fid2=0;
	FILE		*op;
	int		len;
	int		i,j;
	int		maxval;
	double		scale=1;
	char		*p;
	int		startframe,numframes;

	/* decode switches */
	while ((c = getopt(argc,argv,"Ii:l:r:o:81:a2s:e:")) != EOF) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Convert SFS to WAV format V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'o' :	/* output name */
			strcpy(wavname,optarg);
			break;
		case 'a' :	/* automatic gain */
			doagc++;
			doscale12=0;
			break;
		case '1' :	/* 12 bit input */
			if (*optarg=='2')
				doscale12++;
			else
				error("unknown switch -1%s",optarg);
			doagc=0;
			break;
		case '8' :	/* 8 bit output */
			do8bit++;
			break;
		case '2' :	/* fake stereo */
			domakestereo=1;
			dostereo=0;
			break;
		case 'i':	/* input item specification */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it == SP_TYPE) ||
				    (it == LX_TYPE)) {
				    	if (nitem==0) {
						initem = it;
						intype = ty;
						dostereo=0;
						nitem=1;
					}
					else {
						initem2 = it;
						intype2 = ty;
						dostereo=1;
					}
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'l':	/* input item specification - left channel */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it == SP_TYPE) ||
				    (it == LX_TYPE)) {
					initem = it;
					intype = ty;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			dostereo=1;
			domakestereo=0;
			break;
		case 'r':	/* input item specification - right channel */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it == SP_TYPE) ||
				    (it == LX_TYPE)) {
					initem2 = it;
					intype2 = ty;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			dostereo=1;
			domakestereo=0;
			break;
		case 's' :
			starttime = atof(optarg);
			break;
		case 'e' :
			endtime = atof(optarg);
			break;
		default:	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-l item) (-r item) (-2) (-o outfile) (-8) (-12|-a) (-s start) (-e end) sfsfile",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* create output file name */
	if (wavname[0]=='\0') {
		strcpy(wavname,filename);
		if ((p=strrchr(wavname,'.')))
			strcpy(p,".wav");
		else
			strcat(wavname,".wav");
	}

	/* open sfs file */
	if ((fid=sfsopen(filename,"r",NULL)) < 0)
		error("access error on '%s'",filename);
	if (dostereo) fid2 = sfsdup(fid);

	/* locate input item */
	if (!sfsitem(fid,initem,intype,&item))
		error("could not find input item in '%s'",filename);
	if (dostereo) {
		if (!sfsitem(fid2,initem2,intype2,&item2))
			error("could not find input item in '%s'",filename);
	}

	/* get buffer */
	if ((sig=(short *)sfsbuffer(&item,BUFSIZE))==NULL)
		error("unable to get memory buffer");
	if (dostereo) {
		if ((sig2=(short *)sfsbuffer(&item2,BUFSIZE))==NULL)
			error("unable to get memory buffer");
	}
	
	/* check for agc mode */
	if (doagc) {
		/* need to read through waveform to get max */
		maxval=1;
		for (i=0;(len=sfsread(fid,i,BUFSIZE,sig));i+=len) {
			for (j=0;j<len;j++)
				if (sig[j] > maxval)
					maxval = sig[j];
				else if (sig[j] < -maxval)
					maxval = -sig[j];
		}
		if (dostereo) for (i=0;(len=sfsread(fid2,i,BUFSIZE,sig2));i+=len) {
			for (j=0;j<len;j++)
				if (sig2[j] > maxval)
					maxval = sig2[j];
				else if (sig2[j] < -maxval)
					maxval = -sig2[j];
		}
		scale = 24576.0/(double)maxval;
	}

	/* check start and end times */
	if (starttime >= 0)
		startframe = (int)((starttime-item.offset)/item.frameduration);
	else
		startframe = 0;
	if (endtime >= 0)
		numframes = (int)((endtime - item.offset)/item.frameduration - startframe+1);
	else
		numframes = item.numframes - startframe;
	if (dostereo && ((startframe+numframes) > item2.numframes))
		error("channels do not match");

fprintf(stderr,"%s: copying started at frame %d for %d frames\n",PROGNAME,startframe,numframes);

	/* open output file */
#ifdef DOS
	if ((op=fopen(wavname,"wb"))==NULL)
#else
	if ((op=fopen(wavname,"w"))==NULL)
#endif
		error("could not open '%s'",wavname);

	/* write RIFF main header */
	RIFFheader(op,numframes,(dostereo||domakestereo)?2:1);

	/* write RIFF wave header */
	RIFFwaveheader(op,numframes,1.0/item.frameduration,(dostereo||domakestereo)?2:1);

	/* write RIFF data header */
	RIFFdataheader(op,numframes,(dostereo||domakestereo)?2:1);

	/* write data */
	while (numframes > 0) {
		if ((len=sfsread(fid,startframe,(numframes<BUFSIZE)?numframes:BUFSIZE,sig))<=0)
			error("read error on input item");
		if (dostereo)
			if (sfsread(fid2,startframe,len,sig2)!=len)
				error("read mismatch between channels");
		if (doagc) {
			for (j=0;j<len;j++)
				sig[j] = (short)((float)sig[j]*scale);
			if (dostereo) for (j=0;j<len;j++)
				sig2[j] = (short)((float)sig2[j]*scale);
		}
		else if (doscale12) {
			for (j=0;j<len;j++)
				sig[j] = (sig[j] << 4);
			if (dostereo) for (j=0;j<len;j++)
				sig2[j] = (sig2[j] << 4);
		}
		if (do8bit) {
			for (j=0;j<len;j++) {
				if (putc((int)0x80+((int)sig[j]>>8),op)<0)
					error("write error on '%s'",wavname);
				if (dostereo)
					if (putc((int)0x80+((int)sig2[j]>>8),op)<0)
						error("write error on '%s'",wavname);
			}
		}
		else {
			for (j=0;j<len;j++) {
				if (putc(((int)sig[j] & 0xFF),op)<0)
					error("write error on '%s'",wavname);
				if (putc(((int)sig[j] >> 8),op)<0)
					error("write error on '%s'",wavname);
				if (dostereo) {
					if (putc(((int)sig2[j] & 0xFF),op)<0)
						error("write error on '%s'",wavname);
					if (putc(((int)sig2[j] >> 8),op)<0)
						error("write error on '%s'",wavname);
				}
				else if (domakestereo) {
					if (putc(((int)sig[j] & 0xFF),op)<0)
						error("write error on '%s'",wavname);
					if (putc(((int)sig[j] >> 8),op)<0)
						error("write error on '%s'",wavname);
				}
			}
		}
		startframe += len;
		numframes -= len;
	}

	/* that's all folks */
	fclose(op);
	sfsclose(fid);
	exit(0);
}

