/* wdload -- load a word item from a text description of times and labels */

/* M.A.Huckvale - University College London */

/* version 1.0 - July 1998 */

/*-------------------------------------------------------------------*/
/**MAN
.TH WDLOAD SFS1 UCL
.SH NAME
wdload -- load a word item from text description
.SH SYNOPSIS
.B wdload
(-t type) (-s stepsize) textfile file
.SH DESCRIPTION
.I wdload
constructs a new word item in the given file from text
descriptions of word arcs in a text file.  The word item
is a representation of a word lattice with nodes corrseponding
to times in the signal and arcs containing word names and
probabilities.
.PP
Currently, the only lattice format supported is the lattice
output from the Abbot recognition system.
.PP
.B Options
.TP 11
.B -I
Identify the program version.
.TP 11
.BI -s stepsize
Specify the step interval in seconds between the nodes.  Node
numbering is assumed to be fixed step time based.
.TP 11
.BI -t type
Identify the text file format.  Only "ABBOT" is allowed currently
and is the default.
.I type.
.SH OUTPUT ITEMS
.IP WD 11
Loaded word item
.SH VERSION/AUTHOR
1.0 - Mark Huckvale
.SH BUGS
The maximum word name length is set in sfs.h to be 32 characters.
*/
/*---------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "wdload"
#define PROGVERS "1.0"
char	*progname = PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard i-o routines */
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"		/* database filing system structures */

/* supported formats */
#define ABBOT_FORMAT	1

/* global data */
struct item_header 	wditem;			/* annotation item header  */
struct wd_rec		*wd;			/* annotation record */
char			textname[SFSMAXFILENAME];	/* text file name */
char			filename[SFSMAXFILENAME];	/* dbase file name */
double			fdur=0.016;		/* default sample duration */
int			nnode;		/* number of nodes */

/* node information */
struct node_rec {
	double	time;
	char	*wrd;
} *ntab;
int ncnt;

/* save a string in memory */
char *strsave(char *str)
{
	char *ptr=malloc(strlen(str)+1);
	strcpy(ptr,str);
	return(ptr);
}

/* decode a node line */
int decodenode(char *line,int *idx,double *time,char *label)
{
	char	*p;
	char	flag=1;

	*label='\0';

	p = strtok(line," \t\r\n");
	while (p && *p) {
		if (strncmp(p,"I=",2)==0)
			*idx = atoi(p+2);
		else if (strncmp(p,"time=",5)==0)
			*time = atof(p+5);
		else if (strncmp(p,"t=",2)==0)
			*time = atof(p+2);
		else if (strncmp(p,"WORD=",5)==0)
			strcpy(label,p+5);
		else if (strncmp(p,"W=",2)==0)
			strcpy(label,p+2);
		p = strtok(NULL," \t\r\n");
	}
	return flag;
}

/* decode an arc line */
int decodearc(char *line,int *nstart,int *nend,char *label,double *score)
{
	char	*p;
	char	flag=1;

	*label='\0';

	p = strtok(line," \t\r\n");
	while (p && *p) {
		if (strncmp(p,"START=",6)==0)
			*nstart = atoi(p+6);
		else if (strncmp(p,"S=",2)==0)
			*nstart = atoi(p+2);
		else if (strncmp(p,"END=",4)==0)
			*nend = atoi(p+4);
		else if (strncmp(p,"E=",2)==0)
			*nend = atoi(p+2);
		else if (strncmp(p,"WORD=",5)==0)
			strcpy(label,p+5);
		else if (strncmp(p,"W=",2)==0)
			strcpy(label,p+2);
		else if (strncmp(p,"acoustic=",9)==0)
			*score = atof(p+9);
		else if (strncmp(p,"a=",2)==0)
			*score = atof(p+2);

		p = strtok(NULL," \t\r\n");
	}
	return flag;
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;			/* option index */
	extern char	*optarg;		/* option argument ptr */
	int		errflg = 0;		/* option error flag */
	int		c;			/* option switch */

	/* processing variables */
	int		format=ABBOT_FORMAT;
	FILE	*ip;			/* input text file */
	int		ofid;			/* output file descriptors */
	char	iline[512];
	int		s,e,n;
	double	a,t;
	char	label[256];

	/* decode switches */
	while ( (c = getopt(argc,argv,"Is:t:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Load WD item from text V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 't' :	/* format type */
			if (strcmp(optarg,"ABBOT")==0)
				format = ABBOT_FORMAT;
			else
				error("unsupported format type. Types=(ABBOT)");
			break;
		case 's' :	/* step size */
			fdur = atof(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<3))
		error("usage: %s (-I) (-t type) (-s stepsize) textfile file",PROGNAME);

	/* get text file name */
	if (optind < argc)
		strcpy(textname,argv[optind]);
	else
		error("no text file specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open text file */
	if ((ip=fopen(textname,"r"))==NULL)
		error("cannot open %s",textname);

	/* open data file */
	if ((ofid=sfsopen(filename,"w",NULL)) < 0) {
		if (ofid==-1)
			error("cannot find file %s",filename);
		else
			error("access error on %s",filename);
	}
	sfsclose(ofid);

	/* create output item header */
	sfsheader(&wditem,WD_TYPE,-1,1,-1,fdur,0.0,0,0,1);
	sprintf(wditem.history,"%s(file=%s)",PROGNAME,textname);

	/* open output channel */
	if ((ofid=sfschannel(filename,&wditem)) < 0)
		error("unable to open temporary file",NULL);

	/* get output record buffer */
	if ((wd=(struct wd_rec *)sfsbuffer(&wditem,1))==NULL)
		error("could not get memory buffer",NULL);

	/* skip to size line */
	nnode=0;
	while (fgets(iline,sizeof(iline),ip)) {
		if (strncmp(iline,"N=",2)==0) {
			nnode = atoi(iline+2);
			break;
		}
	}
	if (nnode==0)
		error("could not find 'N=' size line");

	/* allocate space for node information */
	if ((ntab = (struct node_rec *)calloc(nnode,sizeof(struct node_rec)))==NULL)
		error("no memory for nodes buffer");

	/* read in node times and arcs */
	while (fgets(iline,sizeof(iline),ip)) {
		if (strncmp(iline,"I=",2)==0) {
			if (decodenode(iline,&n,&t,label) && (n >= 0) && (n < nnode)) {
				/* save time on node */
				ntab[n].time = t;
				if (label[0])
					ntab[n].wrd = strsave(label);
				else
					ntab[n].wrd = NULL;
			}
		}
		else if (strncmp(iline,"J=",2)==0) {
			if (decodearc(iline,&s,&e,label,&a) &&
				(s >= 0) && (s < nnode) &&
				(e >= 0) && (e < nnode)) {
				/* add word arc record */
				wd->start = (int)(0.5+ntab[s].time/fdur);
				wd->end = (int)(0.5+ntab[e].time/fdur);
				wd->score = (float)a;
				if (label[0])
					strncpy(wd->symbol,label,MAXWDLEN);
				else
					strncpy(wd->symbol,ntab[e].wrd,MAXWDLEN);
				strcpy(wd->attr,"");
				if (sfswrite(ofid,1,wd)!=1)
					error("output write error");
			}
		}
	}

	/* update data file */
	if (!sfsupdate(filename))
		error("backup error on %s",filename);

	/* that's all folks */
	exit(0);

}
