/* annotate -- automatic annotation of utterance given transcription */

/* m.a.huckvale - may 1990 */

/* version 2.0 - March 1998 */

/* version 2.2 - April 2000
	- add -e switch to remove average energy - doesn't help
	- use offsets in CO items from dictionary - doesn't work
*/

/* program name and version */
#define	PROGNAME "annotate"
#define PROGVERS "2.2"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH ANNOTATE SFS1 UCL
.SH NAME
annotate - automatic speech annotation by transcription alignment
.SH SYNOPSIS
.B annotate
(-i item) (-2) (-d dictionary) (-p prondict) (-t transcription) (-e) file
.SH DESCRIPTION
.I annotate
performs a non-linear time alignment between a coefficient item
in the specified file and a coefficient item 'built' from a supplied
transcription using a coefficient dictionary.  The resulting time
alignment is used to construct an annotation item in the file, aligned
to the original signal.
.PP
The dictionary is simply an SFS file with a number of coefficient items, one
per transcription symbol. To identify the coefficient items they should have a
'label=<symbol>' history component.  The program
.I andict
generates a file in the appropriate form.  If no dictionary is specified on the command line
the program looks for a file 'dict' in the current directory.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version.
.TP 11
.BI -i item
Select input item numbers.
.TP 11
.BI -d dictionary
Use the specified file as the symbol dictionary rather than 'dict'.
.TP 11
.BI -t transcription
Specify the transcription to be aligned.  Use quotes to enclose the transcription
and a space between each symbol.  If no transcription is specified the 'token'
field in the main header is used instead.
.TP 11
.BI -p prondict
Specify that the transcriptions are given in orthography, and that the
correct transcriptions can be found by searching this text file of
transcriptions.  The file has one transcription per line with the
orthography, a TAB, and the transcription.
.TP 11
.B -2
Apply slope constraints of 2:1 on the DP match.  This limits compression
or expansion during matching.
.TP 11
.B -e
Remove the mean from the coefficient vectors before matching, adding
the mean as an additional scalar value to the vector.  This is useful
if filterbank energies are being used as coefficients.
.SH INPUT ITEMS
.SS DICTIONARY
.IP 11.xx 11
Coefficient item.
.SS DATA FILE
.IP 11.xx 11
Coefficient item matching dictionary.
.SH OUTPUT ITEMS
.IP 5 11
Time aligned annotation item in data file.
.SH VERSION/AUTHOR
2.2 - Mark Huckvale
.SH SEE ALSO
vcalign andict
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard i-o routines */
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>
#include <math.h>
#include "sfs.h"		/* database filing system structures */

/* manfest constants */
#define MAXANN		1000	/* max length of transcription */

/* global data */
struct main_header	head;
struct item_header coitem;	/* filing system item header for data file */
struct co_rec		*cod;	/* coefficients from data file */
struct co_rec		*cot;	/* coefficient from transcription */
int	ncoeff;			/* # coefficients */
struct item_header anitem;	/* filing system item header for an */
struct an_rec		*an;	/* annotations to xfer */

/* global data */
#define	DIAG 	1		/* DP path values */
#define UP 	2
#define LEFT	3
unsigned char	*direc;		/* DP directions */
double	*cudist1,*cudist2;	/* DP distance matrix lines */
double	*dist1,*dist2;		/* DP distance matrix lines */
double 	*reg;			/* DP time registration path */
int	*align;			/* DP frame alignment path */
char	*transcription;		/* input transcription */
char	*dictfile="dict";	/* dictionary file */
int	numd,numt;		/* size of coefficient items */
char	filename[SFSMAXFILENAME];
int	cosave=0;
char	wfilename[SFSMAXFILENAME];
int	ortho=0;
int	slopeconst=0;
int	remenergy=0;		/* remove vector means */

#define EDGE	1E10
#define MINOF(x,y,z) (((x) < (y)) ? (((x) < (z)) ? (x) : (z)) : (((y) < (z)) ? (y) : (z)))
#define MINOFIDX(x,y,z) (((x) < (y)) ? (((x) < (z)) ? 1 : 3) : (((y) < (z)) ? 2 : 3))
#define MIN(x,y)	(((x)<(y))?(x):(y))
#define MAX(x,y)	(((x)>(y))?(x):(y))

/* distance metric for vocoder frames */
double metric(struct co_rec *a,struct co_rec *b)
{
	int i;
	float	*ap,*bp;
	double sumsq=0,val;

	ap = &a->data[0];
	bp = &b->data[0];
	for (i=ncoeff-1;i;i--) {
		val = *ap++ - *bp++;
		sumsq += val*val;
	}
	if (remenergy) sumsq += (a->gain - b->gain) * (a->gain - b->gain);
	return(sqrt(sumsq));
}

/* Dynamic Programming match */
void dpmatch(int refnum,int newnum)
{
	int		i,j;
	double		diag,up,left;
	unsigned char	*dptr;
	double		*ftemp;
	double		*cudist1p,*cudist2p;
	double		*dist1p,*dist2p;
	struct co_rec	*codp,*cotp;
	int		ledge,redge;

	/* initialisation */
	ftemp=cudist1;
	for (i=0;i<refnum;i++) *ftemp++=EDGE;
	codp = &cod[0];

	/* DP match */
	for (i=0;i<newnum;i++,codp++) {
		if (slopeconst) {
			ledge = MAX(i/2,refnum-2*newnum+2*i)-1;
			ledge = MAX(ledge,0);
			redge = MIN(2*i,refnum-newnum/2+i/2)+1;
			redge = MIN(redge,refnum);
		}
		else {
			ledge = 0;
			redge = refnum;
		}
		cotp = &cot[ledge];
		cudist1p = &cudist1[ledge];
		cudist2p = &cudist2[ledge];
		dist1p = &dist1[ledge];
		dist2p = &dist2[ledge];
		dptr=direc + (i*refnum) + ledge;
		if (ledge > 0) {
			cudist2[ledge-1] = EDGE;
			dist2[ledge-1] = EDGE;
		}
		for (j=ledge;j<redge;j++,cotp++,cudist1p++,cudist2p++,dist1p++,dist2p++) {
			if ((i==0) && (j==0)) {
				diag=0;
				left=EDGE;
			}
			else if (j==0) {
				diag=EDGE;
				left=EDGE;
			}
			else {
				diag= *(cudist1p-1) + *(dist1p-1);
				left= *(cudist2p-1) + *(dist2p-1);
			}
			up = *cudist1p + *dist1p;
			*dist2p = metric(codp,cotp);
			*cudist2p = *dist2p + MINOF(diag,up,left);
			*dptr++ = MINOFIDX(diag,up,left);
		}
		if (redge < refnum) {
			cudist2[redge] = EDGE;
			dist2[redge] = EDGE;
		}
		if ((redge+1) < refnum) {
			cudist2[redge+1] = EDGE;
			dist2[redge+1] = EDGE;
		}
		/* swap over buffers */
		ftemp=cudist1;
		cudist1=cudist2;
		cudist2=ftemp;
		ftemp=dist1;
		dist1=dist2;
		dist2=ftemp;
		if (((i%10)==9) && ttytest()) {
			printf("%d/%d\r",i+1,newnum);
			fflush(stdout);
		}
	}
}

/* Dynamic Programming time registration path */
void timereg(int refnum,int newnum,double *reg)
{
	register int	i;
	unsigned char	*dptr;

	/* initialise at end of table */
	dptr = direc + refnum*newnum - 1;
	reg[refnum] = (cod[numd-1].posn+cod[numd-1].size)*coitem.frameduration;
	i = refnum - 1;

	/* pass backwards up registration path */
	while ((dptr > direc) && (i >= 0)) {
		switch (*dptr) {
		case DIAG:
			reg[i--] = cod[(dptr-direc) / refnum].posn*coitem.frameduration;
			dptr -= refnum + 1;
			break;
		case UP:
			dptr -= refnum;
			break;
		case LEFT:
			reg[i--] = cod[(dptr-direc) / refnum].posn*coitem.frameduration;
			dptr--;
			break;
		default:
			error("bad pointer in time registration path");
		}
#ifdef IAG
printf("dptr offset = %d,%d i =%d\n",(int)(dptr-direc)/refnum,(int)(dptr-direc)%refnum,i);
#endif
	}
}

/* get new time instant from time registration path */
double	newtime(double t,double *reg,int refnum)
		/* time to xfer */
		/* registration path */
		/* length of registration path */
{
	int	p1,p2;
	double	mix,newt;

	/* get pointers into registration path */
	p2=0;
	while ((p2<refnum) && (t >= cot[p2].posn*coitem.frameduration)) p2++;
	p1= p2 - 1;

#ifdef IAG
	printf("t=%g,p1=%d,p2=%d,refnum=%d\n",t,p1,p2,refnum);
#endif

	/* check not off end */
	if (p2 >= refnum) return(reg[refnum]);

	/* get addmixture coefficient */
	mix = t - cot[p1].posn*coitem.frameduration;

	/* interpolate answer */
	newt=(1.0-mix)*reg[p1]+mix*reg[p2];

#ifdef IAG
	printf("t=%g,cot=%g,reg[p1]=%g,reg[p2]=%g,newt=%g\n",
		t,cot[p1].posn*coitem.frameduration,reg[p1],reg[p2],newt);
#endif

	return(newt);
}

/* getold time instant from time registration path */
double	oldtime(
double		t,		/* time to xfer */
double		*reg,		/* registration path */
int		refnum)		/* length of registration path */
{
	int	p1,p2;
	double	mix,newt;

	/* get pointers into registration path */
	p2=0;
	while ((p2<refnum) && (t >= reg[p2])) p2++;
	p1= p2 - 1;

	/* check not off end */
	if (p2 >= refnum) return(cod[refnum-1].posn*coitem.frameduration);

	/* get addmixture coefficient */
	mix = t - reg[p1];

	/* interpolate answer */
	newt=(1.0-mix)*cod[p1].posn*coitem.frameduration+mix*cod[p2].posn*coitem.frameduration;

#ifdef IAG
	printf("t=%g,cot=%g,reg[p1]=%g,reg[p2]=%g,newt=%g\n",
		t,cod[p1].posn*coitem.frameduration,reg[p1],reg[p2],newt);
#endif

	return(newt);
}

/* data area to hold transcription processing */
struct anproc_rec {
	char	*label;		/* symbol */
	char	*trip;		/* triphone */
	char	*bip;		/* biphone */
	char	*monp;		/* monophone */
	int	itemno;		/* item number found */
	int	fcnt;		/* duration in CO frames */
	int	posn;		/* frame position in output */
	double	offset;		/* original offset from coitem */
} antab[MAXANN];
int	antablen;

/* look up orthography in file to get transcription */
char *filetrans(char *fname,char *ortho)
{
	FILE	*ip;
	static char iline[16*MAXANN];
	char	*p,*q;
	
	if ((ip=fopen(fname,"r"))==NULL)
		error("could not open '%s'",fname);
	while (fgets(iline,16*MAXANN,ip)) {
		p = strtok(iline,"\t");
		if (strcmp(ortho,p)==0) {
			q = strtok(NULL,"\n");
			fclose(ip);
			return(q);
		}
	}
	fclose(ip);
	error("could not find transcription '%s' in '%s'\n",ortho,fname);
	return(NULL);
}

/* save string in memory */
char *strsave(char *str)
{
	char *ptr = malloc(strlen(str)+1);
	if (!ptr) {
		fprintf(stderr,"out of memory\n");
		exit(1);
	}
	strcpy(ptr,str);
	return(ptr);
}

/* parse transcription */
void parsetrans(char *tran)
{
	char	*lp;
	char	*tp;
	char	*np;
	char	annot[256];

	antablen=0;
	lp = "";
	tp = strtok(tran," ");
	while (tp && *tp) {
		np = strtok(NULL," ");
		if (np==NULL) np="";
		antab[antablen].label = tp;
		sprintf(annot,"%s\\%s_%s",tp,lp,np);
		antab[antablen].trip = strsave(annot);
		sprintf(annot,"%s\\%s_",tp,lp);
		antab[antablen].bip = strsave(annot);
		sprintf(annot,"%s\\",tp);
		antab[antablen].monp = strsave(annot);
		antab[antablen].itemno = 0;
		antab[antablen].fcnt = 0;
		antablen++;
		lp = tp;
		tp = np;
	}
printf("Found %d annotations in transcription\n",antablen);
}

/* build artificial coefficient item from dictionary */
int buildcoeff(
char			*dict,		/* dictionary file name */
char			*tran,		/* input transcription */
struct item_header	*icoitem,	/* input coeff header */
struct co_rec		**cot,		/* returned co data */
struct an_rec		**an,		/* returned an data */
int			*anlen)		/* returned # annotations */
{
	int			i,j,ncoeff;
	int			fid;
	struct item_header	item;
	int			tfreq,dfreq;
	char			*temp,*lab;
	int			err1,totfcnt;
	int			numtogo;
	int			itemno=0;
	double			sum;
	int			n0=0,n1=0,n2=0,n3=0;

	/* parse transcription */
	temp = (char *)malloc(strlen(tran)+1);
	strcpy(temp,tran);
	parsetrans(temp);
	*anlen = antablen;

	/* open dictionary */
	if ((fid = sfsopen(dict,"r",NULL)) < 0)
		error("access error on '%s'",dict);

	/* do first pass over dictionary checking availability of triphones */
	numtogo = antablen;
	while (sfsnextitem(fid,&item)) {
		itemno++;
		if ((item.datatype==CO_TYPE) && (item.subtype > 0) && (icoitem->framesize==item.framesize)) {
			/* potential coeff item */
			dfreq = (int)(0.5 + 0.01/icoitem->frameduration);
			tfreq = (int)(0.5 + 0.01/item.frameduration);
			if (dfreq==tfreq) {
				lab = params(item.history,"label","");
				for (i=0;i<antablen;i++) if (antab[i].fcnt==0)
					if (strcmp(antab[i].trip,lab)==0) {
/* printf("%d.%02d found triphone '%s'\n",item.datatype,item.subtype,lab); */
						antab[i].itemno = itemno;
						antab[i].fcnt = item.numframes;
						numtogo--;
						n3++;
					}
			}
		}
	}

	if (numtogo != 0) {
		/* do second pass over dictionary checking availability of biphones */
		sfsnextitem(fid,NULL);
		itemno=0;
		while (sfsnextitem(fid,&item)) {
			itemno++;
			if ((item.datatype==CO_TYPE) && (item.subtype > 0) && (icoitem->framesize==item.framesize)) {
				/* potential coeff item */
				dfreq = (int)(0.5 + 0.01/icoitem->frameduration);
				tfreq = (int)(0.5 + 0.01/item.frameduration);
				if (dfreq==tfreq) {
					lab = params(item.history,"label","");
					for (i=0;i<antablen;i++) if (antab[i].fcnt==0)
						if (strncmp(antab[i].bip,lab,strlen(antab[i].bip))==0) {
/* printf("%d.%02d found biphone '%s'\n",item.datatype,item.subtype,antab[i].bip); */
							antab[i].itemno = itemno;
							antab[i].fcnt = item.numframes;
							numtogo--;
							n2++;
						}
				}
			}
		}
	}

	if (numtogo != 0) {
		/* do third pass over dictionary checking availability of monophones */
		sfsnextitem(fid,NULL);
		itemno=0;
		while (sfsnextitem(fid,&item)) {
			itemno++;
			if ((item.datatype==CO_TYPE) && (item.subtype > 0) && (icoitem->framesize==item.framesize)) {
				/* potential coeff item */
				dfreq = (int)(0.5 + 0.01/icoitem->frameduration);
				tfreq = (int)(0.5 + 0.01/item.frameduration);
				if (dfreq==tfreq) {
					lab = params(item.history,"label","");
					for (i=0;i<antablen;i++) if (antab[i].fcnt==0)
						if (strncmp(antab[i].monp,lab,strlen(antab[i].monp))==0) {
/* printf("%d.%02d found monophone '%s'\n",item.datatype,item.subtype,antab[i].monp); */
							antab[i].itemno = itemno;
							antab[i].fcnt = item.numframes;
							numtogo--;
							n1++;
						}
				}
			}
		}
	}

	if (numtogo != 0) {
		/* do fourth pass over dictionary checking availability of labels */
		sfsnextitem(fid,NULL);
		itemno=0;
		while (sfsnextitem(fid,&item)) {
			itemno++;
			if ((item.datatype==CO_TYPE) && (item.subtype > 0) && (icoitem->framesize==item.framesize)) {
				/* potential coeff item */
				dfreq = (int)(0.5 + 0.01/icoitem->frameduration);
				tfreq = (int)(0.5 + 0.01/item.frameduration);
				if (dfreq==tfreq) {
					lab = params(item.history,"label","");
					for (i=0;i<antablen;i++) if (antab[i].fcnt==0)
						if (strcmp(antab[i].label,lab)==0) {
/* printf("%d.%02d found label '%s'\n",item.datatype,item.subtype,lab); */
							antab[i].itemno = itemno;
							antab[i].fcnt = item.numframes;
							numtogo--;
							n0++;
						}
				}
			}
		}
	}

	/* check all transcription is OK */
	err1=0;
	totfcnt=0;
	for (i=0;i<antablen;i++) {
		if (!antab[i].fcnt) {
			if (!err1) {
				fprintf(stderr,"the following symbol(s) were not found in '%s':\n",dict);
				err1++;
			}
			fprintf(stderr,"%s\n",antab[i].label);
		}
		antab[i].posn = totfcnt;
		totfcnt += antab[i].fcnt;
	}
	if (err1) exit(1);

	/* report degree of match */
	printf("Found %d trigrams, %d bigrams, %d unigrams, %d labels in dictionary\n",n3,n2,n1,n0);

	/* ok allocate memory for co data */
	if ((*cot=(struct co_rec *)sfsbuffer(icoitem,totfcnt))==NULL)
		error("could not get memory buffer");

	/* one last pass, get data for real */
	sfsnextitem(fid,NULL);
	itemno=0;
	while (sfsnextitem(fid,&item)) {
		itemno++;
		for (i=0;i<antablen;i++) {
			if (antab[i].itemno == itemno) {
				sfsread(fid,0,item.numframes,(*cot)+antab[i].posn);
				antab[i].offset = item.offset;
			}
		}
	}

	/* fix up frame times */
	for (i=0;i<totfcnt;i++)
		(*cot)[i].posn=i*item.windowsize;

	/* remove means if required */
	if (remenergy) {
		ncoeff = SFSRECSIZE(icoitem);
		for (i=0;i<totfcnt;i++) {
			sum=0;
			for (j=0;j<ncoeff;j++) sum += (*cot)[i].data[j];
			sum /= ncoeff;
			for (j=0;j<ncoeff;j++) (*cot)[i].data[j] /= sum;
			(*cot)[i].gain = sum;
		}
	}

	/* save to file if required */
	if (cosave) {
		sfsheader(&item,CO_TYPE,-1,4,icoitem->framesize,icoitem->frameduration,
			icoitem->offset,icoitem->windowsize,icoitem->overlap,icoitem->lxsync);
		sprintf(item.history,"%s/CO(%d.%02d;dict=%s)",
				PROGNAME,
				icoitem->datatype,icoitem->subtype,
				dict);
		putitem(filename,&item,totfcnt,*cot);
	}

	/* build annotation item too */
	sfsheader(&anitem,AN_TYPE,-1,1,-1,icoitem->frameduration,0.0,0,0,1);
	sprintf(anitem.history,"%s/AN(%d.%02d;dict=%s)",
			PROGNAME,
			icoitem->datatype,icoitem->subtype,
			dict);
	if ((*an = (struct an_rec *)sfsbuffer(&anitem,antablen))==NULL)
		error("could not get memory buffer");

	for (i=0;i<antablen;i++) {
		strcpy((*an)[i].label,antab[i].label);
		(*an)[i].posn = antab[i].posn * icoitem->windowsize;
		(*an)[i].size = antab[i].fcnt * icoitem->windowsize;
	}

	if (cosave) putitem(filename,&anitem,antablen,*an);

	return(totfcnt);
}

/* main program */
int main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	char		*cotype="0";	/* input co-types */
	int		c;		/* option switch */
	int		it;		/* item/sub-type specifiers */
	char		*ty;		/* sub-type match */
	/* file variables */
	int		fid,ofid;
	double		postime,lentime;
	int		i,numf,anlen;
	struct an_rec	*annot;
#ifdef IAG
	int		j;
	unsigned char 	*dptr;
#endif

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:d:t:Cp:2e")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Automatic transcription alignment V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == CO_TYPE)
					cotype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'd' :	/* dictionary */
			dictfile = optarg;
			break;
		case 't' :	/* transcription */
			transcription = optarg;
			break;
		case 'p' :
			strcpy(wfilename,optarg);
			ortho=1;
			break;
		case 'C' :	/* save artificial coefficients */
			cosave++;
			break;
		case '2' :	/* 2:1 slope constraints */
			slopeconst=1;
			break;
		case 'e' :	/* remove vector means */
			remenergy=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check for option decoding error */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-2) (-e) (-d coeffdict) (-p prondict) (-t transcription) file",PROGNAME);

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no SFS file specified",NULL);

	/* get transcription */
	if (!transcription) {
		if ((fid=sfsopen(filename,"w",&head)) < 0)
			error("access error on '%s'",filename);
		sfsclose(fid);
		transcription = head.token;
	}
	if (!*transcription)
		error("no transcription specified on command line or in header");

	/* transcribe orthography if required */
	if (ortho)
		transcription = filetrans(wfilename,transcription);

	/* read in coefficient data */
	getitem(filename,CO_TYPE,cotype,&coitem,(void *)&cod);
	ncoeff = SFSRECSIZE(&coitem);
	numd = coitem.numframes;
	if (coitem.lxsync)
		error("coefficients must be fixed frame length");

	/* build artificial coefficients from dictionary */
	numt = buildcoeff(dictfile,transcription,&coitem,&cot,&an,&anlen);

	/* check ratios */
	if ((numd >= 3*numt) || (numt >= 3*numd))
		error("alignment too steep: %d:%d frames",numd,numt);

	/* get memory for calculations */
	numf = (numd>numt) ? numd : numt;
	dist1=(double *)calloc(numf+1,sizeof(double));
	dist2=(double *)calloc(numf+1,sizeof(double));
	cudist1=(double *)calloc(numf+1,sizeof(double));
	cudist2=(double *)calloc(numf+1,sizeof(double));
	direc=(unsigned char *)malloc(numd*numt);
	if ((dist1==NULL) || (dist2==NULL) || (cudist1==NULL) || (cudist2==NULL) || (direc==NULL))
		error("unable to get sufficient memory for buffers",NULL);
	memset(direc,0,numd*numt);

	/* do DP matching */
	dpmatch(numt,numd);

#ifdef IAG
	/* print out directions matrix */
	dptr=direc;
	for (i=0;i<numd;i++) {
		for (j=0;j<numt;j++) {
			switch (*dptr++) {
			case 1:		printf("\\"); break;
			case 2:		printf("|"); break;
			case 3:		printf("-"); break;
			default:	printf("?"); break;
			}
		}
		printf("\n");
	}
#endif
	/* create output item header */
	sfsheader(&anitem,AN_TYPE,-1,1,-1,coitem.frameduration,coitem.offset,0,0,1);
	if (strlen(transcription)<100)
		sprintf(anitem.history,"%s(%d.%02d;dict=%s%s,trans='%s',type=auto)",
			PROGNAME,
			coitem.datatype,coitem.subtype,
			dictfile,(slopeconst)?",slope=2":"",transcription);
	else
		sprintf(anitem.history,"%s(%d.%02d;dict=%s%s,type=auto)",
			PROGNAME,
			coitem.datatype,coitem.subtype,
			dictfile,(slopeconst)?",slope=2":"");

	/* open output channel */
	if ((ofid=sfschannel(filename,&anitem)) < 0)
		error("could not open output channel to %s",filename);

	/* calculate time registration path */
	reg = (double *)calloc(numt+1,sizeof(double));
	timereg(numt,numd,reg);

#ifdef IAG
	/* print out alignment */
	for (i=0;i<numt;i++) 
		printf("%d %10.1f ->%10.1f\n",i,
			1000*cot[i].posn*coitem.frameduration,
			1000*reg[i]);
#endif

	/* transfer annotations */
	for (i=0;i<anlen;i++) {
/* printf("annotation %d/%d\n",i,anlen); */
		annot = &an[i];

		/* calculate new position and length */
		postime = annot->posn*anitem.frameduration;
		lentime = postime + annot->size*anitem.frameduration;
		annot->posn = (int)(0.5 + newtime(postime,reg,numt)/anitem.frameduration);
		annot->size = (int)(0.5 + newtime(lentime,reg,numt)/anitem.frameduration) - annot->posn;

		/* put annotation */
		if (sfswrite(ofid,1,annot) != 1)
			error("write error on temporary file ",NULL);
#ifdef IAG
		printf("posn=%d,size=%d,text=%s\n",annot->posn,annot->size,annot->label);
#endif
	}

	/* close output file and update */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks */
	free(dist1);
	free(dist2);
	free(cudist1);
	free(cudist2);
	free(direc);
	exit(0);
}

