/* antr -- annotations to tracks via mapping file */

/* M.A.Huckvale - January 1989 */

/* version 1.0
	- binary features
*/
/* version 1.1
	- debug mode
*/
/* version 1.2
	- hi and lo target specifications
*/
/* version 1.3 - September 1989
	- improved representational accuracy of short annotations
*/
/* version 1.4 - December 1989
	- transitory response to features
*/

#define PROGNAME "antr"
#define PROGVERS "1.4"
char *progname=PROGNAME;

/*-------------------------------------------------------------------*/
/**MAN
.TH ANTR 1 UCL SFS
.SH NAME
antr -- annotations to tracks via mapping file
.SH SYNOPSIS
.B antr
(-i item) (-r samplingrate) (-f feature) (-t ontime) (-h hi) (-l lo) (-o) mapfile (file)
.SH DESCRIPTION
.I antr
creates track items from annotation items using a file which maps
each annotation onto a list of binary features.
Each track is the value of a feature for each instant of time covered by
the annotation item, with value 0 if the current annotation does not
have the given feature, and 1 if it does.  A transient switch allows the
feature to be on for a fixed period of time after a matching annotation.
If no SFS data file is provided, the program simply reports the
contents of the annotation mapping file.
.PP
.I Options:
.TP 11
.B -I
Identify the program name and version.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -r sampling rate
Select the sampling rate for the output track item.  Default is 100Hz.
Choose sampling rate of -1 to obtain sampling rate of input annotation item.
.TP 11
.BI -f feature
Select to create a track for an individual feature.  Default is to create tracks
for all features.
.TP 11
.BI -t ontime
For each annotation that has the output feature, switch the output
trace on for this length of time (in seconds).  Default is to turn
on for duration of annotation.
.TP 11
.BI -h hi
High target value.  Default 1.
.TP 11
.BI -l lo
Low target value.  Default 0.
.TP 11
.B -o
Re-write annotation mapping file, after sorting.
.SH MAPPING FILE FORMAT
Each line in the mapping file must have the following format:
.nf
      <annotation> (<feature>)
.fi
where <annotation> is the name of an annotation to be converted, and
(<feature>) is a list of feature names separated by spaces.  Where a feature
is listed after an annotation, then this feature will be converted to
a track value of 1 when the annotation is present.  If the feature is not
listed it will be converted to a value of 0 when the annotation is present.
.SH VERSION/AUTHOR
1.1 - Mark Huckvale
.br
1.2 - Ian Howard, hi and lo target values.
.br
1.4 - Mark Huckvale
.SH BUGS
Maximum of 32 features.
*/
/*---------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <math.h>
#include "sfs.h"

/* default target values */
#define HIGH 1.0
#define LOW 0.0

/* global data */
struct item_header	anitem;		/* input annotation item */
struct an_rec		*an;
struct item_header	tritem;		/* output track item */
char			mapname[80];
char			filename[80];
double			srate = 0.0;
double			ontime = -1;

/* feature map */
char	**label;			/* annotation names */
int	numlab;
long	*attr;				/* annotation attributes */
char	*feature[32];			/* feature names */
int	fcount[32];			/* feature use count */
int	numfeat;			/* # features */
long	BIT[32]={0x1L,0x2L,0x4L,0x8L,
		 0x10L,0x20L,0x40L,0x80L,
		 0x100L,0x200L,0x400L,0x800L,
		 0x1000L,0x2000L,0x4000L,0x8000L,
		 0x10000L,0x20000L,0x40000L,0x80000L,
		 0x100000L,0x200000L,0x400000L,0x800000L,
		 0x1000000L,0x2000000L,0x4000000L,0x8000000L,
		 0x10000000L,0x20000000L,0x40000000L,0x80000000L};

/* list of errors */
#define NUMERROR	100
char	*anerr[NUMERROR];
int	numerr=0;

/* map file line */
char	mline[256];

/* save string in dynamic memory */
char	*strsave(s)
char	*s;
{
	char	*p;
	int	len=strlen(s)+1;
	if ((p=malloc(len))==NULL)
		error("cannot save string",NULL);
	strcpy(p,s);
	return(p);
}

/* binary string search */
int strfind(s,t,num)
char	*s;
char	*t[];
int	num;
{
	int	i,j,k;
	int	c;

	if (num==0)
		return(-1);
	else {
		i=0;
		j=num-1;
		do {
			k=(i+j)/2;
			if ((c=strcmp(s,t[k])) > 0)
				i=k+1;
			else
				j=k-1;
		} while (c && (i <= j));
		if (c)
			return(-1);
		else
			return(k);
	}
}

/* maintain string table */
void strtable(s,t,num)
char	*s;
char	*t[];
int	*num;
{
	int	i;

	/* see if string in table */
	if (strfind(s,t,*num) < 0) {
		/* add to table */
		i = *num;
		while ((i>0) && (strcmp(s,t[i-1])<0)) {
			t[i] = t[i-1];
			i--;
		}
		t[i]=strsave(s);
		(*num)++;
	}
}

/* get feature names */
int	getfeatnames(ip)
FILE	*ip;
{
	char	*p;
	int	cnt=0;

	numfeat=0;
	while (fgets(mline,256,ip)) if (mline[0]!='\n') {
		p=strtok(mline," \t\n");
		p=strtok(NULL," \t\n");
		while (p && *p) {
			strtable(p,feature,&numfeat);
			if (numfeat >= 32)
				error("too many features found in '%s'",mapname);
			p=strtok(NULL," \t\n");
		}
		cnt++;
	}
	return(cnt);
}

/* get annotation names */
void getlabelnames(ip)
FILE	*ip;
{
	char	*p;

	numlab=0;
	while (fgets(mline,256,ip)) {
		p=strtok(mline," \t\n");
		strtable(p,label,&numlab);
	}
}

/* load features */
void loadfeatures(ip)
FILE	*ip;
{
	char	*p;
	int	a,f;

	while (fgets(mline,256,ip)) {
		p=strtok(mline," \t\n");
		a=strfind(p,label,numlab);
		p=strtok(NULL," \t\n");
		while (p && *p) {
			f=strfind(p,feature,numfeat);
			fcount[f]++;
			attr[a] |= BIT[f];
			p=strtok(NULL," \t\n");
		}
	}
}

/* re-save map */
void resavemap(fname)
char	*fname;
{
	FILE	*op;
	int	i,j;

	/* open map file */
	if ((op=fopen(fname,"w"))==NULL)
		error("cannot write to '%s'",fname);

	/* save annotation map */
	for (i=0;i<numlab;i++) {
		fprintf(op,"%s\t",label[i]);
		for (j=0;j<numfeat;j++)
			if (attr[i] & BIT[j])
				fprintf(op,"%s ",feature[j]);
		fprintf(op,"\n");
	}

	fclose(op);
}

/* process single feature into a track */
void process(fid,feat,high,low)
int	fid;
int	feat;
float	high;	/* hi target value */
float	low;	/* low target value */
{
	int	ofid;
	int	set;
	float	val;
	double	trtime,antime,fdur,halfdur;
	double	trset;
	int	i,posn,idx;
	int	oncount;

	/* select output rate */
	if (srate < 0)
		fdur = anitem.frameduration;
	else if (srate==0)
		fdur = 0.01;
	else
		fdur = 1.0/srate;
	halfdur = fdur/2.0;

	/* calculate frame count for transients */
	if (ontime > 0)
		oncount = (int)(0.5 + ontime/fdur);
	else
		oncount = -1;

	/* create output item header */
	sfsheader(&tritem,TR_TYPE,1,4,1,fdur,anitem.offset,1,0,0);
	if (ontime > 0)
		sprintf(tritem.history,"%s(%d.%02d;map=%s,rate=%g,feature=%s,transient=%g)",
			PROGNAME,
			anitem.datatype,anitem.subtype,
			mapname,1.0/fdur,feature[feat],ontime);
	else
		sprintf(tritem.history,"%s(%d.%02d;map=%s,rate=%g,feature=%s)",
			PROGNAME,
			anitem.datatype,anitem.subtype,
			mapname,1.0/fdur,feature[feat]);

	/* open output channel */
	if ((ofid=sfschannel(filename,&tritem)) < 0)
		error("could not open output channel to '%s'",filename);

	/* read through annotation, outputting feature value */
	trtime=fdur;		/* trtime is end of current frame */
	set=0;
	antime=0;
	posn=0;
	trset=0;
	for (i=0;sfsread(fid,i,1,an);i++) {
		/* set output track up to the start of this annotation */
		while (posn < an->posn) {
			antime += anitem.frameduration;
			if (set) trset += anitem.frameduration;
			if (antime >= trtime) {
				val = (trset > halfdur) ? high : low;
				if (sfswrite(ofid,1,&val) != 1)
					error("write error on '%s'",filename);
				trtime += fdur;
				if (set > 0) set--;
				trset=0;
			}
			posn++;
		}
		/* look up value of this annotation */
		idx = strfind(an->label,label,numlab);
		if (idx < 0) {
			strtable(an->label,anerr,&numerr);
			if (numerr >= NUMERROR)
				error("too many annotations not processed",NULL);
			set=0;
		}
		else if (attr[idx] & BIT[feat])
			set=oncount;
		else if (set < 0)
			set=0;
	}

	/* last annotation */
	while (posn < (an->posn+an->size)) {
		antime += anitem.frameduration;
		if (set) trset += anitem.frameduration;
		if (antime >= trtime) {
			val = (trset > halfdur) ? high : low;
			if (sfswrite(ofid,1,&val) != 1)
				error("write error on '%s'",filename);
			trtime += fdur;
			if (set > 0) set--;
			trset=0;
		}
		posn++;
	}

	/* don't close 'ofid' in here */
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* processing variables */
	int		fid;
	FILE		*ip;
	int		it;
	char		*ty;
	char		*antype="0";
	int		i;
	float		high=HIGH;	/* high target value */
	float		low=LOW;	/* low target value */

	/* control variables */
	int		spfeat=0;	/* special feature */
	char		*spfeatname="";	/* special feature name */
	int		anout=0;	/* re-save annotation map */
	int		debug=0;	/* debug mode */

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:r:f:h:l:ot:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Convert annotations to track data V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item number */
			if (itspec(optarg,&it,&ty)==0) {
				if (it==AN_TYPE)
					antype=ty;
				else
					error("bad item specification",NULL);
			}
			else
				error("illegal item specification",NULL);
			break;
		case 'r' :	/* sampling rate */
			srate = atof(optarg);
			break;
		case 'h' :	/* high target value */
			high = atof(optarg);
			break;
		case 'l' :	/* low target value */
			low = atof(optarg);
			break;
		case 'f' :	/* specific feature */
			spfeatname=optarg;
			spfeat++;
			break;
		case 't' :	/* transient time */
			ontime = atof(optarg);
			if (ontime <= 0.0)
				error("transient time must be > 0");
			break;
		case 'o' :	/* re-save annotation map */
			anout++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-r samprate) (-f feature) (-t ontime) (-o) mapname (file)",PROGNAME);

	/* get map filename */
	if (optind < argc)
		strcpy(mapname,argv[optind]);
	else
		error("no annotation mapfile specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		debug=1;

	/* open map file */
	if ((ip=fopen(mapname,"r"))==NULL)
		error("cannot open map file '%s'",mapname);

	/* get list of features from map */
	numlab=getfeatnames(ip);

	/* get space for annotation table */
	if ((label=(char **)calloc(numlab,sizeof(char *)))==NULL)
		error("cannot get memory for annotation table",NULL);
	if ((attr=(long *)calloc(numlab,sizeof(long)))==NULL)
		error("cannot get memory for annotation table",NULL);

	/* get list of annotations from map */
	rewind(ip);
	getlabelnames(ip);

	/* load features from map */
	rewind(ip);
	loadfeatures(ip);
	fclose(ip);
	fprintf(stderr,"%s: %d labels, %d features loaded from %s\n",
		PROGNAME,
		numlab,numfeat,mapname);

	/* re-save map if required */
	if (anout) resavemap(mapname);

	/* dump features if required */
	if (debug) {
		printf("Use:\tFeatures:\n");
		for (i=0;i<numfeat;i++)
			printf("%d\t%s\n",fcount[i],feature[i]);
		exit(0);
	}

	/* open data file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on '%s'",filename);

	/* find input annotation item */
	if (!sfsitem(fid,AN_TYPE,antype,&anitem))
		error("cannot find input AN item in '%s'",filename);

	/* get annotation buffer */
	if ((an=(struct an_rec *)sfsbuffer(&anitem,1))==NULL)
		error("could not get annotation buffer",NULL);

	/* loop through features */
	if (spfeat) {
		/* process selected feature */
		i = strfind(spfeatname,feature,numfeat);
		if (i < 0)
			error("cannot find feature '%s' in map",spfeatname);
		else
			process(fid,i,high,low);
	}
	else {
		/* process all features */
		for (i=0;i<numfeat;i++) {

			process(fid,i,high,low);

			/* update file periodically */
			if ((i%10)==9)
				if (!sfsupdate(filename))
					error("update error on '%s'",filename);
		}
	}

	/* report any errors */
	if (numerr) {
		fprintf(stderr,"%s: the following annotations could not be processed:\n",PROGNAME);
		for (i=0;i<numerr;i++) printf("%s\n",anerr[i]);
	}

	/* that's all folks */
	sfsclose(fid);
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);
	exit(0);
}

