/*--------------------------------------------------------------------------
COMBINE.C - merge speech items from one or two SFS files.

Andrew Simpson - University College London

Tidied up by Mark Huckvale

--------------------------------------------------------------------------*/

#define PROGNAME "combine"
#define PROGVERS "1.1"
char	*progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH COMBINE SFS1 UCL
.SH NAME
combine - add two speech signals
.SH SYNOPSIS
.B combine
(-i item1) (-i item2) (-2) file
.br
.B combine
(-i item1) (-N item2) (-2) -n file2 file1
.SH DESCRIPTION
.I combine
is a program to add together two waveforms.  It is useful for adding
noise to a speech signal.  A signal in file 1 is added to a signal in
file 2 and the result put back into file 1.  Signals must match
exactly in sample rate.  The second signal is padded with zeros if
it is too short.  Two items from one file can also be added, by
using one filename only.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -n noise_file
Specify name of file containing second signal.  Default is to use
same file.
.TP 11
.BI -N noise_item
Specify item number of signal in second file.
.TP 11
.B -2
Halve the size of each waveform first to prevent overflow.
.SH INPUT ITEMS
.IP SP
Signal in file 1.
.IP SP
Signal in file 2.
.SH OUTPUT ITEMS
.IP SP
Sum of two signals.
.SH VERSION/AUTHOR
.IP 1.0
Andrew Simpson
.IP 1.1
Mark Huckvale
.SH SEE ALSO
level(SFS1)
.SH BUGS
If the resulting signal overloads, the program dies.
*/
/*--------------------------------------------------------------------------*/


#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"

/* level constants */
#define MAXSAMPLE 32767.0
#define TRUE -1
#define FALSE 0

/* manifest constants */
#define BUFSIZE	4096

/* global data */
short	 buff1[BUFSIZE];
short	 buff2[BUFSIZE];
short	 obuff[BUFSIZE];
char	filename1[SFSMAXFILENAME];
char	filename2[SFSMAXFILENAME];
char	*ty1=NULL;
char	*ty2=NULL;
struct item_header spitem1;
struct item_header spitem2;
struct item_header opitem;
int	dodiv2=0;

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int 	c,errflg=0;		/* option char */
	int	it;
	char	*ty="0";
	int	fid1,fid2,ofid;
	int	len1,len2,i,sample;
	int	j;

	/* decode switches */
	while ((c=getopt(argc,argv,"Ii:n:N:2")) != EOF) switch (c) {
	case 'I':
		fprintf(stderr,"%s: Combine (add) two speech signals V%s\n",
       			PROGNAME,PROGVERS);
       		exit(0);
       	case 'i':
		/* signal item */
		if (itspec(optarg,&it,&ty) == 0) {
			if (it == SP_TYPE) {
				if (ty1==NULL)
					ty1=ty;
				else
					ty2=ty;
			}
			else
				error("unsuitable item specification %s",optarg);
		}
		else
			error("illegal item specification %s",optarg);
		break;
	case 'N' :
		/* item number in the noise file */
		if (itspec(optarg,&it,&ty) == 0) {
			if (it == SP_TYPE) {
				ty2=ty;
			}
			else
				error("unsuitable item specification %s",optarg);
		}
		else
			error("illegal item specification %s",optarg);
		break;

	case 'n':
		strncpy(filename2,optarg,SFSMAXFILENAME);
		break;

	case '2':
		dodiv2=1;
		break;

	case '?' :	/* unknown */
		errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-i item1) (-N item2) (-2) (-n file2) filename",PROGNAME);

	/* get filename1 */
	if (optind < argc)
		strcpy(filename1,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open file1 */
	if ((fid1=sfsopen(filename1,"w",NULL)) < 0) {
		if (fid1==-1)
			error("unable to find file '%s'",filename1);
		else
			error("access error on '%s'",filename1);
	}

	/* get filename2 */
	if (filename2[0]=='\0') strcpy(filename2,filename1);

	/* open file2 */
	if ((fid2=sfsopen(filename2,"r",NULL)) < 0) {
		if (fid2==-1)
			error("unable to find file '%s'",filename2);
		else
			error("access error on file '%s'",filename2);
	}

	/* locate input item in file 1*/
	if (ty1==NULL) {
		if (strcmp(filename1,filename2)==0)
			ty1="*";
		else
			ty1="0";
	}
	if (!sfsitem(fid1,SP_TYPE,ty1,&spitem1))
		error("unable to find input item in '%s'",filename1);

	/* locate input item in file 2 */
	if (ty2==NULL) ty2="0";
	if (!sfsitem(fid2,SP_TYPE,ty2,&spitem2))
		error("unable to find input item in '%s'",filename2);

	/* check that they have the same sampling rates */
	if ((int)(0.5+0.1/spitem1.frameduration) !=
	    (int)(0.5+0.1/spitem2.frameduration))
		error("signals have different sampling rates");

	/* create output item header */
	sfsheader(&opitem,spitem1.datatype,spitem1.floating,
		spitem1.datasize,spitem1.framesize,
		spitem1.frameduration,spitem1.offset,
		spitem1.windowsize,spitem1.overlap,spitem1.lxsync);

	if (strcmp(filename1,filename2)==0)
		sprintf(opitem.history,"%s(%d.%02d,%d.%02d%s)",
			PROGNAME,
			spitem1.datatype,spitem1.subtype,
			spitem2.datatype,spitem2.subtype,
			(dodiv2)?";halved":"");
	else
		sprintf(opitem.history,"%s(%d.%02d;file=%s,item=%d.%02d%s)",
			PROGNAME,
			spitem1.datatype,spitem1.subtype,
			filename2,
			spitem2.datatype,spitem2.subtype,
			(dodiv2)?",halved":"");

	/* open output channel */
	if ((ofid=sfschannel(filename1,&opitem)) < 0)
		error("unable to open output channel",NULL);

	for (i=0;(len1=sfsread(fid1,i,BUFSIZE,buff1))>0;i+=len1) {

		len2=sfsread(fid2,i,len1,buff2);	/* get matching block */
		for (j=len2;j<len1;j++) buff2[j]=0;	/* use zero if too short */

		for (sample=0;sample<len1;sample++) {
			j = buff1[sample] + buff2[sample];
			if (dodiv2) j = j >> 1;
			if (abs(j) > MAXSAMPLE)
				error("combined signal overloaded");
			obuff[sample] = j;
		}

		/* write out result of processing */
		if (sfswrite(ofid,len1,obuff) != len1)
			error("write error on output file",NULL);
	}

	sfsclose(fid2);

	/* update file */
	if (!sfsupdate(filename1))
		error("update error on %s",filename1);

	exit(0);
}
