/* dfxhist - display fx histogram from tx data */

/* Mark Huckvale - University College London */

/* version 1.0 - February 1998 */

/* version 1.1 - January 2000
	- log compress on scatter plot
*/
/* version 1.2 - December 2006
	- add statistics only output
*/

#define PROGNAME "dfxhist"
#define PROGVERS "1.3"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH DFXHIST SFS1 UCL
.SH NAME
dfxhist - display fundamental frequency histograms
.SH SYNOPSIS
.B dfxhist
(-i item) (-p) (-t title) (-s) file
.SH DESCRIPTION
.I dfxhist
is a program to display a standard set of analyses of a Tx data
set.  The program produces a graphical display containing (i) a
histogram of all tx periods (elapsed time, logarithmic scale),
(ii) a second order histogram of all tx periods occurring within
regular voicing (is within 10% of one of periods on either side),
(iii) a scatter-plot between adjacent tx periods on a grey-scale,
(iv) a panel of statistics.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.B -p
Send output directly to printer
.TP 11
.BI -t title
Display given title at top of page.
.TP 11
.B -s
Print table of statistics only - to standard ouput.
.SH INPUT ITEMS
.IP TX
Fundamental period data set.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
vtx
.SH BUGS
*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"
#include "dig.h"
#include "digdata.h"
#include "dgraph.h"

/* manifest constants */
#define MINFREQ	60.0		/* minimum Fx */
#define MAXFREQ 600.0		/* maximum Fx */
#define MINTIME	0.05		/* smallest time to plot */
#define MAXBINS	128		/* maximum # histogram bins */
#define BINFACT	0.1		/* numbins = namsamp * binfact */

/* graph layout */

/* global data */
struct main_header	head;
struct item_header	txitem;
int32			*tx;
char			tbuf[1024];
char			tbuf2[1024];
char			title[1024];

/* histogram data */
int	numbins=MAXBINS;
double	dhist1[MAXBINS];
double	dhist2[MAXBINS];
int	chist[MAXBINS][MAXBINS];
char	ghist[MAXBINS*MAXBINS];
double	mintime=MINTIME;
double	maxtime;
double	tottime;
double	voitime;
double	regtime;
int	numhist1;
int	numhist2;

/* convert fx value to bin # */
int32 fx2bin(double fx)
{
	double	factor = (log(fx)-log(MINFREQ))/(log(MAXFREQ)-log(MINFREQ));
	return (int32)(0.5+factor*numbins);
}

/* convert bin # to fx value */
double bin2fx(int bin,double portion)
{
	double lo,hi;
	lo = exp(log(MINFREQ)+bin*(log(MAXFREQ)-log(MINFREQ))/numbins);
	hi = exp(log(MINFREQ)+(bin+1)*(log(MAXFREQ)-log(MINFREQ))/numbins);
	return((1.0-portion)*lo+portion*hi);
}

/* calculate simple histogram */
void calchist1(int32 *tx,int32 len,double dur)
{
	int32	i;
	double	fx;
	int32	idx;

	for (i=0;i<len;i++) {
		tottime += tx[i]*dur;
		fx = 1.0/(tx[i]*dur);
		idx = fx2bin(fx);
		if ((0 <= idx) && (idx < numbins)) {
			dhist1[idx] += tx[i]*dur;
			numhist1++;
			voitime += tx[i]*dur;
		}
	}

	for (i=0;i<numbins;i++) {
		if (dhist1[i] > maxtime)
			maxtime = dhist1[i];
		if (dhist1[i] < dur)
			dhist1[i] = dur;
	}

}

/* check fx value against neighbours */
int	isregular(double fx1,double fx2,double fx3)
{
	if ((0.9 <= fx1/fx2) && (fx1/fx2 <= 1.1))
		return(1);
	if ((0.9 <= fx3/fx2) && (fx3/fx2 <= 1.1))
		return(1);
	return(0);
}

/* calculate regular only histogram */
void calchist2(int32 *tx,int32 len,double dur)
{
	int32	i;
	double	fx1,fx2,fx3;
	int32	idx;

	for (i=1;i<len-1;i++) {
		fx1 = 1.0/(tx[i-1]*dur);
		fx2 = 1.0/(tx[i]*dur);
		fx3 = 1.0/(tx[i+1]*dur);
		if (isregular(fx1,fx2,fx3)) {
			idx = fx2bin(fx2);
			if ((0 <= idx) && (idx < numbins)) {
				dhist2[idx] += tx[i]*dur;
				numhist2++;
				regtime += tx[i]*dur;
			}
		}
	}

	for (i=0;i<numbins;i++) {
		if (dhist2[i] > maxtime)
			maxtime = dhist2[i];
		if (dhist2[i] < dur)
			dhist2[i] = dur;
	}

}

double mylog(double val)
{
	if (val<=1.0)
		return(0.0);
	else
		return(log(val));
}

/* calculate scatterplot histogram */
void calcscatter(int32 *tx,int32 len,double dur)
{
	int32	i,j,maxcount;
	double	fx1,fx2;
	int32	idx1,idx2;

	for (i=1;i<len;i++) {
		fx1 = 1.0/(tx[i-1]*dur);
		fx2 = 1.0/(tx[i]*dur);
		idx1 = fx2bin(fx1);
		idx2 = fx2bin(fx2);
		if ((0 <= idx1) && (idx1 < numbins) &&
		    (0 <= idx2) && (idx2 < numbins))
			chist[idx1][idx2]++;
	}

	/* find max */
	maxcount=0;
	for (i=0;i<numbins;i++)
		for (j=0;j<numbins;j++)
			if (chist[i][j] > maxcount)
				maxcount = chist[i][j];

	/* convert to grey scale */
	for (i=0;i<numbins;i++)
		for (j=0;j<numbins;j++)
			ghist[i*numbins+j] = (char)(((digdata.digdev.greylevels-1)*mylog(1.0+chist[i][j]))/mylog(maxcount));
}

/* find mode */
double findmode()
{
	int	i;
	double	mtime=0;
	int	midx=0;
	for (i=0;i<numbins;i++) {
		if (dhist1[i] > mtime) {
			mtime = dhist1[i];
			midx = i;
		}
	}
	return(bin2fx(midx,0.5));
}

/* calculate percentile frequencies */
void percentiles(double dur,double *fx0,double *fx5,double *fx10,double *fx50,double *fx90,double *fx95,double *fx100)
{
	double 	ctim=0;
	int	bin=0;

	ctim = dhist1[0];
	while ((bin < (numbins-1)) && (ctim==0))
		if (dhist2[++bin]>dur)
			ctim += dhist2[bin];
	*fx0 = bin2fx(bin,0.0);

	while ((bin < (numbins-1)) && (ctim < 0.05*regtime))
		if (dhist2[++bin]>dur)
			ctim += dhist2[bin];
	*fx5 = bin2fx(bin,0.05*regtime/ctim);

	while ((bin < (numbins-1)) && (ctim < 0.1*regtime))
		if (dhist2[++bin]>dur)
			ctim += dhist2[bin];
	*fx10 = bin2fx(bin,0.1*regtime/ctim);

	while ((bin < (numbins-1)) && (ctim < 0.5*regtime))
		if (dhist2[++bin]>dur)
			ctim += dhist2[bin];
	*fx50 = bin2fx(bin,0.5*regtime/ctim);

	while ((bin < (numbins-1)) && (ctim < 0.9*regtime))
		if (dhist2[++bin]>dur)
			ctim += dhist2[bin];
	*fx90 = bin2fx(bin,0.9*regtime/ctim);

	while ((bin < (numbins-1)) && (ctim < 0.95*regtime))
		if (dhist2[++bin]>dur)
			ctim += dhist2[bin];
	*fx95 = bin2fx(bin,0.95*regtime/ctim);

	bin=numbins;
	ctim=regtime;
	while ((bin > 0) && (ctim==regtime))
		if (dhist2[--bin]>dur)
			ctim -= dhist2[bin];
	*fx100 = bin2fx(bin,1.0);

}

/* main program */
int main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char		*ty;		/* item sub type */
	char		*txtype="0";
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;
	int		doprint=0;
	float		x[2],y[2];
	float		charwidth,charheight;
	char		messg[256];
	double		xpos,ypos,spacing;
	double		fx0,fx5,fx10,fx50,fx90,fx95,fx100;
	int			dostatsonly=0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:pt:s")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Display Fx histograms V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == TX_TYPE)
					txtype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'p' :	/* send graphics to printer */
			doprint++;
			break;
		case 't' :	/* title */
			strncpy(title,optarg,256);
			break;
		case 's' :
			dostatsonly=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-p) (-t title) (-s) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file to get header */
	if ((fid=sfsopen(filename,"r",&head))<0)
		error("access error on '%s'",filename);
	sfsclose(fid);

	/* load tx data */
	getitem(filename,TX_TYPE,txtype,&txitem,(void **)&tx);

	/* calculate number of bins to use */
	numbins = (int)(txitem.numframes * BINFACT);
	if (numbins < 10) numbins=10;
	if (numbins > MAXBINS) numbins=MAXBINS;

	/* calculate first histogram */
	calchist1(tx,txitem.numframes,txitem.frameduration);

	/* calculate second histogram */
	calchist2(tx,txitem.numframes,txitem.frameduration);

	/* open graphics */
	if (!dostatsonly) {
		digstart((doprint)?DIG_DEFAULT_PRINTER:DIG_DEFAULT_TERM,NULL,1);
		digscale(1.0,1.05,0);
		digclearscreen();
		digbox(20,0.0,0.0,1.0,1.0);
		charheight = digdata.chheight/digdata.yscale;

		sprintf(tbuf,"file=%s ",filename);
		if (head.speaker[0]!='\0') {
			sprintf(tbuf2,"speaker=%s ",head.speaker);
			strcat(tbuf,tbuf2);
		}
		if (head.token[0]!='\0') {
			sprintf(tbuf2,"token=%s ",head.token);
			strcat(tbuf,tbuf2);
		}
		if (title[0]!='\0') {
			sprintf(tbuf2,"title=%s",title);
			strcat(tbuf,tbuf2);
		}
		digtext(20,0.0,1.05-3*charheight/2,tbuf);
		sprintf(tbuf,"SFS/%s V%s",PROGNAME,PROGVERS);
		digtext(20,(float)1.0-digtextlen(tbuf),(float)(1.05-3*charheight/2),tbuf);

		/* display first histogram */
		/* do axes */
		x[0] = MINFREQ;
		x[1] = MAXFREQ;
		y[0] = mintime;
		y[1] = maxtime;

		digscale(2.0,2.1,0);
		digorigin(0.0,1.0);
		dgraph((short *)x,(short *)y,2,
			DGfloat+DGlowax+DGlast+DGlogax,
			DGfloat+DGlowax+DGlogax,DGaxonly,
			23242423,
			"","Frequency (Hz)","Time (s)");

		dgraph((short *)x,(short *)dhist1,numbins,
			DGfloat+DGlowax+DGlast+DGlogax+DGprev,
			DGdouble+DGlowax+DGlogax+DGprev,DGhist,
			23242423,
			"","","");
		digtext(20,dgraphp->right-digtextlen("All Tx"),1.0-2*charheight,"All Tx");

		/* display second histogram */
		/* do axes */
		x[0] = MINFREQ;
		x[1] = MAXFREQ;
		y[0] = mintime;
		y[1] = maxtime;

		digscale(2.0,2.1,0);
		digorigin(1.0,1.0);
		dgraph((short *)x,(short *)y,2,
			DGfloat+DGlowax+DGlast+DGlogax,
			DGfloat+DGlowax+DGlogax,DGaxonly,
			23242423,
			"","Frequency (Hz)","Time (s)");

		dgraph((short *)x,(short *)dhist2,numbins,
			DGfloat+DGlowax+DGlast+DGlogax+DGprev,
			DGdouble+DGlowax+DGlogax+DGprev,DGhist,
			23242423,
			"","","");
		digtext(20,dgraphp->right-digtextlen("Regular Tx"),1.0-2*charheight,"Regular Tx");

		/* calculate scatter histogram */
		calcscatter(tx,txitem.numframes,txitem.frameduration);

		/* display scatter plot */
		/* do axes */
		x[0] = MINFREQ;
		x[1] = MAXFREQ;
		y[0] = MINFREQ;
		y[1] = MAXFREQ;

		digscale(2.0,2.1,0);
		digorigin(0.0,0.0);
		dgraph((short *)x,(short *)y,2,
			DGfloat+DGlowax+DGlast+DGlogax,
			DGfloat+DGlowax+DGlogax,DGaxonly,
			23242423,
			"","Frequency (Hz)","Frequency (Hz)");

		diggreyscale(20,dgraphp->left,dgraphp->bot,dgraphp->right,dgraphp->top,
			ghist,numbins,numbins,digdata.digdev.greylevels);

		/* print statistics */
		digscale(2.0,2.1,0);
		digorigin(1.0,0.0);
		charwidth = digdata.chwidth/digdata.xscale;

		spacing = 1.0/10;
		ypos = 1.0 - spacing;
		xpos = 0.5 - 17*charwidth;

		sprintf(messg,"      #Tx,Time Total : %d,%.1fs",txitem.numframes,tottime);
		digtext(20,xpos,ypos,messg);
		ypos -= spacing;

		sprintf(messg,"     #Tx,Time Voiced : %d,%.1fs",numhist1,voitime);
		digtext(20,xpos,ypos,messg);
		ypos -= spacing;

		sprintf(messg,"    #Tx,Time Regular : %d,%.1fs",numhist2,regtime);
		digtext(20,xpos,ypos,messg);
		ypos -= spacing;

		sprintf(messg,"          Regularity : %d%%",(int)(100.0*regtime/voitime));
		digtext(20,xpos,ypos,messg);
		ypos -= spacing;

		percentiles(txitem.frameduration,&fx0,&fx5,&fx10,&fx50,&fx90,&fx95,&fx100);
		sprintf(messg,"    Mode,Median (Hz) : %d,%d",(int)findmode(),(int)fx50);
		digtext(20,xpos,ypos,messg);
		ypos -= spacing;

		/* percentiles */
		sprintf(messg,"    0,5,10 %%ile (Hz) : %d,%d,%d",(int)fx0,(int)fx5,(int)fx10);
		digtext(20,xpos,ypos,messg);
		ypos -= spacing;
		sprintf(messg," 100,95,90 %%ile (Hz) : %d,%d,%d",(int)fx100,(int)fx95,(int)fx90);
		digtext(20,xpos,ypos,messg);
		ypos -= spacing;
		sprintf(messg,"100,90,80 range (Hz) : %d,%d,%d",(int)(fx100-fx0),(int)(fx95-fx5),(int)(fx90-fx10));
		digtext(20,xpos,ypos,messg);
		ypos -= spacing;

		/* that's all folks */
		digflush();
		digquit(1);
	}
	else {
		printf("      #Tx,Time Total : %d,%.1fs\n",txitem.numframes,tottime);
		printf("     #Tx,Time Voiced : %d,%.1fs\n",numhist1,voitime);
		printf("    #Tx,Time Regular : %d,%.1fs\n",numhist2,regtime);
		printf("          Regularity : %d%%\n",(int)(100.0*regtime/voitime));
		percentiles(txitem.frameduration,&fx0,&fx5,&fx10,&fx50,&fx90,&fx95,&fx100);
		printf("    Mode,Median (Hz) : %d,%d\n",(int)findmode(),(int)fx50);
		printf("    0,5,10 %%ile (Hz) : %d,%d,%d\n",(int)fx0,(int)fx5,(int)fx10);
		printf(" 100,95,90 %%ile (Hz) : %d,%d,%d\n",(int)fx100,(int)fx95,(int)fx90);
		printf("100,90,80 range (Hz) : %d,%d,%d\n",(int)(fx100-fx0),(int)(fx95-fx5),(int)(fx90-fx10));
	}
	exit(0);
}
