/* fxbuzz -- convert Fx contour to a buzz */

/* M.A.Huckvale - University College London */

/* version 1.0 - April 2003 */

#define PROGNAME "fxbuzz"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH FXBUZZ 1 UCL
.SH NAME
fxbuzz - convert FX contour to a buzz with matching frequency
.SH SYNOPSIS
.B fxlist
(-i item) (-e) (-S samplerate) sfsfile
.SH DESCRIPTION
.I fxbuzz
is a program to convert an FX item into a SP item as a buzz with a
matching fundamental frequency.  The buzz can be of fixed amplitude
or it can follow the amplitude envelope of a supplied speech signal.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item. If a speech item is selected, then its amplitude
envelope will be used for the buzz.
.TP 11
.B -e
Buzz will follow the amplitude envelope of a given speech signal.
.TP 11
.BI -S samplerate
Produce the buzz with the given sampling rate.  Default: 10000Hz;
.SH INPUT ITEMS
.IP FX 11
Any fundamental frequency item.
.IP SP 11
Any speech item (for envelope option).
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/
/* global structures */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <fcntl.h>
#include <string.h>
#include "sfs.h"

#define MIN(x,y) (((x)<(y))?(x):(y))

/* global data */
struct item_header	fxitem;
short	*fx;
struct item_header	spitem;
short	*sp;
struct item_header	opitem;
short	*osp;

/* parameters */
double	odur=0.0001;
int		doenv=0;

/* find signal envelope */
double	envelope(double start,double stop)
{
	int	bstart,numf;
	int	i,cnt;
	double	val,sumsq=0;

	if (start < 0) start=0;
	bstart = (int)(start/spitem.frameduration);
	numf = (int)((stop-start)/spitem.frameduration);

	cnt=0;
	for (i=bstart;(i<bstart+numf)&&(i<spitem.numframes);i++) {
		val = sp[i];
		sumsq += val * val;
		cnt++;
	}

	if (cnt==0) {
		/* printf("envelope(%g,%g)=%g\n",start,stop,0); */
		return(0);
	}
	else {
		/* printf("envelope(%g,%g)=%g\n",start,stop,sqrt(sumsq/cnt)); */
		return(sqrt(sumsq/cnt));
	}
}

/* get FX value at time */
int getfx(double t)
{
	int	idx;

	idx = (int)(0.5+t/((fxitem.windowsize-fxitem.overlap)*fxitem.frameduration));

	if ((0 <= idx) && (idx < fxitem.numframes))
		return(fx[idx]);
	else
		return(0);
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* option decoding */
	extern int	optind;
	extern char	*optarg;
	int		errflg=0;
	int		c;
	int32		it;		/* selected item type */
	char		*ty;		/* item specification */
	char		*fxtype="0";
	char		*sptype="0";

	/* file variables */
	char		filename[SFSMAXFILENAME];	/* data file */
	int		ocnt;
	int		pos;
	int		nsamp;
	double	f,p;
	double	amp,gradient;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:eS:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Convert FX to buzz V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i':	/* input item specification */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == FX_TYPE) {
					fxtype = ty;
				}
				else if (it == SP_TYPE) {
					sptype = ty;
					doenv=1;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'e' :	/* do envelope */
			doenv++;
			break;
		case 'S' :	/* sample rate */
			if (atof(optarg)<=0.0)
				error("bad sample rate");
			odur = 1.0/atof(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: fxbuzz (-I) (-i item) (-e) (-S srate) sfsfile\n",NULL);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* load FX */
	getitem(filename,FX_TYPE,fxtype,&fxitem,&fx);

	/* load speech */
	if (doenv)
		getitem(filename,SP_TYPE,sptype,&spitem,&sp);

	/* build output item */
	sfsheader(&opitem,SP_TYPE,0,1,2,odur,fxitem.offset,1,0,0);
	if (doenv)
		sprintf(opitem.history,"%s(%d.%02d,%d.%02d;srate=%d)",
			PROGNAME,
			fxitem.datatype,fxitem.subtype,
			spitem.datatype,spitem.subtype,
			(int)(0.5+1.0/odur));
	else
		sprintf(opitem.history,"%s(%d.%02d;srate=%d)",
			PROGNAME,
			fxitem.datatype,fxitem.subtype,
			(int)(0.5+1.0/odur));

	/* create output buffer */
	ocnt = (int)(fxitem.numframes * (fxitem.windowsize-fxitem.overlap) * fxitem.frameduration / odur);
	osp = sfsbuffer(&opitem,ocnt);

	/* build contour cycle by cycle */
	pos=0;
	nsamp=0;

	while (pos < ocnt) {
		if (nsamp <= 0) {
			f = getfx(pos*odur);
			if (f==0) {
				osp[pos++]=0;
			}
			else {
				p = 1.0/f;
				nsamp = (int)(p/odur);
				if (doenv)
					amp=(int)(2*envelope(pos*odur-p,pos*odur+p));
				else
					amp=16000;
				gradient = (2*amp)/nsamp;

				osp[pos++]=(int)amp;
				amp -= gradient;
				nsamp--;
			}
		}
		else {
			osp[pos++] = (int) amp;
			amp -= gradient;
			nsamp--;
		}
	}

	/* save item back to file */
	putitem(filename,&opitem,ocnt,osp);

	/* that's all folks */
	exit(0);
}


