/* fxsmooth -- smooth Fx contour using various techniques */

/* M.A.Huckvale - University College London */

/* version 1.0 - June 2002 */

#define PROGNAME "fxsmooth"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH FXSMOOTH 1 UCL
.SH NAME
fxsmooth - smooth a fundamental frequency contour
.SH SYNOPSIS
.B fxsmooth
(-i item) (-m median_smoother_ms) (-d) (-s smoothing_window_ms) (-j) sfsfile
.SH DESCRIPTION
.I fxsmooth
is a program to smooth a fundamental frequency contour stored in an fx item.
A number of smoothing technqiues are available: (i) the deglitch algorithm
attempts to remove sudden jumps in the contour; (ii) the median smoother replaces
each Fx value by the median of the values found in a window symmetrically placed
around the sample; (iii) the linear smoother applies a raised cosine window
symmetrically about each voiced value, (iv) the interpolation function joins up
the voiced regions so that a continuous curve is obtained.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item.
.TP 11
.BI -m median_smooth_ms
Specify size of window used for median smoothing in ms.
.TP 11
.B -d
Deglitch function.  Try to remove spikes and sudden jumps.
.TP 11
.BI -s smoothing_window_ms
Specify size of window used for linear (raised cosine window) smoothing in ms.
.TP 11
.B -j
Join voiced regions with an interpolated contour to create a continuous
curve.
.SH INPUT ITEMS
.IP FX 11
Any fundamental frequency item.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
.SH SEE ALSO
fxmomel
*/
/*--------------------------------------------------------------------------*/
/* global structures */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <fcntl.h>
#include <string.h>
#include "sfs.h"

#define MIN(x,y) (((x)<(y))?(x):(y))

/* global data */
struct item_header	fxitem;
struct item_header	ofxitem;
short	*fx;
int		fxhist[1000];

/* find IQR */
void findIQR(int *fxlo,int *fxhi)
{
	int	i,cnt,sum;

	memset(fxhist,0,sizeof(fxhist));
	cnt=0;
	for (i=0;i<fxitem.numframes;i++) {
		if ((0<fx[i])&&(fx[i]<1000)) {
			fxhist[fx[i]]++;
			cnt++;
		}
	}

	for (sum=0,*fxlo=1;sum<cnt/4;(*fxlo)++) sum += fxhist[*fxlo];
	for (sum=0,*fxhi=999;sum<cnt/4;(*fxhi)--) sum += fxhist[*fxhi];

	//printf("IQR: %d..%d\n",*fxlo,*fxhi);
}

/* median smoothing */
void mediansmoother(int wsize,double *fxin,double *fxout)
{
	double	*window;
	int		i,j,k;
	double	t;
	int		nzero;

	window = (double *)calloc(wsize+1,sizeof(double));

	for (i=0;i<fxitem.numframes;i++) {
		/* get samples */
		nzero=0;
		for (j=i-wsize/2,k=0;j<=i+wsize/2;j++,k++) {
			if ((j<0)||(j>=fxitem.numframes))
				window[k]=0;
			else if (fxin[j]<10) {
				window[k]=((nzero % 2)==0) ? 0 : 1000;
				nzero++;
			}
			else
				window[k]=fxin[j];
		}
		/* sort samples */
		for (j=1;j<wsize;j++) {
			k=j;
			t=window[k];
			while ((k>0)&&(t < window[k-1])) {
				window[k] = window[k-1];
				k--;
			}
			window[k]=t;
		}
		/* output median */
		if (nzero > wsize/2)
			fxout[i]=0;
		else
			fxout[i] = window[wsize/2];
	}

	free(window);
}

/* linear smoothing */
void linearsmoother(int wsize,double *fxin,double *fxout)
{
	double	*window;
	int		i,j,k;
	double	sum;
	int		wused;
	double	PI=4.0*atan(1.0);

	window = (double *)calloc(wsize+1,sizeof(double));

	for (i=0;i<fxitem.numframes;i++) {
		/* get samples */
		wused=0;
		for (j=i-wsize/2,k=0;j<=i+wsize/2;j++,k++) {
			if ((j<0)||(j>=fxitem.numframes))
				/* skip */;
			else if (fxin[j]<10)
				/* skip */;
			else
				window[wused++]=fxin[j];
		}
		/* apply window */
		if (wused==0)
			fxout[i]=0;
		else if (wused<wsize/2)
			fxout[i]=fxin[i];
		else {
			sum=0;
			for (j=0;j<wused;j++)
				sum += window[j] * (1.0-cos(2.0*PI*(j+1)/(wused+1)))/(wused+1);
			fxout[i] = sum;
		}
	}

	free(window);
}

/* join voiced regions with an interpolation */
void interpolate(double *fxbuf)
{
	double	sval,eval,gradient;
	int		start,stop;
	int		i,j;

	start=0;
	for (i=1;i<fxitem.numframes;i++) {
		if ((fxbuf[i-1]==0)&&(fxbuf[i]!=0)) {
			stop=i;
			sval = fxbuf[start];
			if (sval==0) sval=fxbuf[i];
			eval = fxbuf[i];
			gradient = (eval-sval)/(stop-start);
			for (j=start;j<stop;j++)
				fxbuf[j] = sval+gradient*(j-start);
		}
		else if ((fxbuf[i-1]!=0)&&(fxbuf[i]==0)) {
			start=i-1;
			sval=fxbuf[start];
		}
		else if ((i==fxitem.numframes-1)&&(fxbuf[i]==0)) {
			for (j=start;j<=i;j++)
				fxbuf[j] = sval;
		}
	}
}

/* main program */
void main(int argc,char **argv)
{
	/* option decoding */
	extern int	optind;
	extern char	*optarg;
	int		errflg=0;
	int		c;
	int32		it;		/* selected item type */
	char		*ty;		/* item specification */
	int		initem=FX_TYPE;
	char		*intype="0";

	/* file variables */
	char		filename[SFSMAXFILENAME];	/* data file */
	int		fid;
	int		i;
	double	medtime=0.05;
	int		medsize;
	int		domedian=0;
	int		deglitch=0;
	int		dolinear=0;
	int		dojoin=0;
	double	lintime=0.05;
	int		linsize;
	double	*fxbuf1,*fxbuf2;
	double	sum,mean;
	int		somechange,cnt;
	char	tbuf[256];
	int		fxlo,fxhi;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:m:ds:j")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: FX contour smoothing V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i':	/* input item specification */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == FX_TYPE) {
					initem = it;
					intype = ty;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'm' : /* median smoother size */
			domedian=1;
			medtime = atof(optarg);
			break;
		case 's' : /* median smoother size */
			dolinear=1;
			lintime = atof(optarg);
			break;
		case 'd' :	/* deglitch */
			deglitch=1;
			break;
		case 'j' :	/* join up contour */
			dojoin=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: fxsmooth (-I) (-i item) (-m mediansmoothtime_ms) (-d) (-j) (-s linearsmoothtime_ms) sfsfile\n",NULL);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* get fx data */
	getitem(filename,FX_TYPE,intype,&fxitem,&fx);

	/* get some processing buffers */
	fxbuf1 = (double *)calloc(fxitem.numframes,sizeof(double));
	fxbuf2 = (double *)calloc(fxitem.numframes,sizeof(double));

	/* copy data into buffer */
	for (i=0;i<fxitem.numframes;i++) fxbuf1[i] = fx[i];

	/* do deglitching */
	if (deglitch) {
		/* find IQR */
		findIQR(&fxlo,&fxhi);
		/* eliminate outliers */
		for (i=0;i<fxitem.numframes;i++) {
			if (fx[i] > fxhi+2*(fxhi-fxlo)) {
				fxbuf1[i]=0;
			}
			if (fx[i] < fxlo-2*(fxhi-fxlo)) {
				fxbuf1[i]=0;
			}
		}
		/* find mean */
		sum=0;
		cnt=0;
		for (i=0;i<fxitem.numframes;i++)
			if (fx[i] > 10) {
				sum += fxbuf1[i];
				cnt++;
			}
		if (cnt > 0)
			mean=sum/cnt;
		else
			mean=125;
		//fprintf(stderr,"Mean Fx = %.1fHz\n",mean);
		/* eliminate sudden jumps */
		do {
			somechange=0;
			for (i=1;i<fxitem.numframes;i++) {
				if ((fxbuf1[i]==0)||(fxbuf1[i-1]==0))
					/* skip */;
				else if (fxbuf1[i] > 1.70*fxbuf1[i-1]) {
					if (fabs(fxbuf1[i]-mean) < fabs(fxbuf1[i-1]-mean))
						fxbuf1[i-1] = fxbuf1[i];
					else
						fxbuf1[i] = fxbuf1[i-1];
					somechange++;
				}
				else if (fxbuf1[i] < fxbuf1[i-1]/1.70) {
					if (fabs(fxbuf1[i]-mean) < fabs(fxbuf1[i-1]-mean))
						fxbuf1[i-1] = fxbuf1[i];
					else
						fxbuf1[i] = fxbuf1[i-1];
					somechange++;
				}
			}
		} while (somechange);
	}

	/* do median smoothing */
	if (domedian) {
		medsize = (int)(0.5+0.001*medtime/(fxitem.frameduration*(fxitem.windowsize-fxitem.overlap)));
		if ((medsize%2)==0) medsize++;
		fprintf(stderr,"Median filter window = %d samples\n",medsize);
		if (medsize <= 2)
			error("median smoothing window too small");
		if (medsize > fxitem.numframes/4)
			error("median smoothing window too large");
		mediansmoother(medsize,fxbuf1,fxbuf2);
		for (i=0;i<fxitem.numframes;i++) fxbuf1[i] = fxbuf2[i];
	}

	/* do linear smoothing */
	if (dolinear) {
		linsize = (int)(0.5+0.001*lintime/(fxitem.frameduration*(fxitem.windowsize-fxitem.overlap)));
		fprintf(stderr,"Linear filter window = %d samples\n",linsize);
		if (linsize <= 2)
			error("linear smoothing window too small");
		if (linsize > fxitem.numframes/4)
			error("linear smoothing window too large");
		linearsmoother(linsize,fxbuf1,fxbuf2);
		for (i=0;i<fxitem.numframes;i++) fxbuf1[i] = fxbuf2[i];
	}

	/* do join contour */
	if (dojoin)
		interpolate(fxbuf1);

	/* make output header */
	sfsheader(&ofxitem,FX_TYPE,0,2,1,fxitem.frameduration,
		fxitem.offset,fxitem.windowsize,fxitem.overlap,fxitem.lxsync);
	sprintf(ofxitem.history,"%s(%d.%02d;",PROGNAME,
		fxitem.datatype,fxitem.subtype);
	if (deglitch) strcat(ofxitem.history,"deglitch");
	if (domedian) {
		if (deglitch) strcat(ofxitem.history,",");
		sprintf(tbuf,"median=%g",medtime);
		strcat(ofxitem.history,tbuf);
	}
	if (dolinear) {
		if (deglitch||domedian) strcat(ofxitem.history,",");
		sprintf(tbuf,"smooth=%g",lintime);
		strcat(ofxitem.history,tbuf);
	}
	if (dojoin) {
		if (deglitch||domedian||dolinear) strcat(ofxitem.history,",");
		sprintf(tbuf,"join",lintime);
		strcat(ofxitem.history,tbuf);
	}
	strcat(ofxitem.history,")");

	/* copy data */
	for (i=0;i<fxitem.numframes;i++)
		fx[i] = (short)fxbuf1[i];
	putitem(filename,&ofxitem,fxitem.numframes,fx);

	/* that's all folks */
	free(fxbuf1);
	free(fxbuf2);
	exit(0);
}


