/* g711 - simulate a telephone line using bandpass filtering and a-law companding */

/* Mark Huckvale - University College London */

/* version 1.0 - January 2010 */

#undef IAG

#define PROGNAME "g711"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH G711 SFS1 UCL
.SH NAME
g711 -- simulate a telephone line using bandpass filtering and a-law companding
.SH SYNOPSIS
.B g711
(-i item) file
.SH DESCRIPTION
.I g711
is a program to simulate the effect of a plain telephone line on a speech
signal. The input signal is band-pass filtered, resampled to 8000samples/sec, then
encoded with 8-bit logarithmic quantisation, and then decoded again to create
the output signal.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item.
.SH INPUT ITEMS
.IP 1.xx 11
Any speech item.
.SH VERSION/AUTHOR
.nf
1.0 - Mark Huckvale.
.fi
*/
/*--------------------------------------------------------------------------*/


/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"			/* header structures */
#include "filter.h"

/* global data */
struct item_header spitem;	/* input item header data */
short	*sp;
struct item_header opitem;	/* output item header */

#define min(x,y)	(((x)<(y))?(x):(y))
#define max(x,y)	(((x)>(y))?(x):(y))

/*
 * This source code is a product of Sun Microsystems, Inc. and is provided
 * for unrestricted use.  Users may copy or modify this source code without
 * charge.
 *
 * SUN SOURCE CODE IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING
 * THE WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 *
 * Sun source code is provided with no support and without any obligation on
 * the part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 *
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY THIS SOFTWARE
 * OR ANY PART THEREOF.
 *
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 *
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */

/*
 * g711.c
 *
 * u-law, A-law and linear PCM conversions.
 */

#define	SIGN_BIT	(0x80)		/* Sign bit for a A-law byte. */
#define	QUANT_MASK	(0xf)		/* Quantization field mask. */
#define	NSEGS		(8)		/* Number of A-law segments. */
#define	SEG_SHIFT	(4)		/* Left shift for segment number. */
#define	SEG_MASK	(0x70)		/* Segment field mask. */

static short seg_end[8] = {0xFF, 0x1FF, 0x3FF, 0x7FF,
			    0xFFF, 0x1FFF, 0x3FFF, 0x7FFF};

/* copy from CCITT G.711 specifications */
unsigned char _u2a[128] = {			/* u- to A-law conversions */
	1,	1,	2,	2,	3,	3,	4,	4,
	5,	5,	6,	6,	7,	7,	8,	8,
	9,	10,	11,	12,	13,	14,	15,	16,
	17,	18,	19,	20,	21,	22,	23,	24,
	25,	27,	29,	31,	33,	34,	35,	36,
	37,	38,	39,	40,	41,	42,	43,	44,
	46,	48,	49,	50,	51,	52,	53,	54,
	55,	56,	57,	58,	59,	60,	61,	62,
	64,	65,	66,	67,	68,	69,	70,	71,
	72,	73,	74,	75,	76,	77,	78,	79,
	81,	82,	83,	84,	85,	86,	87,	88,
	89,	90,	91,	92,	93,	94,	95,	96,
	97,	98,	99,	100,	101,	102,	103,	104,
	105,	106,	107,	108,	109,	110,	111,	112,
	113,	114,	115,	116,	117,	118,	119,	120,
	121,	122,	123,	124,	125,	126,	127,	128};

unsigned char _a2u[128] = {			/* A- to u-law conversions */
	1,	3,	5,	7,	9,	11,	13,	15,
	16,	17,	18,	19,	20,	21,	22,	23,
	24,	25,	26,	27,	28,	29,	30,	31,
	32,	32,	33,	33,	34,	34,	35,	35,
	36,	37,	38,	39,	40,	41,	42,	43,
	44,	45,	46,	47,	48,	48,	49,	49,
	50,	51,	52,	53,	54,	55,	56,	57,
	58,	59,	60,	61,	62,	63,	64,	64,
	65,	66,	67,	68,	69,	70,	71,	72,
	73,	74,	75,	76,	77,	78,	79,	79,
	80,	81,	82,	83,	84,	85,	86,	87,
	88,	89,	90,	91,	92,	93,	94,	95,
	96,	97,	98,	99,	100,	101,	102,	103,
	104,	105,	106,	107,	108,	109,	110,	111,
	112,	113,	114,	115,	116,	117,	118,	119,
	120,	121,	122,	123,	124,	125,	126,	127};

static int
search(
	int		val,
	short		*table,
	int		size)
{
	int		i;

	for (i = 0; i < size; i++) {
		if (val <= *table++)
			return (i);
	}
	return (size);
}

/*
 * linear2alaw() - Convert a 16-bit linear PCM value to 8-bit A-law
 *
 * linear2alaw() accepts an 16-bit integer and encodes it as A-law data.
 *
 *		Linear Input Code	Compressed Code
 *	------------------------	---------------
 *	0000000wxyza			000wxyz
 *	0000001wxyza			001wxyz
 *	000001wxyzab			010wxyz
 *	00001wxyzabc			011wxyz
 *	0001wxyzabcd			100wxyz
 *	001wxyzabcde			101wxyz
 *	01wxyzabcdef			110wxyz
 *	1wxyzabcdefg			111wxyz
 *
 * For further information see John C. Bellamy's Digital Telephony, 1982,
 * John Wiley & Sons, pps 98-111 and 472-476.
 */
unsigned char
linear2alaw(
	int		pcm_val)	/* 2's complement (16-bit range) */
{
	int		mask;
	int		seg;
	unsigned char	aval;

	if (pcm_val >= 0) {
		mask = 0xD5;		/* sign (7th) bit = 1 */
	} else {
		mask = 0x55;		/* sign bit = 0 */
		pcm_val = -pcm_val - 8;
	}

	/* Convert the scaled magnitude to segment number. */
	seg = search(pcm_val, seg_end, 8);

	/* Combine the sign, segment, and quantization bits. */

	if (seg >= 8)		/* out of range, return maximum value. */
		return (0x7F ^ mask);
	else {
		aval = seg << SEG_SHIFT;
		if (seg < 2)
			aval |= (pcm_val >> 4) & QUANT_MASK;
		else
			aval |= (pcm_val >> (seg + 3)) & QUANT_MASK;
		return (aval ^ mask);
	}
}

/*
 * alaw2linear() - Convert an A-law value to 16-bit linear PCM
 *
 */
int
alaw2linear(
	unsigned char	a_val)
{
	int		t;
	int		seg;

	a_val ^= 0x55;

	t = (a_val & QUANT_MASK) << 4;
	seg = ((unsigned)a_val & SEG_MASK) >> SEG_SHIFT;
	switch (seg) {
	case 0:
		t += 8;
		break;
	case 1:
		t += 0x108;
		break;
	default:
		t += 0x108;
		t <<= seg - 1;
	}
	return ((a_val & SIGN_BIT) ? t : -t);
}

/* cubic interpolation */
short cubint(short y0,short y1,short y2,short y3,double mu)
{
	int	a0 = y3 - y2 - y0 + y1;
	int a1 = y0 - y1 - a0;
	int a2 = y2 - y0;
	int a3 = y1;
	return((short)(a0*mu*mu*mu+a1*mu*mu+a2*mu+a3));
}

/* resample a signal by cubic interpolation */
int resample(short *sp,int len,double irate,double orate)
{
	double rfact=irate/orate;
	int		nframe=(int)(len/rfact);
	short 	*osp=(short *)calloc(nframe,sizeof(short));
	int		i,idx1,idx2,idx3,idx4;
	double	mu;
	FILTER	*lpfilt;

	/* low-pass filter */
	lpfilt = filter_design(FILTER_LOW_PASS,4,0.9*orate,0,irate);
	filter_signal(lpfilt,sp,sp,len);

	/* calculate each output sample in turn */
	for (i=0;i<nframe;i++) {
		idx2=(int)(i*rfact);
		idx1=max(0,idx2-1);
		idx3=min(len-1,idx2+1);
		idx4=min(len-1,idx3+1);
		mu=i*rfact-idx2;
		osp[i] = cubint(sp[idx1],sp[idx2],sp[idx3],sp[idx4],mu);
	}

	/* copy back */
	for (i=0;i<nframe;i++) sp[i]=osp[i];

	free(osp);
	return(nframe);
}

/* main program */
void main(int argc,char *argv[])
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	int		c;		/* option char */
	int		it;
	char	*ty;
	char	*sptype="0";
	char	filename[SFSMAXFILENAME]; 		/* database file name */
	int		fid;
	int		i;
	int		srate;
	FILTER	*bpfilt;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: telephone simulation V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item spec */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else
					error("unsuitable item specification %s",optarg);
			}
			else
				error("illegal item specification %s",optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) file\n",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("unable to find file '%s'",filename);
		else
			error("access error on '%s'",filename);
	}

	/* locate input item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("unable to find input item in '%s'",filename);

	/* check */
	srate=100*(int)(0.01/spitem.frameduration);
	if (srate < 8000)
		error("input signal must be sampled at 8000Hz at least");

	/* load signal */
	sp = (short *)sfsbuffer(&spitem,spitem.numframes);
	sfsread(fid,0,spitem.numframes,sp);
	sfsclose(fid);

	/* bandpass filter */
	bpfilt = filter_design(FILTER_BAND_PASS,8,300,3200,1.0/spitem.frameduration);
	filter_signal(bpfilt,sp,sp,spitem.numframes);

	/* resample if required */
	if (srate!=8000) {
		spitem.numframes=resample(sp,spitem.numframes,1.0/spitem.frameduration,8000);
		spitem.frameduration = 1.0/8000;
	}

	/* encode and decode */
	for (i=0;i<spitem.numframes;i++)
		sp[i] = alaw2linear(linear2alaw(sp[i]));

	/* create output */
	sfsheader(&opitem,SP_TYPE,0,2,1,spitem.frameduration,spitem.offset,0,0,1);
	sprintf(opitem.history,"%s(%d.%02d)",PROGNAME,spitem.datatype,spitem.subtype);

	/* write output */
	putitem(filename,&opitem,spitem.numframes,sp);

	/* that's all folks */
	exit(0);
}
