/* lpcdecomp -- decompose a speech signal into filter, pulses and noise */

/* M.A.Huckvale - University College London */

/* version 1.0 - January 2010 */

#undef IAG
#undef OLDCODE

#define PROGNAME "lpcdecomp"
#define PROGVERS "1.0"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH LPCDECOMP SFS1 UCL
.SH NAME
lpcdecomp - decomposes a speech signal into a filter plus residual
.SH SYNOPSIS
.B lpcdecomp
(-i item) (-n ncoeffs) (-p|-r) file
.SH DESCRIPTION
.I lpcdecomp
performs a fixed frame LPC analysis on sections of a speech waveform to
generate an LP filter and a residual waveform.
Optionally, the residual is then decomposed further into a pulse-only signal and a noise-only
signal.
The matching program lpcrecomp puts the bits back together again.
With no manipulation of the analysed components, recomposition is close to
lossless.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -n ncoeff
Specify number of predictor coefficients to use.  Default is (2+2*sample_rate/1000).
.TP 11
.B -p
Decompose the residual into two parts, one containing pulses and one noise.
.TP 11
.B -r
Randomise phase of noise residual signal.
.SH INPUT ITEMS
.IP 1.xx 11
Any speech waveform
.SH OUTPUT ITEMS
.IP PC 11
Filter coefficients.
.IP LX 11
Pulses
.IP LX 11
Noise
.SH VERSION/AUTHOR
.nf
1.0 - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include "complex.h"
#include "filter.h"
#include "lsys.h"
#include "lsp.h"
#include "fft.h"
#include "sfs.h"

#define MAXCOEFF	100	/* max # LPC coefficients */
#define	PREEMP		0.95	/* pre-emphasis coeff */
#define MAXWINDOW	22050	/* maximum analysis window size = 50ms @ 44.1kHz */
#define FFTSIZE 	512
#define RANPHASEFACTOR	3.0	/* i.e. +/- 0.5 radians */
#define MINSCALE	0.5

#define min(x,y)	(((x)<(y))?(x):(y))
#define max(x,y)	(((x)>(y))?(x):(y))

/* global data */
char		filename[SFSMAXFILENAME];	/* dbase file name */
struct item_header	spitem;		/* speech item header */
short		*sp;				/* speech buffer */
struct item_header	rspitem;	/* residual item header */
double		*rsp;				/* residual buffer */
struct item_header	pspitem;	/* pulses item header */
short		*psp;				/* pulses buffer */
struct item_header	nspitem;	/* noise item header */
short		*nsp;				/* noise buffer */
struct item_header	pcitem;		/* LPC item header */
struct co_rec	*lpcrec;		/* buffer for LPC data */
int			ncoeff;				/* # LPC coefficients */
int			*pcnt;

/* analysis parameters */
double		witime=0.030;
double		sttime=0.010;
int			wisize;
int			stsize;
double		pc[MAXCOEFF];
float		fftbuf[FFTSIZE];
float		fftbuf1[FFTSIZE];
float		fftbuf2[FFTSIZE];
float		fftbuf3[FFTSIZE];
int			fftsize=FFTSIZE;

/* create random noise at quantisation level */
double quantnoise()
{
	int v=rand()%10000;
	return ((v-5000)/10000.0);
}

/* sin(x)/x */
double sinc(double x)
{
	if (x==0)
		return(1.0);
	else
		return(sin(x)/x);
}


/* spectral analysis */
void spectanal(float *fbuf,double *rsp,int idx,int len,double *win)
{
	int	i;
	float	mag,arg;

	for (i=0;i<len;i++) fbuf[i] = (float)(rsp[idx+i]*win[i]);
	for (;i<fftsize;i++) fbuf[i]=0;
	REALFFT(fbuf,fftsize/2,1);
	fbuf[0]=fbuf[1]=0;
	for (i=1;i<fftsize/2;i++) {
		mag = (float)sqrt(fbuf[2*i]*fbuf[2*i]+fbuf[2*i+1]*fbuf[2*i+1]);
		arg = (float)(atan2(fbuf[2*i+1],fbuf[2*i]));
		fbuf[2*i] = mag;
		fbuf[2*i+1] = arg;
	}
}

/* spectral synthesis */
void spectsynth(float *fbuf,short *sig,int idx,int len,double *win)
{
	int	i;
	float	mag,arg;

	for (i=1;i<fftsize/2;i++) {
		mag = fbuf[2*i];
		arg = fbuf[2*i+1];
		fbuf[2*i] = (float)(mag*cos(arg));
		fbuf[2*i+1] = (float)(mag*sin(arg));
	}
	REALFFT(fbuf,fftsize/2,-1);
	for (i=0;i<len;i++)
		sig[idx+i] += (short)(win[i]*2*fbuf[i]/fftsize);
}

void triwin(double *buf,int len)
{
	int	i;
	for (i=0;i<len;i++)
		buf[i] = 2*(len/2-fabs(i-(len-1)/2.0))/len;
}


/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selections */
	char		*ty;
	char		*sptype="0";	/* default sub-type = last */
	/* file variables */
	int		fid,ofid1,ofid2,ofid3,ofid4;
	/* data variables */
	int		i,j,len;
	double	omega,sum,sumsq,val;
	double	*fsp,*frsp,*fwsp,*dp,*wsp;
	int		nframes;
	short	xbuf[4096];
	int		psize,psize2,nsize,nsize2;
	FILTER	*lpfilt,*hpfilt;
	double	peak;
	int		dopulse=0;
	int		doranphase=0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:n:rp")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: decompose speech signal to filter, pulses & noise V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'n' :	/* specify # coeffs */
			ncoeff = atoi(optarg);
			if (ncoeff < 2)
				error("too few poles: '%s'",optarg);
			if (ncoeff > MAXCOEFF) ncoeff=MAXCOEFF;
			break;
		case 'p' :	/* output pulse & noise residuals */
			dopulse=1;
			break;
		case 'r' :	/* randomise noise phase */
			doranphase=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-n numcoeff) (-p|-r) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* check file ok for writing */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on %s",filename);

	/* locate and load speech data */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("unable to find input speech item in %s",filename);
	sp = (short *)sfsbuffer(&spitem,spitem.numframes);
	sfsread(fid,0,spitem.numframes,sp);

	/* create buffer for LPC coefficients */
	if (ncoeff==0) ncoeff = (int)(2 + 0.001/spitem.frameduration);
	if (ncoeff > MAXCOEFF) ncoeff=MAXCOEFF;
	wisize=(int)(0.5+witime/spitem.frameduration);
	stsize=(int)(0.5+sttime/spitem.frameduration);
	nframes = (spitem.numframes-wisize)/stsize;

	sfsheader(&pcitem,PC_TYPE,-1,4,sfsstruct[PC_TYPE]/4 + ncoeff,
			spitem.frameduration,spitem.offset,wisize,wisize-stsize,0);
	sprintf(pcitem.history,"%s/PC(%d.%02d;win=%g;step=%g,ncoeff=%d)",
		PROGNAME,
		spitem.datatype,spitem.subtype,
		witime,sttime,ncoeff);
	if ((lpcrec=(struct co_rec *)sfsbuffer(&pcitem,nframes))==NULL)
		error("cannot get LPC buffer",NULL);
	ofid1=sfschannel(filename,&pcitem);

	/* get input signal into float buffer */
	fsp = (double *)calloc(spitem.numframes,sizeof(double));

	/* pre-emp whole signal */
	fsp[0]=0;
	for (j=1;j<spitem.numframes;j++) fsp[j]=sp[j]-PREEMP*sp[j-1];

	/* get data buffers */
	rsp = (double *)calloc(spitem.numframes,sizeof(double));
	psp = (short *)calloc(spitem.numframes,sizeof(short));
	pcnt = (int *)calloc(spitem.numframes,sizeof(int));
	nsp = (short *)calloc(spitem.numframes,sizeof(short));
	fwsp = (double *)calloc(wisize,sizeof(double));
	frsp = (double *)calloc(wisize,sizeof(double));
	wsp = (double *)calloc(wisize,sizeof(double));

//	triwin(wsp,wisize);
//	printf("window="); for (i=0;i<wisize;i++) printf("%g,",wsp[i]); printf("\n");
	for (i=0;i<wisize;i++) wsp[i] = 0.50-0.50*cos(2*M_PI*i/(wisize-1));

	/* do LPC analysis */
	for (i=0;i<nframes;i++) {

		/* get speech */
		for (j=0;j<wisize;j++) fwsp[j]=fsp[i*stsize+j] + 0.01*quantnoise();

		/* remove mean & window */
		sum=0;
		for (j=0;j<wisize;j++) sum += fwsp[j];
		sum /= wisize;
		omega=(2*M_PI)/(wisize-1);
		for (j=0;j<wisize;j++)
			fwsp[j] = (fwsp[j]-sum)*wsp[j];

		/* calculate RMS energy per sample */
		sum=0;
		for (j=0;j<wisize;j++) sum += fwsp[j]*fwsp[j];
		sum = sqrt(sum/wisize);

		/* perform LPC analysis on this window */
		lsp_auto_lpc(fwsp,wisize,pc,ncoeff);

		/* save */
		lpcrec[i].posn=i*stsize;
		lpcrec[i].size=wisize;
		lpcrec[i].flag=0;
		lpcrec[i].mix=0;
		lpcrec[i].gain=(float)sum;
		for (j=0;j<ncoeff;j++)
			lpcrec[i].data[j]=(float)pc[j+1];
		sfswrite(ofid1,1,&lpcrec[i]);

		/* calculate residual */
		lsp_residual_lpc(fwsp,wisize,pc,ncoeff,frsp);
		for (j=0;j<wisize;j++)
			rsp[i*stsize+j] += frsp[j];

	}

	if (dopulse==0) {
		/* save whole residual */
		sfsheader(&rspitem,LX_TYPE,0,2,1,spitem.frameduration,spitem.offset,1,0,0);
		sprintf(rspitem.history,"%s/LX(%d.%02d;residual,ncoeff=%d)",
			PROGNAME,
			spitem.datatype,spitem.subtype,ncoeff);
		ofid4 = sfschannel(filename,&rspitem);
		len=spitem.numframes;
		dp=rsp;
		while (len > 0) {
			for (i=0;(i<4096)&&(len > 0);i++,len--) xbuf[i]=(short)(*dp++);
			sfswrite(ofid4,i,xbuf);
		}
	}
	else {

#ifdef OLDCODE

		/* separate residual into pulses + noise */
		nsize = (int)(0.5+0.004/spitem.frameduration);
		nsize2 = nsize/2;
		sum=0;
		for (i=0;i<nsize;i++) sum += 0.5-0.5*cos(2*M_PI*i/(nsize-1));
		for (i=0;i<nsize;i++) fwsp[i]= -(0.5-0.5*cos(2*M_PI*i/(nsize-1)))/sum;

		psize = (int)(0.5+0.001/spitem.frameduration);
		psize2 = psize/2;
		sum=0;
		for (i=0;i<psize;i++) sum += 0.5-0.5*cos(2*M_PI*i/(psize-1));
		for (i=0;i<psize;i++) fwsp[i+nsize2-psize2] += (0.5-0.5*cos(2*M_PI*i/(psize-1)))/sum;


	//	lpfilt = filter_design(FILTER_LOW_PASS,4,4000,4000,1.0/spitem.frameduration);
	//	for (i=0;i<spitem.numframes;i++)
	//		rsp[i] = filter_sample(lpfilt,rsp[i]);

		for (i=nsize2;i<spitem.numframes-1-nsize2;i++) {
			sumsq=0;
			for (j=0;j<nsize;j++) {
				val= rsp[i-nsize2+j];
				sumsq += val*val*fwsp[j];
			}
			psp[i] = (short)sqrt(sumsq);
	//		sumsq=0;
	//		for (j=0;j<nsize;j++) {
	//			val= rsp[i-nsize2+j];
	//			sumsq += val*val*fwsp[j];
	//		}
	//		nsp[i] = (short)sqrt(sumsq);
		}

		/* do peak follower */
		peak=psp[0];
		for (i=1;i<spitem.numframes;i++) {
			peak = 0.99 * peak;
			if (peak < psp[i]) {
				peak=psp[i];
				nsp[i]=(short)peak;
			}
			else
				nsp[i]=0;
		}

		/* copy over residual */
		memset(psp,0,spitem.numframes*sizeof(short));
		psize = (int)(0.5+0.002/spitem.frameduration);
		psize2 = psize/2;
		for (i=0;i<psize;i++) frsp[i]=0.5-0.5*cos(2*M_PI*i/(psize-1));
		for (i=psize/2;i<(spitem.numframes-1-psize2);i++) {
			if (nsp[i]!=0) {
				for (j=0;j<psize;j++) {
					psp[i-psize2+j] += (short)(rsp[i-psize2+j]*frsp[j]);
					pcnt[i-psize2+j]++;
				}
			}
		}

		/* normalise */
		for (i=0;i<spitem.numframes;i++) {
			if (pcnt[i]>0)
				psp[i] /= pcnt[i];
		}

		/* create noise residual */
		memset(nsp,0,spitem.numframes*sizeof(short));
		if (doranphase) {
			/* randomise noise phase */
			psize = (int)(0.5+0.0075/spitem.frameduration);
			for (i=0;i<psize;i++) fwsp[i] = sqrt(0.5-0.5*cos(2*M_PI*i/(psize-1)));
			for (i=0;i<spitem.numframes-1-psize;i+=psize/2) {
				for (j=0;j<psize;j++) fftbuf[j]=(float)((rsp[i+j]-psp[i+j])*fwsp[j]);
				for (;j<FFTSIZE;j++) fftbuf[j]=0;
				REALFFT(fftbuf,FFTSIZE/2,1);
				fftbuf[0]=fftbuf[1]=0;
				for (j=1;j<FFTSIZE/2;j++) {
					double r = sqrt(fftbuf[2*j]*fftbuf[2*j]+fftbuf[2*j+1]*fftbuf[2*j+1]);
					double arg = atan2(fftbuf[2*j+1],fftbuf[2*j]);
					arg += RANPHASEFACTOR * (rand()%2000-1000)/1000.0;
					fftbuf[2*j] = (float)(r*cos(arg));
					fftbuf[2*j+1] = (float)(r*sin(arg));
				}
				REALFFT(fftbuf,FFTSIZE/2,-1);
				for (j=0;j<psize;j++)
					nsp[i+j] += (short)((2*fftbuf[j]/FFTSIZE) * fwsp[j]);
			}
		}
		else {
			/* subtract to get noise */
			for (i=0;i<spitem.numframes;i++) {
				nsp[i] = (short)(rsp[i] - psp[i]);
			}
		}
#else
		/* separate residual into foreground + background */
		memset(psp,0,spitem.numframes*sizeof(short));
		memset(nsp,0,spitem.numframes*sizeof(short));

		/* choose window size */
		nsize = (int)(0.5+0.003/spitem.frameduration);
		nsize2 = nsize/2;
		for (i=0;i<nsize;i++) fwsp[i]= sqrt(0.54-0.46*cos(2*M_PI*i/(nsize-1)));
		fftsize=16;
		while (fftsize < nsize) fftsize *= 2;
		if (fftsize > FFTSIZE) fftsize=FFTSIZE;

		spectanal(fftbuf1,rsp,0,nsize,fwsp);
		spectanal(fftbuf2,rsp,nsize2,nsize,fwsp);
		for (i=nsize;i<spitem.numframes-nsize-1;i+=nsize/2) {
			spectanal(fftbuf3,rsp,i,nsize,fwsp);

			/* find minimum energy in each frame triplet */
			fftbuf[0]=fftbuf2[0];
			fftbuf[1]=fftbuf2[1];
			for (j=1;j<fftsize/4;j++) {
				if (fftbuf1[2*j]<fftbuf2[2*j]) {
					if (fftbuf1[2*j]<fftbuf3[2*j]) {
						fftbuf[2*j] = (float)(MINSCALE * fftbuf1[2*j]);
					}
					else {
						fftbuf[2*j] = (float)(MINSCALE * fftbuf3[2*j]);
					}
				}
				else {
					if (fftbuf2[2*j]<fftbuf3[2*j]) {
						fftbuf[2*j] = (float)(MINSCALE * fftbuf2[2*j]);
					}
					else {
						fftbuf[2*j] = (float)(MINSCALE * fftbuf3[2*j]);
					}
				}
				fftbuf[2*j+1] = fftbuf2[2*j+1];
			}
			for (;j<fftsize/2;j++) {
				fftbuf[2*j] = (float)(/*MINSCALE * */ fftbuf2[2*j]);
				fftbuf[2*j+1] = fftbuf2[2*j+1];
			}

			/* subtract minimum energy */
			for (j=0;j<fftsize/2;j++) {
				fftbuf1[2*j] = fftbuf2[2*j] - fftbuf[2*j];
				fftbuf1[2*j+1] = fftbuf2[2*j+1];
			}

			/* synthesize and overlap add foreground */
			spectsynth(fftbuf1,psp,i-nsize2,nsize,fwsp);

			/* randomise background */
			if (doranphase) {
				for (j=1;j<fftsize/2;j++) {
					fftbuf[2*j+1] += (float)(RANPHASEFACTOR * (rand()%2000-1000)/1000.0);
				}
			}

			/* synthesize and overlap add background */
			spectsynth(fftbuf,nsp,i-nsize2,nsize,fwsp);

			memcpy(fftbuf1,fftbuf2,FFTSIZE*sizeof(float));
			memcpy(fftbuf2,fftbuf3,FFTSIZE*sizeof(float));
		}


#endif

		/* save pulse signal */
		sfsheader(&pspitem,LX_TYPE,0,2,1,spitem.frameduration,spitem.offset,1,0,0);
		sprintf(pspitem.history,"%s/LX(%d.%02d;pulses,ncoeff=%d)",
			PROGNAME,
			spitem.datatype,spitem.subtype,ncoeff);
		ofid2 = sfschannel(filename,&pspitem);
		sfswrite(ofid2,spitem.numframes,psp);

		/* save noise signal */
		sfsheader(&nspitem,LX_TYPE,0,2,1,spitem.frameduration,spitem.offset,1,0,0);
		sprintf(nspitem.history,"%s/LX(%d.%02d;noise,ncoeff=%d%s)",
			PROGNAME,
			spitem.datatype,spitem.subtype,ncoeff,(doranphase)?",randomphase":"");
		ofid3 = sfschannel(filename,&nspitem);
		sfswrite(ofid3,spitem.numframes,nsp);

	}

	/* update */
	if (!sfsupdate(filename) != 0)
		error("update error on %s",filename);

	/* that's all folks ... */
	exit(0);
}

