/* lpcrecomp - recompose signal from lpcdecomp analysis into filter, pulses & noise */

/* M.A.Huckvale - University College London */

/* version 1.0 - January 2010 */

#define PROGNAME "lpcrecomp"
#define PROGVERS "1.0"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH LPCRECOMP SFS1 UCL
.SH NAME
lpcrecomp - recompose speech signal from filter and residual
.SH SYNOPSIS
.B lpcrecomp
(-i item) file
.SH DESCRIPTION
.I lpcrecomp
reverses the analysis of lpcdecomp to resynthesize a waveform from the
LP filter and residual, or filter plus pulse residual and noise residual.
If two residual waveforms are selected, they are added together
then passed through the filter to obtain the output signal.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item number for LP coefficients, and one or two residual waveforms.
.SH INPUT ITEMS
.IP 14.xx 11
Predictor coefficients
.IP 2.xx 11
Pulse residual, Noise residual
.SH OUTPUT ITEMS
.IP 1 11
Resynthesized speech signal
.SH VERSION/AUTHOR
.nf
1.0 - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "complex.h"
#include "lsys.h"
#include "lsp.h"
#include "sfs.h"

#define MAXCOEFF	50	/* max # LPC coefficients */
#define	PREEMP		0.95	/* pre-emphasis coeff */
#define MAXWINDOW	2205	/* maximum analysis window size = 50ms @ 44.1kHz */

/* global data */
struct item_header	spitem;		/* speech item header */
short			*sp;		/* speech buffer */
struct item_header	lxitem;		/* residual item header */
short			*lx;		/* residual buffer */
struct item_header	lxitem2;		/* residual item header */
short			*lx2;		/* residual buffer */
struct item_header	pcitem;		/* LPC item header */
struct co_rec	*lpcrec;	/* buffer for LPC data */
int			ncoeff;				/* # LPC coefficients */
double		pc[MAXCOEFF];
double		lsp[MAXCOEFF];

void triwin(double *buf,int len)
{
	int	i;
	for (i=0;i<len;i++)
		buf[i] = 2*(len/2-fabs(i-(len-1)/2.0))/len;
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selections */
	char		*ty;
	char		*pctype="0";	/* default sub-type = last */
	char		*lxtype1="0";	/* default sub-type = last */
	char		*lxtype2="0";	/* default sub-type = last */
	int			lxcnt=0;
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	int			fid,ofid;
	/* data variables */
	int		i,j;
	double	omega,factor,sum;
	double	*fsp,*frsp,*wsp;
	int		lspcoded=0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: LPC recomposition from filter and residual(s) V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == PC_TYPE)
					pctype = ty;
				else if ((it == LX_TYPE)&&(lxcnt==0)) {
					lxtype1 = ty;
					lxcnt=1;
				}
				else if (it == LX_TYPE) {
					lxtype2 = ty;
					lxcnt=2;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* check file ok for writing */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on %s",filename);

	/* locate residual */
	if (!sfsitem(fid,LX_TYPE,lxtype1,&lxitem))
		error("unable to find input lx item in %s",filename);

	/* create buffer and load */
	lx=(short *)sfsbuffer(&lxitem,lxitem.numframes);
	sfsread(fid,0,lxitem.numframes,lx);

	if (lxcnt==2) {
		/* locate residual */
		if (!sfsitem(fid,LX_TYPE,lxtype2,&lxitem2))
			error("unable to find input lx item in %s",filename);

		if ((int)(1.0/lxitem.frameduration)!=(int)(1.0/lxitem2.frameduration))
			error("two residual waveforms do not match in sample rate");

		if (lxitem2.numframes > lxitem.numframes) lxitem2.numframes=lxitem.numframes;

		/* create buffer and load */
		lx2=(short *)sfsbuffer(&lxitem2,lxitem2.numframes);
		sfsread(fid,0,lxitem2.numframes,lx2);

		/* add */
		for (i=0;i<lxitem2.numframes;i++) lx[i] += lx2[i];
	}

	/* locate PC data */
	if (!sfsitem(fid,PC_TYPE,pctype,&pcitem))
		error("unable to find input PC item in %s",filename);
	if (strcmp(params(pcitem.params,"format",""),"LSP")==0) lspcoded=1;

	if ((int)(0.5+0.01/pcitem.frameduration)!=(int)(0.5+0.01/lxitem.frameduration))
		error("PC sample rate does not match residual");

	/* create buffer for LPC coefficients */
	ncoeff = SFSRECSIZE(&pcitem);
	if ((lpcrec=(struct co_rec *)sfsbuffer(&pcitem,1))==NULL)
		error("cannot get LPC buffer",NULL);

	/* create output item */
	sfsheader(&spitem,SP_TYPE,0,2,1,lxitem.frameduration,lxitem.offset,1,0,0);
	if (lxcnt==2)
		sprintf(spitem.history,"%s(%d.%02d,%d.%02d,%d.%02d)",
			PROGNAME,
			pcitem.datatype,pcitem.subtype,lxitem.datatype,lxitem.subtype,
			lxitem2.datatype,lxitem2.subtype);
	else
		sprintf(spitem.history,"%s(%d.%02d,%d.%02d)",
			PROGNAME,
			pcitem.datatype,pcitem.subtype,lxitem.datatype,lxitem.subtype);

	/* open output channel */
	if ((ofid=sfschannel(filename,&spitem))<0)
		error("could not open temporary file",NULL);

	/* get data buffers */
	sp = (short *)sfsbuffer(&spitem,lxitem.numframes);
	for (i=0;i<lxitem.numframes;i++) sp[i]=0;

	fsp = (double *)calloc((int)(0.5/spitem.frameduration),sizeof(double));
	wsp = (double *)calloc((int)(0.5/spitem.frameduration),sizeof(double));
	frsp = (double *)calloc((int)(0.5/spitem.frameduration),sizeof(double));

	/* process speech frame by frame */
	for (i=0;i<pcitem.numframes;i++) {

		/* get PC frame */
		sfsread(fid,i,1,lpcrec);

		/* check residual long enough */
		if ((lpcrec->posn+lpcrec->size) < lxitem.numframes) {

//			omega=(2*M_PI)/(lpcrec->size-1);
//			for (j=0;j<lpcrec->size;j++) wsp[j] = (0.50-0.50*cos(j*omega));
			triwin(wsp,lpcrec->size);

			for (j=0;j<lpcrec->size;j++)
				frsp[j]=lx[lpcrec->posn+j];

			if (lspcoded) {
				for (j=0;j<ncoeff;j++) lsp[j]=lpcrec->data[j];
				lsp_synthesis(frsp,lpcrec->size,lsp,ncoeff,fsp);
			}
			else {
				pc[0]=1.0;
				for (j=0;j<ncoeff;j++) pc[j+1]=lpcrec->data[j];
				lsp_synth_lpc(frsp,lpcrec->size,pc,ncoeff,fsp);
			}

			/* get scaling factor */
			sum=0;
			for (j=0;j<lpcrec->size;j++) sum += fsp[j]*fsp[j]*wsp[j]*wsp[j];
			sum = sqrt(sum/lpcrec->size);
			factor = lpcrec->gain/sum;

			/* overlap add */
			for (j=0;j<lpcrec->size;j++)
				sp[lpcrec->posn+j] = (short)(sp[lpcrec->posn+j]+fsp[j]*factor*wsp[j]);

		}

		/* print progress */
		if (ttytest()) {
			printf("\rFrame %d/%d",i+1,pcitem.numframes);
			fflush(stdout);
		}
	}
	if (ttytest()) printf("\r                     \r");

	/* de-emphasis */
	sp[0]=0;
	for (j=1;j<lxitem.numframes;j++) sp[j]=(short)(sp[j]+PREEMP*sp[j-1]);

	/* write speech */
	sfswrite(ofid,lxitem.numframes,sp);

	/* update */
	if (!sfsupdate(filename) != 0)
		error("update error on %s",filename);

	free(fsp);
	free(frsp);
	free(wsp);

	/* that's all folks ... */
	exit(0);
}

