/* lspsynth -- line spectral pairs + residual to speech waveform */

/* M.A.Huckvale - University College London */

/* version 1.0 - November 2006 */

#define PROGNAME "lspsynth"
#define PROGVERS "1.0"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH LSPSYNTH SFS1 UCL
.SH NAME
lspsynth - synthesize speech waveform from line spectral pairs + residual
.SH SYNOPSIS
.B lspsynth
(-i item) (-r) (-l dblevel) file
.SH DESCRIPTION
.I lspsynth
reverses the analysis of lspanal(SFS1) to resynthesize a waveform from the
line spectral frequencies and a residual.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.B -r
Synthesize the signal backwards.  Pitch synchronous analysis only.
.TP 11
.BI -l dblevel
Change level by this amount (decibels).
.SH INPUT ITEMS
.IP 14.xx 11
Predictor coefficients stored as LSPs
.IP 2.xx 11
Residual
.SH OUTPUT ITEMS
.IP 1 11
Resynthesized speech signal
.SH VERSION/AUTHOR
.nf
1.0 - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "complex.h"
#include "lsys.h"
#include "lsp.h"
#include "sfs.h"

#define MAXCOEFF	50	/* max # LPC coefficients */
#define	PREEMP		0.95	/* pre-emphasis coeff */
#define MAXWINDOW	2205	/* maximum analysis window size = 50ms @ 44.1kHz */

/* global data */
struct item_header	spitem;		/* speech item header */
short			*sp;		/* speech buffer */
struct item_header	lxitem;		/* residual item header */
short			*lx;		/* residual buffer */
struct item_header	pcitem;		/* LPC item header */
struct co_rec	*lpcrec;	/* buffer for LPC data */
int			ncoeff;				/* # LPC coefficients */
double		lsp[MAXCOEFF];

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selections */
	char		*ty;
	char		*pctype="0";	/* default sub-type = last */
	char		*lxtype="0";	/* default sub-type = last */
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	int			fid,ofid;
	/* data variables */
	int		i,j;
	int		doreverse=0;
	double	omega,factor,sum;
	double	*fsp,*frsp;
	double	scale=1.0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:rl:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Line spectral pair synthesis V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == PC_TYPE)
					pctype = ty;
				else if (it == LX_TYPE) {
					lxtype = ty;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'r':
			doreverse=1;
			break;
		case 'l':
			scale = pow(10.0,atof(optarg)/20);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-r) (-l scaledb) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* check file ok for writing */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on %s",filename);

	/* locate residual */
	if (!sfsitem(fid,LX_TYPE,lxtype,&lxitem))
		error("unable to find input lx item in %s",filename);

	/* create buffer and load */
	lx=(short *)sfsbuffer(&lxitem,lxitem.numframes);
	sfsread(fid,0,lxitem.numframes,lx);

	/* locate PC data */
	if (!sfsitem(fid,PC_TYPE,pctype,&pcitem))
		error("unable to find input PC item in %s",filename);
	if (strcmp(params(pcitem.params,"format","LSP"),"LSP")!=0)
		error("PC item not coded as LSPs");
	if ((int)(0.5+0.01/pcitem.frameduration)!=(int)(0.5+0.01/lxitem.frameduration))
		error("PC sample rate does not match residual");

	/* create buffer for LPC coefficients */
	ncoeff = SFSRECSIZE(&pcitem);
	if ((lpcrec=(struct co_rec *)sfsbuffer(&pcitem,1))==NULL)
		error("cannot get LPC buffer",NULL);

	/* create output item */
	sfsheader(&spitem,SP_TYPE,0,2,1,lxitem.frameduration,lxitem.offset,1,0,0);
	sprintf(spitem.history,"%s(%d.%02d,%d.%02d%s)",
		PROGNAME,
		pcitem.datatype,pcitem.subtype,lxitem.datatype,lxitem.subtype,
		(doreverse)?";reverse":"");

	/* open output channels */
	if ((ofid=sfschannel(filename,&spitem))<0)
		error("could not open temporary file",NULL);

	/* get data buffers */
	sp = (short *)sfsbuffer(&spitem,lxitem.numframes);
	for (i=0;i<lxitem.numframes;i++) sp[i]=0;

	fsp = (double *)calloc((int)(0.5/spitem.frameduration),sizeof(double));
	frsp = (double *)calloc((int)(0.5/spitem.frameduration),sizeof(double));

	/* process speech frame by frame */
	for (i=0;i<pcitem.numframes;i++) {

		/* get PC frame */
		if (doreverse)
			sfsread(fid,pcitem.numframes-i-1,1,lpcrec);
		else
			sfsread(fid,i,1,lpcrec);

		/* check residual long enough */
		if ((lpcrec->posn+lpcrec->size) < lxitem.numframes) {
			for (j=0;j<ncoeff;j++)
				lsp[j]=lpcrec->data[j];
			omega=(2*M_PI)/(lpcrec->size-1);
			for (j=0;j<lpcrec->size;j++)
				frsp[j]=lx[lpcrec->posn+j]*(0.50-0.50*cos(omega*j));
			lsp_synthesis(frsp,lpcrec->size,lsp,ncoeff,fsp);

			/* get scaling factor */
			sum=0;
			for (j=0;j<lpcrec->size;j++) sum += fsp[j]*fsp[j];
			sum = sqrt(sum/lpcrec->size);
			factor = scale*lpcrec->gain/sum;

			if (doreverse) {
				lpcrec->posn = pcitem.lastposn-(lpcrec->posn+lpcrec->size);
				if (lpcrec->posn < 0) lpcrec->posn=0;
			}

			for (j=0;j<lpcrec->size;j++)
				sp[lpcrec->posn+j] = (short)(sp[lpcrec->posn+j]+fsp[j]*factor);

		}

		/* print progress */
		if (ttytest()) {
			printf("\rFrame %d/%d",i+1,pcitem.numframes);
			fflush(stdout);
		}
	}
	if (ttytest()) printf("\r                     \r");

	/* de-emphasis */
	sp[0]=0;
	for (j=1;j<lxitem.numframes-1;j++) sp[j]=(short)(sp[j]+PREEMP*sp[j-1]);

	/* write speech */
	sfswrite(ofid,lxitem.numframes,sp);

	/* update */
	if (!sfsupdate(filename) != 0)
		error("update error on %s",filename);

	/* that's all folks ... */
	exit(0);
}

