/* phosynth -- formant synthesis from MBROLA .PHO file format */

/* M.A.Huckvale - University College London */

/* version 1.0 - May 2000 */

#undef IAG

#define PROGNAME "phosynth"
#define PROGVERS "1.4"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH PHOSYNTH 1 SFS UCL
.SH NAME
phosynth - formant synthesis by rule from MBROLA format control file
.SH SYNOPSIS
.B phosynth
(-I) (-l|-s) (-m file.pho|-t transcription) (-a) (-p patchfile) (out.sfs)
.SH DESCRIPTION
.I phosynth
is a program to perform synthesis by rule for a formant synthesizer.
It takes as input a phonetic transcription and a fundamental frequency
contour and produces synthesizer control data.  The input can come from
an input AN item and FX item or from a .PHO file as used by the MBROLA
diphone synthesis
system.  Output is direct to the synthesizer, or to a listing file or
to an SFS file.
.PP
The formant parameter tracks are produced using the Holmes, Mattingly
and Shearme algorithm (1964) and target values are adapted from the JSRU
text to speech system.
.PP
Here is an example for you to try:
.nf
_ 50
t 95 0 150
e 133 0 150 99 110
s 150
t 135 0 110
_ 50
.fi
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input AN and FX items.
.TP 11
.B -l
Send synthesizer control data to standard output in format
compatible with sylist(SFS1) and syload(SFS1).
.TP 11
.B -s
Save synthesizer control data to SFS file.
.TP 11
.BI -m mbrola.pho
Take input from MBROLA format .PHO file.
.TP 11
.BI -t transcription
Take input from a phonetic transcription supplied on command line.  For example
"_ t e s t _".
.TP 11
.B -a
Save transcription as annotations to SFS file specified (must exist).
For transcription (-t) input only.
.TP 11
.BI -p patchfile
Reads in patches to the internal segment parameter table from the
supplied text file.  Commands in the file are separated by white
space, commas or semi-colons.  Each command is of the form
<parameter>=<value>.  Available parameters are:
segment, rank, target<PAR>, externdur<PAR>, interndur<PAR>.
The PAR values are; FN, F1, F2, F3, AL, A1, A2, A3, AH, VR.
The segment command
sets the context for the subsequent commands until the next segment
command.  For example:
.nf
segment=A:, targetF1=700, targetF2=1060, targetF3=2950
.fi
Lines starting with '#' are considered comments.
.SH VERSION/AUTHOR
.IP 1.4
Mark Huckvale
.SH SEE ALSO
repros
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include "sfs.h"
#include "phonetic.h"		/* phonetic data table */

char *paramnames[NUM_PARAM]={
	"FN", "F1", "F2", "F3", "AL", "A1", "A2", "A3", "AH", "VR"
};

/* identities of parameters in SY frame */
#define	SYFX	0
#define SYAH	1
#define SYMS	2
#define SYF1	3
#define SYA1	4
#define SYB1	5
#define SYF2	6
#define SYA2	7
#define SYB2	8
#define SYF3	9
#define SYA3	10
#define SYB3	11
#define SYF4	12
#define SYA4	13
#define SYB4	14
#define SYFN	15
#define SYAN	16
#define SYBN	17
#define SYVR	18

/* label sequence */
struct label_rec {
	char	*label;			/* phonetic label */
	double	stime;			/* start time */
	double	etime;			/* end time */
};

/* synthesis unit sequence */
struct synth_rec {
	int	phidx;			/* index in phonetic table */
	double	stime;			/* start time */
	double	etime;			/* end time */
	double	ibound[NUM_PARAM];	/* incoming boundary (Hz/dB) */
	double	itrans[NUM_PARAM];	/* incoming transition duration (s) */
	double	target[NUM_PARAM];	/* target values (Hz/dB) */
	double	ebound[NUM_PARAM];	/* outgoing boundary (Hz/dB) */
	double	etrans[NUM_PARAM];	/* outgoing transition duration (s) */
};

/* global data */
double			totdur=0;	/* total duration */
struct label_rec	*itab;		/* input table */
int			icount;		/* # entries */
struct synth_rec	*stab;		/* synthesis table */
int			scount;		/* # entries */
short			*fx;		/* fx contour */
int			fxcount;	/* # samples */
short			*sy;		/* encoded synthesizer control data */
int			sycount;	/* #frames in sy data */
int			dosfs=1;	/* input from SFS file */
int			dombrola=0;	/* input from mbrola file */
int			dotrans=0;	/* input from transcription */
int			dosynth=1;	/* output to synthesizer */
int			dolist=0;	/* output control data */
int			dosavesfs=0;	/* output to SFS file */
int			dosavetxt=0;	/* output to TXT file */
int			doannot=0;		/* save annotation item */
int			ofid;		/* output file id */
struct item_header	syitem;		/* output SY item header */
struct item_header	anitem;
struct item_header	fxitem;

char			mfilename[SFSMAXFILENAME]; /* control file name */
char			sfilename[SFSMAXFILENAME]; /* SFS file name */
char			ofilename[SFSMAXFILENAME]; /* output file name */
char			pfilename[SFSMAXFILENAME]; /* patch file name */
char		transcription[1024]="_ t e s t _";

/* symbol mapping and default durations */
struct map_rec {
	char	*in;
	char	*out;
	int		dur;
} symmap[]={
	{ "_",	"_",	100 },
	{ "\"",	"_",	0 },
	{ "/",	"_",	100 },
	{ "&",	"&",	140 },
	{ ",",	"_",	100 },
	{ ".",	"_",	200 },
	{ "3",	"3:",	120 },
	{ "3:",	"3:",	240 },
	{ "5",	"5",	 80 },		/* alias of l~ */
	{ "@",	"@",	 90 },
	{ "@U",	"@U",	230 },
	{ "A",	"A:",	110 },
	{ "A:",	"A:",	220 },
	{ "D",	"D",	100 },
	{ "I",	"I",	100 },
	{ "I@",	"I@",	210 },
	{ "N",	"N",	130 },
	{ "O",	"O:",	105 },
	{ "O:",	"O:",	210 },
	{ "OI",	"OI",	240 },
	{ "Q",	"Q",	140 },
	{ "S",	"S",	180 },
	{ "T",	"T",	160 },
	{ "U",	"U",	100 },
	{ "V",	"V",	155 },
	{ "Z",	"Z",	 70 },
	{ "aI",	"aI",	220 },
	{ "aU",	"aU",	240 },
	{ "b",	"b",	115 },
	{ "d",	"d",	 75 },
	{ "dZ",	"dZ",	170 },
	{ "e",	"e",	125 },
	{ "e@",	"e@",	270 },
	{ "eI",	"eI",	230 },
	{ "f",	"f",	130 },
	{ "g",	"g",	 90 },
	{ "h",	"h",	160 },
	{ "i",	"i:",	 70 },
	{ "i:",	"i:",	140 },
	{ "j",	"j",	110 },
	{ "k",	"k",	140 },
	{ "l",	"l",	 80 },
	{ "l~",	"l~",	 80 },
	{ "m",	"m",	110 },
	{ "n",	"n",	130 },
	{ "p",	"p",	130 },
	{ "r",	"r",	 80 },
	{ "s",	"s",	125 },
	{ "t",	"t",	130 },
	{ "tS",	"tS",	210 },
	{ "u",	"u:",	 80 },
	{ "u:",	"u:",	155 },
	{ "v",	"v",	 85 },
	{ "w",	"w",	 80 },
	{ "z",	"z",	140 },
	{ "{",	"{",	140 },
};
#define NUMSYM (sizeof(symmap)/sizeof(struct map_rec))

/* save string in memory */
char *strsave(char *str)
{
	char *ptr=malloc(strlen(str)+1);
	if (ptr==NULL)
		error("out of memory");
	strcpy(ptr,str);
	return(ptr);
}

/* load AN and FX item from SFS file */
void loadsfs(char *fname,char *antype,char *fxtype)
{
	struct an_rec		*an;
	short			*ifx;
	int			i,j;

	/* get SFS items */
	getitem(fname,AN_TYPE,antype,&anitem,&an);
	getitem(fname,FX_TYPE,fxtype,&fxitem,&ifx);

	/* allocate buffer for output Fx */
	fxcount = (int)(1+(fxitem.numframes*fxitem.frameduration/0.01));
	if ((fx=(short *)calloc(fxcount,sizeof(short)))==NULL)
		error("out of memory");

	/* copy fx */
	for (i=0;i<fxcount;i++) {
		j = (int)(i*0.01/fxitem.frameduration);
		if ((0<=j)&&(j<fxitem.numframes))
			fx[i] = ifx[j];
	}

	/* allocate buffer for labels */
	icount = anitem.numframes;
	totdur = (an[icount-1].posn+an[icount-1].size)*anitem.frameduration;
	if ((itab=(struct label_rec *)calloc(icount,sizeof(struct label_rec)))==NULL)
		error("out of memory");

	/* copy annotations across */
	for (i=0;i<icount;i++) {
		itab[i].stime = an[i].posn*anitem.frameduration;
		itab[i].label = strsave(an[i].label);
		if (i < icount-1)
			itab[i].etime = an[i+1].posn*anitem.frameduration;
		else
			itab[i].etime = (an[i].posn+an[i].size)*anitem.frameduration;
	}
}

/* load MBROLA control file */
void loadmbrola(char *fname)
{
	FILE	*ip;
	char	line[1024];
	char	label[128];
	char	*p;
	int	dur;
	int	i,j,idx;
	double	t,pc;
	int	fxval;
	int	f1,f2;

	if (strcmp(fname,"-")==0) {
		/* copy to temp file */
		ip = fopen("phosynth.$$$","w+");

		/* find total duration and # segments */
		icount=0;
		while (fgets(line,1024,stdin)) if ((line[0]!=';') && (line[0]!='\n')) {
			fputs(line,ip);
			sscanf(line,"%s %d",label,&dur);
			if (dur==0)
				error("zero length segment in '%s'",fname);
			totdur += dur/1000.0;
			icount++;
		}
	}
	else {
		if ((ip=fopen(fname,"r"))==NULL)
			error("could not find '%s'",fname);

		/* find total duration and # segments */
		icount=0;
		while (fgets(line,1024,ip)) if ((line[0]!=';') && (line[0]!='\n')) {
			if (sscanf(line,"%s %d",label,&dur)!=2) break;
			if (dur==0)
				error("zero length segment in '%s'",fname);
			totdur += dur/1000.0;
			icount++;
		}
	}
#ifdef IAG
	printf("totdur=%g\n",totdur);
#endif
	rewind(ip);

	/* allocate buffer for output Fx */
	fxcount = (int)(1+(totdur/0.01));
	if ((fx=(short *)calloc(fxcount,sizeof(short)))==NULL)
		error("out of memory");

	/* allocate buffer for labels */
	if ((itab=(struct label_rec *)calloc(icount,sizeof(struct label_rec)))==NULL)
		error("out of memory");

	/* run through control lines, relating to segments on file */
	for (i=0;fgets(line,1024,ip)&&(i<icount);) if ((line[0]!=';') && (line[0]!='\n')) {

		/* calculate start time */
		if (i>0)
			itab[i].stime = itab[i-1].etime;
		else
			itab[i].stime = 0;

		/* get label */
		p = strtok(line," \t\n");
		if (!p || !*p) break;
		itab[i].label = strsave(p);

		/* get duration */
		p = strtok(NULL," \t\n");
		itab[i].etime = itab[i].stime + atof(p)/1000.0;

		/* store FX values */
		p = strtok(NULL," \t\n");
		while (p && *p && isdigit(*p)) {
			pc = atoi(p)/100.0;
			p = strtok(NULL," \t\n");
			if (p && *p && isdigit(*p)) {
				fxval = atoi(p);
				t = itab[i].stime + pc*(itab[i].etime-itab[i].stime);
				idx = (int)(100*t);
				if ((0 <= idx) && (idx < fxcount))
					fx[idx]=fxval;
			}
			p = strtok(NULL," \t\n");
		}
		i++;
	}

	/* now join up Fx values with linear interpolation */
	idx=0;
	f1=-1;
	for (i=0;i<fxcount;i++) {
		f2 = fx[i];
		if (f2 > 0) {
			if (f1<0) f1=f2;
			for (j=idx;j<i;j++)
				fx[j] = f1 + (j-idx)*(f2-f1)/(i-idx);
			f1 = f2;
			idx = i;
		}
	}
	for (j=idx;j<fxcount;j++) fx[j] = f1;

	fclose(ip);

	if (strcmp(fname,"-")==0) remove("phosynth.$$$");
}

/* load table from simple transcription */
void loadtranscription(char *trans)
{
	char	*p;
	int		i,cnt=0;
	int		idx;
	char	buf[32];
	double	t=0;

	/* count number of symbols */
	icount=1;
	for (i=0;trans[i];i++) if (isspace(trans[i])) icount++;

	/* allocate buffer for labels */
	if ((itab=(struct label_rec *)calloc(icount,sizeof(struct label_rec)))==NULL)
		error("out of memory");

	/* parse symbols */
	p = strtok(strsave(trans)," \t\r\n");
	cnt=0;
	while (p && *p && (cnt < icount)) {
		idx=-1;
		for (i=0;i<NUMSYM;i++)
			if (strcmp(p,symmap[i].in)==0) {
				idx=i;
				break;
			}
		if (idx >= 0) {
			itab[cnt].label = strsave(symmap[idx].out);
			itab[cnt].stime = t;
			itab[cnt].etime = t + symmap[idx].dur/1000.0;
			cnt++;
			t += symmap[idx].dur/1000.0;
		}
		p = strtok(NULL," \t\r\n");
	}
	icount=cnt;
	totdur=t;

	/* build a simple fx contour */
	fxcount = (int)(1+t/0.01);
	if ((fx=(short *)calloc(fxcount,sizeof(short)))==NULL)
		error("out of memory");
	for (i=0;i<fxcount;i++) {
		fx[i] = 150 + i*(100-150)/fxcount;
	}

#ifdef IAG
for (i=0;i<icount;i++) printf("%s: %g %g\n",itab[i].label,itab[i].stime,itab[i].etime);
#endif
}


/* find symbol */
int findsymbol(char *label)
{
	int	i,j,k,c;

	i=0;
	j=NUM_PHONE-1;
	do {
		k = (i+j)/2;
		if ((c=strcmp(label,phone_tab[k].label)) < 0)
			j = k - 1;
		else if (c > 0)
			i = k + 1;
		else
			return(k);
	} while (i <= j);
	return(-1);
}

/* look up input transcription for synthesis */
void lookup()
{
	int	i,j,idx1,idx2,idx3;
	char	tlabel[256];
	double	dur;

	/* scan input, counting up duplex segments */
	scount=0;
	for (i=0;i<icount;i++) {
#ifdef IAG
printf("%d '%s'\n",i,itab[i].label);
#endif
		if (findsymbol(itab[i].label)<0)
			error("could not find symbol '%s'",itab[i].label);
		scount++;
		sprintf(tlabel,"%s2",itab[i].label);
		if (findsymbol(tlabel) >= 0) {
			scount++;
			sprintf(tlabel,"%s3",itab[i].label);
			if (findsymbol(tlabel) >= 0) scount++;
		}
	}

	/* allocate memory for synthesis table */
	stab = (struct synth_rec *)calloc(scount,sizeof(struct synth_rec));
	if (stab==NULL)
		error("out of memory");

	/* now transfer labels, finding table indices */
	for (i=0,j=0;i<icount;i++) {
		idx1 = stab[j].phidx = findsymbol(itab[i].label);
		stab[j].stime = itab[i].stime;
		stab[j].etime = itab[i].etime;
		dur = itab[i].etime - itab[i].stime;
		j++;
		sprintf(tlabel,"%s2",itab[i].label);
		if ((idx2=findsymbol(tlabel)) >= 0) {
			stab[j].phidx = idx2;
			stab[j].stime = stab[j-1].etime = itab[i].stime + (dur*phone_tab[idx1].inhdur)/(phone_tab[idx1].inhdur+phone_tab[idx2].inhdur);
			stab[j].etime = itab[i].etime;
			j++;
			sprintf(tlabel,"%s3",itab[i].label);
			if ((idx3=findsymbol(tlabel)) >= 0) {
				stab[j].phidx = idx3;
				stab[j-1].stime = stab[j-2].etime = itab[i].stime + (dur*phone_tab[idx1].inhdur)/(phone_tab[idx1].inhdur+phone_tab[idx2].inhdur+phone_tab[idx3].inhdur);
				stab[j].stime   = stab[j-1].etime = stab[j-1].stime + (dur*phone_tab[idx2].inhdur)/(phone_tab[idx1].inhdur+phone_tab[idx2].inhdur+phone_tab[idx3].inhdur);
				stab[j].etime = itab[i].etime;
				j++;
			}
		}
	}
}

/* copy target parameters */
void copytarget(int sidx)
{
	int	i;
	int	cidx = stab[sidx].phidx;

	for (i=0;i<NUM_PARAM;i++)
		stab[sidx].target[i] = phone_tab[cidx].target[i];
}

/* copy entry parameters */
void copyentry(int sidx,int pidx)
{
	int	i;
	int	cidx = stab[sidx].phidx;

	if (pidx < 0) {
		for (i=0;i<NUM_PARAM;i++) {
			stab[sidx].ibound[i] = phone_tab[cidx].target[i];
			stab[sidx].itrans[i] = 0;
		}
	}
	else if (phone_tab[cidx].rank > phone_tab[pidx].rank) {
		for (i=0;i<NUM_PARAM;i++) {
			stab[sidx].ibound[i] =
				phone_tab[cidx].boundfix[i] +
				phone_tab[cidx].boundvar[i] * phone_tab[pidx].target[i]/100.0;
			stab[sidx].itrans[i] =
				phone_tab[cidx].interndur[i]/1000.0;
		}
	}
	else {
		for (i=0;i<NUM_PARAM;i++) {
			stab[sidx].ibound[i] =
				phone_tab[pidx].boundfix[i] +
				phone_tab[pidx].boundvar[i] * phone_tab[cidx].target[i]/100.0;
			stab[sidx].itrans[i] =
				phone_tab[pidx].externdur[i]/1000.0;
		}
	}
}

/* copy exit parameters */
void copyexit(int sidx,int nidx)
{
	int	i;
	int	cidx = stab[sidx].phidx;

	if (nidx < 0) {
		for (i=0;i<NUM_PARAM;i++) {
			stab[sidx].ebound[i] = phone_tab[cidx].target[i];
			stab[sidx].etrans[i] = 0;
		}
	}
	else if (phone_tab[cidx].rank > phone_tab[nidx].rank) {
		for (i=0;i<NUM_PARAM;i++) {
			stab[sidx].ebound[i] =
				phone_tab[cidx].boundfix[i] +
				phone_tab[cidx].boundvar[i] * phone_tab[nidx].target[i]/100.0;
			stab[sidx].etrans[i] =
				phone_tab[cidx].interndur[i]/1000.0;
		}
	}
	else {
		for (i=0;i<NUM_PARAM;i++) {
			stab[sidx].ebound[i] =
				phone_tab[nidx].boundfix[i] +
				phone_tab[nidx].boundvar[i] * phone_tab[cidx].target[i]/100.0;
			stab[sidx].etrans[i] =
				phone_tab[nidx].externdur[i]/1000.0;
		}
	}
}

/* Holmes, Mattingly & Shearme transition rules */
void hms()
{
	int	i,j;

	/* copy targets and calculate boundary values */
	if (scount==0) return;
	else if (scount==1) {
		copytarget(0);
		copyentry(0,-1);
		copyexit(0,-1);
	}
	else {
		copytarget(0);
		copyentry(0,-1);
		copyexit(0,stab[1].phidx);
		for (i=1;i<scount-1;i++) {
			copytarget(i);
			copyentry(i,stab[i-1].phidx);
			copyexit(i,stab[i+1].phidx);
		}
		copytarget(scount-1);
		copyentry(scount-1,stab[scount-2].phidx);
		copyexit(scount-1,-1);
	}

	/* check everything */
	for (i=0;i<scount;i++)
		for (j=0;j<NUM_PARAM;j++) {
			if (stab[i].ibound[j] < 0) stab[i].ibound[j]=0;
			if (stab[i].ebound[j] < 0) stab[i].ebound[j]=0;
		}
}

/* calculate a parameter value */
double paramval(int pno,double currt,int curri)
{
	double	t,theta;

	if (currt <= (stab[curri].stime+stab[curri].itrans[pno])) {
		/* in inbound transition */
		if (stab[curri].itrans[pno]==0)
			return(stab[curri].ibound[pno]);
		else {
			t = currt - stab[curri].stime;
			theta = (t*t)/(stab[curri].itrans[pno]*stab[curri].itrans[pno]) - 2*t/stab[curri].itrans[pno] + 1;
			return(stab[curri].target[pno] + theta*(stab[curri].ibound[pno]-stab[curri].target[pno]));
		}
	}
	else if (currt >= (stab[curri].etime-stab[curri].etrans[pno])) {
		/* in outbound transition */
		if (stab[curri].etrans[pno]==0)
			return(stab[curri].ebound[pno]);
		else {
			t = stab[curri].etime - currt;
			theta = (t*t)/(stab[curri].etrans[pno]*stab[curri].etrans[pno]) - 2*t/stab[curri].etrans[pno] + 1;
			return(stab[curri].target[pno] + theta*(stab[curri].ebound[pno]-stab[curri].target[pno]));
		}
	}
	else
		/* at target value */
		return(stab[curri].target[pno]);
}

/* generate synthesizer control data */
void gensy(FILE *op)
{
	double	currt;
	int	curri=0;
	int	fidx=0;
	short	frame[32];
	time_t	tim=time((time_t *)0);

	if (dolist) {
		fprintf(op,"* generated by %s Vs %s from '%s' on %s",
			PROGNAME,PROGVERS,mfilename,ctime(&tim));
		fprintf(op,"*  fx   f1   a1   f2   a2   f3   a3   f4   a4   fn   an    v\n");
		fprintf(op,"*\n");
		fprintf(op,"* %s %g\n",phone_tab[stab[0].phidx].label,stab[0].stime);
	}
	for (currt=0.0;currt <= totdur;currt += 0.01,fidx++) {
		/* move to next segment if required */
		while ((currt > stab[curri].etime)&&(curri<scount-1)) {
			curri++;
			if (dolist) fprintf(op,"* %s %g\n",phone_tab[stab[curri].phidx].label,stab[curri].stime);
		}
		if ((0<=fidx)&&(fidx<fxcount))
			frame[SYFX] = fx[fidx];
		else
			frame[SYFX] = 0;
		frame[SYAH] = -1;
		frame[SYMS] = -1;
		frame[SYF1] = (int)paramval(F1_PARAM,currt,curri);
		frame[SYA1] = (int)(10.0*paramval(A1_PARAM,currt,curri));
		frame[SYB1] = -1;
		frame[SYF2] = (int)paramval(F2_PARAM,currt,curri);
		frame[SYA2] = (int)(10.0*paramval(A2_PARAM,currt,curri));
		frame[SYB2] = -1;
		frame[SYF3] = (int)paramval(F3_PARAM,currt,curri);
		frame[SYA3] = (int)(10.0*paramval(A3_PARAM,currt,curri));
		frame[SYB3] = -1;
		frame[SYF4] = 4000;
		frame[SYA4] = (int)(10.0*paramval(AH_PARAM,currt,curri));
		frame[SYB4] = -1;
		frame[SYFN] = (int)paramval(FN_PARAM,currt,curri);
		frame[SYAN] = (int)(10.0*paramval(AL_PARAM,currt,curri));
		frame[SYBN] = -1;
		frame[SYVR] = (int)paramval(VR_PARAM,currt,curri);

		if (dolist) {
			fprintf(op,"%5d",frame[0]);
			fprintf(op,"%5d%5.1f",frame[3],frame[4]/10.0);
			fprintf(op,"%5d%5.1f",frame[6],frame[7]/10.0);
			fprintf(op,"%5d%5.1f",frame[9],frame[10]/10.0);
			fprintf(op,"%5d%5.1f",frame[12],frame[13]/10.0);
			fprintf(op,"%5d%5.1f",frame[15],frame[16]/10.0);
			fprintf(op,"%5d\n",frame[18]);
		}
		if (dosynth) {
			if ((0<=fidx)&&(fidx<sycount))
				hwsynthcode(frame,sy+16*fidx);
		}
		if (dosavesfs) sfswrite(ofid,1,frame);
	}
}

/* get a PARAM number */
int getparam(char *str)
{
	if (stricmp(str,"FN")==0)
		return(FN_PARAM);
	else if (stricmp(str,"F1")==0)
		return(F1_PARAM);
	else if (stricmp(str,"F2")==0)
		return(F2_PARAM);
	else if (stricmp(str,"F3")==0)
		return(F3_PARAM);
	else if (stricmp(str,"AL")==0)
		return(AL_PARAM);
	else if (stricmp(str,"A1")==0)
		return(A1_PARAM);
	else if (stricmp(str,"A2")==0)
		return(A2_PARAM);
	else if (stricmp(str,"A3")==0)
		return(A3_PARAM);
	else if (stricmp(str,"AH")==0)
		return(AH_PARAM);
	else if (stricmp(str,"VR")==0)
		return(VR_PARAM);
	else
		error("No such parameter '%s' in patch file",str);
	return(0);
}

/* process a path file */
void processpatch(char *filename)
{
	FILE	*ip;
	char	line[1024];
	char	param[16];
	char	*p;
	int		segidx=-1;
	int		lineno=0;
	int		val,ipar;
	int		i;

	if ((ip=fopen(filename,"r"))==NULL)
		error("unable to open '%s'",filename);

	while (fgets(line,1024,ip)) {
		lineno++;
		if (line[0]!='#') {
			p = strtok(line," \t\r\n,;");
			while (p && *p) {
				for (i=0;(i<16)&&(p[i])&&(p[i]!='=');i++) param[i]=p[i];
				param[i]='\0';
				if (p[i]=='=') {
					val = atoi(p+i+1);
				}
				else
					error("format error in '%s' line %d",filename,lineno);
				if (strcmp(param,"segment")==0) {
					for (segidx=0;segidx<NUM_PHONE;segidx++)
						if (strcmp(phone_tab[segidx].label,p+i+1)==0)
							break;
				}
				else if (segidx==-1)
					error("no current segment in '%s' line %d",filename,lineno);
				else if (strcmp(param,"rank")==0)
					phone_tab[segidx].rank = val;
				else if (strncmp(param,"target",6)==0) {
					ipar=getparam(param+6);
					if (phone_tab[segidx].boundfix[ipar]==((phone_tab[segidx].target[ipar]*(100-phone_tab[segidx].boundvar[ipar]))/100)) {
						phone_tab[segidx].boundfix[ipar]=(val*(100-phone_tab[segidx].boundvar[ipar]))/100;
					}
					phone_tab[segidx].target[ipar] = val;
				}
				else if (strncmp(param,"externdur",9)==0) {
					ipar=getparam(param+9);
					phone_tab[segidx].externdur[ipar] = val;
				}
				else if (strncmp(param,"interndur",9)==0) {
					ipar=getparam(param+9);
					phone_tab[segidx].interndur[ipar] = val;
				}
				else
					error("illegal command in '%s' line %d",filename,lineno);

				p=strtok(NULL," \t\r\n,;");
			}
		}

	}

	fclose(ip);
}

/* dump the entire parameter table in patch format */
void dumptable(char *filename)
{
	FILE	*op;
	int		i,j;

	if ((op=fopen(filename,"w"))==NULL)
		error("could not open '%s'",filename);

	for (i=0;i<NUM_PHONE;i++) {
		fprintf(op,"segment=%s\n",phone_tab[i].label);
		fprintf(op," rank=%d\n",phone_tab[i].rank);
		for (j=0;j<NUM_PARAM;j++)
			fprintf(op," target%s=%d;",paramnames[j],phone_tab[i].target[j]);
		fprintf(op,"\n");
		for (j=0;j<NUM_PARAM;j++)
			fprintf(op," externdur%s=%d;",paramnames[j],phone_tab[i].externdur[j]);
		fprintf(op,"\n");
		for (j=0;j<NUM_PARAM;j++)
			fprintf(op," interndur%s=%d;",paramnames[j],phone_tab[i].interndur[j]);
		fprintf(op,"\n");
		for (j=0;j<NUM_PARAM;j++)
			fprintf(op," boundfix%s=%d;",paramnames[j],phone_tab[i].boundfix[j]);
		fprintf(op,"\n");
		for (j=0;j<NUM_PARAM;j++)
			fprintf(op," boundvar%s=%d;",paramnames[j],phone_tab[i].boundvar[j]);
		fprintf(op,"\n");
	}

	fclose(op);
}

/* save annotations from transcription */
void saveAN(char *trans,char *fname)
{
	char	*p;
	int		i,cnt=0;
	int		t=0;
	int		idx;
	char	buf[128];
	struct an_rec *an;
	int		ofid;

	sfsheader(&anitem,AN_TYPE,-1,1,-1,0.001,0.0,0,0,1);
	strncpy(buf,trans,127);
	strcpy(buf+90,"...");
	sprintf(anitem.history,"%s(type=transcription,trans=%s)",
		PROGNAME,buf);
	if ((ofid=sfschannel(fname,&anitem))<0)
		error("could not open output channel to '%s'",fname);
	an = sfsbuffer(&anitem,1);

	for (i=0;i<icount;i++) {
		an->posn = (int)(0.5+itab[i].stime/0.001);
		an->size = (int)(0.5+(itab[i].etime-itab[i].stime)/0.001);
		strncpy(an->label,itab[i].label,250);
		sfswrite(ofid,1,an);
	}

}


/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		i,j;
	/* input items */
	int		it;
	char		*ty;
	char		*antype="0";
	char		*fxtype="0";
	FILE		*op;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:lsm:o:p:d:t:a")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Synthesis by rule of .PHO file V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* input items */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == AN_TYPE) {
					antype = ty;
					dosfs=1;
					dombrola=0;
					dotrans=0;
				}
				else if (it == FX_TYPE) {
					fxtype = ty;
					dosfs=1;
					dombrola=0;
					dotrans=0;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'm' :	/* input MBROLA file */
			strcpy(mfilename,optarg);
			dosfs=0;
			dombrola=1;
			dotrans=0;
			break;
		case 't' :	/* input transcription */
			strncpy(transcription,optarg,sizeof(transcription));
			dosfs=0;
			dombrola=0;
			dotrans=1;
			break;
		case 'l' :	/* list data */
			dosynth=0;
			dolist=1;
			dosavesfs=0;
			break;
		case 's' :	/* save to SFS file */
			dosynth=0;
			dolist=0;
			dosavesfs=1;
			dosavetxt=0;
			break;
		case 'o' :	/* output to TXT file */
			dosynth=0;
			dolist=1;
			dosavesfs=0;
			dosavetxt=1;
			strcpy(ofilename,optarg);
			break;
		case 'p':	/* patch file */
			if (pfilename[0]) strcat(pfilename,",");
			strcat(pfilename,optarg);
			processpatch(optarg);
			break;
		case 'd':	/* dump parameter file */
			dumptable(optarg);
			break;
		case 'a':	/* save annotations */
			doannot=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-l|-s) (-m file.pho|-t transcription) (-o output.txt) (-p patchfile) (-d dumpfile) (file.sfs)",PROGNAME);

	/* get SFS filename */
	if (optind < argc)
		strcpy(sfilename,sfsfile(argv[optind]));
	else
		dosavesfs=0;

	/* load control file */
	if (dosfs)
		loadsfs(sfilename,antype,fxtype);
	else if (dombrola)
		loadmbrola(mfilename);
	else
		loadtranscription(transcription);

#ifdef IAG
printf("Segment table:\n");
for (i=0;i<icount;i++)
	printf("%d. %s %g %g\n",
		i,itab[i].label,itab[i].stime,itab[i].etime);
#endif
#ifdef IAG
printf("Fx contour:\n");
for (i=0;i<fxcount;i++) {
	printf("%3d ",fx[i]);
	if ((i%10)==9) printf("\n");
}
printf("\n");
#endif

	/* look up transcription in phonetic table */
	lookup();

#ifdef IAG
printf("Synthesis table:\n");
for (i=0;i<scount;i++)
	printf("%d. %s %g %g\n",
		i,phone_tab[stab[i].phidx].label,stab[i].stime,stab[i].etime);
#endif

	/* calculate boundary values and transition durations */
	hms();

#ifdef IAG
printf("Synthesis table:\n");
for (i=0;i<scount;i++) {
	printf("%d. %s %g %g\n",
		i,phone_tab[stab[i].phidx].label,stab[i].stime,stab[i].etime);
	for (j=0;j<NUM_PARAM;j++)
		printf("param%d %g (%g) %g (%g) %g\n",
			j,stab[i].ibound[j],stab[i].itrans[j],stab[i].target[j],stab[i].etrans[j],stab[i].ebound[j]);
}
#endif

	/* get output table */
	if (dosynth) {
		sy = (short *)calloc(fxcount,16*sizeof(short));
		if (sy==NULL)
			error("out of memory");
		sycount = fxcount;
	}

	/* open fid to SFS file */
	if (dosavesfs) {
		sfsheader(&syitem,SY_TYPE,0,2,19,0.01,0.0,1,0,0);
		if (pfilename[0]) {
			if (dombrola)
				sprintf(syitem.history,"%s(file=%s,patch=%s)",PROGNAME,mfilename,pfilename);
			else if (dotrans) {
				strcpy(transcription+64,"...");
				sprintf(syitem.history,"%s(trans=%s,patch=%s)",PROGNAME,transcription,pfilename);
			}
			else
				sprintf(syitem.history,"%s(%d.%02d,%d.%02d;patch=%s)",
					PROGNAME,
					anitem.datatype,anitem.subtype,
					fxitem.datatype,fxitem.subtype,
					pfilename);
		}
		else {
			if (dombrola)
				sprintf(syitem.history,"%s(file=%s)",PROGNAME,mfilename);
			else if (dotrans) {
				strcpy(transcription+128,"...");
				sprintf(syitem.history,"%s(trans=%s)",PROGNAME,transcription);
			}
			else
				sprintf(syitem.history,"%s(%d.%02d,%d.%02d)",
					PROGNAME,
					anitem.datatype,anitem.subtype,
					fxitem.datatype,fxitem.subtype);
		}
		if ((ofid=sfschannel(sfilename,&syitem)) < 0)
			error("unable to create temporary file",NULL);
	}

	/* open output stream */
	if (dosavetxt) {
		if ((op=fopen(ofilename,"w"))==NULL)
			error("could not open '%s'",ofilename);
	}
	else
		op=stdout;

	/* generate synthesizer control data */
	gensy(op);
	if (op!=stdout) fclose(op);

	/* produce synthesis */
	if (dosynth) hwsynth(sy,sycount);

	/* that's all folks ! */
	if (dosavesfs) {
		if (doannot) saveAN(transcription,sfilename);
		sfsupdate(sfilename);
	}

	exit(0);
}
