/* qx - calculate Tx open markers and CQ track */

/* M.A.Huckvale - University College London */

/* version 1.0 - December 2006 */

#define PROGNAME "qx"
#define PROGVERS "1.0"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH QX SFS1 UCL SFS
.SH NAME
qx - calculate Tx opening markers and CQ track
.SH SYNOPSIS
.B qx
(-i item) (-f framerate) file
.SH DESCRIPTION
.I qx
is a program to locate the point of larynx opening given a Laryngograph
waveform and a set of Tx closing markers.  It outputs a Tx item that marks
the opening points and a track indicating the closed quotient.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item.
.TP 11
.BI -f framerate
Select sampling frequency of track item. Default 200.
.SH INPUT ITEMS
.IP LX.xx 11
Laryngograph waveform.
.IP TX.xx 11
Any excitation period data item.
.SH OUTPUT ITEMS
.IP TX 11
Tx opening markers.
.IP TR 11
Closed quotient track.
.SH VERSION/AUTHOR
.IP 1.0 11
Mark Huckvale
.SH BUGS
Tracks are sampled, not averaged.
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"		/* database structures */

/* input file variables */
char		filename[SFSMAXFILENAME]; /* sfs file name */
struct item_header	lxitem;	/* item header for speech data */
short	*lx;
struct item_header txitem;	/* item header for tx data */
int		*tx;
struct item_header qxitem;	/* item header for qx data */
int		*qx;
struct item_header tritem;	/* item header for CQ data */

/* output file variables */
int		ffreq = 200;		/* output frame rate */
double	fdur;				/* output frame duration */
int		txlimit;			/* largest interval that is voiced */

/* calculate the opening point for a region of the Lx */
double findopening(double stime,double etime)
{
	int	sidx=(int)(0.5+stime/lxitem.frameduration);
	int	eidx=(int)(0.5+etime/lxitem.frameduration);
	int tlen=eidx-sidx;
	int	i,n;
	int		imin1,imax1,imin2,imax2;
	int		lmin1,lmax1,lmin2,lmax2;
	double	x1,y1,y1a;
	double	x2,y2,y2a;
	double	x3,y3,y3a;
	double	x4,y4,y4a;
	double	m;

	if ((sidx >= lxitem.numframes)||(eidx >= lxitem.numframes)) return(stime);

	/* find minimum before sidx */
	imin1=sidx;
	lmin1=lx[imin1];
	for (i=sidx;(i>sidx-tlen/2)&&(i>0);i--)
		if (lx[i]<lmin1) {
			imin1=i;
			lmin1=lx[imin1];
		}

	/* find minimum between sidx and eidx */
	imin2=eidx;
	lmin2=lx[imin2];
	for (i=eidx;i>eidx-tlen/2;i--)
		if (lx[i]<lmin2) {
			imin2=i;
			lmin2=lx[imin2];
		}

	/* find maximum after sidx but before minimum */
	imax1=sidx;
	lmax1=lx[imax1];
	for (i=sidx;i<imin2;i++)
		if (lx[i]>lmax1) {
			imax1=i;
			lmax1=lx[imax1];
		}

	/* find maximum after eidx */
	imax2=eidx;
	lmax2=lx[imax2];
	for (i=eidx;(i<eidx+tlen/2)&&(i<lxitem.numframes);i++)
		if (lx[i]>lmax2) {
			imax2=i;
			lmax2=lx[imax2];
		}

#if 0
	printf("%g:%g imin1=%g (%d) imax1=%g (%d) imin2=%g (%d) imax2=%g (%d)\n",
		stime,etime,
		imin1*lxitem.frameduration,lx[imin1],
		imax1*lxitem.frameduration,lx[imax1],
		imin2*lxitem.frameduration,lx[imin2],
		imax2*lxitem.frameduration,lx[imax2]);
#endif

	x1=imin1; y1=lx[imin1];
	x2=imax1; y2=lx[imax1];
	x3=imin2; y3=lx[imin2];
	x4=imax2; y4=lx[imax2];

	y1a=y1+(x2-x1)*(y3-y1)/(x3-x1);
	y3a=y3+(x4-x3)*(y3-y1)/(x3-x1);

	y2a=(y1a+y2)/2;
	y4a=(y3a+y4)/2;

	n=0;
	for (i=sidx;i<eidx;i++) {
		m=y2a+(i-x2)*(y4a-y2a)/(x4-x2);
		if (lx[i] > m) n++;
	}

	return stime+n*lxitem.frameduration;
}

/* output the CQ track */
void outtrack(int fid)
{
	int		i;
	int		cumtx=0;
	int		cumqx=0;
	double	t=0,txtime=0;
	float	val=0;

	cumtx=tx[0];
	for (i=1;i<txitem.numframes;i++) {

		/* write value to file */
		while (t < cumtx*txitem.frameduration) {

			if (sfswrite(fid,1,&val) != 1)
				error("write error on temporary file",NULL);

			t += fdur;
		}

		cumqx += qx[i-1];
		if (tx[i] >= txlimit)
			val = 0;
		else
			val = (float)(100.0*(cumqx-cumtx)/tx[i]);

		if (val < 0) val=0;
		if (val > 100) val=100;

		cumtx += tx[i];
	}

	/* write value to file */
	while (t < cumtx*txitem.frameduration) {

		if (sfswrite(fid,1,&val) != 1)
			error("write error on temporary file",NULL);

		t += fdur;
	}
}

/* main program */
void main(int argc,char **argv)
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item type selection */
	char		*ty;		/* item match selection */
	char		*lxtype="0";
	char		*txtype="0";

	/* output file variables */
	int		ofid;		/* output file descriptor */
	int		tfid;		/* output file descriptor */
	int		cumtx=0;
	int		cumqx=0;
	int		i,t;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:f:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Qx statistics V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == LX_TYPE)
					lxtype=ty;
				else if (it == TX_TYPE)
					txtype=ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'f' :	/* output sampling freq */
			if ((ffreq=atoi(optarg)) <= 0)
				error("illegal sampling frequency: %s",optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-f frame rate) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no file specified",NULL);

	/* load Tx data */
	getitem(filename,TX_TYPE,txtype,&txitem,&tx);
	getitem(filename,LX_TYPE,lxtype,&lxitem,&lx);
	txlimit = (int)(0.025/txitem.frameduration);
	fdur = 1.0/ffreq;

	/* create output item headers and channels */
	/* make header */
	sfsheader(&qxitem,TX_TYPE,0,4,1,txitem.frameduration,txitem.offset,1,0,0);
	sprintf(qxitem.history,"%s/TX(%d.%02d,%d.%02d)",
			PROGNAME,
			lxitem.datatype,lxitem.subtype,
			txitem.datatype,txitem.subtype);

	/* open output channel */
	if ((ofid=sfschannel(filename,&qxitem)) < 0)
		error("cannot open output channel",NULL);

	sfsheader(&tritem,TR_TYPE,1,4,1,1.0/ffreq,txitem.offset,1,0,0);
	sprintf(tritem.history,"%s/TR(%d.%02d,%d.%02d;rate=%d)",
			PROGNAME,
			lxitem.datatype,lxitem.subtype,
			txitem.datatype,txitem.subtype,ffreq);

	/* open output channel */
	if ((tfid=sfschannel(filename,&tritem)) < 0)
		error("cannot open output channel",NULL);

	/* get buffer */
	qx=(int *)sfsbuffer(&qxitem,txitem.numframes);

	/* calculate the QX positions */
	cumtx=tx[0];
	cumqx=0;
	for (i=1;i<txitem.numframes;i++) {
		if (tx[i] < txlimit) {
			t = (int)(0.5+findopening(cumtx*txitem.frameduration,(cumtx+tx[i])*txitem.frameduration)/txitem.frameduration);
		}
		else if ((i>0)&&(tx[i-1]<txlimit)) {
			t = (int)(0.5+findopening(cumtx*txitem.frameduration,(cumtx+tx[i-1])*txitem.frameduration)/txitem.frameduration);
		}
		else {
			t = cumtx;
		}
		cumtx += tx[i];
		if (t <= cumqx) t=cumqx+1;
		if (t > cumtx) t=cumtx;
		qx[i-1] = t-cumqx;
		cumqx += qx[i-1];
	}
	qx[txitem.numframes-1]=cumtx-cumqx;

	/* output the QX */
	sfswrite(ofid,txitem.numframes,qx);

	/* output the CQ track */
	outtrack(tfid);

	/* update data file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks */
	if (qx) free(qx);
	if (tx) free(tx);
	if (lx) free(lx);
	exit(0);
}


