/* spgate -- produce gated speech stimuli */

/* M.A.Huckvale - University College London */

/* version 1.0 - November 2001 */

#define PROGNAME "spgate"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH SPGATE SFS1 UCL
.SH NAME
spgate -- produce gated speech stimuli
.SH SYNOPSIS
.B spgate
(-n numgate) (-t timegate|-a) (-w windowtime) (-S|-N) file
.SH DESCRIPTION
.I spgate
is a program to build a number of gated speech stimuli from a
single speech recording.  The gates can be defined by time or by
the position of annotations in the file.  The lost portion can be replaced
with silence or with noise.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -n numgate
Specify the number of stimuli to be produced.  Default set by
the number of annotations or by the size of the file and size of
gate.
.TP 11
.BI -t timegate
Specify the size of each gate in seconds.  Default 0.25s.
.TP 11
.B -a
Use annotations to specify where gates are to go.
.TP 11
.BI -w windowtime
Specify duration of window cutting off signal in seconds.  Default 0.025s.
.TP 11
.B -S
Replace the excised portion with silence (so that output files
are all the same size)
.TP 11
.B -N
Replace the excised portion with noise.  The size of the noise is chosen to match the speech energy.
.SH INPUT ITEMS
.IP SP
Any speech items
.IP AN
Gate annotations.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include "sfs.h"

/* global data */
struct item_header	spitem;

short			*sp;
struct item_header	anitem;
struct an_rec		*antab;
char		filename[SFSMAXFILENAME]; /* SFS data file name */

/* operational modes */
int	maxgate = 0x7FFFFFF;
double	timegate=0.25;
int		sizegate;
double	timewin=0.025;
int		sizewin;			/* maximum 2048 samples */
int		doannot=0;
int		dosilence=0;
int		donoise=0;
int		maxamp;

/* measure energy */
void calcenergy(int fid)
{
	short	*ebuf;
	int		elen;
	int		i,j;
	double	val,sumsq;

	elen = (int)(0.05/spitem.frameduration);
	ebuf = calloc(elen,sizeof(short));
	for (i=0;sfsread(fid,i,elen,ebuf)==elen;i+=elen) {
		sumsq=0;
		for (j=0;j<elen;j++) {
			val = ebuf[j];
			sumsq += val*val;
		}
		sumsq = sqrt(sumsq/elen);
		if (sumsq > maxamp) maxamp=(int)sumsq;
	}
}

/* generate noise buffer */
void getnoise(short *buf,int len)
{
	int	i;

	for (i=0;i<len;i++)
		buf[i] = ((rand()%(2*maxamp))-maxamp);
}

/* create a gate */
void creategate(int fid,double etime)
{
	struct item_header	opitem;
	static int		gate=0;
	int		totlen;
	int		xlen;
	short	xbuf[2048],nbuf[2048];
	int		i,ofid;
	double	omega;
	int		ncopy;

	/* create a header */
	sfsheader(&opitem,SP_TYPE,0,2,1,spitem.frameduration,spitem.offset,1,0,0);
	if (doannot) {
		sprintf(opitem.history,"%s(%d.%02d,%d.%02d;gate=%d,window=%g%s%s)",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			anitem.datatype,anitem.subtype,
			++gate,
			timewin,
			(dosilence)?",silencefill":"",
			(donoise)?",noisefill":"");
	}
	else {
		sprintf(opitem.history,"%s(%d.%02d;gate=%d,size=%g,window=%g%s%s)",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			++gate,
			timegate,
			timewin,
			(dosilence)?",silencefill":"",
			(donoise)?",noisefill":"");
	}

	/* open an output channel */
	if ((ofid=sfschannel(filename,&opitem))<0)
		error("could not open output channel to '%s'",filename);

	/* get a buffer */
	if (dosilence || donoise)
		totlen = spitem.numframes;
	else
		totlen = (int)(0.5+etime/spitem.frameduration);

	/* get copy size */
	ncopy = (int)(0.5+etime/spitem.frameduration)-sizewin;
	if (ncopy < 0) ncopy=0;

	/* copy out the data up to the window */
	i=0;
	while (i < ncopy) {
		xlen = sfsread(fid,i,(ncopy-i>2048)?2048:ncopy-i,xbuf);
		sfswrite(ofid,xlen,xbuf);
		i += xlen;
	}

	/* get the signal for the window */
	xlen = sfsread(fid,ncopy,sizewin,xbuf);

	/* get a noise buffer if required */
	if (donoise) getnoise(nbuf,xlen);

	/* perform the cut */
	omega = 8.0*atan(1.0)/(2*xlen-1);
	for (i=0;i<xlen;i++) {
		double w = 0.54-0.46*cos(i*omega);
		if (donoise)
			xbuf[i] = (short)((1-w)*xbuf[i] + w*nbuf[i]);
		else
			xbuf[i] = (short)((1-w)*xbuf[i]);
	}

	/* write out the cut */
	sfswrite(ofid,xlen,xbuf);

	/* pad if required */
	if (dosilence) {
		memset(xbuf,0,2048*sizeof(short));
		i = ncopy+xlen;
		while (i < totlen) {
			xlen = sfswrite(ofid,(totlen-i>2048)?2048:totlen-i,xbuf);
			i += xlen;
		}
	}
	else if (donoise) {
		i = ncopy+xlen;
		while (i < totlen) {
			getnoise(nbuf,2048);
			xlen = sfswrite(ofid,(totlen-i>2048)?2048:totlen-i,nbuf);
			i += xlen;
		}
	}

	if ((gate%10)==0) sfsupdate(filename);
}


/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char		*ty;		/* item sub type */
	char		*sptype="0";
	char		*antype="0";
	/* file variables */
	int		fid;		/* input file descriptor */
	int		i;
	double	etime;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:n:t:w:aSN")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: gated stimuli generation V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else if (it == AN_TYPE)
					antype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'n' :	/* number of gates */
			maxgate = atoi(optarg);
			break;
		case 't' :	/* time for each gate */
			timegate = atof(optarg);
			break;
		case 'w' :	/* window time */
			timewin = atof(optarg);
			break;
		case 'a' :	/* use annotations */
			doannot=1;
			break;
		case 'S' :	/* pad with silence */
			dosilence=1;
			donoise=0;
			break;
		case 'N' :	/* pad with noise */
			dosilence=0;
			donoise=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-n numgate) (-t timegate|-a) (-w wintime) (-s|-n) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on '%s'",filename);

	/* find annotations if required */
	if (doannot) {
		if (!sfsitem(fid,AN_TYPE,antype,&anitem))
			error("could not find annotation item in '%s'",filename);

		/* load annotations */
		if ((antab = (struct an_rec *)sfsbuffer(&anitem,anitem.numframes))==NULL)
			error("could not get buffer");
		if (sfsread(fid,0,anitem.numframes,antab)!=anitem.numframes)
			error("read error on '%s'",filename);
		if (maxgate > anitem.numframes) maxgate=anitem.numframes;
	}

	/* find the speech item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
			error("could not find speech item in '%s'",filename);

	/* check the window time */
	sizewin = (int)(0.5+timewin/spitem.frameduration);
	if (sizewin > 2048)
		error("window size too large (max 2048 samples)");

	/* measure the energy in the speech */
	if (donoise)
		calcenergy(fid);

	/* process the gates */
	for (i=0;i<maxgate;i++) {
		if (doannot)
			etime = antab[i].posn * anitem.frameduration + anitem.offset;
		else
			etime = (i+1)*timegate;
		if (etime > spitem.numframes * spitem.frameduration + spitem.offset)
			break;
		creategate(fid,etime);
	}

	/* that's all folks ! */
	sfsclose(fid);
	sfsupdate(filename);
	exit(0);
}

