/* sv6 - Speech Voltmeter as CCITT Recommendation */

/* Mark Huckvale - University College London
   - using code written by Laurie Moye */

/* version 1.0 - July 1998 */

#define PROGNAME "sv6"
#define PROGVERS "1.0"
char *progname=PROGNAME;

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include "sfs.h"

/*-------------------------------------------------------------------------*/
/**MAN
.TH SV6 SFS1 UCL
.SH NAME
sv6 -- Speech voltmeter - speech level metering
.SH SYNOPSIS
.B sv6
(-i item) file
.SH DESCRIPTION
.I sv6
is a program to make a dB level reading of a speech signal
using the speech voltmeter recommendation of CCITT blue book p56.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.SH INPUT ITEMS
.IP SPEECH
Speech signal.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale (with code from Laurie Moye)
.SH SEE ALSO
level
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* support routines for converting between power and deciBels */
double powtodB(double val)
{
//	return(10.0 * log10((val*2)/(32767.0*32767.0)));
	return(10.0 * log10(val));
}
double dBtopow(double val)
{
//	return((32767.0*32767.0)*pow(10.0,val/10.0)/2.0);
	return(pow(10.0,val/10));
}

/*-------------------------------------------------------------------------
 * TITLE: sv6_funcs.c
 * AUTHOR: Laurie Moye
 * ENVIR: ANSI `C'
 * USAGE: called by wrapper function which handles local files
 * TASKS: speech level measrurement according to CCITT Recommendation p.56
 * (C)    Crown Copyright $Date: 1998/07/02 13:10:16 $
 * DATES:
 * $Log: sv6_funcs.c,v $
 * Revision 1.1  1998/07/02 13:10:16  moye
 * Initial revision
 *
 *
sv6_funcs.c - sv6 speech voltmeter routines
v2.1	lsm	29-oct-91	return ambiguous results
v2.0	lsm	18-sep-91
These routines are intended to allow measurements to be made to
CCITT Recommendation p.56 (in the Blue Book)
the variable names and equations follow it as closely as possible

The intention is that these routines are machine independent whilst the main
program that drives them may depend on the machine and will depend on the
format of the files.

It is assumed that the input signal is a 16-bit integer, and the reference
level (0dB) is that of a peak sine wave,
i.e. one with a peak value of +-32767.

There is potentially a problem in accumulating sums over very large numbers
of samples. This has been dealt with by accumulating in double floats which
have a mantissa of 55 bits (VAX) or 53 bits (DECstation), each with a
suppressed leading zero; equivalent to 56 and 54 bits respectively.
signal is			< 2**15
squared signal is		< 2**30
rms is			< 2**29
20dB down on that is 10**-2 or 2**-7 less
which is			< 2**22
accumulate these over < 2**31 samples
and the sum is		< 2**53
So, it should be possible to acculmulate the squares over the longest files
allowed by the current (signed int) sample count without losing any accuracy
for typical speech files, and without losing any SIGNIFICANT accuracy on
file with higher signal levels.
Note that this sample count allows for files 30 Hr long at 20 k samples/s
which is probably enough for most speech applications for the forseeable
future even if the technology allows us to store longer files!

p56 says you can take offset off x squared, but then you have to correct the
envelope. This is not really practicable because the offset might be near
the threshold and completely upset the measurement of activity. For this
reason, the program goes through the signal first to take off the offset.

USAGE:
you must
   #include "sv6.h"

then there are four routines:

first, initialise the storage with the sample rate in Hz
   void sv6_init(float sample_rate);

second pass all the signal (in buffers of any size you choose) by repeatedly
calling:
   void sv6_do1 (short *buffer, long count);

that will have measured the offset, then then pass the same lot of signal by
calling:
   void sv6_do2 (short *buffer, long count);

finally, to get the results, call:
RESULTS *sv6_results();

look in sv6.h to see what is available to you in the structure RESULTS

The routines can be used repeatedly to measure different parts of a signal
file, so long as sv6_init is called at the start of each measurement.
A new structure will be returned each time with the results in.

NOTE: Ambiguous results.
The measured speech level is a function of the activity threshold. The
result required by the specification is that the speech level while active
is the level measured with an activity threshold MdB below the speech level
while active. It is found by estimating, by interpolation, the point at
which level(threshold) - M =threshold. If the level is not a monotonic
function of threshold, there may be more than one such point.
This situation could occur with a speech signal with background noise at
just below -MdB, in which case the higher result would be correct; or it
could occur in a quiet speech signal with one or two very loud words in it,
in which case, the lower result would be the correct one.
Test signals displaying these properties can easily be generated, and a
quiet speech file producing a false result from a not very large vowel sound
has been found.
Because ambiguous results are always a possibility, all ambiguous results
are returned.
It should normally be obvious from the activity factor which one is required.
 *
 *----------------------------------------------------------------------***/

typedef struct results{
   int errflg;		/* 0 for success */
   float L;		/* overall mean power */
   float A;		/* interpolated result */
   float act;		/* activity as a fraction */
   short offset;	/* mean of complete input signal */
   short maxp;		/* largest positive sample in input */
   short maxn;		/* largest negative sample in input */
   struct results *next;	/* link pointer for ambiguous results */
} RESULTS;
#define MAXOFFERR -1

/* the routines */
void sv6_init(float sample_rate);	/* initialise with samples/second */
void sv6_pass1 (short *buffer, long count);	/* first pass (offset) */
void sv6_pass2 (short *buffer, long count);	/* second pass (sp_ volts) */
RESULTS *sv6_results();		/* calculate results and return in struct */

         /* constants defined in CCITT p.56 */
         /* desired threshold in dB (equiv. to 15dB with rms envelope) */
#define M 15.9
         /* envelope integration time in seconds */
#define T 0.03
         /* hangover time in seconds */
#define H 0.2

         /* results are dB re sine wave of peak +-MAXMUN */
#define MAXNUM  ((int) 0x7fff)
         /* number of thresholds in operation */
#define N 13
          /* each DIFF apart - diff is ratio of LEVEL, not power, not dB */
#define DIFF 2.0

         /* longs and doubles used for sample counts and sums */
static long n1, n2;	/* sample counts */
static double s;	/* sum of x */
static double sq;	/* sum of x squared */
static double p;	/* first envelope smoothing */
static double q;	/* second envelope smoothing */
static long a[N];	/* activity counts */
static int h[N];	/* hangover counts */
static float A[N];	/* active power */
static float c[N];	/* threshold */
static float delta[N];	/* diff re desired threshold */
static RESULTS *resptr;	/* for return of results */
static RESULTS **resptrptr; /* and more results */

/* calculated  constants */
static float g;		/* integrator decay rate */
static float g1;	/* 1.0 - g */
static float I;		/* hangover count */
static float ref;	/* reference level */

/* flags */
static int first_time = 0;	/* to calc offset at start of pass 2 */

/* init requires sample rate in samples/sec */
void sv6_init(float sample_rate) {	/* initialise */
   int j;

   /* calculate the constants */
   g = (float)exp(-1.0/(T*sample_rate));
   g1 = (float)(1.0 - g);
   I = (float)(H*sample_rate);
   /* top threshold is < 15.9 db below peak sine wave */
   /* so make it 12dB (i.e. 1/4) */
   c[N-1] = (float)(0.25*(MAXNUM/sqrt(2.0)));
   /* successive thresholds DIFF apart */
   for (j=N-2; j>=0; j--) c[j] = (float)(c[j+1] / DIFF);
   ref = (float)powtodB((float)MAXNUM*(float)MAXNUM/2.0);	/* to normalise the result */


   /* get a struct for return of results */
   /* malloc a new one each time in case user wants to hang on to them */
   resptr = (RESULTS*)calloc(1,sizeof(RESULTS));
   if (!resptr) {
      fprintf(stderr, "SV6_RESULTS - malloc failure - first results\n");
      exit(0);
      }
   resptrptr = &resptr;

   /* zero all the other variables */
   resptr->offset = 0;	/* offset */
   n1 = n2 = 0;		/* total samples */
   s = sq = 0.0;	/* sums of x and x squared */
   p = q = 0.0;		/* partially and fully smoothed envelope */
	      		/* activity and hangover counts */
   for (j=0; j<N; j++) a[j] = h[j] = 0;	/* threshold counts */
   resptr->maxp = resptr->maxn = 0;	/* peak signals */
   }

void sv6_pass1 (short *buffer, long count) {
   long i;
   for (i=0; i<count; i++, buffer++) {
      short x;
      x = *buffer;
      if (x>resptr->maxp) resptr->maxp = x;
      if (x<resptr->maxn) resptr->maxn = x;
      s += x;
      }
   n1 += count;
   first_time = 1;	/* tells pass 2 to calculate offset */
   }

void sv6_pass2 (short *buffer, long count) {	/* returns success flag */
   float x;
   float modx;
   int i,j;

   if (first_time) {	/* calculate the offset (rounded) */
      resptr->offset = (short)(s/n1 + ( (s>=0.0)? 0.5: -0.5 ));
      first_time = 0;
      }

   for (i=0; i<count; i++, buffer++) {
      x = (float)((*buffer) - resptr->offset);	/* remove offset */
      sq += x*x;
      modx = (x>=0)? x: -x;
      p = g*p + g1*modx;		/* envelope filter */
      q = g*q + g1*p;
      for (j=0; j<N; j++) {	/* activity counts */
         if (q >= c[j]) {	/* active */
            a[j]++;
            h[j] = 0;
            }
         else if (h[j]<I) {	/* hangover */
            a[j]++;
            h[j]++;
            }
         else {/* do nothing */}	/* not active */
         }
      }
      n2 += count;
   }

RESULTS *sv6_results() {	/* calculate results and return in struct */
   int j;
   int deltapos = 1;

   /* check for consistency of sample counts */
   /* a zero count on pass 1 is allowed for those who KNOW their signal */
   /* has had the offset removed and omit pass 1 for speed */
   /* this single pass use MUST NOT be incorporate into general purpose progs */
   if (n1 && n1!=n2) {
      fprintf(stderr, "SV6_RESULTS - WARNING - %ld samples on pass 1, %ld on pass 2\n", n1, n2);
      }

#ifdef DEBUG
   fprintf (stderr, "Total samples %d\n", n2);
   fprintf (stderr, "a[j], A[j], C[j], delta[j]\n");
#endif
   for (j=0; j<N; j++) {
      A[j] = (float)powtodB(sq/a[j]);
      /* A-C decreases with increasing j */
      /* 2.0*powtodB(c[j]) is called C[j] in p56 */
      delta[j] = (float)(A[j] - 2.0*powtodB(c[j]) - M);
      if (delta[j] > 0.0) deltapos = 1;
      else {
         if (deltapos) {	/* we have a pos/neg transition */
            if (j==0) {
               fprintf(stderr, "SV6_RESULTS - delta is negative at lowest "
		       "threshold.\nThis means your signal is too small to "
		       "be accurately measured.\n");
               exit(0);
               }
            if (!(*resptrptr)) {	/* for second (nth?) crossing results */
               (*resptrptr) = (RESULTS*)malloc(sizeof(RESULTS));
               if (!(*resptrptr)) {
                  fprintf(stderr, "SV6_RESULTS - malloc failure - nth results\n");
                  exit(0);
                  }
               }
            (*resptrptr)->L = (float)(powtodB(sq/n2) - ref);
            /* interpolate between A[j] and A[j-1]*/
            (*resptrptr)->A = (A[j]*delta[j-1] - A[j-1]*delta[j]) / (delta[j-1] - delta[j]) - ref;
            (*resptrptr)->act = (float)(dBtopow((*resptrptr)->L-(*resptrptr)->A));
            (*resptrptr)->errflg = 0;	/* provided for a (future) check on adequacy of arithmetic */
            resptrptr = &((*resptrptr)->next);	/* moved down linked list */
            *resptrptr = NULL;			/* and terminate it */
            }
         deltapos = 0;
         }
#ifdef DEBUG
      fprintf(stderr, "%d, %f, %f, %f\n", a[j], A[j], 2.0*powtodB(c[j]), delta[j]);
#endif
      }
   return resptr;
}

/*------------------ SFS wrapper for SV6 functions -----------------*/

struct item_header spitem;
short	*sp;

/* main program */
void main(int argc,char **argv)
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char		*ty;		/* item sub type */
	char		*sptype="0";
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */

	float		sprate;
	int		blocklen;
	int		len;
	int		i;
	RESULTS		*resp;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"r",NULL)) < 0)
		error("access error on '%s'",filename);

	/* find item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("could not find input item in '%s'",filename);

	/* get basic params */
	sprate = (float)(1.0/spitem.frameduration);
	blocklen = (int)(10.0 * sprate);	/* 10s worth at a time */
	if (blocklen > spitem.numframes) blocklen=spitem.numframes;
	if ((sp=(short *)sfsbuffer(&spitem,blocklen))==NULL)
		error("could not get memory buffer");

	/* initialise Speech voltmeter */
	sv6_init(sprate);

	/* first pass */
	for (i=0;(len=sfsread(fid,i,blocklen,sp))>0;i+=len)
		sv6_pass1(sp,len);

	/* second pass */
	for (i=0;(len=sfsread(fid,i,blocklen,sp))>0;i+=len)
		sv6_pass2(sp,len);

	/* report results */
	resp = sv6_results();
	if (resp->errflg)
		error("speech voltmeter error code %d", resp->errflg);

	if (resp->next)
		printf("Main result:\n");
	printf("Active speech level : %7.1fdB  Activity : %5.1f%%\n", resp->A, 100.0 * resp->act);
#ifdef EBUG
printf("errflg=%d L=%g A=%g act=%g offset=%d maxp=%d maxn=%d\n",
	resp->errflg,resp->L,resp->A,resp->act,resp->offset,resp->maxp,resp->maxn);
#endif
	if (resp->next) {
		printf("Other results:\n");
		while (resp->next) {
			resp = resp->next;
			printf("Active speech level : %7.1fdB  Activity : %5.1f%%\n", resp->A, 100.0 * resp->act);
#ifdef EBUG
printf("errflg=%d L=%g A=%g act=%g offset=%d maxp=%d maxn=%d\n",
	resp->errflg,resp->L,resp->A,resp->act,resp->offset,resp->maxp,resp->maxn);
#endif
		}
	}

	sfsclose(fid);
	exit(0);
}

#if 0
/* this is the orignal wrapper from Laurie Moye */

#define SV6PASSES 3
#define MINSEGDUR 10.0
#define DEFMINPROP 1.0

#define BUFFER_SIZE   64*1024

void main (int argc, char *argv[])
{
	extern int optind, opterr;
	extern char *optarg;
	char ifname[FNAMELEN+1];
	char extifname[FNAMELEN+1];
	char ofname[FNAMELEN+1];

  FILE *ofile;	       	/* output file pointer */
  int genofname = FALSE;     	/* -O */
  int gotofname = FALSE;     	/* -o */
  float segdur;		/* segment duration */
  int gotdur = FALSE;	/* -d */
  float segper;		/* segment repetition period */
  int gotper = FALSE;	/* -p */
  int verbose = FALSE;	/* verbose flag, -v */
  float minprop = DEFMINPROP;	/* last seg min proportion, (-m) */
  int16 *inbuf = NULL;	/* input signal buffer */
  int reqbuf = FALSE;	/* -b */
  int reqbufsize;	/* requested buffer size (samples) (may use less) */
  char ch;		/* general purpose */
  int i;		/* general purpose */

  /* parse command line */
  while ( (ch = getopt(argc, argv, "Oo:d:p:vm:b:")) !=  EOF
	 && ch != '?') {
    switch(ch) {
    case 'O':
      genofname = TRUE;
      break;
    case 'o':
      gotofname = TRUE;
      strcpy(ofname, optarg);
      break;
    case 'd':
      segdur = atof(optarg);
      gotdur = TRUE;
      break;
    case 'p':
      segper = atof(optarg);
      gotper = TRUE;
      break;
    case 'v':
      verbose = TRUE;
      break;
    case 'm':
      minprop = atof(optarg);
      break;
    case 'b':
      reqbufsize = atoi(optarg);
      reqbuf = TRUE;
      break;
    default:
      fprintf (stderr,
"*WARNING - internal error, some command line options may have been ignored\n"
"the getopt switch statement is broken and doesn't recognise ch = '%c'\n",
	       ch);
    }
  }	/* end of   while ( ch = getopt(argc, argv, "Oo:d:p:vm:b:") ... */

  if (ch == '?' || optind >= argc) {
    printf(
"Usage: sv6 [-O] [-o <output file>] [-d <segment duration (sec)>]\n"
"[-p <segment repetion period (sec)>] [-v] [-m <min proportion>]\n"
"[-b <buffer size (samples)>] <input file> [<input file>] ... ...\n\n"
"    defaults: analyses whole of each input file, output defaults to stdout\n"
"-o <output file> = output to named file\n"
"-O = output to file of same name as first input file with .sv6 ext'n \n"
"    warning given if -O and >1 input file\n"
"-d <segment duration (sec)> = analyse file in short segments\n"
"    this should not be less than 10sec, WARNING on stderr if it is\n"
"-p <segment repetition period (sec)>, if -d and no -p, period = duration\n"
"    WARNING on stderr if period less than duration, but goes ahead\n"
"-v = verbose output, normal output is terse one-liner, easy to parse\n"
"    but BEWARE, even terse output may contain ambiguous results\n"
"-m <min proportion>, bit of signal at file end less than\n"
"    <min proportion>*<segment duration> long is ignored, default = 1.0\n"
"-b <buffer size (samples)> = use up to this size buffer, -b 0 will give\n"
"    buffer big enough for a seg, illegal without -d\n"
"for more info, especially about abmiguous results, see research note\n");
    return;
  }
  /* several checks for silly option values */
  if (minprop > 1.0) {
    fprintf (stderr, "Minprop is %f, should be <= 1.0\n", minprop);
    exit(2);
  }
  if (gotper && !gotdur) {
    fprintf (stderr,
	     "You've specified a period (-p), but not  a duration (-d)\n");
    exit(2);
  }
  if (gotper && gotdur && segper>segdur) {
    fprintf (stderr,
	     "*WARNING - segments do not overlap, %f segs repeated at %f\n",
	     segdur, segper);
  }
  if (reqbuf && reqbufsize==0 && !gotdur) {
    fprintf (stderr,
	     "You can only specify optimimum buffer size (-b 0)\n"
	     "if you also specify a segment duration (-d sss)\n");
    exit(2);
  }


  /* sort out the output file name*/
  if (genofname) {
    if (gotofname) {
      fprintf (stderr,
"You've specified a file name (-o), and asked to generate one (-O)\n");
      exit(2);
    }
    else {
      if (optind != argc-1) {
	fprintf (stderr,
"Generating an output file name (-O) only allowed for single input file\n");
	exit(2);
      }
      else {
	char *chp;
	strcpy(ofname, argv[optind]);
	/* swap any input extension */
	if (chp = strstr(ofname, IDEFEXT)) strcpy(chp, ODEFEXT);
	gotofname = TRUE;
      }
    }
  }	/* end if (genofname) */
  if (gotofname) {
    /* make sure filename has correct extension */
    if(!strstr(ofname, ODEFEXT)) strcat(ofname, ODEFEXT);
  }
  /* open output file, defalting to stdout if no -o or -O option */
  ofile = (!gotofname)? stdout: fopen(ofname, "w");
  if (!ofile) {
    printf("failure to open o/p file %s\n", ofname);
    exit(2);
  }

  fprintf (ofile,
"! Power while active is measured in dB re peak sine wave\n");
  if (!verbose) fprintf(ofile,
"!Filename                                                Power(dB)"
"   Activity(%%)\n");

  /* loop round input files */
  for ( ; optind<argc; optind++) {
    int16 *inptr;	/* input signal buffer pointer */
    long seglen;		/* segment length in samples */
    long replen;		/* segment repetition period in samples */
    long minlength;	/* min length of last seg in samples */
    int bufsize;	/* the actual buffer size in samples */
    int oldbufsize;	/* used in finding final size of buffer */
    int segwillfit;	/* whole segment will fit in buffer */
    /* all buffer and segment navigation is done with `beyondish' numbers */
    /* first in this lump is `start', first not in this lump is `beyond' */
    /* this means that beyond-start is the length of a lump and to move */
    /* to next consecutive lump you just say start=beyond */
    long buf_start, buf_beyond;	/* sample numbers in buffer */
    long seg_start, seg_beyond;	/* sample numbers of segment */
    /* open input file, possibly with added extension */
    strcpy (ifname, argv[optind]);
    ifile = srfopen(ifname,"rb");
    if (!ifile) {
      strcpy (extifname, ifname);
      strcat (extifname, IDEFEXT);
      if (!( ifile = srfopen(extifname,"rb") )) {
	fprintf(stderr, "can't open input file %s or %s\n",
		ifname, extifname);
	exit(2);
      }
      else strcpy(ifname, extifname);
    }
    /* check we have right sort of file */
    if(ifile->srh.file_type != APM_C_SAMP_DAT) {
      fprintf (stderr, "file %s is of type %d, sampled data is type %d\n", ifile,
	       ifile->srh.file_type, APM_C_SAMP_DAT);
      exit(2);
    }
    /* if no sample rate, assume 10kHz */
    if (ifile->frame_rate <= 0) {
      ifile->frame_rate=10000;
      fprintf(stderr, "*WARNING - sample rate is 0, set to 10kHz\n");
    }
    /* this code requires to know length of file */
    if (ifile->number_frames == LONG_MAX) {
      fprintf (stderr, "File header does not contain number of samples\n");
      exit(2);
    }

    /* set duration  and period in samples */
    /* default to whole file */
    if (!gotdur) {
      seglen = ifile->number_frames;
      replen = seglen;
    }
    /* default to consecutive segments */
    else {
      if (!gotper) segper = segdur;
      seglen = segdur * ifile->frame_rate;
      replen = segper * ifile->frame_rate;
    }
    minlength = seglen * minprop;

#ifdef DIAG
    printf("sample_rate %f, file_length %ld\n"
	   "seglen %ld, replen %ld, minlength %ld\n",
	   ifile->frame_rate, ifile->number_frames,
	   seglen, replen, minlength );
#endif


    /* find out how big a buffer we can use */
    oldbufsize = (inbuf)? bufsize: 0;
    bufsize = (reqbuf)? reqbufsize: BUFFER_SIZE;
    segwillfit = FALSE;
    if (gotdur) {
      int biggest = MAX(seglen, replen);
      if (biggest <= bufsize) {
	bufsize = biggest;
	segwillfit = TRUE;
      }
    }
    /* ***** is this neccessary ????? */
    else bufsize = MIN(seglen, bufsize);


    /* civilised handling of failure to allocate huge buffers */
    if (bufsize > oldbufsize)
      inbuf = (int16 *) realloc(inbuf, bufsize*sizeof(int16));
    oldbufsize = bufsize;
    while (!inbuf){
      bufsize /= 2;
      inbuf = (int16 *) calloc(bufsize, sizeof(int16));
    }

#ifdef DIAG
    printf("bufsize %d, oldbufsize %d\n", bufsize, oldbufsize);
#endif

    /* let the user know what happened */
    if (bufsize!=oldbufsize)
      fprintf (stderr,
"*WARNING - Not able to allocate optimum buffer of %d samples, only %d\n",
	       oldbufsize, bufsize);

    /* o/p headings for this file */
    if (verbose) {
      fprintf(ofile, "Input file: %s\n", ifname);
      fprintf(ofile,
"!             Start  Finish  Max pos  Max neg  Offset     Power    Activity\n");
    }
    else	/* try to make it look neat if filename not too long */
      if (strlen(ifname) <= 55)	/* left justify and fill with spaces */
	fprintf(ofile, "%-55.55s", ifname);
      else 			/* truncate from left to right justify */
	fprintf(ofile, "... %s", &ifname[strlen(ifname)-51]);

    /* loop for segments in this file */
    for (buf_start=-1, buf_beyond=0, seg_start=0, seg_beyond=0;
	 seg_beyond < ifile->number_frames;
	 seg_start += replen) {
      int num_read;		/* samples read this time */
      int pass;		/* index through three passes of sv6 */
      RESULTS *resp;	/* structure ptr returned from sv6_results */

      /* find out where this seg should end */
      seg_beyond = seg_start+seglen;	/* normal end */
      /* should we do the remainder of the file in this seg */
      /* because we allow overlap, this is a bit complicated */
      /* either not enough left for a full-length seg */
      if (seg_beyond > ifile->number_frames ||
	  /* or the next seg would be too short, so this one is long */
	  seg_start+replen+minlength > ifile->number_frames)
	seg_beyond = ifile->number_frames;

#ifdef DIAG
      printf("seg_start %ld, seg_beyond %ld\n", seg_start, seg_beyond);
#endif

      /* just make sure I've got that right */
      /* this warning produced if segment*minprop specified longer than file */
      /* or if I've got the end of file segment loop code wrong */
      if (seg_beyond-seg_start < minlength)
	fprintf (stderr,
"*WARNING - this segment is too short, perhaps file is very short?\n"
"*WARNING - seg_start = %d, seg_beyond = %d, seglen = %d\n"
"*WARNING - replen = %d, ifile->number_frames = %d\n",
		 seg_start, seg_beyond, seglen, replen, ifile->number_frames);

      /* initialise the measurements */
      sv6_init(ifile->frame_rate);

      /* loop around the passes  of sv6 */
      for (pass = 1; pass<SV6PASSES; pass++) {
	long chunk_start;	/* start of seg chunk being */
	int thislen;		/* actual length to process in buffer */

	/* loop for bufferfuls to make a segment */
	for (chunk_start=seg_start; chunk_start < seg_beyond;
	     chunk_start += thislen) {

#ifdef DIAG
	  printf("chunk_start %ld, thislen (at loop head) %d\n",
		 chunk_start, thislen);
#endif


	  /* is chunk_start in buffer */
	  if (chunk_start >= buf_start && chunk_start < buf_beyond) {
	    if (segwillfit) {	/* move to bottom and top up buffer */
	      int shift = chunk_start - buf_start;
	      memmove(inbuf, inbuf+shift, (bufsize-shift)*sizeof(int16));
	      buf_start = chunk_start;
	      num_read = sruread(inbuf+bufsize-shift, shift, ifile);

#ifdef DIAG
	      printf("just read %d after moving buffer contents down\n",
		     num_read);
#endif

	      buf_beyond += num_read;
	    }
	    else {/* just use what we've got */}
	  }
	  else {	/* we must read a new bufferful */
	    /* check for consecutive reads */
	    if (chunk_start == buf_beyond) {
	      num_read = sruread(inbuf, bufsize, ifile);

#ifdef DIAG
	      printf("just read %d after consecutive read\n",
		     num_read);
#endif

	    }
	    else {	/* seek and read from the relevant place */
	      int result;
	      result = sruseek(ifile, chunk_start, 0);
	      if (result==EOF) {
		fprintf (stdout,
			 "Ouch, trying to seek beyond end of data, file %s\n",
			 ifname);
		exit(2);
	      }
	      num_read = sruread(inbuf, bufsize, ifile);

#ifdef DIAG
	      printf("just read %d after seek to %d\n",
		     num_read, chunk_start);
#endif

	    }
	    buf_start = chunk_start;
	    buf_beyond = buf_start + num_read;
	  }		/* should have some valid data now */

	  /* if we've read EOF, we should have completed last seg */
	  if (srfeof(ifile) && buf_beyond != ifile->number_frames) {
	    fprintf (stderr, "EOF found before all signal read, file %s\n",
		     ifile);
	    exit(2);
	  }

	  /* what part of buffer do we need? */
	  inptr = inbuf + chunk_start - buf_start;
	  thislen = MIN(seg_beyond, buf_beyond) - chunk_start;

#ifdef DIAG
	  printf("buf_start %ld, buf_beyond %ld, thislen %d\n",
		 buf_start, buf_beyond, thislen);
	  {
	    int16 *pp;
	    int ii;
	    int mp=0;
	    int mn=0;
	    long posat=0;
	    long negat=0;
	    for (pp=inptr, ii=0; ii<thislen; pp++, ii++) {
	      if (*pp > mp) {
		mp = *pp;
		posat = ii;
	      }
	      else if (*pp < mn) {
		mn = *pp;
		negat = ii;
	      }
	    }
	    printf("***segmax %d, segmin %d, posat %ld, negat %ld\n",
		   mp, mn, posat, negat);
	  }
#endif


	  /* now do relevant pass */
	  switch (pass) {
	  case 1:
            /* first pass, calculate offset */
            sv6_pass1(inptr, thislen);
            break;
	  case 2:
            /* second pass, do meter calculations */
            sv6_pass2(inptr, thislen);
	    break;
	  }
	}	/* end of if (chunk_start >= buf_start && chunk_start ... */
      }
      resp = sv6_results();

      if (resp->errflg)
	fprintf(ofile,  "***** Result in error (flag = %d)\n", resp->errflg);
      else {
	int ambig;
	if (verbose) {
	  fprintf(ofile, "            %7.2f %7.2f %8d %8d",
		  seg_start/(float)ifile->frame_rate,
		  seg_beyond/(float)ifile->frame_rate,
		  (int)resp->maxp, (int)resp->maxn);
	}
	/* loop to print ambiguous results */
	for (ambig=0 ; resp; resp=resp->next, ambig++) {
	  if (verbose) {
	    if (ambig==1) fprintf(ofile,
"further results:                             ");
	    else if (ambig>1) fprintf(ofile,
"                                             ");
	    if (resp->errflg)
	      fprintf(ofile, "***** Result in error (flag = %d)\n",
		      resp->errflg);
	    else fprintf(ofile, " %7d %7.1fdB   %8.1f%%\n",
			 resp->offset, resp->A, 100.*resp->act);
	  }
	  else {
	    /* identify segment starts and ambiguous results */
	    if (!ambig) {
	      if (seg_start != 0) fprintf(ofile,
"*seg@%-7.3f                                           ",
					  seg_start/(float)ifile->frame_rate);
	    }
	    else fprintf(ofile,
"*ambiguous                                             ");
	    fprintf(ofile, " %7.1f     %8.1f\n",
		    resp->A, 100.*resp->act);
	  }
	}
      }
    }	/* end of for (buf_start=-1, buf_beyond=0; seg_start ... */
    srfclose(ifile);
  }	/* end of for ( ; optind<argc; optind++) */
}

#endif
