/* wordlink -- link annotated regions as new items */

/* M.A.Huckvale - University College London */

/* version 1.0 - January 2012 */

#define PROGNAME "wordlink"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH WORDLINK SFS1 UCL
.SH NAME
wordlink -- link annotated regions of speech as new items
.SH SYNOPSIS
.B wordplay
(-n wnum) (-s wnum) (-e wnum) (-i item) file
.SH DESCRIPTION
.I wordlink
is a program to link annotated regions of a speech item as new items.  It assumes that individual
words are annotated at the start and end.  If an annotation is labelled simply '/', then
it is skipped.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -n wnum
Link word number wnum only.  Default: link all words.
.TP 11
.BI -s wnum
Start linkink at word wnum.  Default: word 1.
.TP 11
.BI -e enum
Finished linking word wnum.  Default: last word.
.SH INPUT ITEMS
.IP SP
Any speech item.
.IP AN
Word annotations.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
wordchop
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include "sfs.h"
#include "sfsdata.h"

/* global data */
char	filename[SFSMAXFILENAME];
struct item_header	spitem;
short			*sp;
int			spsize;
struct item_header	anitem;
struct an_rec		*antab;
struct item_header	lnitem;
struct link_header	lnlink;
struct link_header	lhead;


/* operational modes */
int	wnum = -1;	/* link all words */
int	snum = 1;
int	fnum = 1000000;

/* do the link */
void dolink(double stime,double etime)
{
	int	ssamp,nsamp;
	int		nbeep,nsilence;
	int		totsamp;
	short	*spbuff;
	int		i;

	ssamp = (int)((stime-spitem.offset)/spitem.frameduration);
	nsamp = (int)((etime-spitem.offset)/spitem.frameduration - ssamp);

	if (ssamp < 0) ssamp = 0;
	if (nsamp > (spitem.numframes - ssamp)) nsamp = spitem.numframes-ssamp;
	if (nsamp <= 0) return;

	/* take copy of item header */
	sfsheader(&lnitem,spitem.datatype,spitem.floating,
			spitem.datasize,spitem.framesize,
			spitem.frameduration,spitem.offset,
			spitem.windowsize,spitem.overlap,spitem.lxsync);

	/* do item history */
	sprintf(lnitem.history,"%s(file=%s,item=%d.%02d,start=%d,end=%d,history=%s)",
		PROGNAME,
		filename,
		spitem.datatype,
		spitem.subtype,
		ssamp,ssamp+nsamp,
		spitem.history);
	strcpy(lnitem.params,spitem.params);

	/* check input item not linked itself ! */
	if (spitem.datapresent==2) {
		/* take copy of link header */
		lnlink = lhead;
		lnlink.offset += (ssamp*spitem.datasize*spitem.framesize)*(lnlink.multiplex+1);
	}
	else {
		/* build link header */
		memset(&lnlink,0,sizeof(struct link_header));
		strcpy(lnlink.filename,pathname(filename));
		lnlink.filetype = SFS_TYPE;
		lnlink.datatype = spitem.datatype;
		lnlink.subtype = spitem.subtype;
		lnlink.offset = (int)(((stime-spitem.offset)/spitem.frameduration)*spitem.datasize*spitem.framesize);
		lnlink.linkdate = spitem.processdate;
		lnlink.machine = spitem.machine;
	}

	/* write link to file */
	if (!sfswritelink(&lnitem,nsamp,&lnlink,filename))
		error("write failed on temporary file",NULL);


}

/* main program */
void main(int argc,char *argv[])
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char		*ty;		/* item sub type */
	char		*sptype="0";
	char		*antype="0";
	/* file variables */
	int		fid;		/* input file descriptor */
	int		i,wcnt;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:n:s:e:1:BS")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Replay words in file V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else if (it == AN_TYPE)
					antype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'n' :	/* link nth words */
			wnum = atoi(optarg);
			break;
		case 's' :	/* start word # */
			snum = atoi(optarg);
			break;
		case 'e' :	/* end word # */
			fnum = atoi(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-n wnum) (-s wnum) (-e wnum) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"r",NULL)) < 0)
		error("access error on '%s'",filename);

	/* find annotations */
	if (!sfsitem(fid,AN_TYPE,antype,&anitem))
		error("could not find annotation item in '%s'",filename);

	/* load annotations */
	if ((antab = (struct an_rec *)sfsbuffer(&anitem,anitem.numframes))==NULL)
		error("could not get buffer");
	if (sfsread(fid,0,anitem.numframes,antab)!=anitem.numframes)
		error("read error on '%s'",filename);

	/* locate speech */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("could not find speech item in '%s'",filename);

	/* check this item is not itslef linked */
	if (spitem.datapresent==2) {
		/* item is linked - get link header */
		lseek(fid,sfsdata[fid]->datastart,0);
		read(fid,&lhead,sizeof(struct link_header));
		sfsdata[fid]->currpos=lseek(fid,0L,1);
	}

	/* do linking */
	if (wnum > 0) snum = fnum = wnum;
	wcnt=1;
	for (i=0;i<anitem.numframes-1;i++) {
		if (strcmp(antab[i].label,"/")) {
			if ((wcnt >= snum) && (wcnt <= fnum)) {
				printf("%d. %s",wcnt,antab[i].label);
				fflush(stdout);
				dolink(antab[i].posn*anitem.frameduration+anitem.offset,
					  antab[i+1].posn*anitem.frameduration+anitem.offset);
				printf("\n");
			}
			wcnt++;
		}
	}

	/* that's all folks ! */
	sfsupdate(filename);
	sfsclose(fid);
	exit(0);
}

