/* code -- code for machine that executes program */

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include "sml.h"
#ifdef DOS
#include "y_tab.h"
#include <conio.h>
#else
#include "y.tab.h"
#endif
#include <setjmp.h>

/* global data initialisation */
double	timerange[10][2]={{0.0,9999.9E99},{0.0,0.0}};
double	ERRVAL=21.1e-86;	/* unlikely (!) number for error handling */
char 	ERRSTRING[10]="E=R=R=O=R";
jmp_buf jumpenv;

/* install code in machine */
void code(inst)
int	inst;
{
	if (pc>=MAXCODE) {
		warning("too much code generated","");
		exit(1);
	}
#ifdef IAG
	if (inst < 1000)
		fprintf(stderr,"variable %d\n",inst);
	else {
		printinstr(stderr,inst);
		fprintf(stderr,"\n");
	}
#endif
	memory[pc++]=inst;
}

/* generate code for function call */
void callcode(pnum,t,inst,entry)
int	*pnum,t,inst,entry;
{
	int	ptype,pstat;
	(*pnum)++;	/* move to next parameter */
	ptype=stab[*pnum].type;
	pstat=stab[*pnum].stat;
	if (t==NUMBER) {
		if (ptype!=VAR)
			warning("incorrect parameter type in call to",stab[entry].name);
		else if (pstat==ADRPAR)
			warning("variable access only in call to",stab[entry].name);
	}
	else if (t==STRING) {
		if (ptype!=STRVAR)
			warning("incorrect parameter type in call to",stab[entry].name);
		else if (pstat==ADRPAR)
			warning("variable access only in call to",stab[entry].name);
	}
	else if (t != ptype)
		warning("incorrect parameter type in call to",stab[entry].name);
	else if ((pstat==VALPAR) && (inst != STOP))
		code(inst);
}

/* dump source and machine code */
void dumpscode(filename)
char	*filename;
{
	int	i,num,count=0,lineno=0;
	char	line[MAXSTRING],*ctime();
	time_t	tim;

	/* header */
	tim=time((time_t *)0);
	printf("Compilation of \"%s\" on %s\n",filename,ctime(&tim));
	rewind(ip);
	sline[slent].pc=pc;
	printf("Line PC   Instruction/Source\n");
	printf("-------------------------------------------------------------------\n");
	for (i=0;i<slent;i++) {
		for (;lineno<sline[i].lineno;lineno++) {
			fgets(line,MAXSTRING,ip);
			printf("%4d      %s",lineno+1,line);
		}
		for (;count<sline[i+1].pc;count++) {
			printf("     %04d ",count);
			if (memory[count] < 1000)
				printf("variable %d\n",memory[count]);
			else {
				num=printinstr(stdout,memory[count]);
				while (num > 0) {
					printf(" %d,",memory[++count]);
					num--;
				}
				printf("\n");
			}
		}
	}
}

/* dump machine code only */
void dumpcode()
{
	int	i,num;
	printf("======== Code ========\n");
	for (i=0;i<pc;i++) {
		printf("%4d) ",i);
		if (memory[i] < 1000)
			printf("variable %d\n",memory[i]);
		else {
			num=printinstr(stdout,memory[i]);
			while (num > 0) {
				printf(" %d,",memory[++i]);
				num--;
			}
			printf("\n");
		}
	}
}

/* print text version of instruction */
int printinstr(fil,i)
FILE	*fil;
int	i;
{
	switch(i) {
		case ADDRVAR:		fprintf(fil,"ADDRVAR"); return(1);
		case ADDRARRAY:		fprintf(fil,"ADDRARRAY"); return(1);
		case ADDRELEMENT:	fprintf(fil,"ADDRELEMENT"); return(1);
		case LOADVAR:		fprintf(fil,"LOADVAR"); return(0);
		case STOREVAR:		fprintf(fil,"STOREVAR"); return(0);
		case LOADVARCONST:	fprintf(fil,"LOADVARCONST"); return(1);

		case ADDRSTR:		fprintf(fil,"ADDRSTR"); return(1);
		case ADDRSTRARRAY:	fprintf(fil,"ADDRSTRARRAY"); return(1);
		case ADDRSTRELEMENT:	fprintf(fil,"ADDRSTRELEMENT"); return(1);
		case LOADSTR:		fprintf(fil,"LOADSTR"); return(0);
		case STORESTR:		fprintf(fil,"STORESTR"); return(0);
		case LOADSTRCONST:	fprintf(fil,"LOADSTRCONST"); return(1);

		case ADDRSTAT:		fprintf(fil,"ADDRSTAT"); return(1);
		case ADDRSTATFIELD:	fprintf(fil,"ADDRSTATFIELD"); return(2);
		case ADDRSTATARRAY:	fprintf(fil,"ADDRSTATARRAY"); return(1);
		case ADDRSTATELEMENT:	fprintf(fil,"ADDRSTATELEMENT"); return(1);
		case ADDRSTATELEMENTFIELD:	fprintf(fil,"ADDRSTATELEMENTFIELD"); return(2);
		case UPDATESTAT:	fprintf(fil,"UPDATESTAT"); return(0);

		case LOADFILE:		fprintf(fil,"LOADFILE"); return(1);
		case ASSNFILE:		fprintf(fil,"ASSNFILE"); return(2);

		case BLOCK:		fprintf(fil,"BLOCK"); return(1);
		case CALL:		fprintf(fil,"CALL"); return(1);
		case RETURN:		fprintf(fil,"RETURN"); return(0);
		case SRETURN:		fprintf(fil,"SRETURN"); return(0);
		case NEXTIPFILE:	fprintf(fil,"NEXTIPFILE"); return(0);
		case CLOSEDOWN:		fprintf(fil,"CLOSEDOWN"); return(1);
		case STOP:		fprintf(fil,"STOP"); return(0);
		case EXIT:		fprintf(fil,"EXIT"); return(0);

		case GOTO:		fprintf(fil,"GOTO"); return(1);
		case CONDITION:		fprintf(fil,"CONDITION"); return(3);
		case WHILECODE:		fprintf(fil,"WHILECODE"); return(2);
		case FORCODE:		fprintf(fil,"FORCODE"); return(3);
		case WITH:		fprintf(fil,"WITH"); return(0);
		case WITHN:		fprintf(fil,"WITHN"); return(0);
		case WITHIN:		fprintf(fil,"WITHIN"); return(0);
		case CLEAR:		fprintf(fil,"CLEAR"); return(1);
		case CLEARARRAY:	fprintf(fil,"CLEARARRAY"); return(1);
		case CLEARELEMENT:	fprintf(fil,"CLEARELEMENT"); return(1);
		case LOBOUND:		fprintf(fil,"LOBOUND"); return(1);
		case HIBOUND:		fprintf(fil,"HIBOUND"); return(1);
		case ABORT:		fprintf(fil,"ABORT"); return(0);
		case ABORTTEST:		fprintf(fil,"ABORTTEST"); return(0);
		case TRACEON:		fprintf(fil,"TRACEON"); return(0);
		case TRACEOFF:		fprintf(fil,"TRACEOFF"); return(0);
		case BREAK:		fprintf(fil,"BREAK");return(0);

		case SWITCH:		fprintf(fil,"SWITCH"); return(2);
		case CASE:		fprintf(fil,"CASE"); return(2);
		case STRCASE:		fprintf(fil,"STRCASE"); return(2);
		case PATCASE:		fprintf(fil,"PATCASE"); return(2);
		case RANGECASE:		fprintf(fil,"RANGECASE"); return(3);
		case DEFCASE:		fprintf(fil,"DEFCASE"); return(0);

		case PLUS:		fprintf(fil,"PLUS"); return(0);
		case MINUS:		fprintf(fil,"MINUS"); return(0);
		case TIMES:		fprintf(fil,"TIMES"); return(0);
		case DIVIDE:		fprintf(fil,"DIVIDE"); return(0);
		case MODULUS:		fprintf(fil,"MODULUS"); return(0);
		case NEGATE:		fprintf(fil,"NEGATE"); return(0);

		case GTCODE:		fprintf(fil,"GT"); return(0);
		case GECODE:		fprintf(fil,"GE"); return(0);
		case LTCODE:		fprintf(fil,"LT"); return(0);
		case LECODE:		fprintf(fil,"LE"); return(0);
		case EQCODE:		fprintf(fil,"EQ"); return(0);
		case NECODE:		fprintf(fil,"NE"); return(0);
		case ORCODE:		fprintf(fil,"OR"); return(0);
		case ANDCODE:		fprintf(fil,"AND"); return(0);
		case NOTCODE:		fprintf(fil,"NOT"); return(0);
		case ERRCHECK:		fprintf(fil,"ERRCHECK"); return(0);

		case STRCAT:		fprintf(fil,"STRCAT"); return(0);
		case STRSCAT:		fprintf(fil,"STRSCAT"); return(0);
		case STRCUT:		fprintf(fil,"STRCUT"); return(0);
		case STRCUT2:		fprintf(fil,"STRCUT2"); return(0);

		case PRINTVAL:		fprintf(fil,"PRINTVAL"); return(1);
		case PRINTVAL1:		fprintf(fil,"PRINTVAL1"); return(1);
		case PRINTVAL2:		fprintf(fil,"PRINTVAL2"); return(1);
		case PRINTSTR:		fprintf(fil,"PRINTSTR"); return(1);
		case PRINTSTR1:		fprintf(fil,"PRINTSTR1"); return(1);
		case INPUTLINE:		fprintf(fil,"INPUTLINE"); return(1);
		case READVAL:		fprintf(fil,"READVAL"); return(0);
		case READSTR:		fprintf(fil,"READSTR"); return(0);
		case READLINE:		fprintf(fil,"READLINE"); return(0);

		default:	fprintf(fil,"code=%d",i);return(0);
	}
}

/* push and pop stack routines */
void push(d)
double	d;
{
	if (sp>=MAXSTACK) error("stack overflow","");
	stack[sp].type=DOUBLEVAL;
	stack[sp++].u.d=d;
}
double pop()
{
	if (sp<=0) error("stack underflow","");
	return (stack[--sp].u.d);
}
void vpush(v)
void	*v;
{
	if (sp>=MAXSTACK) error("stack overflow","");
	stack[sp].type=ADDRVAL;
	stack[sp++].u.v=v;
}
void *vpop()
{
	if (sp<=0) error("stack underflow","");
	return (stack[--sp].u.v);
}
void fpush(f)
Symbol	*f;
{
	if (sp>=MAXSTACK) error("stack overflow","");
	stack[sp].type=FILEVAL;
	stack[sp++].u.f=f;
}
Symbol *fpop()
{
	if (sp<=0) error("stack underflow","");
	return (stack[--sp].u.f);
}
void spush(s)
char	*s;
{
	if (sp>=MAXSTACK) error("stack overflow","");
	if (s != ERRSTRING) {
		stack[sp].u.s = Malloc(strlen(s)+1);
		stack[sp].type=STRINGVAL;
		strcpy(stack[sp++].u.s,s);
	}
	else
		stack[sp++].u.s=s;
}
char *spop()
{
	if (sp<=0) error("stack underflow","");
	return(stack[--sp].u.s);
}

int arrayindex(lo,hi,val)
int	lo,hi;
double	val;
{
	int	i;
	if (val==ERRVAL) {
		runtimewarning("subscript error","");
		return(0);
	}
	i=(int)val;
	if ((i<lo) || (i>hi)) {
		runtimewarning("subscript out of range","");
		return(0);
	}
	else
		return(i-lo);
}

/* prepare machine */
void prepare()
{
	int	v;
	time_t	curtime;

	sline[slent].pc=pc;

	v=lookup("$date");
	curtime = time((time_t *) 0);
	stab[v].u.str=ctime(&curtime);
	v=lookup("$filecount");
	stab[v].u.val=0.0;
	v=lookup("$filename");
	stab[v].u.str=ERRSTRING;
	v=lookup("ERROR");
	stab[v].u.val=ERRVAL;
}

/* run machine */
#define TRUE 1.0		/* conditional codes */
#define FALSE 0.0
#define OK 0			/* return codes */
#define BRK 1
#define FAIL 2
int	currp=0;
int run(p)
int	p;	/* program counter */
{
	double 	d,e,f,atof();
	FILE 	*fil;
	double	*dp,*st,**dpp;
	int	i,j,v,w,*ptr,s1,s2,len,num,func,block;
	char	*str,*str1,*str2,**strp,***strpp;
	char	inputstring[MAXSTRING],*inputpos,*inputend;
	Symbol	*symp;
	int	exitmode=0,loc;
	static int runlevel=0;

	if (runtrace>=2) fprintf(stderr,"\n**** Machine entered at location %d level %d\n",p,runlevel);

	inputpos = inputend = inputstring;

	/* the machine ! */
	while (memory[p] != STOP) {
		currp=p;
		if (runtrace>=1) {
			loc=0;
			while (currp >= sline[loc+1].pc) loc++;
			if ((currp==sline[loc].pc) ||
			    ((memory[sline[loc].pc]==STOP) &&
			     (currp==sline[loc].pc+1))) {
				fseek(ip,sline[loc].fptr,0);
				fgets(inputstring,MAXSTRING,ip);
				fprintf(stderr,"++++%4d %s",sline[loc].lineno,inputstring);
			}
		}
		if (runtrace>=2) {
			fprintf(stderr,"\n**** Stack    = ");
			for (i=sp-1;i>=0;i--) switch (stack[i].type) {
				case DOUBLEVAL: fprintf(stderr,"%g ",stack[i].u.d);
						break;
				case STRINGVAL:	fprintf(stderr,"\"%s\" ",stack[i].u.s);
						break;
				case ADDRVAL:	fprintf(stderr,"(%d) ",(int)stack[i].u.v);
						break;
				case FILEVAL:	fprintf(stderr,"[%d] ",(int)stack[i].u.f);
						break;
			}
			fprintf(stderr,"\n**** Step %3d = ",p);
			num=printinstr(stderr,memory[p]);
			for (i=0;i<num;i++) fprintf(stderr," %d,",memory[p+i+1]);
			fprintf(stderr,"\n");
			fflush(stderr);
		}
		switch(memory[p++]) {
	case ADDRVAR:
	case ADDRSTR:
	case ADDRSTAT:
			v=memory[p++];
			vpush((void *)(*(stab[v].u.entry)));
			break;
	case ADDRSTATFIELD:
			v=memory[p++];
			w=memory[p++];
			vpush((void *)(((double *)*(stab[v].u.entry)) + stab[w].vtype));
			break;
	case ADDRARRAY:
	case ADDRSTRARRAY:
	case ADDRSTATARRAY:
			v=memory[p++];
			vpush((void *)(*(stab[v].u.entry)));
			vpush((void *)(stab[v].lo));
			vpush((void *)(stab[v].hi));
			break;
	case ADDRELEMENT:
			d=pop();
			v=memory[p++];
			i=arrayindex(stab[v].lo,stab[v].hi,d);
			vpush((void *)(((double *) *(stab[v].u.entry))+i));
			break;
	case ADDRSTRELEMENT:
			d=pop();
			v=memory[p++];
			i=arrayindex(stab[v].lo,stab[v].hi,d);
			vpush((void *)(((char **) *(stab[v].u.entry))+i));
			break;
	case ADDRSTATELEMENT:
			d=pop();
			v=memory[p++];
			i=arrayindex(stab[v].lo,stab[v].hi,d);
			vpush((void *)(((double *) *(stab[v].u.entry))+6*i));
			break;
	case ADDRSTATELEMENTFIELD:
			d=pop();
			v=memory[p++];
			w=memory[p++];
			i=arrayindex(stab[v].lo,stab[v].hi,d);
			vpush((void *)(((double *) *(stab[v].u.entry))+6*i+stab[w].vtype));
			break;
	case LOADVAR:	push(*(double *)(vpop()));
			break;
	case LOADVARCONST:
			v=memory[p++];
			push(stab[v].u.val);
			break;
	case STOREVAR:	d=pop();
			if (runtrace==1) fprintf(stderr,"++++ Evaluates to %g\n",d);
			*(double *)(vpop())=d;
			break;
	case LOADSTR:	strp=(char **)(vpop());
			if (*strp == NULL)
				spush("");
			else
				spush(*strp);
			break;
	case LOADSTRCONST:
			v=memory[p++];
			spush(stab[v].u.str);
			break;
	case STORESTR:	str=spop();
			if (runtrace==1) fprintf(stderr,"++++ Evaluates to \"%s\"\n",str);
			strp=(char **)(vpop());
			if ((*strp != NULL) &&
			    (*strp != ERRSTRING)) Free(*strp);
			*strp=str;
			break;
	case UPDATESTAT:
			d=pop();
			if (runtrace==1) fprintf(stderr,"++++ Evaluates to %g\n",d);
			st=(double *)vpop();
			if (d != ERRVAL) {
				st[COUNT]++;		/* update counter */
				st[SUM] += d;		/* update sum */
				st[SUMSQ] += d*d;	/* update sum squared */
				st[MEAN] = st[SUM]/st[COUNT];
				if (st[COUNT] > 1.0) {
					st[VARIANCE] =
						(st[SUMSQ] -
						(st[SUM]*st[SUM]/st[COUNT])) /
						(st[COUNT] - 1.0);
					if (st[VARIANCE] < 0.0) st[VARIANCE] = 0.0;
					st[STDDEV] = sqrt(st[VARIANCE]);
				}
				else {
					st[VARIANCE] = ERRVAL;
					st[STDDEV] = ERRVAL;
				}
			}
			break;
	case LOADFILE:	v=memory[p++];
			fpush(&stab[v]);
			break;
	case ASSNFILE:	v=memory[p++];
			w=memory[p++];
			stab[v].u.fil=stab[w].u.fil;
			stab[v].vtype=stab[w].vtype;
			break;
	case BLOCK:	block=memory[p++];
			s1=stab[block].lo;
			s2=stab[block].hi;
			if (s2<s1) break;
			ptr=(int *)Calloc(s2-s1+2,sizeof(int));
			*ptr=(int)stab[block].u.entry;
			stab[block].u.entry=ptr++;
			for (i=s1;i<=s2;i++) {
/*
				fprintf(stderr,"BLOCK on %-10s, type %d, lo=%d, hi=%d\n",
					stab[i].name,stab[i].type,stab[i].lo,stab[i].hi);
*/
				stab[i].u.entry=ptr;
				switch (stab[i].type) {
				case VAR:	dpp=(double **)ptr;
						*dpp=(double *)Calloc(1,sizeof(double));
						**dpp=0.0;
						break;
				case STATVAR:	dpp=(double **)ptr;
						*dpp=(double *)Calloc(6,sizeof(double));
						for (j=0;j<6;j++) (*dpp)[j]=0.0;
						break;
				case STRVAR:	strpp=(char ***)ptr;
						*strpp=(char **)Calloc(1,sizeof(char *));
						**strpp=NULL;
						break;
				case FILEVAR:	stab[i].u.fil = (FILE *)NULL;
						stab[i].vtype = BADFIL;
						break;
				case ARRAY:	len=stab[i].hi-stab[i].lo+1;
						dpp=(double **)ptr;
						*dpp=(double *)Calloc(len,sizeof(double));
						for (j=0;j<len;j++) (*dpp)[j]=0.0;
						break;
				case STATARRAY:	len=6*(stab[i].hi-stab[i].lo+1);
						dpp=(double **)ptr;
						*dpp=(double *)Calloc(len,sizeof(double));
						for (j=0;j<len;j++) (*dpp)[j]=0.0;
						break;
				case STRARRAY:	len=stab[i].hi-stab[i].lo+1;
						strpp=(char ***)ptr;
						*strpp=(char **)Calloc(len,sizeof(char *));
						for (j=0;j<len;j++) (*strpp)[j]=NULL;
						break;
				case NUMBER:	break;
				default:	runtimewarning("BLOCKing error","");
						fprintf(stderr,"variable type=%d\n",stab[i].type);
				} /* switch */
				ptr++;
			} /* for */
			break;
	case CALL:	func=memory[p++];
			if (stab[func].stat==BLTIN) {
				if (stab[func].type==FUNCTION)
					push((stab[func].u.func()));
				else
					spush((stab[func].u.sfunc()));
				break;
			}
			s1=stab[func].lo;
			s2=stab[func].hi;
			ptr=(int *)Calloc(s2-s1+2,sizeof(int));
			*ptr=(int)stab[func].u.entry;
			stab[func].u.entry=ptr++;
			for (i=s2;i>=s1;i--) {
/*
				fprintf(stderr,"CALL on %-10s, type %d, lo=%d, hi=%d\n",
					stab[i].name,stab[i].type,stab[i].lo,stab[i].hi);
*/
				stab[i].u.entry=ptr;
				switch (stab[i].type) {
				case VAR:	if (stab[i].stat==ADRPAR) {
							dpp=(double **)ptr;
							*dpp=(double *)vpop();
						}
						else {
							dpp=(double **)ptr;
							*dpp=(double *)Calloc(1,sizeof(double));
							**dpp=pop();
						}
						break;
				case STATVAR:	if (stab[i].stat==ADRPAR) {
							dpp=(double **)ptr;
							*dpp=(double *)vpop();
						}
						else {
							dpp=(double **)ptr;
							*dpp=(double *)Calloc(6,sizeof(double));
							dp=(double *)vpop();
							for (j=0;j<6;j++) (*dpp)[j]=dp[j];
						}
						break;
				case STRVAR:	if (stab[i].stat==ADRPAR) {
							strpp=(char ***)ptr;
							*strpp=(char **)vpop();
						}
						else {
							str=spop();
							strpp=(char ***)ptr;
							*strpp=(char **)Calloc(1,sizeof(char *));
							**strpp=(char *)Malloc(strlen(str)+1);
							strcpy(**strpp,str);
						}
						break;
				case FILEVAR:	symp = fpop();
						stab[i].vtype=symp->vtype;
						stab[i].u.fil=symp->u.fil;
						break;
				case ARRAY:	stab[i].hi=(int)vpop();
						stab[i].lo=(int)vpop();
						if (stab[i].stat==ADRPAR) {
							dpp=(double **)ptr;
							*dpp=(double *)vpop();
						}
						else {
							len=stab[i].hi-stab[i].lo+1;
							dpp=(double **)ptr;
							*dpp=(double *)Calloc(len,sizeof(double));
							dp=(double *)vpop();
							for (j=0;j<len;j++) (*dpp)[j]=dp[j];
						}
						break;
				case STATARRAY:	stab[i].hi=(int)vpop();
						stab[i].lo=(int)vpop();
						if (stab[i].stat==ADRPAR) {
							dpp=(double **)ptr;
							*dpp=(double *)vpop();
						}
						else {
							len=6*stab[i].hi-stab[i].lo+1;
							dpp=(double **)ptr;
							*dpp=(double *)Calloc(len,sizeof(double));
							dp=(double *)vpop();
							for (j=0;j<len;j++) (*dpp)[j]=dp[j];
						}
						break;
				case STRARRAY:	stab[i].hi=(int)vpop();
						stab[i].lo=(int)vpop();
						if (stab[i].stat==ADRPAR) {
							strpp=(char ***)ptr;
							*strpp=(char **)vpop();
						}
						else {
							len=stab[i].hi-stab[i].lo+1;
							strpp=(char ***)ptr;
							*strpp=(char **)Calloc(len,sizeof(char *));
							strp=(char **)vpop();
							for (j=0;j<len;j++) {
								(*strpp)[j]=(char *)Malloc(strlen(strp[j])+1);
								strcpy((*strpp)[j],strp[j]);
							}
						}
						break;
				default:	runtimewarning("CALLing error","");
						fprintf(stderr,"variable type=%d\n",stab[i].type);
				} /* switch */
				ptr++;
			} /* for */
			/* run function */
			runlevel++;
			run(stab[func].vtype);
			runlevel--;
			/* delete function parameter block */
			s1=stab[func].lo;
			s2=stab[func].hi;
			unblock(s1,s2);
			ptr=stab[func].u.entry;
			stab[func].u.entry=(int *)*ptr;
			Free((char *)ptr);
			ptr=stab[func].u.entry;
			if (ptr != NULL) {
				ptr++;
				for (i=s2;i>=s1;i--) if (stab[i].type!=FILEVAR)
					stab[i].u.entry=ptr++;
			}
			/* delete local block */
			block=memory[stab[func].vtype+1];
			s1=stab[block].lo;
			s2=stab[block].hi;
			if (s2<s1) break;
			unblock(s1,s2);
			ptr=stab[block].u.entry;
			stab[block].u.entry=(int *)*ptr;
			Free((char *)ptr);
			ptr=stab[block].u.entry;
			if (ptr != NULL) {
				ptr++;
				for (i=s1;i<=s2;i++) if (stab[i].type!=FILEVAR)
					stab[i].u.entry=ptr++;
			}
			if (runtrace==1) {
				i=sp-1;
				if (stack[i].type==DOUBLEVAL)
					fprintf(stderr,"++++ Evaluates to %g\n",stack[i].u.d);
				else
					fprintf(stderr,"++++ Evaluates to \"%s\"\n",stack[i].u.s);
			}
			break;
	case RETURN:	push(0.0);
			break;
	case SRETURN:	spush("");
			break;
	case NEXTIPFILE:
			inmain=1;
			/* set escape jump buffer */
			if (setjmp(jumpenv)==0)
				mainpc=p;
			else
				p=mainpc;
			/* skip to next valid file */
			filename = nextfile(gargc,gargv);
			while ((filename != NULL) && (opendbfile(filename,1) != 0)) {
				/* skip bad files */
				filename = nextfile(gargc,gargv);
			}
			if (filename == NULL) {
				filecount=0;
				inmain=0;
				p=memory[p];
			}
			else {
				v=lookup("$filecount");
				filecount++;
				stab[v].u.val=filecount;
				v=lookup("$filename");
				stab[v].u.str=filename;
				loadann();
				p++;
			}
			break;
	case CLOSEDOWN: block = memory[p++];
			s1=stab[block].lo;
			s2=stab[block].hi;
			if (s2 >= s1) {
				unblock(s1,s2);
				ptr = stab[block].u.entry;
				stab[block].u.entry = (int *)*ptr;
				Free((char *)ptr);
			}
			closedbfile();
			break;
	case GOTO:	p=memory[p];
			break;
	case EXIT:	return(FAIL);
			break;
	case BREAK:	if ((runlevel==0) && inmain) {
				/* at top most level in main */
				block = maindata;
				s1=stab[block].lo;
				s2=stab[block].hi;
				if (s2 >= s1) {
					unblock(s1,s2);
					ptr = stab[block].u.entry;
					stab[block].u.entry = (int *)*ptr;
					Free((char *)ptr);
				}
				closedbfile();
				longjmp(jumpenv,1);
			}
			else
				return(BRK);
			break;
	case CONDITION:	run(p+3);
			d=pop();
			if (d!=ERRVAL) {	/* abort "if" on eval error */
				if (d==TRUE)
					exitmode=run(memory[p]);
				else
					exitmode=run(memory[p+1]);
				if (exitmode) return(exitmode);
			}
			p=memory[p+2];
			break;
	case WHILECODE:	run(p+2);
			d=pop();
			while (d==TRUE) {
				runlevel++;
				exitmode=run(memory[p]);
				runlevel--;
				if (exitmode==FAIL)
					return(exitmode);
				else if (exitmode==BRK)
					break;
				run(p+2);
				d=pop();
			}
			p=memory[p+1];
			break;
	case FORCODE:	run(p+3);
			d=pop();
			while (d==TRUE) {
				runlevel++;
				exitmode=run(memory[p]);
				runlevel--;
				if (exitmode==FAIL)
					return(exitmode);
				else if (exitmode==BRK)
					break;
				run(memory[p+1]);
				run(p+3);
				d=pop();
			}
			p=memory[p+2];
			break;
	case SWITCH:	run(p+2);
			exitmode=run(memory[p]);
			vpop();
			if (exitmode) return(exitmode);
			p=memory[p+1];
			break;
	case CASE:	d=pop();
			push(d);
			if (d==stab[memory[p]].u.val) {
				exitmode=run(p+2);
				if (exitmode) return(exitmode);
				return(0);
			}
			else
				p=memory[p+1];
			break;
	case STRCASE:	str=spop();
			spush(str);
			if (strcmp(stab[memory[p]].u.str,str)==0) {
				Free(str);
				exitmode=run(p+2);
				if (exitmode) return(exitmode);
				return(0);
			}
			else {
				Free(str);
				p=memory[p+1];
			}
			break;
	case PATCASE:	str=spop();
			spush(str);
			if (regex(stab[memory[p]].u.str,str)!=0) {
				Free(str);
				exitmode=run(p+2);
				if (exitmode) return(exitmode);
				return(0);
			}
			else {
				Free(str);
				p=memory[p+1];
			}
			break;
	case RANGECASE:	d=pop();
			push(d);
			if ((d>=stab[memory[p]].u.val) && (d<stab[memory[p+1]].u.val)) {
				exitmode=run(p+3);
				if (exitmode) return(exitmode);
				return(0);
			}
			else
				p=memory[p+2];
			break;
	case DEFCASE:	exitmode=run(p);
			if (exitmode) return(exitmode);
			return(0);
	case WITH:	str=spop();
			spush(str);
			e = Time();
			if (e != ERRVAL) {
				spush(str);
				f = Length();
				tr++;
				timerange[tr][0] = e;
				timerange[tr][1] = e + f;
				exitmode=run(p+1);
				tr--;
			}
			if (str != ERRSTRING) Free(str);
			if (exitmode) return(exitmode);
			p=memory[p];
			break;
	case WITHN:	d=pop();
			str=spop();
			if (d != ERRVAL) {
				spush(str);
				push(d);
				e = Timen();
				if (e != ERRVAL) {
					spush(str);
					push(d);
					f = Lengthn();
					tr++;
					timerange[tr][0] = e;
					timerange[tr][1] = e + f;
					exitmode=run(p+1);
					tr--;
				}
			}
			if (str != ERRSTRING) Free(str);
			if (exitmode) return(exitmode);
			p=memory[p];
			break;
	case WITHIN:	d=pop();	/* end time */
			e=pop();	/* start time */
			if ((d != ERRVAL) && (e != ERRVAL)) {
				tr++;
				timerange[tr][0] = e;
				timerange[tr][1] = d;
				exitmode=run(p+1);
				tr--;
			}
			if (exitmode) return(exitmode);
			p=memory[p];
			break;
	case CLEAR:	v=memory[p++];
			if (stab[v].type==VAR) {
				dp=(double *)*(stab[v].u.entry);
				*dp=0.0;
			}
			else if (stab[v].type==STATVAR) {
				st=(double *)*(stab[v].u.entry);
				for (i=0;i<6;i++) st[i]=0.0;
			}
			else if (stab[v].type==STRVAR) {
				strp=(char **)*(stab[v].u.entry);
				if ((*strp != NULL) &&
				    (*strp != ERRSTRING)) Free(*strp);
				*strp=NULL;
			}
			break;
	case CLEARARRAY:v=memory[p++];
			if (stab[v].type==ARRAY) {
				dp=(double *)*(stab[v].u.entry);
				for (i=0;i<=stab[v].hi-stab[v].lo;i++)
					dp[i]=0.0;
			}
			else if (stab[v].type==STATARRAY) {
				for (i=0;i<=stab[v].hi-stab[v].lo;i++) {
					st = ((double *)*(stab[v].u.entry)) + i*6;
					for (j=0;j<6;j++) st[j]=0.0;
				}
			}
			else if (stab[v].type==STRARRAY) {
				for (i=0;i<=stab[v].hi-stab[v].lo;i++) {
					strp = ((char **)*(stab[v].u.entry)) + i;
					if ((*strp != NULL) &&
					    (*strp != ERRSTRING)) Free(*strp);
					*strp=NULL;
				}
			}
			break;
	case CLEARELEMENT:
			v=memory[p++];
			d=pop();
			i = arrayindex(stab[v].lo,stab[v].hi,d);
			if (stab[v].type==ARRAY) {
				dp=(double *)*(stab[v].u.entry);
				dp[i]=0.0;
			}
			else if (stab[v].type==STATARRAY) {
				st = ((double *)*(stab[v].u.entry)) + i*6;
				for (j=0;j<6;j++) st[j]=0.0;
			}
			else if (stab[v].type==STRARRAY) {
				strp = ((char **)*(stab[v].u.entry)) + i;
				if ((*strp != NULL) &&
				    (*strp != ERRSTRING)) Free(*strp);
				*strp = NULL;
			}
			break;
	case LOBOUND:	v=memory[p++];
			push((double)stab[v].lo);
			break;
	case HIBOUND:	v=memory[p++];
			push((double)stab[v].hi);
			break;
	case ABORT:	error("abort: %s",spop());
			break;
	case ABORTTEST:	d = pop();
			if (d == ERRVAL) {
				runtimewarning("function returned error","");
				error("abort",NULL);
			}
			break;
	case TRACEON:	runtrace++;
			break;
	case TRACEOFF:	runtrace--;
			break;
	case PLUS:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL)) push(ERRVAL);
				else push(d+e);
			break;
	case MINUS:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL)) push(ERRVAL);
				else push(e-d);
			break;
	case TIMES:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL)) push(ERRVAL);
				else push(d*e);
			break;
	case DIVIDE:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL))
				push(ERRVAL);
			else {
				if (d==0.0) {
					runtimewarning("division by zero","");
					push(ERRVAL);
				}
				else
					push(e/d);
			}
			break;
	case MODULUS:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL))
				push(ERRVAL);
			else {
				if (d==0.0) {
					runtimewarning("division by zero","");
					push(ERRVAL);
				}
				else {
					i = (int)(e/d);
					push(e - i*d);
				}
			}
			break;
	case NEGATE:	d = pop();
			if (d==ERRVAL) push(ERRVAL); else push(-d);
			break;
	case GTCODE:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL)) push(ERRVAL);
			else if (e > d) push(TRUE); else push(FALSE);
			break;
	case GECODE:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL)) push(ERRVAL);
			else if (e >= d) push(TRUE); else push(FALSE);
			break;
	case LTCODE:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL)) push(ERRVAL);
			else if (e < d) push(TRUE); else push(FALSE);
			break;
	case LECODE:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL)) push(ERRVAL);
			else if (e <= d) push(TRUE); else push(FALSE);
			break;
	case EQCODE:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL)) push(ERRVAL);
			else if (e == d) push(TRUE); else push(FALSE);
			break;
	case NECODE:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL)) push(ERRVAL);
			else if (e != d) push(TRUE); else push(FALSE);
			break;
	case ORCODE:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL)) push(ERRVAL);
			else if ((e == TRUE) || (d == TRUE)) push(TRUE); else push(FALSE);
			break;
	case ANDCODE:	d=pop();
			e=pop();
			if ((d==ERRVAL) || (e==ERRVAL)) push(ERRVAL);
			else if ((e == TRUE) && (d == TRUE)) push(TRUE); else push(FALSE);
			break;
	case NOTCODE:	d=pop();
			if (d==ERRVAL) push(ERRVAL);
			else if (d == FALSE) push(TRUE); else push(FALSE);
			break;
	case ERRCHECK:	d=pop();
			if (d==ERRVAL)
				push(FALSE);
			else
				push(TRUE);
			break;
	case STRCAT:	str1=spop();
			str2=spop();
			str=Malloc(strlen(str1)+strlen(str2)+1);
			strcpy(str,str2);
			strcat(str,str1);
			spush(str);
			if (str != ERRSTRING) Free(str);
			if (str1 != ERRSTRING) Free(str1);
			if (str2 != ERRSTRING) Free(str2);
			break;
	case STRSCAT:	str1=spop();
			str2=spop();
			str=Malloc(strlen(str1)+strlen(str2)+2);
			strcpy(str,str2);
			strcat(str,ANNSEPS);
			strcat(str,str1);
			spush(str);
			if (str != ERRSTRING) Free(str);
			if (str1 != ERRSTRING) Free(str1);
			if (str2 != ERRSTRING) Free(str2);
			break;
	case STRCUT:	d = pop();
			str1 = spop();
			if ((str1==ERRSTRING) || (d==ERRVAL) ||
			     (d < -MAXSTRING) || (d > MAXSTRING)) {
				Free(str1);
				spush(ERRSTRING);
				break;
			}
			v=abs((int)d);
			str = Malloc(v+1);
			for (i=0;i<v;i++) str[i]=' ';
			str[v]='\0';
			len=strlen(str1);
			if (d>0) {
				for (i=0;(i<len) && (i<v);i++) str[i]=str1[i];			}
			else {
				for (i=0;(i<len) && (i<v);i++) str[v-i-1]=str1[len-i-1];
			}
			spush(str);
			Free(str);
			Free(str1);
			break;
	case STRCUT2:	e = pop();
			d = pop();
			str1 = spop();
			if ((str1==ERRSTRING) || (d==ERRVAL) ||
			     (e==ERRVAL) || (d<=0) || (e<d) ||
			     (d > MAXSTRING) || ((e-d) > MAXSTRING)) {
				Free(str1);
				spush(ERRSTRING);
				break;
			}
			v=(int)(e-d+1);
			str = Malloc(v+1);
			for (i=0;i<v;i++) str[i]=' ';
			str[v]='\0';
			len=strlen(str1);
			w=(int)(d-1);
			for (i=w;(i<len) && (i<e);i++) str[i-w]=str1[i];
			spush(str);
			Free(str);
			Free(str1);
			break;
	case PRINTVAL:	v = memory[p++];
			if ((stab[v].vtype != OUTFIL) && (stab[v].vtype != OUTPIPE))
				error("cannot write to %s",stab[v].name);
			fil=stab[v].u.fil;
			d=pop();
			if (d==ERRVAL)
				fprintf(fil," %s",ERRSTRING);
			else if ((d!=0) && ((fabs(d) >= 1.0E6) || (fabs(d) <= 1.0E-4)))
				fprintf(fil,"%11g",d);
			else if ((v=(int)d) == d)
				fprintf(fil,"%5d     ",v);
			else
				fprintf(fil,"%10.4f",d);
			if ((fil==stdout) || (fil==stderr)) fflush(fil);
			break;
	case PRINTVAL1:	v = memory[p++];
			if ((stab[v].vtype != OUTFIL) && (stab[v].vtype != OUTPIPE))
				error("cannot write to %s",stab[v].name);
			fil=stab[v].u.fil;
			d=pop();	/* field width */
			e=pop();	/* value */
			if ((d==ERRVAL) || (e==ERRVAL))
				fprintf(fil," %s",ERRSTRING);
			else {
				i=(int)d;
				j=(int)e;
				fprintf(fil,"%*d",i,j);
			}
			if ((fil==stdout) || (fil==stderr)) fflush(fil);
			break;
	case PRINTVAL2:	v = memory[p++];
			if ((stab[v].vtype != OUTFIL) && (stab[v].vtype != OUTPIPE))
				error("cannot write to %s",stab[v].name);
			fil=stab[v].u.fil;
			d=pop();	/* second field width */
			e=pop();	/* first field width */
			f=pop();	/* value */
			if ((d==ERRVAL) || (e==ERRVAL) || (f==ERRVAL))
				fprintf(fil," %s",ERRSTRING);
			else {
				i=(int)d;
				j=(int)e;
				fprintf(fil,"%*.*f",j,i,f);
			}
			if ((fil==stdout) || (fil==stderr)) fflush(fil);
			break;
	case PRINTSTR:	v = memory[p++];
			if ((stab[v].vtype != OUTFIL) && (stab[v].vtype != OUTPIPE))
				error("cannot write to %s",stab[v].name);
			fil=stab[v].u.fil;
			str=spop();
			fprintf(fil,"%s",str);
			if ((fil==stdout) || (fil==stderr)) fflush(fil);
			if ((str != NULL) &&
			    (str != ERRSTRING)) Free(str);
			break;
	case PRINTSTR1:	v = memory[p++];
			if ((stab[v].vtype != OUTFIL) && (stab[v].vtype != OUTPIPE))
				error("cannot write to %s",stab[v].name);
			fil=stab[v].u.fil;
			d=pop();
			str=spop();
			if (d==ERRVAL)
				fprintf(fil,ERRSTRING);
			else {
				i=(int)d;
				fprintf(fil,"%*s",i,str);
			}
			if ((fil==stdout) || (fil==stderr)) fflush(fil);
			if ((str != NULL) &&
			    (str != ERRSTRING)) Free(str);
			break;
	case INPUTLINE:	v=memory[p++];
			if ((stab[v].vtype != INPFIL) && (stab[v].vtype != INPIPE))
				error("cannot read from file %s",stab[v].name);
			inputpos=inputstring;
#ifdef DOS
			if (graphicson) {
				int	c;
				while ((c=getch())!='\r') {
					*inputpos++ = c;
					if (inputpos==inputstring+MAXSTRING-1) break;
				}
				*inputpos = '\0';
				inputend = inputstring + strlen(inputstring) -1;
				inputpos=inputstring;
				while ((isspace(*inputpos))
					&& (inputpos < inputend)) inputpos++;
			}
			else
#endif
			if (fgets(inputstring,MAXSTRING,stab[v].u.fil) == NULL)
				inputpos = NULL;
			else {
				inputend = inputstring + strlen(inputstring) -1;
				if (*inputend == '\n') *inputend = '\0';
				while ((isspace(*inputpos))
					&& (inputpos < inputend)) inputpos++;
			}
			break;
	case READVAL:	if (inputpos == NULL) {
				*(double *)(vpop())=ERRVAL;
				break;
			}
			if (sscanf(inputpos,"%lf",&d) != 1)
				*(double *)(vpop())=ERRVAL;
			else
				*(double *)(vpop())=d;
			while (!(isspace(*inputpos))
				&& (inputpos < inputend)) inputpos++;
			while ((isspace(*inputpos))
				&& (inputpos < inputend)) inputpos++;
			break;
	case READSTR:	if (inputpos == NULL) {
				strp=(char **)(vpop());
				if ((*strp != NULL) &&
				    (*strp != ERRSTRING)) Free(*strp);
				*strp=ERRSTRING;
			}
			else {
				str = inputpos;
				while (!(isspace(*inputpos))
					&& (inputpos < inputend)) inputpos++;
				*inputpos++='\0';
				strp=(char **)(vpop());
				if ((*strp != NULL) &&
				    (*strp != ERRSTRING)) Free(*strp);
				*strp=Malloc(strlen(str)+1);
				strcpy(*strp,str);
				while ((isspace(*inputpos))
					&& (inputpos < inputend)) inputpos++;
			}
			break;
	case READLINE:	if (inputpos == NULL) {
				strp=(char **)(vpop());
				if ((*strp != NULL) &&
				    (*strp != ERRSTRING)) Free(*strp);
				*strp=ERRSTRING;
			}
			else {
				strp=(char **)(vpop());
				if ((*strp != NULL) &&
				    (*strp != ERRSTRING)) Free(*strp);
				*strp=Malloc(strlen(inputstring)+1);
				strcpy(*strp,inputstring);
			}
			break;
	default:	runtimewarning("unknown op-code!","");
			fprintf(stderr,"code=%d\n",memory[--p]);
			exit(1);
		}
	}
	if (runtrace>=2) fprintf(stderr,"\n**** Machine exited\n");
	return(0);
}

/* unblock variables - freeing memory */
void unblock(s1,s2)
int	s1,s2;
{
	int	i,j;
	for (i=s1;i<=s2;i++) {
		if (stab[i].stat==ADRPAR)
			/* nothing doing */;
		else switch (stab[i].type) {
		case VAR:
		case STATVAR:
		case ARRAY:
		case STATARRAY:
				Free((char *)(*(double **)(stab[i].u.entry)));
				break;
		case STRVAR:
				Free(**(char ***)(stab[i].u.entry));
				Free(*(char **)(stab[i].u.entry));
				break;
		case STRARRAY:
				for (j=0;j<=stab[i].hi-stab[i].lo;j++)
					Free((*(char ***)(stab[i].u.entry))[j]);
				Free(*(char **)(stab[i].u.entry));
				break;
		}
	}
}

/* run time error report */
void runtimewarning(s,t)
char	*s,*t;
{
	int	loc;
	loc=0;
	while (currp >= sline[loc+1].pc) loc++;
	fprintf(stderr,"%s: %s",progname,s);
	if (strlen(t) > 0) fprintf(stderr," %s",t);
	fprintf(stderr," near line %d\n",sline[loc].lineno);
}

static int	argind=1;

#include <sys/types.h>
#include <sys/stat.h>

#ifdef _MSC_VER
/* dirent -- replacement dirent routines for WIN32 */

#include <windows.h>

typedef struct dirent_handle {
	HANDLE	dp;
	int	first;
	WIN32_FIND_DATA	wfd;
} DIR;

struct dirent {
  char d_namlen;
  char d_name[256];
};

DIR * opendir(const char *dirname)
{
	char	dirpattern[256];
	DIR	*hdir;

	hdir = (DIR *)malloc(sizeof(DIR));
	sprintf(dirpattern,"%s\\*.*",dirname);
	hdir->dp = FindFirstFile(dirpattern,&(hdir->wfd));
	if (hdir->dp!=INVALID_HANDLE_VALUE) {
		hdir->first=1;
		return(hdir);
	}
	else {
		free(hdir);
		return(NULL);
	}
}

struct dirent *	readdir(DIR *dirp)
{
	static struct dirent dd;

	if (dirp->first==1) {
		strncpy(dd.d_name,dirp->wfd.cFileName,256);
		dd.d_namlen = strlen(dd.d_name);
		dirp->first=0;
		return(&dd);
	}
	else if (FindNextFile(dirp->dp,&(dirp->wfd))) {
		strncpy(dd.d_name,dirp->wfd.cFileName,256);
		dd.d_namlen = strlen(dd.d_name);
		return(&dd);
	}
	else
		return(NULL);
}


int closedir(DIR *dirp)
{
	FindClose(dirp->dp);
	free(dirp);
	return(0);
}

int isdir(char *fname)
{
	DWORD flags = GetFileAttributes(fname);
	if (flags == 0xFFFFFFFF)
		return(0);
	else
		return(flags & FILE_ATTRIBUTE_DIRECTORY);
}


#if 0
char *nextfile(int argc,char *argv[])
{
	if (argind < argc) {
		argind++;
		return(argv[argind-1]);
	}
	else
		return(NULL);
}
#endif

#else
#include <dirent.h>
#endif

static struct {
	DIR		*dirp;		/* directory pointer */
	struct dirent	*dp;		/* filename pointer */
	char		*dname;		/* directory name */
} dirstack[100];
static int	dirstackptr = -1;

/* list contents of a directory */
char	*nextfile(argc,argv)
int		argc;
char		*argv[];
{
#ifdef _MSC_VER
	struct _stat	status;
#else
	struct stat	status;
#endif
	static char	filename[256];
	char		*Malloc();

	while ((dirstackptr >= 0) || (argind < argc)) {
		if (dirstackptr < 0) {
			/* not already started a directory */
			strcpy(filename,argv[argind]);
			argind++;
#ifdef _MSC_VER
			if (!isdir(filename)) {
#else
			stat(filename,&status);
			if ((status.st_mode & S_IFDIR) == 0) {
#endif
				/* not a directory */
				return(filename);
			}
			else {
				/* is a directory */
				dirstackptr++;
				dirstack[dirstackptr].dirp = opendir(filename);
				dirstack[dirstackptr].dname = Malloc(strlen(filename)+1);
				strcpy(dirstack[dirstackptr].dname,filename);
				readdir(dirstack[dirstackptr].dirp);	/* skip "." */
				dirstack[dirstackptr].dp = readdir(dirstack[dirstackptr].dirp); /* skip ".." */
			}
		}
		/* already started on a directory */
		dirstack[dirstackptr].dp = readdir(dirstack[dirstackptr].dirp);
		if (dirstack[dirstackptr].dp==NULL) {
			/* no more files in this directory */
			closedir(dirstack[dirstackptr].dirp);
			Free(dirstack[dirstackptr].dname);
			dirstackptr--;
		}
		else {
			sprintf(filename,"%s/%s",dirstack[dirstackptr].dname,dirstack[dirstackptr].dp->d_name);
#ifdef _MSC_VER
			if (!isdir(filename)) {
#else
			stat(filename,&status);
			if ((status.st_mode & S_IFDIR) == 0) {
#endif
				/* not a directory */
				return(filename);
			}
			else {
				/* is a directory */
				dirstackptr++;
				dirstack[dirstackptr].dirp = opendir(filename);
				dirstack[dirstackptr].dname = Malloc(strlen(filename)+1);
				strcpy(dirstack[dirstackptr].dname,filename);
				readdir(dirstack[dirstackptr].dirp);	/* skip "." */
				dirstack[dirstackptr].dp = readdir(dirstack[dirstackptr].dirp); /* skip ".." */
			}
		}
	}
	return(NULL);
}
