/* init.c - table initialisation for SML */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "sml.h"
#ifdef DOS
#include "y_tab.h"
#else
#include "y.tab.h"
#endif

/* reserved words */
static struct {
	char	*name;
	int	tok;
} reserved[] = {			/* KEEP SORTED */
{	"abort",	ABORTTOK },
{	"break",	BREAKTOK },
{	"case",		CASETOK },
{	"clear",	CLEARTOK },
{	"default",	DEFAULTTOK },
{	"else",		ELSETOK },
{	"file",		FILETOK },
{	"for",		FORTOK },
{	"function",	FUNCTOK },
{	"hibound",	HIBOUNDTOK },
{	"if",		IFTOK },
{	"init",		INIT },
{	"input",	INPUTTOK },
{	"inputline",	INPUTLINETOK },
{	"library",	LIBTOK },
{	"lobound",	LOBOUNDTOK },
{	"main",		MAIN },
{	"pattern",	PATTERNTOK },
{	"print",	PRINTTOK },
{	"range",	RANGETOK },
{	"return",	RETURNTOK },
{	"stat",		STATTOK },
{	"string",	STRINGTOK },
{	"summary",	SUMMARY },
{	"switch",	SWITCHTOK },
{	"traceoff",	TRACEOFFTOK },
{	"traceon",	TRACEONTOK },
{	"var",		VARTOK },
{	"while",	WHILETOK },
{	"with",		WITHTOK },
{	"within",	WITHINTOK },
{	"withn",	WITHNTOK },
{	0,		0 }		/* KEEP SORTED */
};

/* stat variable field names */
static struct {
	char	*name;
	int	ftyp;
} fnames[] = {
{	"count",	COUNT },
{	"sum",		SUM },
{	"sumsq",	SUMSQ },
{	"mean",		MEAN },
{	"variance",	VARIANCE },
{	"stddev",	STDDEV },
{	0,		0 }
};

/* predeclared files */
static struct {
	char	*name;
	FILE	*filptr;
	int	filtype;
} files[] = {
{	"stdout", 	NULL,	OUTFIL },
{	"stderr", 	NULL,	OUTFIL },
{	"stdin",	NULL,	INPFIL },
{	0,	0 }
};

/* predeclared variables */
static struct {
	char	*name;
	int	tok;
} vars[] = {
{	"$filecount",	NUMBER },
{	"$filename",	STRING },
{	"$date",	STRING },
{	0,	0 }
};

/* predeclared contants */
static struct {
	char	*name;
	double	val;
} consts[] = {
{	"SP",	SP_TYPE },
{	"LX",	LX_TYPE },
{	"TX",	TX_TYPE },
{	"FX",	FX_TYPE },
{	"AN",	AN_TYPE },
{	"LP",	LP_TYPE },
{	"SY",	SY_TYPE },
{	"CO",	CO_TYPE },
{	"DI",	DI_TYPE },
{	"FM",	FM_TYPE },
{	"PC",	PC_TYPE },
{	"TR",	TR_TYPE },
{	"VU",	VU_TYPE },
{	"ERROR",0.0 },
{	0,	0 }
};

/* dbfile main header variables */
static struct {
	char	*name;
	char	*field;
} headvars[] = {
{	"$head_username",	head.username },
{	"$head_source",		head.source },
{	"$head_dbase",		head.dbase },
{	"$head_speaker",	head.speaker },
{	"$head_session",	head.session },
{	"$head_sessdate",	head.sessdate },
{	"$head_token",		head.token },
{	"$head_rep",		head.rep },
{	0,	0 }
};

/* arithmetic functions */
static struct {
	char	*name;
	double	(*func)();
	int	numpar;
	int	type[6];
} builtins[] = {
/* annotation */
{	"length",	Length,		1,{ STRVAR,0,0,0,0,0 } },
{	"lengthn",	Lengthn,	2,{ STRVAR,VAR,0,0,0,0 } },
{	"numberof",	Numberof,	1,{ STRVAR,0,0,0,0,0 } },
{	"time", 	Time,		1,{ STRVAR,0,0,0,0,0 } },
{	"timen",	Timen,		2,{ STRVAR,VAR,0,0,0,0 } },
/* arithmetic */
{	"abs",		Abs,		1,{ VAR,0,0,0,0,0 } },
{	"exp",		Exp,		1,{ VAR,0,0,0,0,0 } },
{	"log",		Log,		1,{ VAR,0,0,0,0,0 } },
{	"sqrt",		Sqrt,		1,{ VAR,0,0,0,0,0 } },
{	"trunc",	Int,		1,{ VAR,0,0,0,0,0 } },
/* files */
{	"close",	Close,		1,{ FILEVAR,0,0,0,0,0 } },
{ 	"openappend",	Openappend,	2,{ FILEVAR,STRVAR,0,0,0,0 } },
{	"openin",	Openin,		2,{ FILEVAR,STRVAR,0,0,0,0 } },
{	"openout",	Openout,	2,{ FILEVAR,STRVAR,0,0,0,0 } },
/* items */
{	"next",		Next,		2,{ VAR,VAR,0,0,0,0 } },
{	"select",	Select,		1,{ VAR,0,0,0,0,0 } },
{	"selectnext",	Selectnext,	1,{ VAR,0,0,0,0,0 } },
{	"selectmatch",	Selectmatch,	1,{ STRVAR,0,0,0,0,0 } },
/* measurement */
{	"energy",	Energy,		2,{ VAR,VAR,0,0,0,0 } },
{	"f1",		F1,		1,{ VAR,0,0,0,0,0 } },
{	"a1",		A1,		1,{ VAR,0,0,0,0,0 } },
{	"b1",		B1,		1,{ VAR,0,0,0,0,0 } },
{	"f2",		F2,		1,{ VAR,0,0,0,0,0 } },
{	"a2",		A2,		1,{ VAR,0,0,0,0,0 } },
{	"b2",		B2,		1,{ VAR,0,0,0,0,0 } },
{	"f3",		F3,		1,{ VAR,0,0,0,0,0 } },
{	"a3",		A3,		1,{ VAR,0,0,0,0,0 } },
{	"b3",		B3,		1,{ VAR,0,0,0,0,0 } },
{	"f4",		F4,		1,{ VAR,0,0,0,0,0 } },
{	"a4",		A4,		1,{ VAR,0,0,0,0,0 } },
{	"b4",		B4,		1,{ VAR,0,0,0,0,0 } },
{	"fm",		Fm,		2,{ VAR,VAR,0,0,0,0 } },
{	"fx",		Fx,		1,{ VAR,0,0,0,0,0 } },
{	"lp",		Lp,		2,{ STRVAR,VAR,0,0,0,0 } },
{	"sy",		Sy,		2,{ VAR,VAR,0,0,0,0 } },
{	"tr",		Tr,		1,{ VAR,0,0,0,0,0 } },
{	"tx",		Tx,		1,{ VAR,0,0,0,0,0 } },
{	"sp",		Sp,		1,{ VAR,0,0,0,0,0 } },
{	"vu",		Vu,		2,{ VAR,VAR,0,0,0,0 } },
{	"pc",		Pc,		2,{ VAR,VAR,0,0,0,0 } },
{	"co",		Co,		2,{ VAR,VAR,0,0,0,0 } },
/* string */
{	"ascii",	Ascii,		1,{ STRVAR,0,0,0,0,0 } },
{	"compare",	Compare,	2,{ STRVAR,STRVAR,0,0,0,0 } },
{	"entry",	Entry,		2,{ STRVAR,STRARRAY,0,0,0,0 } },
{	"index",	Index,		2,{ STRVAR,STRVAR,0,0,0,0 } },
{	"strlen",	Strlen,		1,{ STRVAR,0,0,0,0,0 } },
{	"val",		Val,		1,{ STRVAR,0,0,0,0,0 } },
/* utility */
{	"stopwatch",	Stopwatch,	1,{ VAR,0,0,0,0,0 } },
{	"system",	System,		1,{ STRVAR,0,0,0,0,0 } },
/* plotting */
{	"plotparam",	Plotparam,	1,{ STRVAR,0,0,0,0,0 } },
{	"plotxdata",	Plotxdata,	2,{ ARRAY,VAR,0,0,0,0 } },
{	"plot",		Plot,		4,{ FILEVAR,VAR,ARRAY,VAR,0,0 } },
{	"plotaxes",	Plotaxes,	6,{ FILEVAR,VAR,VAR,VAR,VAR,VAR } },
{	"plottitle",	Plottitle,	2,{ FILEVAR,STRVAR,0,0,0,0 } },
{	"plotclear",	Plotclear,	1,{ FILEVAR,0,0,0,0,0 } },
{	"plotwait",	Plotwait,	1,{ FILEVAR,0,0,0,0,0 } },
{	"plotend",	Plotend,	1,{ FILEVAR,0,0,0,0,0 } },
{	0,		0,		0,{ 0,0,0,0,0,0 } }
};

/* string functions */
static struct {
	char	*name;
	char	*(*func)();
	int	numpar;
	int	type[3];
} sbuiltins[] = {
/* annotation */
{	"match",	Match,		1,{ STRVAR,0,0 } },
{	"matchn",	Matchn,		2,{ STRVAR,VAR,0 } },
/* item */
{	"history",	History,	1,{ VAR,0,0 } },
/* string */
{	"istr",		Istr,		1,{ VAR,0,0 } },
{	"str",		Str,		3,{ VAR,VAR,VAR } },
{	"char",		Char,		1,{ VAR,0,0 } },
{	0,		0,		0,{ 0,0,0 } }
};

/* global variables */
extern Symbol stab[];
extern int	scnt;
extern int	symreserve;

/* install reserved words and predeclared functions */
int	startnew=1;
void init()
{
	int	i,j,s,t;

	/* install reserved words */
	for (i=0;reserved[i].name;i++) {
		s = install(reserved[i].name,reserved[i].tok,0.0);
		stab[s].stat=RESERVED;
	}
	symreserve=scnt;
	/* install stat variable field names */
	for (i=0;fnames[i].name;i++) {
		s = install(fnames[i].name,FIELD,0.0);
		stab[s].vtype=fnames[i].ftyp;
		stab[s].stat=BLTIN;
	}
	/* install predeclared files */
	for (i=0;files[i].name;i++) {
		s = install(files[i].name,FILEVAR,0.0);
		stab[s].stat = BLTIN;
		if (strcmp(files[i].name,"stdin")==0)
			stab[s].u.fil = stdin;
		else if (strcmp(files[i].name,"stderr")==0)
			stab[s].u.fil = stderr;
		else
			stab[s].u.fil = stdout;
		stab[s].vtype = files[i].filtype;
	}
	/* install predeclared variables */
	for (i=0;vars[i].name;i++) {
		s = install(vars[i].name,vars[i].tok,0.0);
		stab[s].stat = BLTIN;
	}
	/* install predeclared constants */
	for (i=0;consts[i].name;i++) {
		s = install(consts[i].name,NUMBER,consts[i].val);
		stab[s].stat = BLTIN;
	}
	/* install main header variables */
	for (i=0;headvars[i].name;i++) {
		s = install(headvars[i].name,STRING,0.0);
		stab[s].stat = BLTIN;
		stab[s].u.str = headvars[i].field;
	}
	/* install builtin functions */
	for (i=0;builtins[i].name;i++) {
		s = install(builtins[i].name,FUNCTION,0.0);
		stab[s].u.func = builtins[i].func;
		stab[s].stat=BLTIN;
		t=s+1;
		for (j=0;j<builtins[i].numpar;j++) {
			t = install("",builtins[i].type[j],0.0);
			if ((builtins[i].type[j]==FILEVAR) ||
			    (builtins[i].type[j]==ARRAY) ||
			    (builtins[i].type[j]==STRARRAY))
				stab[t].stat=ADRPAR;
			else
				stab[t].stat=VALPAR;
			stab[t].vtype= CLEARED;
		}
		stab[s].lo=s+1;
		stab[s].hi=t;
	}
	for (i=0;sbuiltins[i].name;i++) {
		s = install(sbuiltins[i].name,SFUNCTION,0.0);
		stab[s].u.sfunc = sbuiltins[i].func;
		stab[s].stat=BLTIN;
		t=s+1;
		for (j=0;j<sbuiltins[i].numpar;j++) {
			t = install("",sbuiltins[i].type[j],0.0);
			if ((sbuiltins[i].type[j]==FILEVAR) ||
			    (sbuiltins[i].type[j]==ARRAY) ||
			    (sbuiltins[i].type[j]==STRARRAY))
				stab[t].stat=ADRPAR;
			else
				stab[t].stat=VALPAR;
			stab[t].vtype= CLEARED;
		}
		stab[s].lo=s+1;
		stab[s].hi=t;
	}
	if (startnew>0) startnew=scnt+1;
}
