/* sfsfunc -- SFS functions for SPC */

/* Mark Huckvale - University College London - July 1991 */

/* waveform definitions in 'wave.inc' */
/* SFS definitions in 'sfs.inc' */
/* Speech definitions in 'speech.inc' */
/* Annotation definitions in 'annot.inc' */
/* filter definitions in 'filter.inc' */
/* track definitions in 'track.inc' */

/* external definitions */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <malloc.h>
#include <time.h>

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include "spc.h"
#include "spcc.h"
#include "sfs.h"
#include "filter.h"

/* create a new waveform */
waveform newwaveform_(size,rate)
int	size;
double	rate;
{
	waveform	w;

	w.smr = rate;
	w.len = size;
	w.buf.ar = wavmalloc(w.len);
	memset(w.buf.ar,0,w.len<<1);
	return(w);
}

/* replay a waveform */
void replaywaveform_(w)
waveform *w;
{
/*
	fprintf(stderr,"replaying buffer of length %ld at rate %g\n",
		w->len,w->smr);
*/
	if (dac_open(NULL) > 0) {
		dac_playback(w->buf.ar,w->len,w->smr,16,1,1);
		dac_close(0);
	}
}

/* record a waveform */
waveform recordwaveform_(size,rate)
int	size;
double	rate;
{
	extern double adc_selected_rate;
	waveform	w;

	w.smr = rate;
	w.len = 0;
	w.buf.ar = NULL;
	if (adc_open(NULL) >= 0) {
		w.buf.ar = wavmalloc(size);
		w.len = adc_record(w.buf.ar,size,rate,1,0);
		if (w.len != size)
			w.buf.ar = realloc(w.buf.ar,w.len*2);
		w.smr = adc_selected_rate;
		adc_close(0);
	}
	return(w);
}

/* cut out a part of a waveform */
waveform cutwaveform_(win,start,stop)
waveform *win;
int	start;
int	stop;
{
	waveform wout;

	if (start < 0) start=0;
	if (stop >= win->len) stop = win->len-1;
	if (stop < start) stop=start;
	
	wout.len = (stop - start + 1);
	wout.smr = win->smr;
	wout.buf.ar = wavmalloc(wout.len);
	wavcopy(wout.buf.ar,win->buf.ar + start,wout.len);
	
	return(wout);
}

/* filter a waveform */
waveform filterwaveform_(win,ftype,flo,fhi,forder)
waveform *win;
int	ftype;
double	flo;
double	fhi;
int	forder;
{
	FILTER		*flt;
	waveform	wout;

	if ((flt=filter_design(ftype,forder,flo,fhi,win->smr))==NULL)
		error("failed to design filter");

	wout = newwaveform_(win->len,win->smr);

	filter_signal(flt,win->buf.ar,wout.buf.ar,win->len);

	filter_free(flt);

	return(wout);
}

/* pause function */
void pause_(dur)
double	dur;
{
	time_t	etime;
	etime = time((time_t *)0) + (time_t)dur;
	while (time((time_t *)0) < etime) {
#ifdef DOS
		/* wait */;
#else
		sleep(1);
#endif
	}
}

/* SFS static data */
static int sfs_fid;
static struct main_header sfs_head;
static struct item_header sfs_item[MAXDATATYPE+1];
static int sfs_curitem;

/* Check and Open an SFS file */
unsigned char sfsfile_(fname)
string fname;
{
	if (sfs_fid) sfsclose(sfs_fid);

	memset((char *)&sfs_head,0,sizeof(struct main_header));
	memset((char *)sfs_item,0,(MAXDATATYPE+1)*sizeof(struct item_header));
	
	sfs_fid = sfsopen(fname.buf.ar,"r",&sfs_head);
	_sfree(fname);

	if (sfs_fid <= 0) {
		sfs_fid=0;
		return(FALSE);
	}
	else
		return(TRUE);
}

/* Locate a particular item */
unsigned char sfsitem_(iname)
string iname;
{
	int	code;
	int	it;
	char	*ty;
	
	if (!sfs_fid) return(FALSE);

	if (itspec(iname.buf.ar,&it,&ty) != 0) {
		_sfree(iname);
		return(FALSE);
	}

	code = sfsitem(sfs_fid,it,ty,&sfs_item[it]);
	_sfree(iname);
	if (code == 0) {
		sfs_curitem=0;
		return(FALSE);
	}
	else {
		sfs_curitem=it;
		return(TRUE);
	}
}

/* abort, closing sfs files */
void abort_(messg)
string	messg;
{
	error(messg.buf.ar);
}

/* return size of current speech waveform */
int speechsize_()
{
	if (!sfs_fid) return(0L);

	if (sfs_item[SP_TYPE].datatype != SP_TYPE) {
		if (sfsitem_((_spushstr("sp"),_spop()))==FALSE)
			return(0L);
	}

	return(sfs_item[SP_TYPE].numframes);
}

/* return rate of current speech waveform */
double speechrate_()
{
	if (!sfs_fid) return(0L);

	if (sfs_item[SP_TYPE].datatype != SP_TYPE) {
		if (sfsitem_((_spushstr("sp"),_spop()))==FALSE)
			return(0L);
	}

	return(1.0/sfs_item[SP_TYPE].frameduration);
}

/* load current speech waveform */
waveform speechload_()
{
	waveform w;

	if (!sfs_fid) return(waveformNULL);

	if (sfs_item[SP_TYPE].datatype != SP_TYPE) {
		if (sfsitem_((_spushstr("sp"),_spop()))==FALSE)
			return(waveformNULL);
	}
	else if (sfs_curitem!=SP_TYPE) {
		sfsitem(sfs_fid,SP_TYPE,sfs_item[SP_TYPE].history,&sfs_item[SP_TYPE]);
		sfs_curitem=SP_TYPE;
	}

	w.smr = speechrate_();
	w.len = speechsize_();
	w.buf.ar = wavmalloc(w.len);
	if (sfsread(sfs_fid,0,w.len,(char *)w.buf.ar) != w.len)
		error("read error on input SFS file");

	return(w);
}

/* load part of current speech waveform */
waveform speechpart_(start,stop)
int	start;
int	stop;
{
	waveform w;
	int	len;

	if (!sfs_fid) return(waveformNULL);

	if (sfs_item[SP_TYPE].datatype != SP_TYPE) {
		if (sfsitem_((_spushstr("sp"),_spop()))==FALSE)
			return(waveformNULL);
	}
	else if (sfs_curitem!=SP_TYPE) {
		sfsitem(sfs_fid,SP_TYPE,sfs_item[SP_TYPE].history,&sfs_item[SP_TYPE]);
		sfs_curitem=SP_TYPE;
	}

	len = speechsize_();
	if (start < 0) start=0;
	if (stop >= len) stop = len-1;
	if (stop < start) stop=start;
	
	w.len = (stop - start + 1);
	w.smr = (1.0/sfs_item[SP_TYPE].frameduration);
	w.buf.ar = wavmalloc(w.len);
	if (sfsread(sfs_fid,start,w.len,(char *)w.buf.ar) != w.len)
		error("read error on input SFS file");

	return(w);
}

/* save a waveform to file */
void speechsave_(fname,hist,wave)
string fname;
string hist;
waveform *wave;
{
	struct item_header item;
	extern char *progname;

	sfsheader(&item,SP_TYPE,0,2,1,1.0/wave->smr,0.0,1,0,0);
	sprintf(item.history,"%s(%s)",progname,hist.buf.ar);
	putitem(fname.buf.ar,&item,wave->len,wave->buf.ar);

	_sfree(hist);
	_sfree(fname);
}

/* load annotations from current SFS file */
unsigned char loadannotations_(baseptr,samprate)
struct annotation_rec **baseptr;
double	samprate;
{
	int	i;
	struct an_rec *an;
	struct annotation_rec *root,*ptr;
	
	if (!sfs_fid)
		return(FALSE);

	if (sfs_item[AN_TYPE].datatype != AN_TYPE) {
		if (sfsitem_((_spushstr("an"),_spop()))==FALSE)
			return(FALSE);
	}
	else if (sfs_curitem!=AN_TYPE) {
		sfsitem(sfs_fid,AN_TYPE,sfs_item[AN_TYPE].history,&sfs_item[AN_TYPE]);
		sfs_curitem=AN_TYPE;
	}

	root=NULL;
	an = (struct an_rec *)sfsbuffer(&sfs_item[AN_TYPE],1);

	for (i=0;sfsread(sfs_fid,i,1,(char *)an)==1;i++) {
		if ((ptr = (struct annotation_rec *)calloc(1,sizeof(struct annotation_rec)))==NULL)
			error("out of memory");
		_sassign(&(ptr->annot),(_spushstr(an->label),_spop()));
		ptr->sample = (int)(0.5+(sfs_item[AN_TYPE].offset + an->posn*sfs_item[AN_TYPE].frameduration)*samprate);
		ptr->next = NULL;
		if (!root)
			*baseptr = ptr;
		else
			root->next = ptr;

		root = ptr;
	}

	free(an);

	return(TRUE);	
}

/* return size of current track waveform */
int tracksize_()
{
	if (!sfs_fid) return(0L);

	if (sfs_item[TR_TYPE].datatype != TR_TYPE) {
		if (sfsitem_((_spushstr("tr"),_spop()))==FALSE)
			return(0L);
	}

	return(sfs_item[TR_TYPE].numframes);
}

/* return rate of current track waveform */
double trackrate_()
{
	if (!sfs_fid) return(0L);

	if (sfs_item[TR_TYPE].datatype != TR_TYPE) {
		if (sfsitem_((_spushstr("tr"),_spop()))==FALSE)
			return(0L);
	}

	return(1.0/sfs_item[TR_TYPE].frameduration);
}

tr_item_rec trackNULL;

/* load current track */
tr_item_rec * trackload_()
{
	tr_item_rec *tr;
	float	*fp;
	double	*dp;
	int	i;

	if (!sfs_fid) return(&trackNULL);

	if (sfs_item[TR_TYPE].datatype != TR_TYPE) {
		if (sfsitem_((_spushstr("tr"),_spop()))==FALSE)
			return(&trackNULL);
	}
	else if (sfs_curitem!=TR_TYPE) {
		sfsitem(sfs_fid,TR_TYPE,sfs_item[TR_TYPE].history,&sfs_item[TR_TYPE]);
		sfs_curitem=TR_TYPE;
	}

	if ((tr = (tr_item_rec *)calloc(1,sizeof(tr_item_rec)))==NULL)
		error("out of memory");
	tr->smr = trackrate_();
	tr->len = tracksize_();
	if ((tr->buf = (double *)calloc(tr->len,sizeof(double)))==NULL)
		error("no memory for track item");
	if (sfsread(sfs_fid,0,tr->len,(char *)(tr->buf)) != tr->len)
		error("read error on input SFS file");

	fp = (float *)(tr->buf) + tr->len;
	dp = tr->buf + tr->len;
	for (i=0;i<tr->len;i++)
		*--dp = (double) *--fp;

	return(tr);
}

/* load part of current track waveform */
tr_item_rec * trackpart_(start,stop)
int	start;
int	stop;
{
	tr_item_rec	*tr;
	int		len;
	float		*fp;
	double		*dp;
	int		i;

	if (!sfs_fid) return(&trackNULL);

	if (sfs_item[TR_TYPE].datatype != TR_TYPE) {
		if (sfsitem_((_spushstr("tr"),_spop()))==FALSE)
			return(&trackNULL);
	}
	else if (sfs_curitem!=TR_TYPE) {
		sfsitem(sfs_fid,TR_TYPE,sfs_item[TR_TYPE].history,&sfs_item[TR_TYPE]);
		sfs_curitem=TR_TYPE;
	}

	len = tracksize_();
	if (start < 0) start=0;
	if (stop >= len) stop = len-1;
	if (stop < start) stop=start;
	
	if ((tr = (tr_item_rec *)calloc(1,sizeof(tr_item_rec)))==NULL)
		error("out of memory");
	tr->len = (stop - start + 1);
	tr->smr = (1.0/sfs_item[TR_TYPE].frameduration);
	if ((tr->buf = (double *) calloc(tr->len,sizeof(double)))==NULL)
		error("no memory for track item");
	if (sfsread(sfs_fid,start,tr->len,(char *)tr->buf) != tr->len)
		error("read error on input SFS file");
	fp = (float *)tr->buf + tr->len;
	dp = tr->buf + tr->len;
	for (i=0;i<tr->len;i++) *--dp = *--fp;

	return(tr);
}

/* save a waveform to file */
void tracksave_(fname,hist,tr)
string fname;
string hist;
tr_item_rec *tr;
{
	struct item_header item;
	extern char *progname;
	float		*fp;
	double		*dp;
	int		i;
	
	sfsheader(&item,TR_TYPE,1,4,1,1.0/tr->smr,0.0,1,0,0);
	sprintf(item.history,"%s(%s)",progname,hist.buf.ar);
	if ((fp=(float *)calloc(tr->len,sizeof(float)))==NULL)
		error("out of memory");
	dp = tr->buf;
	for (i=0;i<tr->len;i++) fp[i] = *dp++;
	putitem(fname.buf.ar,&item,tr->len,fp);
	free(fp);

	_sfree(hist);
	_sfree(fname);
}

void trackfree_(tr)
tr_item_rec *tr;
{
	if (tr) {
		if (tr->buf) free(tr->buf);
		free(tr);
	}
}

