/* ps -- DIG routines for PostScript compatible graphics */

/* Mike Edgington, Colin Ayer and Mark Huckvale */

/* version 2.0 - May 1993
	- adapted for Unix DIG
*/

/* version 2.1 - July 1994
	- add support for A5 picture outputs
	- with subtype >= 50
*/
/* version 2.2 - June 1995
	- add support for EPS output
	- with subtype >= 200
	- subtypes >= 100 go to file
*/
/* version 2.3 - April 1996
	- add subtypes 0/1=stdprn, 2/3=prn, 4/5=lpt1, 6/7=lpt2 for DOS
*/
/* version 2.4 - May 1996
	- add support for GSIZE and GFILE variables
*/

#include "SFSCONFG.h"
#ifdef	EVp

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include <string.h>
#include "dig.h"
#include "digdata.h"

#define DIG_PS_VERSION "2.4"

/* Page Layout stuff */

/* Dots per inch of target device */
#define  DPI            ((int32)300)

/* Dimensions of paper, and of border around drawing area, 
   in points (72 points == 1 inch) assuming portrait orientation. */
#define	A4PAPERWIDTH     594
#define	A4PAPERHEIGHT    840
#define	A5PAPERWIDTH     420
#define	A5PAPERHEIGHT    594
static int PAPERWIDTH=A4PAPERWIDTH;
static int PAPERHEIGHT=A4PAPERHEIGHT;
#define	BORDERWIDTH    22
#define	BORDERHEIGHT   32

/* Specifies the minimum size of text in points */
#define	MINTEXTSIZE	9

/* offset to baseline in pixels */
#define	TEXTOFFSET	10

#define PSWHITE		1.0   		/* grey scale value for white */
#define PSBLACK		0.0		/* grey scale value for black */

/* PS system modes */
#define PS_INITIALISED	1
#define PS_PORTRAIT	2
#define PS_A5		4
#define PS_TOFILE	8
#define PS_EPS		16

/* local storage of PS parameters */
struct {
	int	mode;		/* PS system modes */
	int	subtype;	/* subtype as supplied from digmap */
	int	myfile;		/* PS system opened output file */
} digDEVp;

/* forward declarations */
static void ps_colour();

/* get graphics parameters */
int ps_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	char	*p,cbuf[32];
	int charheight, height, width;

	digDEVp.subtype = subtype;
	if (subtype >= 200) {
		digDEVp.mode |= PS_EPS;
		subtype -= 100;
	}
	if (subtype >= 100) {
		digDEVp.mode |= PS_TOFILE;
		subtype -= 100;
	}
	if (subtype >= 50)
		digDEVp.mode |= PS_A5;
	if (subtype & 1)
		digDEVp.mode |= PS_PORTRAIT;

	/* set up image dimensions */
	if ((p=getenv("GSIZE"))!=NULL) {
		strncpy(cbuf,p,31);
		width = atoi(strtok(cbuf,"x"));
		height = atoi(strtok(NULL,""));
		if ((width < 10) || (height < 10))
			error("dig: GSIZE (WIDTHxHEIGHT) too small");
		PAPERWIDTH = 2*BORDERWIDTH + (72*width)/DPI;
		PAPERHEIGHT = 2*BORDERHEIGHT + (72*height)/DPI;
	}
	else {
		if (digDEVp.mode & PS_A5) {
		 	PAPERHEIGHT = A5PAPERHEIGHT;
			PAPERWIDTH = A5PAPERWIDTH;
		}
		width = (DPI * (int32)(PAPERWIDTH - 2 * BORDERWIDTH)) / 72;
		height = (DPI * (int32)(PAPERHEIGHT - 2 * BORDERHEIGHT)) / 72;
	}
	charheight = (DPI * (int32)(MINTEXTSIZE)) / 72;
	
	if (digDEVp.mode & PS_PORTRAIT) {
		/* portrait */
		digdev->nhoriz = width;
		digdev->nvert  = height;
	}
	else {
		/* landscape */
		digdev->nhoriz = height;
		digdev->nvert  = width;
	}

	/* return other image characteristics */
	digdev->aspect = 1.0;
 	digdev->chwidth = (int)(0.6 * charheight);
	digdev->chheight = charheight;
	digdev->greylevels = 16;
	digdev->greypixwidth = 4;
	digdev->greypixheight = 4;
	return(0);
}

/* open PostScript file */
int ps_outfile(fp)
FILE	**fp;
{
	char	*p,ofilename[256];

	if ((p=getenv("GFILE"))!=NULL)
		strncpy(ofilename,p,255);
	else if (digDEVp.mode & PS_EPS)
		strcpy(ofilename,"dig.eps");
	else
		strcpy(ofilename,"dig.ps");

	if (digDEVp.mode & PS_TOFILE) {
		if ((*fp = fopen(ofilename,"w"))==NULL)
			error("could not open '%s'",ofilename);
		digDEVp.myfile=1;
	}
#ifdef DOS
	else {
		int ptr = (digDEVp.subtype % 50)/2;
#ifdef _MSC_VER
		FILE *stdprn = fopen("PRN","wb");
#endif
		digDEVp.myfile=0;
		
		if (ptr==0) {
			*fp = stdprn;
		}
		else if (ptr==1) {
			if ((*fp = fopen("PRN","wb"))==NULL)
				*fp = stdprn;
			else
				digDEVp.myfile=1;
		}
		else if (ptr==2) {
			if ((*fp = fopen("LPT1","wb"))==NULL)
				*fp = stdprn;
			else
				digDEVp.myfile=1;
		}
		else if (ptr==3) {
			if ((*fp = fopen("LPT2","wb"))==NULL)
				*fp = stdprn;
			else
				digDEVp.myfile=1;
		}
#ifdef _MSC_VER
		if (digDEVp.myfile==1) fclose(stdprn);
#endif
	}
#else
	else if (digDEVp.myfile==0) {
		char	pipename[SFSMAXFILENAME];
		sprintf(pipename,"%s/bin/postscript %d",sfsbase(),digDEVp.subtype % 50);
		if ((*fp = popen(pipename,"w"))==NULL)
			error("could not open pipe to '%s'\n",pipename);
		digDEVp.myfile=1;
	}
#endif
	return(0);
}

/* open output */
int ps_open()
{
	time_t	tim;

	if (digDEVp.mode & PS_INITIALISED) return(0);

	/* The PostScript file is assumed to consist of just the one page */
    
	/* PostScript Header, Structuring Convention version 1.0 */
	if (digDEVp.mode & PS_EPS)
		fprintf(digdata.outfile, "%%!PS-Adobe-3.0 EPSF-3.0\n");
	else
		fprintf(digdata.outfile, "%%!PS-Adobe-1.0\n");
	fprintf(digdata.outfile, "%%%%DocumentFonts: Courier\n");
	fprintf(digdata.outfile, "%%%%Title: DIG (SFS) PostScript output\n");
	fprintf(digdata.outfile, "%%%%Creator: DIG PostScript driver V%s, (Mike Edgington/Colin Ayer/Mark Huckvale).\n",DIG_PS_VERSION);
	tim = time((time_t *)0);
	fprintf(digdata.outfile, "%%%%CreationDate: %s",ctime(&tim));
	fprintf(digdata.outfile, "%%%%Pages: 1\n");
	if (digDEVp.mode & PS_A5) {
		if (digDEVp.mode & PS_PORTRAIT)
			fprintf(digdata.outfile, "%%%%BoundingBox: %d %d %d %d\n",
				BORDERWIDTH - 1, 
				BORDERHEIGHT - 1,
				PAPERWIDTH - BORDERWIDTH + 1,
				PAPERHEIGHT - BORDERHEIGHT + 1);
		else
			fprintf(digdata.outfile, "%%%%BoundingBox: %d %d %d %d\n",
				BORDERWIDTH-1,
				BORDERHEIGHT+PAPERWIDTH-2*BORDERWIDTH-1, 
				PAPERHEIGHT-2*BORDERWIDTH+1,
				2*PAPERWIDTH-BORDERWIDTH-BORDERHEIGHT+1);
	}
	else {
		if (digDEVp.mode & PS_PORTRAIT) 
			fprintf(digdata.outfile, "%%%%BoundingBox: %d %d %d %d\n",
				BORDERWIDTH - 1, 
				BORDERHEIGHT - 1,
				PAPERWIDTH - BORDERWIDTH + 1,
				PAPERHEIGHT - BORDERHEIGHT + 1);
		else
			fprintf(digdata.outfile, "%%%%BoundingBox: %d %d %d %d\n",
				BORDERWIDTH - 1, 
				BORDERHEIGHT - 1,
				PAPERWIDTH - BORDERWIDTH - 1,
				PAPERHEIGHT - BORDERHEIGHT - 1);
	}
	fprintf(digdata.outfile, "%%%%EndComments\n\n");

	/* PostScript Prolog */
	fprintf(digdata.outfile, "%% Save initial system state\n");
	fprintf(digdata.outfile, "save\n\n");

	/* Define Page Setup Procedure */
	fprintf(digdata.outfile, "%% Define Page Setup Procedure\n");
	fprintf(digdata.outfile, "/PageSetup {\n");
	fprintf(digdata.outfile, "\t%% Move in by 'border'\n");
	fprintf(digdata.outfile, "\t%d %d translate\n", BORDERWIDTH, BORDERHEIGHT);

	if (digDEVp.mode & PS_A5) {
		if ((digDEVp.mode & PS_PORTRAIT)==0) {
			fprintf(digdata.outfile, "\t%% Change to landscape page A5 layout\n");
			fprintf(digdata.outfile, "\t0 %d translate\n", (PAPERWIDTH - 2 * BORDERWIDTH));
		}
		else {
			fprintf(digdata.outfile, "\t%% Change to portrait page A5 layout\n");
			if ((digDEVp.mode & PS_EPS)==0) {
				fprintf(digdata.outfile, "\t%d 0 translate\n", (PAPERHEIGHT - 2 * BORDERHEIGHT));
				fprintf(digdata.outfile, "\t90 rotate\n");
			}
		}
	}
	else if ((digDEVp.mode & PS_PORTRAIT)==0) {
		fprintf(digdata.outfile, "\t%% Change to landscape page A4 layout\n");
		fprintf(digdata.outfile, "\t%d 0 translate\n", (PAPERWIDTH - 2 * BORDERWIDTH));
		fprintf(digdata.outfile, "\t90 rotate\n");
	}
	fprintf(digdata.outfile, "\t%% Set to %d dpi\n", DPI);
	fprintf(digdata.outfile, "\t72 %d div dup scale\n", DPI);
	fprintf(digdata.outfile, "} def\n\n");
    
	/* Font setup procedure */
	fprintf(digdata.outfile, "%% Define Font Setup Procedure\n");
	fprintf(digdata.outfile, "/FontSetup {\n");
	fprintf(digdata.outfile, "\t/Courier findfont\n");
	fprintf(digdata.outfile, "\t%d scalefont setfont\n", (MINTEXTSIZE * DPI)/72);
	fprintf(digdata.outfile, "} def\n\n");

	/* Various Macros */
	fprintf(digdata.outfile, "%% Define various Macros\n");
	fprintf(digdata.outfile, "/nmt { newpath moveto } def\n");
	fprintf(digdata.outfile, "/lt { lineto } def\n");
	fprintf(digdata.outfile, "/sg { setgray } def\n");

	fprintf(digdata.outfile, "%%%%EndProlog\n");

	/* PostScript Page Script */
	fprintf(digdata.outfile, "%%%%Page: 1 1\n\n");
	fprintf(digdata.outfile, "gsave\n");
	fprintf(digdata.outfile, "PageSetup\n");
	fprintf(digdata.outfile, "FontSetup\n");

	digDEVp.mode |= PS_INITIALISED;
	return(0);

}

void ps_inter()
{
}

/* close PostScript file */
void ps_close()
{
	if (digDEVp.mode & PS_INITIALISED) {

		/* End of page */
		fprintf(digdata.outfile, "grestore\n");
		if (!(digDEVp.mode & PS_EPS))
			fprintf(digdata.outfile, "showpage\n\n");
		
		/* PostScript Trailer Comments */
		fprintf(digdata.outfile, "%%%%Trailer\n");
		fprintf(digdata.outfile, "%% Restore initial system state\n");
		fprintf(digdata.outfile, "restore\n");
		fflush(digdata.outfile);
		
		/* shut down output stream */
		if (digDEVp.myfile && (digDEVp.mode & PS_TOFILE))
			fclose(digdata.outfile);
#ifndef DOS
		else if (digDEVp.myfile)
			pclose(digdata.outfile);
#endif
	}
	digDEVp.mode = 0;
	digDEVp.myfile=0;
}

/* clear screen */
int ps_clear()
{
	/* dummy comment */
	fprintf(digdata.outfile, "%% PC_CLEAR\n");
	return(0);
}

/* set up colour from bundle */
static void ps_colour(bundle)
int	bundle;
{
	static int oldbundle = -1;

	if (bundle==oldbundle) return;
	oldbundle = bundle;

	if (bundle<20)
              fprintf(digdata.outfile,"2 setlinewidth\n");
        else
              fprintf(digdata.outfile,"1 setlinewidth\n");
        if (bundle<35)
              fprintf(digdata.outfile,"[] 0 setdash\n");
        else if (bundle<50)
              fprintf(digdata.outfile,"[2] 0 setdash\n");
        else if (bundle<65)
              fprintf(digdata.outfile,"[3 5] 6 setdash\n");
        else if (bundle<80)
              fprintf(digdata.outfile,"[4] 0 setdash\n");
        else if (bundle<95)
              fprintf(digdata.outfile,"[5 7] 10 setdash\n");
        else 
              fprintf(digdata.outfile,"[3 2 1 2] 0 setdash\n");
}

/* plot poly-line */
void ps_polyline(bundle,buff,len)
int bundle;
short *buff;
int	len;
{
	int     x,y;
	
	/* move to first point */
	x = *buff++;
	y = *buff++;
	len-=(int32)2;
	fprintf(digdata.outfile, "%d %d nmt\n", x, y);

	/* plot lines */
	while (len > (int32)0) {
		x = *buff++;
		y = *buff++;
		len -= (int32)2;
		fprintf(digdata.outfile, "%d %d lt\n", x, y);
	}

	/* set up plot colour */
	ps_colour(bundle);

	fprintf(digdata.outfile, "stroke\n");
}


/* draw text */
void ps_text(bundle,x,y,str)
int bundle;
int x;
int y;
char *str;
{
	/* set up plot colour */
	ps_colour(bundle);

	/* move to start */
	fprintf(digdata.outfile, "%d %d moveto\n(", x, y + TEXTOFFSET);

	/* output string */
	while (*str) {
		/* convert to PostScript form, i.e. 
		   '(' -> '\('
		   ')' -> '\)'
		   '\' -> '\\'
		 */
		switch(*str) {
		case '(':
		case ')':
		case '\\':
			/* add backslash */
			putc('\\',digdata.outfile);
		default:
	    		putc(*str,digdata.outfile);
			break;
		}
		str++;
	}

	/* draw string */
	fprintf(digdata.outfile, ") show\n");

}

/* draw a filled rectangle */
void ps_fillrect(bundle,x1,y1,x2,y2)
int bundle;
int x1;
int y1;
int x2;
int y2;
{
	/* save current gray */
	fprintf(digdata.outfile,"currentgray /gray_level exch def\n");

	/* set fill gray level */
	if (bundle) 
		fprintf(digdata.outfile,"%f setgray\n",PSBLACK);
	else
		fprintf(digdata.outfile,"%f setgray\n",PSWHITE);

	/* draw and fill box */
	fprintf(digdata.outfile, "%d %d nmt ", x1, y1);
	fprintf(digdata.outfile, "%d %d lt ", x2, y1);
	fprintf(digdata.outfile, "%d %d lt ", x2, y2);
	fprintf(digdata.outfile, "%d %d lt closepath\n", x1, y2);
	fprintf(digdata.outfile, "fill\n");
	
	/* restore old gray level */
	fprintf(digdata.outfile,"gray_level setgray\n");

}

/* draw a circle */
void ps_circle(bundle,xc,yc,r)
int bundle;
int xc;
int yc;
int r;
{
	/* set up plot colour */
	ps_colour(bundle);

	/* draw circle */
	fprintf(digdata.outfile, "%% PC_CIRCLE\n");
	fprintf(digdata.outfile, "newpath %d %d %d 0 360 arc stroke\n", xc, yc, r);
}

/* draw a filled circle */
void ps_fcircle(bundle,xc,yc,r)
int bundle;
int xc;
int yc;
int r;
{
	/* save current gray */
	fprintf(digdata.outfile,"currentgray /gray_level exch def\n");

	/* set fill gray level */
	if (bundle) 
		fprintf(digdata.outfile,"%f setgray\n",PSBLACK);
	else
		fprintf(digdata.outfile,"%f setgray\n",PSWHITE);

	/* draw circle */
	fprintf(digdata.outfile, "%% PC_FILLCIRCLE\n");
	fprintf(digdata.outfile, "newpath %d %d %d 0 360 arc fill\n", xc, yc, r);

	/* restore old gray level */
	fprintf(digdata.outfile,"gray_level setgray\n");
}

/* grey-level display */
void ps_gscale(x,y,height,buff)
int x;
int y;
int height;
unsigned char *buff;
{
	int             i;
	unsigned char  *p;

	fprintf(digdata.outfile, "%% PC_GREYSCALE\n");
	p=buff;
	
	fprintf(digdata.outfile, "gsave\n");
	fprintf(digdata.outfile, "%d %d translate\n", x, y);
	fprintf(digdata.outfile, "%d %d scale\n", 4, 4*height);
 	fprintf(digdata.outfile, "%d %d %d ", 1, height, 4);
	fprintf(digdata.outfile, "[1 0 0 %d 0 0]\n", height);
	fprintf(digdata.outfile, "{<");
	p += height;
	for (i = 0; i < height; i++) {
		p--;
		fprintf(digdata.outfile, "%1x%1x", 15 - *p, 15 - *p);
	}
	fprintf(digdata.outfile, ">}\n");
	fprintf(digdata.outfile, "image\n");
	fprintf(digdata.outfile, "grestore\n");
}

/* wait for key routine - NULL don't wait */
void ps_pause()
{
}	
#endif
