/* vgagrx -- VGA graphics using LIBGRX */

/* Mark Huckvale - University College London */

/* version 1.0 - April 1993 */

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include "SFSCONFG.h"
#include "dig.h"
#include "digdata.h"

#ifdef EVv		/* <- letter chosen for device type */

#include <dos.h>
#include "grx.h"
#include "mousex.h"

/* global data */
struct {
	int subtype;		/* graphics subtype */
	int numx,numy;		/* graphics size */
	int numcol;		/* number of colours */
	int mode;		/* 0=?, 1=initialised */
	GrContext * ctx;	/* graphics context */
	int rop;		/* raster operation */
	int col;		/* current colour */
	int ylim;		/* y draw limit */
	int mouseon;		/* mouse initialised */
	int xmouse,ymouse,smouse; /* mouse status */
} digDEVv;

/* VGA colours */
#define NUM_COLOUR	32
struct rgb_rec {
	int	entry;
	int	r,g,b;
} rgbtab[NUM_COLOUR]={
	{ 0,250,250,250},	/* 0 = background */
	{ 0,200, 50, 50},	/* 1 */
	{ 0,100, 75,100},	/* 2 */
	{ 0,100, 75, 50},	/* 3 */
	{ 0,150,200,  0},	/* 4 */
	{ 0, 25, 25,100},	/* 5 = was red - mapped to dark grey/blue */
	{ 0, 50,200,100},	/* 6 */
	{ 0, 50,100,200},	/* 7 */
	{ 0,150,150,  0},	/* 8 */
	{ 0, 50, 50, 50},	/* 9 = was white - mapped to dark grey */
	{ 0,200,100,  0},	/* 10 */
	{ 0,100,  0,100},	/* 11 */
	{ 0,  0,100,100},	/* 12 */
	{ 0,200,100,100},	/* 13 */
	{ 0, 50,100,200},	/* 14 */
	{ 0,  0,  0,  0},	/* 15 = foreground */
	{ 0,250,250,250},	/* 16 */
	{ 0,248,248,248},	/* 17 */
	{ 0,244,244,244},	/* 18 */
	{ 0,238,238,238},	/* 19 */
	{ 0,230,230,230},	/* 20 */
	{ 0,220,220,220},	/* 21 */
 	{ 0,208,208,208},	/* 22 */
	{ 0,194,194,194},	/* 23 */
	{ 0,178,178,178},	/* 24 */
	{ 0,160,160,160},	/* 25 */
	{ 0,140,140,140},	/* 26 */
	{ 0,118,118,118},	/* 27 */
	{ 0, 94, 94, 94},	/* 28 */
	{ 0, 68, 68, 68},	/* 29 */
	{ 0, 40, 40, 40},	/* 30 */
	{ 0, 10, 10, 10},	/* 31 */
};

static int PrintContext(GrContext *,int);
int vga_open(void);
int vga_close(void);
int vga_clear(void);

/* device characteristics */
int vga_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	digDEVv.subtype=subtype;
	if (!(digDEVv.mode)) {		/* may need to open */
		vga_open();
	}
	digdev->nhoriz = GrSizeX();
	digdev->nvert  = GrSizeY();
	digDEVv.ylim = digdev->nvert - 1;
	digdev->aspect = 1.0;
	digdev->chwidth = 8;
	digdev->chheight = 15;
	if (digDEVv.subtype >= 100) {
		digdev->greylevels = 10;
		digdev->greypixwidth = 3;
		digdev->greypixheight = 3;
	}
	else {
		digdev->greylevels = 16;
		digdev->greypixwidth = 1;
		digdev->greypixheight = 1;
	}
	return(0);
}

/* open graphics */
int vga_open()
{
	int	i;
	if (!(digDEVv.mode)) {
		/* change to graphics mode */
		if (digDEVv.subtype==0) {
			/* biggest possible */
			GrSetMode(GR_biggest_graphics);
		}
		else if (digDEVv.subtype==1) {
			/* VGA 640x480 16 colours */
			GrSetMode(GR_width_height_color_graphics,640,480,16);
		}
		else if (digDEVv.subtype==2) {
			/* VGA 640x480 256 colours */
			GrSetMode(GR_width_height_color_graphics,640,480,256);
		}
		else if (digDEVv.subtype==3) {
			/* SVGA 800x600 16 colours */
			GrSetMode(GR_width_height_color_graphics,800,600,16);
		}
		else if (digDEVv.subtype==4) {
			/* SVGA 800x600 256 colours */
			GrSetMode(GR_width_height_color_graphics,800,600,256);
		}
		else if (digDEVv.subtype==5) {
			/* XVGA 1024x768 16 colours */
			GrSetMode(GR_width_height_color_graphics,1024,768,16);
		}
		else if (digDEVv.subtype==6) {
			/* XVGA 1024x768 256 colours */
			GrSetMode(GR_width_height_color_graphics,1024,768,256);
		}
		else if (digDEVv.subtype==100) {
			/* Epson LQ printers */
/*			GrSetMode(GR_default_text); */
			digDEVv.ctx = GrCreateContext(1024,768,NULL,NULL);
			GrSetContext(digDEVv.ctx);
		}

		if (digDEVv.subtype < 100)
			digDEVv.numcol = GrNumColors();
		else
			digDEVv.numcol = 2;
		digDEVv.numx = GrSizeX();
		digDEVv.numy = GrSizeY();

		/* initialise colours */
		for (i=0;i<NUM_COLOUR;i++)
			rgbtab[i].entry = GrAllocColor(
						rgbtab[i].r,
						rgbtab[i].g,
						rgbtab[i].b);
		(digDEVv.mode)++;
		digDEVv.col=GrBlack();
		digDEVv.rop=0;
		vga_clear();
	}
	return(0);
}

/* interrupt graphics */
int vga_inter()
{
	/* deal with interrupted drawing */
	vga_close();
	fprintf(stderr,"Interrupt\n");
	fflush(stderr);
	return(0);
}

/* close graphics - return to text mode */
int vga_close()
{
	if (digDEVv.mode) {
		/* remove mouse */
		if (digDEVv.mouseon) MouseUnInit();
		digDEVv.mouseon=0;
		/* print on printer */
		if (digDEVv.subtype >= 100) {
			PrintContext(digDEVv.ctx,digDEVv.subtype);
			GrDestroyContext(digDEVv.ctx);
		}
		/* return to text mode */
		if (digDEVv.subtype < 100) {
			GrResetColors();
			GrSetMode(GR_default_text);
			fprintf(stderr,"Mode was %dx%d in %d colours\n",
				digDEVv.numx,digDEVv.numy,digDEVv.numcol);
		}
	}
	digDEVv.mode = 0;
	return(0);
}

/* clear screen */
int vga_clear()
{
	/* clear screen to background */
	if (digDEVv.subtype < 100)
		GrClearContext(GrWhite());
	else
		GrClearContext(0);
	return(0);
}

/* set up colour from bundle */
static int vga_colour(bun)
int	bun;
{
	/* bundles are 0=background, 1-100 various colours/line styles, >100 greyscale */
	if (digDEVv.subtype >= 100)
		digDEVv.col = (bun) ? 1 : 0;
	else if (bun==0)
		digDEVv.col=GrWhite();
	else if (bun < 5)
		digDEVv.col=rgbtab[bun].entry;
	else if (bun >= 100)
		digDEVv.col=rgbtab[bun-84].entry;
	else {
		bun = bun % 15;
		if (bun==0) 
			digDEVv.col=rgbtab[15].entry;
		else
			digDEVv.col=rgbtab[bun].entry;
	}
	if (digDEVv.rop)
		digDEVv.col |= GrXOR;
	return(0);
}

/* draw a poly-line */
int vga_polyline(bundle,buff,len)
int	bundle;
short	*buff;
int	len;
{
	int	x,y;
	int	lx,ly;

	/* sort out colours */
	vga_colour(bundle);

	/* position to first point */
	lx = *buff++;
	ly = digDEVv.ylim - *buff++;
	len -= 2;

	/* plot rest of points */
	while (len > 0) {
		x = *buff++;
		y = digDEVv.ylim - *buff++;
		GrLine(lx,ly,x,y,digDEVv.col);
		/* plot line to x,y */
		lx = x;
		ly = y;
		len -= 2;
	}
	return(0);
}

/* draw 'hardware' text */
int vga_text(bundle,x,y,str)
int	bundle;
int	x,y;
char	*str;
{

	/* sort out colours */
	vga_colour(bundle);

	/* write string */
	if (digDEVv.subtype >= 100)
		GrTextXY(x,digDEVv.ylim - y - 14,str,digDEVv.col,GrOR | 0);
	else
		GrTextXY(x,digDEVv.ylim - y - 14,str,digDEVv.col,GrOR | GrBlack());
	return(0);
}

/* draw a filled rectangle */
int vga_fillrect(bundle,x1,y1,x2,y2)
int	bundle;
int	x1,y1,x2,y2;
{
	/* sort out colours */
	vga_colour(bundle);

	/* draw filled box */
	GrFilledBox(x1,digDEVv.ylim-y1,x2,digDEVv.ylim-y2,digDEVv.col);
	return(0);
}

/* change line/text raster operation */
int vga_rasterop(mode)
int	mode;
{
	int	ret;

	/* set new raster op 0=write, 1=xor */
	ret=digDEVv.rop;
	digDEVv.rop=mode;
	return(ret);
}

/* draw a circle */
int vga_circle(bundle,xc,yc,r)
int	bundle;
int	xc,yc,r;
{
	/* sort out colours */
	vga_colour(bundle);

	/* draw a circle */
	GrCircle(xc,digDEVv.ylim-yc,r,digDEVv.col);
	return(0);
}

/* draw a filled circle */
int vga_fcircle(bundle,xc,yc,r)
int	bundle;
int	xc,yc,r;
{
	/* sort out colours */
	vga_colour(bundle);

	/* draw a filled circle */
	GrFilledCircle(xc,digDEVv.ylim-yc,r,digDEVv.col);
	return(0);
	
}

/* initialise pointing device */
int vga_pointer(flag,xm,ym)
int	flag;
int	xm,ym;
{
	if (flag) {
		if (digDEVv.mouseon) return(0);
		/* turn mouse on */
		if (MouseDetect()) {
			MouseEventMode(0);
			MouseInit();
			MouseWarp(xm,digDEVv.ylim - ym);
			digDEVv.mouseon=1;
		}
		else
			digDEVv.mouseon=0;	
	}
	else {
		/* turn mouse off */
		if (digDEVv.mouseon) MouseUnInit();
		digDEVv.mouseon=0;
	}
	return(0);
}

/* get keyboard & mouse input */
int vga_mouse(but,xm,ym,ich)
int	*but;
int	*xm;
int	*ym;
int	*ich;
{
	MouseEvent	ev;

	/* load last known co-ordinates */
	*xm = digDEVv.xmouse;
	*ym = digDEVv.ymouse;
	*but = digDEVv.smouse;
	*ich = 0;

	/* get next event */
	MouseGetEvent(M_KEYPRESS|M_BUTTON_UP|M_BUTTON_DOWN,&ev);

	if (ev.flags & M_KEYPRESS) {
		*ich = ev.key;
		*xm = ev.x;
		*ym = digDEVv.ylim - ev.y;
		return(DIG_KEY);
	}
	else if (ev.flags & M_BUTTON_CHANGE) {
		*xm = ev.x;
		*ym = digDEVv.ylim - ev.y;
		*but =  ((ev.buttons & M_LEFT) ? 4 : 0) |
			((ev.buttons & M_MIDDLE) ? 2 : 0) |
			((ev.buttons & M_RIGHT) ? 1 : 0);
		digDEVv.xmouse = *xm;
		digDEVv.ymouse = *ym;
		digDEVv.smouse = *but;
		return(DIG_MOUSE);
	}

	return(0);
}

/* display text on prompt line */
int vga_prompt(str)
char	*str;
{
	GrFilledBox(0,0,GrMaxX(),15,GrWhite());
	GrTextXY(0,0,str,GrBlack(),GrWhite());
	return(0);
}

/* dot dither */
static unsigned char DITHER[3][3]={{1,4,7},{6,9,2},{3,8,5}};

/* grey-level display */
int vga_gscale(x,y,height,buff)
int	x,y;
int	height;
unsigned char	*buff;
{
	int		i,j,k,oy;
	unsigned char 	*p,lastc;

	if (digDEVv.subtype < 100) {
		/* replace greyscales by colours */
		p=buff;
		oy = digDEVv.ylim-(y+height-1);
		y = oy+1;
		lastc=*p++;

		for (i = 1; i < height; i++, p++, y++) {
			if (*p != lastc) {
				GrVLine(x,oy,y,rgbtab[lastc+16].entry);
				oy = y;
				lastc = *p;
			}
		}
		GrVLine(x,oy,y-1,rgbtab[lastc+16].entry);
	}
	else {
		/* 3x3 dot dither */
		oy = digDEVv.ylim-(y+3*height-1);
		for (j=0;j<3;j++) {
			p=buff;
			for (i=0;i<height;i++) {
				lastc=*p++;
				for (k=0;k<3;k++)
					if (lastc >= DITHER[k][j])
						GrPlot(x+j,oy+3*i+k,1);
			}
		}
	}
	return(0);
}

/* pause for user input */
int vga_pause()
{
	int	c=0;
	if (digDEVv.subtype < 100)
		c=getkey();
	return(c);
}

static int	BITS[8]={0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01};

#define pputc(c) bdos(5,c,0);

static int PrintContext(ctx,ptype)
GrContext	*ctx;
int		ptype;
{
	int	i,j,k,l,m,x,y,val;

	/* dump context */
	for (j=0;j<ctx->gc_ymax;j+=24) {

		/* move to beginning of line and indent */
		pputc('\r');
		pputc('\033');
		pputc('$');
		pputc(72);
		pputc(0);

		for (i=0;i<=ctx->gc_xmax;i+=256) {

			/* set up graphics download */
			pputc('\033');
			pputc('*');
			pputc(39);
			pputc(0);
			pputc(1);

			/* dump a line */
			for (x=i,m=0;m<256;m++) {
				for (k=0;k<3;k++) {
					val = 0;
					for (y=j+k*8,l=0;l<8;l++)
						val |= (GrPixel(x+m,y+l)==0) ? 0 : BITS[l];
					pputc(val);
				}
			}
		}

		/* do a line feed of 24/180" */
		pputc('\033');
		pputc('J');
		pputc(24);
	}

	pputc('\r');
	pputc('\n');
	return(0);
}

#endif
