/* digitfft -- speech spectrogram display */

/* Mark Huckvale - University College London */

/* version 1.0 - August 1993 */

/* version 1.1 - November 1996
	- add proper 'merge' operation for severe compression
	- compile with PROPER_MERGE
*/

#undef PROPER_MERGE

#include "SFSCONFG.h"
#include <stdio.h>
#include <math.h>
#include <fcntl.h>
#include "sfs.h"
#include "dig.h"
#include "digdata.h"
#include "digtable.h"
#include "digitem.h"
#include "fft.h"

#define PREEMP		0.95
#define ANALWIDTH	300.0
#define ANALWIDTH_NARROW	45.0
#define MAXFFT		2048
#define MAXPIXELS	1024
#define DBRANGE		50.0
#define DBHEADROOM	6.0

static float	fftbuf[MAXFFT+2];
static char	pixels[MAXPIXELS];

/* grid reference co-ordinates (pixels) */
extern int digit_ygrid[];
extern int digit_nygrid;
extern int digit_xgrid[];
extern int digit_nxgrid;
static char	grid1[MAXPIXELS];
static char	grid2[MAXPIXELS];

#ifdef __STDC__
int    digitemFFT(int32  bundles,float  xl,float  yb,float xr,float  yt,struct  item_header *item,short  *buff,double  start,double  stop,int  flags)
#else
int digitemFFT(bundles,xl,yb,xr,yt,item,buff,start,stop,flags)
int32		bundles;
float		xl,yb,xr,yt;
struct item_header *item;
short		*buff;
double		start,stop;
int		flags;
#endif
{
	register int	i,j,k,l,hi,lo;
	int		x,y;
	int		numf,bstart;
	int		fftsize,wisize;
	int		pheight,pwidth,cnt;
	char		messg[128];
	float		omega;
	float		*xp;
	short		*sp;
	float		xstep,xend,xpos,xsum,xmax;
	int		ixstart,ixend;
	float		dbnorm,edb;
	int		p;
#ifdef PROPER_MERGE
	int		nmerge,n;
#endif	
	/* sort out bundles */
	digitbundle(bundles);

	/* init box */
	digitbox(xl,yb,xr,yt,start,stop,flags);
	xl += digitab.ixoff/digdata.xscale;
	yt = yval(digitab.iyt);

	/* write y-axis */
	digitab.lo = 0;
	digitab.hi = 0.5/item->frameduration;
	digitemfreq(bundles,digitab.iyb,digitab.iyt,digitab.lo,digitab.hi,1);

	/* check can do grey scale */
	if (!digtable[digdata.select].greyscale) return(0);

	/* check timing of data set */
	bstart = (start-item->offset-0.8/ANALWIDTH)/item->frameduration;
	numf = (stop - start)/item->frameduration;
	if ((bstart+numf) > item->numframes) {
		/* data does not reach to right end */
		xr = xl + (item->numframes-bstart)*item->frameduration*digitab.scalex;
		numf = item->numframes - bstart;
	}
	if (bstart < 0) {
		/* data does not reach to left end */
		xl += (-bstart*item->frameduration)*digitab.scalex;
		numf += bstart;
		bstart=0;
	}

	/* get display size */
	ixstart = xpix(xl) + 1;
	ixend = xpix(xr);
	pheight = (digitab.iyt - digitab.iyb - 1)/digdata.digdev.greypixheight;
	if (pheight > MAXPIXELS) pheight = MAXPIXELS;
	pwidth = (ixend - ixstart - 1)/digdata.digdev.greypixwidth;

	/* get window size - speech chunk */
	if (flags & DIGITEMFFTNARROW)
		wisize = (1.6/ANALWIDTH_NARROW)/item->frameduration;
	else
		wisize = (1.6/ANALWIDTH)/item->frameduration;
	fftsize = 2 * pheight;
	if (fftsize < 128)
		fftsize = 128;
	else if (fftsize < 256)
		fftsize = 256;
	else if (fftsize < 512)
		fftsize = 512;
	else if (fftsize < 1024)
		fftsize = 1024;
	else
		fftsize = 2048;
	if (wisize > fftsize) wisize = fftsize;

        omega = 8.0 * atan(1.0) / (wisize - 1); 

	/* get normalisation estimate from speech power */
	sp = buff + bstart;
	xmax = 0.0;
	for (i=0;i<(numf-wisize);i+=wisize) {
		/* find biggest power in a window */
		xsum = 1.0;
		for (j=0;j<wisize;j++,sp++) 
			xsum += (float)*sp * (float)*sp;
		if (xsum > xmax) xmax = xsum;
	}
	dbnorm = (10.0*log10(xmax) + DBHEADROOM) - DBRANGE;

#ifdef PROPER_MERGE
	/* check for need of merge operation */
	nmerge = ((numf/pwidth)+(wisize/2)-1)/(wisize/2);
	if (nmerge==0) nmerge=1;
#endif
	/* write axis label */
	y = (digitab.iyt + digitab.iyb)/2;
	digtextp(digitab.lbun,digitab.ixl+3,y,"Hz");
	sprintf(messg,"(%ddB)",(int)dbnorm);
	y -= digdata.chheight;
	digtextp(digitab.lbun,digitab.ixl+3,y,messg);

	/* calculate grid pixels */
	if (flags & DIGITEMGRID) {
		for (i=0;i<MAXPIXELS;i++) {
			grid1[i]=0;
			grid2[i]=0;
		}
		for (i=0;i<digit_nygrid;i++) {
			j = pheight-(digit_ygrid[i]-digitab.iyb)/digdata.digdev.greypixheight-1;
			if ((j > 0) && (j < pheight-1)) {
				grid1[j]=1;
				grid2[j-1]=1;
				grid2[j]=1;
				grid2[j+1]=1;
			}
		}
	}

#ifdef PROPER_MERGE
	fprintf(stderr,"pheight=%d pwidth=%d fftsize=%d wisize=%d dbnorm=%g nmerge=%d numf=%d\n",
		pheight,pwidth,fftsize,wisize,dbnorm,nmerge,numf);
#endif
/*
	sprintf(messg,"pheight=%d pwidth=%d fftsize=%d wisize=%d dbnorm=%g",
		pheight,pwidth,fftsize,wisize,dbnorm);
	digprompt(messg);
	digwait();
*/
	hi = -1000; lo = 1000;	
	/* display, if anything left */
	if ((numf>1) && (bstart<item->numframes)) for (j=0;j<pwidth;j++) {

	    memset(pixels,0,pheight);
#ifdef PROPER_MERGE
	    for (n=0;n<nmerge;n++) {
#endif	    	
		/* move a window into FFT buffer */
		xp = fftbuf;
#ifdef PROPER_MERGE
		sp = buff + bstart + (j*numf)/pwidth + 1 - wisize/2 + n*wisize/2;
#else
		sp = buff + bstart + (j*numf)/pwidth + 1 - wisize/2;
#endif
		if (sp < buff) continue;	/* too early */
		if ((sp+wisize) > (buff+item->numframes)) break;	/* insufficient data */
		for (i=0;i<wisize;i++,sp++)
			*xp++ = ((float)*sp - PREEMP*(float)sp[-1]) *
				(0.54 - 0.46 * cos((float)i * omega)); 
		for (;i<=fftsize;i++)
			*xp++ = 0.0;

	        /* perform FFT */ 
	        REALFFT(fftbuf, fftsize/2, 1);

		/* convert to pixels */
		xstep = (float)(fftsize-2)/(float)pheight;
		xend = xstep;
		xpos = 0.0;
		xp = fftbuf+2;
		
		for (i=pheight-1;i>=0;i--) {
			xsum = 1.0;
			cnt = 0;
			while (xpos < xend) {
				xsum += *xp * *xp;
				xp++;
				xsum += *xp * *xp;
				xp++;
				xpos += 2.0;
				cnt++;
			}
			edb = 10.0 * log10(xsum/cnt);
			if (edb > hi) hi = edb;
			if (edb < lo) lo = edb;
			p = (int)(0.5+(float)digdata.digdev.greylevels*(edb - dbnorm)/DBRANGE);
			if (p >= digdata.digdev.greylevels)
				p = digdata.digdev.greylevels-1;
			else if (p < 0)
				p = 0;
			pixels[i] += p;
			xend += xstep;
		}
#ifdef PROPER_MERGE
	    }
	    if (nmerge > 1) for (i=0;i<pheight;i++) pixels[i] /= nmerge;
#endif
		/* add in grid - if required */
		if (flags & DIGITEMGRID) {
			x = ixstart+j*digdata.digdev.greypixwidth;
			for (l=0;l<digdata.digdev.greypixwidth;l++) {
				for (i=0;i<digit_nxgrid;i++) {
					if ((x+l)==digit_xgrid[i]) {
						for (k=0;k<pheight;k++)
							if (grid2[k])
								pixels[k] = (pixels[k] > digdata.digdev.greylevels/3) ? 0 : digdata.digdev.greylevels-1;
						goto griddone;
					}
				}
			}
			for (l=0;l<digdata.digdev.greypixwidth;l++) {
				for (i=0;i<digit_nxgrid;i++) {
					if (((x+l)==digit_xgrid[i]-digdata.digdev.greypixwidth) || ((x+l)==digit_xgrid[i]+digdata.digdev.greypixwidth)) {
						for (k=0;k<pheight;k++)
							if (grid1[k])
								pixels[k] = (pixels[k] > digdata.digdev.greylevels/3) ? 0 : digdata.digdev.greylevels-1;
						goto griddone;
					}
				}
			}
griddone:	;
		}

		/* call device driver greyscale routine */
		(*digtable[digdata.select].greyscale)(ixstart+j*digdata.digdev.greypixwidth,digitab.iyb+1,pheight,pixels);

	}

	/* draw title if required */
	digititle(item,flags);
/*
	sprintf(messg,"lo=%ddB hi=%ddB",lo,hi);
	digprompt(messg);
	digwait();
*/
	/* that's it */
	digflush();
	return(0);
}

#ifdef EMO
char	*progname="digitFFT";
main(argc,argv)
int	argc;
char	*argv[];
{
	struct	item_header item;
	short	*buff;
	double	totime;

	if (strcmp(argv[1],"-p")==0) {
		digstart(DIG_DEFAULT_PRINTER,NULL,1);
		getitem(argv[2],SP_TYPE,"*",&item,(void *)&buff);
	}
	else {
		digstart(DIG_DEFAULT_TERM,NULL,1);
		getitem(argv[1],SP_TYPE,"*",&item,(void *)&buff);
	}

	digscale(10.0,10.0,0);
	digclearscreen();

	totime = item.numframes*item.frameduration;
	digitemtime(23,0.0,9.0,10.0,9.5,0.0,totime,1);
	digitemFFT(20,0.0,7.0,10.0,9.0,&item,buff,0.0,totime,3);

	digitemtime(23,0.0,6.0,10.0,6.5,totime-0.25,totime+0.25,1);
	digitemFFT(20,0.0,5.0,10.0,6.0,&item,buff,totime-0.25,totime+0.25,3);

	digitemtime(23,0.0,4.0,10.0,4.5,0.0,0.1,1);
	item.offset=0.05;
	digitab.hi=1000;
	digitab.lo = -1000;
	digitemFFT(20,0.0,1.0,10.0,4.0,&item,buff,0.0,0.1,7);

	digwait();

	digclearscreen();
	digitemtime(23,0.0,9.0,10.0,9.5,0.0,0.5,1);
	digitemFFT(20,0.0,1.0,10.0,9.0,&item,buff,0.0,0.5,3);

	digwait();
	digend();
	exit(0);
}
#endif
