/* digitemSY -- synthesizer control data display */

/* version 1.1 - November 1987
	- SFS version
*/

#define MAXFREQ 4500.0		/* maximum allowable frequency */
#define MINFREQ 0.0		/* minimum allowable frequency */
#define FX	0		/* Fundamental freq */
#define F1	3		/* first formamt freq */
#define A1	4		/* first formant amp */
#define F2	6		/* second formant freq */
#define A2	7		/* second formant amp */
#define F3	9		/* third formant freq */
#define A3	10		/* third formant amp */
#define F4	12		/* fourth formant freq */
#define A4	13		/* fourth formant amp */
#define VR	18		/* voicing ratio */

/* macros to return the largest or smallest of two values */
#define MAX(x,y)  (((x) > (y)) ? (x) : (y))
#define MIN(x,y)  (((x) < (y)) ? (x) : (y))

/* external definitions */
#include <stdio.h>
#include <fcntl.h>
#include "sfs.h"
#include "dig.h"
#include "digdata.h"
#include "digitem.h"

#ifdef __STDC__
void	displayframe(int,int *,int,int *);
void	displaydots(int,int,int,int);
void	encodeframe(short *,int *,double,int);
#else
void	displayframe();
void	displaydots();
void	encodeframe();
#endif

#ifdef __STDC__
int    digitemSY(int32  bundles,float  xl,float  yb,float xr,float  yt,struct  item_header *item,short  *buff,double  start,double  stop,int  flags)
#else
int digitemSY(bundles,xl,yb,xr,yt,item,buff,start,stop,flags)
int32               bundles;         /* colours for graph */
float              xl,yb,xr,yt;     /* co-ords of box */
struct item_header *item;           /* data definition */
short              *buff;           /* data buffer */
double             start,stop;      /* start and stop times */
int                flags;           /* possible options on display */
#endif
{ 
	register int	i,j;
	int  		frame[32],oframe[32]; /* data frames */
	int		x,y,oldx;
	int		adj;
	int		numf,bstart;
	int		hi,lo;
	float		digtextlen();
	double		scalex,ampscale;

	/* sort out bundles */
	digitbundle(bundles);

	/* init box */
	digitbox(xl,yb,xr,yt,start,stop,flags);
	xl += digitab.ixoff/digdata.xscale;
	yt = yval(digitab.iyt);

	/* write axis label */
	y = (digitab.iyt + digitab.iyb - digdata.chheight)/2;
	digtextp(digitab.lbun,digitab.ixl+3,y,"Hz");

	/* check timing of data set */
	bstart = (start-item->offset)/item->frameduration;
	numf = 1 + (stop - start)/item->frameduration;
	if ((bstart+numf) > item->numframes) {
		/* data does not reach to right hand side */
		xr = xl + (item->numframes-bstart)*item->frameduration*digitab.scalex;
		numf = item->numframes - bstart;
	}
	if (bstart < 0) {
		/* data does not reach to left hand end */
		xl += (-bstart*item->frameduration)*digitab.scalex;
		numf += bstart;
		bstart=0;
	}

	/* display, if anything left */
	if ((numf>0) && (bstart<item->numframes)) {

		/* draw y-axis */
		digitemfreq(bundles,digitab.iyb,digitab.iyt,MINFREQ,MAXFREQ,0);

		/* get max amplitude */
		hi = 400;
		for (i=0;i<numf;i++) {
			hi = MAX(hi,buff[(bstart+i)*item->framesize+A1]);
			hi = MAX(hi,buff[(bstart+i)*item->framesize+A2]);
			hi = MAX(hi,buff[(bstart+i)*item->framesize+A3]);
			hi = MAX(hi,buff[(bstart+i)*item->framesize+A4]);
		}
		lo = hi - 400;	/* top 40db */

		/* get scale factors */
		scalex = item->frameduration*digitab.scalex*digdata.xscale;
		digitab.scaley = ((float)(digitab.iyt-digitab.iyb))/(MAXFREQ-MINFREQ);
		adj=digitab.ixl+digitab.ixoff+(item->offset+bstart*item->frameduration-start)*digitab.scalex*digdata.xscale;
		ampscale=((float)(digitab.iyt-digitab.iyb))/(20*(hi-lo));

		/* draw graph */
		digclippush();
		digclip(xl,yb,xr,yt);
		if (bstart > 0) {
			oldx= adj - scalex;
			encodeframe(&buff[(bstart-1)*item->framesize],oframe,ampscale,lo);
		}
		else {
			oldx = 0;
			encodeframe(&buff[bstart*item->framesize],oframe,ampscale,lo);
		}
		for (i=0;i<numf;i++) {
			x = adj + (float)i*scalex;
			encodeframe(&buff[(bstart+i)*item->framesize],frame,ampscale,lo);
			displayframe(oldx,oframe,x,frame);
			oldx=x;
			for (j=0;j<32;j++) oframe[j]=frame[j];
		}
		if ((bstart+numf) < item->numframes) {
			x = adj + (float)i*scalex;
			encodeframe(&buff[(bstart+i)*item->framesize],frame,ampscale,lo);
			displayframe(oldx,oframe,x,frame);
		}
		digclippop();
	}

	/* draw title if required */
	digititle(item,flags);

	/* that's it */
	digflush();
	return(0);

}

void displayframe(x1,f1,x2,f2)
int	x1,x2;			/* x co-ord of frames */
int	*f1,*f2;		/* pixel values for SY data */
{
	int	dash;		/* amplitude line style */
	
	/* display different voicing ratios as different line styles */
	if (f1[VR] > 192)
		dash = 0;
	else if (f1[VR] > 128)
		dash = 15;
	else if (f1[VR] > 64)
		dash = 30;
	else 
		dash = 45;

	/* Fx display */
	if (f1[VR] > 128) diglinep(digitab.gbun+dash,x1,f1[FX],x2,f2[FX]);

	/* F1 and A1 display */
	diglinep(digitab.gbun+dash+1,x1,f1[F1]+f1[A1],x2,f2[F1]+f2[A1]);
	diglinep(digitab.gbun+dash+1,x1,f1[F1]-f1[A1],x2,f2[F1]-f2[A1]);
	if (f1[VR] > 128) 
		diglinep(digitab.gbun+dash+1,x1,f1[F1]-f1[A1],x1,f1[F1]+f1[A1]);
	else
		displaydots(digitab.gbun+1,x1,f1[F1],f1[A1]);

	/* F2 and A2 display */
	diglinep(digitab.gbun+dash+2,x1,f1[F2]+f1[A2],x2,f2[F2]+f2[A2]);
	diglinep(digitab.gbun+dash+2,x1,f1[F2]-f1[A2],x2,f2[F2]-f2[A2]);
	if (f1[VR] > 128) 
		diglinep(digitab.gbun+dash+2,x1,f1[F2]-f1[A2],x1,f1[F2]+f1[A2]);
	else
		displaydots(digitab.gbun+2,x1,f1[F2],f1[A2]);

	/* F3 and A3 display */
	diglinep(digitab.gbun+dash+3,x1,f1[F3]+f1[A3],x2,f2[F3]+f2[A3]);
	diglinep(digitab.gbun+dash+3,x1,f1[F3]-f1[A3],x2,f2[F3]-f2[A3]);
	if (f1[VR] > 128) 
		diglinep(digitab.gbun+dash+3,x1,f1[F3]-f1[A3],x1,f1[F3]+f1[A3]);
	else
		displaydots(digitab.gbun+3,x1,f1[F3],f1[A3]);

	/* F4 and A4 display */
	if ((f1[F4] > 0) || (f2[F4] > 0)) {
		if ((f1[F4] > 0) && (f2[F4] > 0)) {
			diglinep(digitab.gbun+dash+4,x1,f1[F4]+f1[A4],x2,f2[F4]+f2[A4]);
			diglinep(digitab.gbun+dash+4,x1,f1[F4]-f1[A4],x2,f2[F4]-f2[A4]);
		}
		else if (f1[F4] > 0) {
			diglinep(digitab.gbun+dash+4,x1,f1[F4]+f1[A4],x2,f1[F4]+f1[A4]);
			diglinep(digitab.gbun+dash+4,x1,f1[F4]-f1[A4],x2,f1[F4]-f1[A4]);
		}
		else {
			diglinep(digitab.gbun+dash+4,x1,f2[F4]+f1[A4],x2,f2[F4]+f2[A4]);
			diglinep(digitab.gbun+dash+4,x1,f2[F4]-f1[A4],x2,f2[F4]-f2[A4]);
		}
	}
	if (f1[F4] > 0) {
		if (f1[VR] > 128) 
			diglinep(digitab.gbun+dash+4,x1,f1[F4]-f1[A4],x1,f1[F4]+f1[A4]);
		else
			displaydots(digitab.gbun+4,x1,f1[F4],f1[A4]);
	}

}

void displaydots(bun,x,y,dy)
int	bun;
int	x,y,dy;
{
	int	i,d,ndots;
	ndots = (2*dy)/4;
	for (i=0;i<ndots;i++) {
		d = y - dy + rand()%(2*dy);
		diglinep(bun,x,d,x,d);
	}
}


/* encode synthesizer data into pixels */
void encodeframe(frame,pixel,ampscale,lo)
short		*frame;		/* SY data frame */
int		*pixel;		/* pixel data */
double		ampscale;
int		lo;
{

	/* encode FX */
	pixel[FX] = digitab.iyb + (frame[FX]-MINFREQ) * digitab.scaley;

	/* encode F1 and A1 */
	pixel[F1] = digitab.iyb + (frame[F1]-MINFREQ) * digitab.scaley;
	pixel[A1] = (frame[A1]-lo) * ampscale;
	if (pixel[A1] < 0) pixel[A1] = 0;

	/* encode F2 and A2 */
	pixel[F2] = digitab.iyb + (frame[F2]-MINFREQ) * digitab.scaley;
	pixel[A2] = (frame[A2]-lo) * ampscale;
	if (pixel[A2] < 0) pixel[A2] = 0;

	/* encode F3 and A3 */
	pixel[F3] = digitab.iyb + (frame[F3]-MINFREQ) * digitab.scaley;
	pixel[A3] = (frame[A3]-lo) * ampscale;
	if (pixel[A3] < 0) pixel[A3] = 0;

	/* encode F4 and A4 */
	if (frame[F4] <= 0) {
		if (frame[A4] > 0)
			pixel[F4] = digitab.iyb + (4000-MINFREQ) * digitab.scaley;
		else
			pixel[F4] = -1;
	}
	else
		pixel[F4] = digitab.iyb + (frame[F4]-MINFREQ) * digitab.scaley;
	pixel[A4] = (frame[A4]-lo) * ampscale;
	if (pixel[A4] < 0) pixel[A4] = 0;

	pixel[VR]=frame[VR];
}

#ifdef EMO
char	     *progname="digitemSY";
main(argc,argv)
int          argc;
char         *argv[];
{
	struct      item_header item;          /* data definition */
	short       *buff;                     /* data buffer */
	double      totime;                    /* total time */

	/* to choose between laser printer or terminal, 'k' indicates the */
	/* laser printer, '\0' indicates a terminal */
	if (strcmp(argv[1], "-p") == 0) { 
		digstart('k',NULL,1);
		getitem(argv[2],SY_TYPE,"*", &item, &buff);
	}
	else { 
		digstart('\0',NULL,1);
		getitem(argv[1],SY_TYPE,"*", &item, &buff);
	}

	/* set the scale for display */
	digscale (10.0,12.0,0);
	/* clear the screen ready to display */
	digclearscreen();
	totime = item.numframes * item.frameduration;
	
	/* display the timescale */
	digitemtime(23,0.0,11.0,10.0,12.0,0.0,totime,1);
	/* display the synthesizor control contours and axes */
	digitemSY(20,0.0,8.0,10.0,11.0,&item,buff,0.0,totime,1);
	/* display the timescale */
	digitemtime(23,0.0,7.0,10.0,8.0,totime-0.25,totime+0.25,1);
	/* display the synthesizor control contours and axes */
	digitemSY(20,0.0,4.0,10.0,7.0,&item,buff,totime-0.25,totime+0.25,3);
	/* display the timescale */
	digitemtime(23,0.0,3.0,10.0,4.0,0.0,0.5,1);
	item.offset=0.1;
	/* display the synthesizor control contours and axes */
	digitemSY(20,0.0,0.0,10.0,3.0,&item,buff,0.0,0.5,3);

	digend();
	exit(0);
}
#endif
