/* Forward and Reverse Fast Fourier Transform Routines */
/* using routines as published in 'Numerical Recipes in C'. */

#include "SFSCONFG.h"
#include <math.h>
#ifdef __STDC__
#include <float.h>
#endif
#include <stdlib.h> 
#include <stdio.h>                   
#include "sfs.h"
#include "fft.h"

#ifndef PI
#define PI      3.141592653589793
#endif

#ifdef __STDC__
void four1(float *data, int nn, int isign);
#else
void four1();
#endif

/************************************************ 
 Complex Discrete Inverse Fast Fourier Transform, 
 ************************************************/  
void CDIFFT(x,points)
float *x; int points; 
{       
/* NOTE:     x[0..points+1] allocated memory */
/*       but x[0..points-1] real data */
     int i, n;                   

     n = points >> 1;
     if (n<<1 != points) error("ffs: points MUST be integer power of 2.");

     x[1] = x[points];                  /* last frequency component put in x[1] */

	REALFFT(x, n, REVERSE);

     for (i = 0; i < n; i++) {          /* real data normalised */
       x[i] /= n;
     }
}  

/**************************************** 
 Complex Discrete Fast Fourier Transform, 
 ***************************************/  

void CDFFT(x,points)
float *x; int points; 
{ 
/* NOTE:     x[0..points+1] allocated memory */
/*       but x[0..points-1] real data */
     int n;                   

     n = points >> 1;
     if (n<<1 != points) error("ffs: points MUST be integer power of 2.");

	REALFFT(x, n, FORWARD);

     x[points]   = x[1];                /* last frequency component */
     x[points+1] = (float)0.0;          /* imag. parts of first and last components */
     x[1]        = (float)0.0;
}  

 
/* REALFT - Numerical Recipes in C */
/*
This routine takes a single real function and calculates either its
DFT or its Inverse DFT depending on 'isign' as for 'four1'. 
On Entry: 'data[0..2n-1]' contains the real function of length '2n',
          where 'n' MUST be an integral power of 2. 
On Exit:  'data[0..2n-1]' contains the positive half of the complex DFT.
          'data[0]' contains the real valued first component and
          'data[1]' contains the real valued last component of the
          complex transform. 
This routine also calculates the inverse transform of a complex data
array if it is the transform of real data and 'isign' is set to -1.
(Result in this case must be multiplied by 1/n.)
*/

void REALFFT(data,n,isign)
float *data; int n; int isign;
{
	   int i,i1,i2,i3,i4,n2p3;
        double c1=(double)0.5,c2,h1r,h1i,h2r,h2i;
        double wr,wi,wpr,wpi,wtemp,theta;

        /* the code in this routine expects indexes to go 1..2n */
        data--;
       
        /* Initialise the recurrence. */
        theta = 3.141592653589793/(double) n;
        if (isign == 1) {
          c2 = (double)-0.5; 
          four1(data, n, 1);                 /* Forward Transform */
        } 
        else {
          c2 = (double)0.5;
          theta = -theta;
        }           

        wtemp = sin(0.5*theta);
        wpr   = -2.0*wtemp*wtemp;
        wpi   = sin(theta);
        wr    = 1.0+wpr;
        wi    = wpi;
        n2p3  = 2*n+3;  

        for (i = 2; i <= n/2; i++) {        /* Case i==1 done below */

          /* Separate transforms from the data */
          i4  = 1+(i3=n2p3-(i2=1+(i1=i+i-1)));
          h1r = c1*(data[i1]+data[i3]);
          h1i = c1*(data[i2]-data[i4]);
          h2r = -c2*(data[i2]+data[i4]);
          h2i = c2*(data[i1]-data[i3]);

          /* Recombine to form true transform of original data */
          data[i1] = (float)(h1r+wr*h2r-wi*h2i);
          data[i2] = (float)(h1i+wr*h2i+wi*h2r);
          data[i3] = (float)(h1r-wr*h2r+wi*h2i);
          data[i4] = (float)(-h1i+wr*h2i+wi*h2r);

          /* The recurrence Relations */
          wr = (wtemp = wr)*wpr-wi*wpi+wr;
          wi = wi*wpr+wtemp*wpi+wi;
        }    

        if (isign == 1) { 
          /* Squeeze the first & last data into the original array */ 
          data[1] = (h1r=data[1])+data[2];
          data[2] = h1r-data[2];
        } 
        else {
          data[1] = c1*((h1r=data[1])+data[2]);
          data[2] = c1*(h1r-data[2]);
          four1(data, n, -1);                /* Inverse Transform */
        }
}

/* FOUR1 - Numerical Recipes in C */

/*
FFT routine based on one originally written by N. Brenner of Lincoln
Laboratories.  
On entry: 'data[1..2*nn]' contains 'nn' complex data points, with each
          complex value occupying two consecutive locations, real
          value stored in the lower address.  'nn' MUST be an integer
          power of 2 (this is not checked for).   
On exit:  'data[1..2*nn]' contains the complex transformation output,
          again stored in pairs of real and imaginary values.
'isign' determins the direction of the transform: 
           1 for forward DFT,
          -1 for inverse DFT, scaled by a factor of 'nn'; 
                              (i.e. not normalised by 1/N).
*/

#define SWAP(a,b) tempr=(a);(a)=(b);(b)=tempr

void four1(data,nn,isign)
float *data; int nn; int isign;
{
	   int n,mmax,m,j,istep,i;
        double wtemp,wr,wpr,wpi,wi,theta; /* double precision for the */
                                          /* trigonometric recurrences. */
        float tempr,tempi;

        n=nn << 1;
        j=1;
        
        /* Bit-Reversal routine */
        for (i = 1; i < n; i += 2) {
          if (j > i) {
            SWAP(data[j],data[i]);   /* exchange the two complex numbers */
            SWAP(data[j+1],data[i+1]);
          }
          m = n >> 1;
          while (m >= 2 && j > m) {
            j -= m;
            m >>= 1;
          }
          j += m;
        }                 
        
        /* Danielson-Lanczos loops */
        mmax = 2;
        while (n > mmax) {      
          /* outer loop executed ln(nn) times */
          istep = 2*mmax; 
                  
          /* initialise for the trigonometric recurrence */
          theta = 6.28318530717959/(isign*mmax); 
          wtemp = sin(0.5*theta);
          wpr   = -2.0*wtemp*wtemp;
          wpi   = sin(theta);
          wr    = 1.0;
          wi    = 0.0;
              
          /* two nested inner loops */
          for (m = 1; m < mmax; m += 2) {
            for (i = m; i <= n; i += istep) {   
    
              /* Danielson-Lanczos formula: */
/* printf("nn=%d isign=%d j=%d data[j]=%g data[j+1]=%g\n",nn,isign,j,data[j],data[j+1]); */
              j          = i+mmax;
              tempr      = (float)(wr*data[j]-wi*data[j+1]);
              tempi      = (float)(wr*data[j+1]+wi*data[j]);
              data[j]    = data[i]-tempr;
              data[j+1]  = data[i+1]-tempi;
              data[i]   += tempr;
              data[i+1] += tempi;
            }                                 

            /* trigonometric recurrence */
            wr = (wtemp=wr)*wpr-wi*wpi+wr;
            wi = wi*wpr+wtemp*wpi+wi;
          }
          mmax = istep;
        }
}

#undef SWAP
 
 
 
 

