/* lsys.c -- general Linear System routines */

/*
 * (c) 1994 Mark Huckvale
 * Search Systems Ltd
 * 52-54 High Holborn
 * London WC1E 6BT
 * Tel. 071-430 1685
 */

#include <stdio.h>
#include <math.h>
#include "complex.h"
#include "lsys.h"

#ifndef PI
#define PI	3.14159265359
#endif

/* reset LTI system memory */
void LTISystemClear(ltis)
LTIState *ltis;
{
	int	i;
	for (i=0;i<=MAXLTISIZE;i++) ltis->smem[i] = 0;
}

/* apply LTI system to single input sample - Direct form II implementation */
double	LTISystem(ltis,x)
LTIState *ltis;
double x;
{
	int	i;
	double	out;

	/* shift state memory */
	if (ltis->na > ltis->nb) i=ltis->na; else i=ltis->nb;
	while (i > 0) {
		ltis->smem[i] = ltis->smem[i-1];
		i = i - 1;
	}

	/* compute smem[0] - from y[] coefficients */
	ltis->smem[0] = x;
	for (i=1;i<=ltis->nb;i++)
		ltis->smem[0] -= ltis->b[i] * ltis->smem[i];

	/* compute output - from x[] coefficients */ 
	out = 0.0;
	for (i=0;i<=ltis->na;i++)
		out += ltis->a[i] * ltis->smem[i];

	return(out);
}

/* find response of LTI System at given frequency */
double LTISystemResponse(ltis,f)
LTIState *ltis;		/* LTI System */
double f;		/* freq as fraction of sample rate */
{
	COMPLEX	omega[MAXLTISIZE+1];
	int	i;
	int	ns;
	COMPLEX	rnum,rden;

	/* initialise polynomial values of complex frequency */
	omega[0] = CMake(1.0,0.0);
	omega[1] = CExp(CMake(0.0,2*PI*f));
	if (ltis->na > ltis->nb) ns = ltis->na; else ns = ltis->nb;
	for (i=2;i<=ns;i++)
		omega[i] = CMult(omega[i-1],omega[1]);

	/* compute response of numerator */
	rnum=CMake(0.0,0.0);
	for (i=0;i<=ltis->na;i++)
		rnum = CAdd(rnum,CScale(omega[i],ltis->a[i]));

	/* compute response of denominator */
	rden=omega[0];
	for (i=1;i<=ltis->nb;i++)
		rden = CAdd(rden,CScale(omega[i],ltis->b[i]));

	/* compute ratio */
	if (CMag(rden)==0)
		return(1.0E10);			/* i.e. infinity */
	else
		return(CMag(CDiv(rnum,rden)));
}

