/* hwsynth -- LSI synthesizer replay */

/*--------------------------------------------------------------------------*/
/**MAN
.TH HWSYNTH SFS3 UCL SFS
.SH NAME
hwsynth - hardware synthesizer replay routines
.SH SYNOPSIS
.nf

void     \fBhwsynthcode\fR(inframe,outframe)
short    *inframe;      /\* input frame from SY item *\/
short    *outframe;     /\* encoded frame ready for replay *\/

void     \fBhwsynth\fR(buff,numframes)
short    *buffer;       /\* buffer of encoded frames for replay *\/
int      numframes;     /\* number of frames to replay *\/

.fi
.SH DESCRIPTION
.I hwsynth
provides a standard mechanism for replaying synthesizer control
data.  For machines equipped with a parallel interface connected
to a hardware synthesizer, the routine can control the
hardware directly.  Otherwise the routine uses a software
emulation of the JSRU synthesizer,
.I srusynth.
.SS
The routine
.I hwsynthcode
encodes a standard SY item frame into a 32 byte frame suitable
for replay.  It must be called for each SY item frame before
they may be replayed.
.SS
The routine
.I hwsynth
is called with the encoded data for a complete synthesis.
The data is converted to a signal and replayed, either through
a hardware interface or through a software emulation.  In the
latter case, the DAC environment variable controls the selection
of the replay device (see sfsdac(3)).
.PP
On Sun devices, replay is simultaneous with conversion, reducing
replay delay - see srusynth(1).
.SH DEVICES
.TP 20
/dev/dacp0/pfo0
16-bit Parallel IO on Masscomp computers
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
.SH BUGS
On the LSI synthesizer replay actually starts at the start of the second frame.  Include a dummy
frame to ensure accurate replay (replicating the first frame is easiest).
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"

/* encode a SY item frame */
void hwsynthcode(iframe,oframe)
short	*iframe;
unsigned short	*oframe;
{
	/* encode frame */
	oframe[0] = (((iframe[15] - 90) / 5) & 0xFF);
	oframe[1] = ((iframe[16]/10) & 0xFF);
	oframe[2] = (((iframe[3] - 100) / 25) & 0xFF);
	oframe[3] = ((iframe[4]/10) & 0xFF);
	oframe[4] = (((iframe[6] - 500) / 50) & 0xFF);
	oframe[5] = ((iframe[7]/10) & 0xFF);
	oframe[6] = (((iframe[9] - 1300) / 50) & 0xFF);
	oframe[7] = ((iframe[10]/10) & 0xFF);
	oframe[8] = ((iframe[13]/10) & 0xFF);
	oframe[9] = ((iframe[18]/4) &0xFF);
	oframe[10] = (((int)(23.09*log(1.0*iframe[0]+0.1) - 75.37)) & 0xFF);
	oframe[11] = (32);
	oframe[12] = (0);
	oframe[13] = (1);
	oframe[14] = (0);
	oframe[15] = (255);
}

/* call software routine if PIO not available */
#ifndef PIO_AVAIL
int hwsynth(buff,numf)
short	*buff;
int32	numf;
{
	FILE	*op;
	int	i,j;

#ifndef _MSC_VER
	if ((op=popen("srusynth -t","w"))) {
		for (i=0;i<numf;i++) {
			for (j=0;j<16;j++)
				fprintf(op,"%02X",buff[j] & 0xFF);
			fprintf(op,"\n");
			buff += 16;
		}
		pclose(op);
	}
#endif
	return(0);
}
#else

#ifdef PIO_MASSCOMP

#include <mr.h>
#include <mrerrs.h>

/* error trap */
static int traperror()
{
	return(0);
}

/* replay buffer */
int hwsynth(buff,numf)
short	*buff;
int32	numf;
{
	extern int	traperror();
	int	pathno = -1,status[2],idur,err=0;

	/* set up error handling */
#define ERRDEFAULT 1
	seterrtrap(traperror,ERRDEFAULT);

	/* open channel to pio */
	if ((err=mropen(&pathno,PIO_MASSCOMP_CHAN,0)) < 0) {
		if (errno==DA_OPENFAILED) {
			sleep(1);	/* sleep one second and try again */
			if ((err=mropen(&pathno,PIO_MASSCOMP_CHAN,0)) < 0)
				return;
		}
		else
			return;
	}

	/* initialise parallel modes */
#define FIFO_ENB 1
#define PRIME_BUSY 1
#define SET_GO 1
#define ENB_IN 0
#define ENB_OUT 1
#define MASK1 3
#define MASK2 5
	if (mrpfomod(pathno,FIFO_ENB,PRIME_BUSY,SET_GO,ENB_IN,ENB_OUT,MASK1,MASK2))
		error("unable to set pfo modes",NULL);

	/* transfer data */
	if (mrpdxout(pathno,16*numf,buff))
		error("unable to start pfo transfer",NULL);

	/* wait for transfer */
	idur = (numf * 10) + 1000;
	if (mrevwt(pathno,status,idur))
		error("pfo timeout",NULL);

	/* and quit */
	mrclose(pathno);
}
#endif
#endif

#ifdef IAG
void main(argc,argv)
int	argc;
char	*argv[];
{
	struct	item_header item;
	short		*sy;
	short		*buff;
	int		i,fid;

	if ((fid=sfsopen(argv[1],"r",NULL)) < 0)
		error("cannot find file '%s'",argv[1]);

	if (!sfsitem(fid,SY_TYPE,"*",&item))
		error("cannot find SY item in '%s'",argv[1]);

	sy=(short *)sfsbuffer(&item,1);
	buff=(short *)calloc(item.numframes+1,32);

	sfsread(fid,0,1,sy);
	hwsynthcode(sy,buff);
	for (i=0;sfsread(fid,i,1,sy);i++)
		hwsynthcode(sy,&buff[16*i+16]);

	hwsynth(buff,item.numframes+1);

	exit(0);
}
#endif
