/* sfsdac -- SFS routines for support of Digital-to-Analogue conversion */

/* M.A.Huckvale - University College London */

/* version 1.0 - March 1993 */
/* version 1.1 - July 1993
	- add 8-5 resamp to SUN16 playback
	- add sun16 spkr/phone/line
*/
/* version 1.2 - March 1994
	- add DOS eb12 support
*/
/* version 1.3 - July 1994
	- add Vista Extend (X-Server/PC) support
*/
/* version 1.4 - February 1996
	- SoundBlaster output
	- Pipe output
*/
/* version 1.5 - March 1996
	- Linux DAC device support
*/

/*-------------------------------------------------------------------------*/
/**MAN
.TH SFSDAC 3 SFS UCL
.SH NAME
sfsdac -- SFS support for digital to analogue converters
.SH SYNOPSIS
.nf
int dac_open(name)
char *name;	/\* device name - supply as NULL for auto-detection *\/

int dac_playback(buf,numf,srate,nbit,nchan,ntime)
short *buf;	/\* sample buffer *\/
int numf;	/\* number of samples in buffer *\/
double srate;	/\* sampling rate *\/
int nbit;	/\* number of bits in waveform (=12 or =16 only) *\/
int nchan;	/\* number of channels of data in buffer *\/
int ntime;	/\* number of times to play buffer *\/

void dac_close(rapid)
int rapid;	/\* flag to shut down device immediately *\/

int dac_repeat(offset,numf,ntime)
int offset;	/\* sample offset into last buffer *\/
int numf;	/\* number of sample to replay *\/
int ntime;	/\* number of times to replay *\/

void playback(buff,numf,srate)
short *buff;	/\* sample buffer *\/
int numf;	/\* number of samples *\/
double srate;	/\* sampling rate *\/

extern int dac_available_channels;	/\* # available channels *\/
extern double dac_selected_rate;	/\* selected sampling rate *\/
.fi
.SH DESCRIPTION
The support for Digital-to-Analogue conversion within SFS
is wholly confined to this set of routines.
All programs access any DAC through these routines.
DAC devices are given types which are selected by name.  These
names are coded into sfsdac.c and which types are available
depend on the machine configuration specified in SFSCONFG.h.
The name of the DAC is discovered at open time from a supplied
name, or from a DAC environment variable, or from a file
$(SFSBASE)/data/dactable.
.SS DAC_OPEN
The open routine attempts to find a suitable DAC type and device
and opens it to give the user unique access.  The mnemonic name
for the device may be given in the routine call, or if NULL is
found from the DAC environment variable, or if DAC is NULL from
the file $(SFSBASE)/data/dactable - which identifies DAC types from
specific terminal names.
dac_open() returns the DAC type if successful, otherwise -1.  The external
variable dac_available_channels is set to indicate the maximum number
of channels of simultaneous replay available.
.SS DAC_CLOSE
This routine closes the DAC device and frees any allocated memory.
.SS DAC_PLAYBACK
This routine takes details of a waveform supplied in a short buffer
and replays it a given number of times through the currently opened
DAC device.  Note that many DACs have a limited range of sampling frequencies
and that the required rate may not be available.  In this case, dac_playback()
silently selects the nearest frequency.  The selected frequency may be
recovered from the global variable dac_selected_rate.
Behaviour for multiple-channel
data played through single channel DACs is undefined.
Future dac_getconfig() and dac_setconfig() routines will allow more
control.
A repetition count of zero will only set sampling rate.
.SS DAC_REPEAT
This routine allows parts of the immediately previous buffer
supplied to dac_playback() to be replayed.  It assumes that the
buffer still exists and contains valid data.  This routine can
be used to great effect in systems which allow caching of DAC
data (e.g. the PC 'login' terminal emulator) whereby only a replay
command rather than a replay buffer needs to be transported over
a network.
.SS PLAYBACK
This routine provides backward compatibility with the old SFS
playback() routine.  It opens, replays and closes the default DAC device.
Note that this compatibility routine expects 16-bit single-channel data.
.SH FILES
.IP SFSBASE/data/dacmap
Table of mappings from ttynames to DAC name.
.SH SUPPORTED TYPES
.IP sun	11
SPARC-2 with 8-bit ulaw
.IP sparc2 11
SPARC-2 with 8-bit ulaw
.IP sun8 11
SPARC-2 with 8-bit ulaw
.IP sun8-spkr 11
SPARC-2 with 8-bit ulaw (speaker output)
.IP sun8-phone 11
SPARC-2 with 8-bit ulaw (headphone output)
.IP sun8-line 11
SPARC-2 with 8-bit ulaw (line output)
.IP sparc10 11
SPARC-10 with ISDN 16bit sound interface
.IP sun16	11
SPARC-10 with ISDN 16bit sound interface
.IP dbri	11
SPARC-10 with ISDN 16bit sound interface
.IP sun16-spkr	11
SPARC-10 with ISDN 16bit sound interface (speaker output)
.IP sun16-phone	11
SPARC-10 with ISDN 16bit sound interface (headphone output)
.IP sun16-line	11
SPARC-10 with ISDN 16bit sound interface (line output)
.IP 0..7	11
Masscomp DA08 - channel 0..7
.IP ansi	11
UCL/ANSI stdout transport - 16 bit
.IP ansi16	11
UCL/ANSI stdout transport - 16 bit
.IP ansi12	11
UCL/ANSI stdout transport - 12 bit
.IP pp	11
PC: UCL Parallel printer DAC
.IP dt2811	11
PC: Data Translation DT2811 board
.IP pclx	11
PC: Laryngograph PC/LX board
.IP eb12	11
PC: UCL expansion-bus 12-bit DAC
.IP sb8	11
PC: SoundBlaster 8 bit
.IP sb16	11
PC: SoundBlaster 16 bit
.IP extend	11
Vista Extend, (X-Server/PC) UCL Replay daemon
.IP pipe	11
sends data to $SFSBASE/bin/dacpipe
.IP linux	11
standard LINUX soundcard /dev/dsp
.SH VERSION/AUTHOR
.IP 1.5
Mark Huckvale
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <memory.h>
#include <string.h>
#include <math.h>
#include <fcntl.h>
#include "sfs.h"

#define MIN(x,y)	(((x)<(y))?(x):(y))

/* standard size replay buffer */
#define REPBUFSIZE	4096		/* bytes */

/* device-specific data */
#ifdef DAC_SUN8
#include <sun/audioio.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/ioctl.h>

static audio_info_t	audinfo;		/* audio state */
static unsigned char 	u_law();		/* u-law conversion */
#endif

#ifdef DAC_SUN16
#include <sun/audioio.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/ioctl.h>
#include "multimedia/libaudio.h"
#include "multimedia/audio_device.h"

/* manifest constants */
#if REPBUFSIZE < 16384
#undef REPBUFSIZE
#define REPBUFSIZE 	16384		/* about 8k samples allows elbow room */
#endif
static Audio_hdr	devhdr;		/* audio state */
static Audio_hdr	bufhdr;		/* buffer state */
#endif

/* LINUX standard soundcard */
#ifdef DAC_LINUX
#include <sys/soundcard.h>
#endif

#ifdef DAC_MASSCOMP
#include <mr.h>
#include <mrerrs.h>
#if REPBUFSIZE < 65536
#undef REPBUFSIZE
#define REPBUFSIZE 	65536		/* about a second allows elbow room */
#endif
#define MASSCOMP_BUFSPLIT	4	/* split into four 1/4 second chunks */
int	cfd;				/* clock file handle */
#endif

#ifdef DAC_EXTEND
#include <X11/Intrinsic.h>
#include <X11/Xlib.h>
#ifndef __STDC__
#define sgi
#endif
#include "extend.h"
#ifndef __STDC__
#undef sgi
#endif
#include "rmessage.h"
#if SFSMACHINE==0
#define REVERSESHORT(x) (((x & 0xFF00)>>8) | ((x & 0x00FF) << 8))
#define REVERSELONG(x)	(((x & 0xFF000000)>>24) | ((x & 0x00FF0000) >> 8) | ((x & 0x0000FF00) << 8) | ((x & 0x000000FF) << 24))
#else
#define REVERSESHORT(x)	(x)
#define REVERSELONG(x)	(x)
#endif
extern char *progname;
static XID	dosid;
static Display	*dpy;
static int	maxDOSxfer;		/* size of DOS xfer buf */
static int	maxWRDxfer;		/* max replay daemon xfer size */
static int eXt_major_code, first_event, first_error;
static int32	doshandle;
static int	dosrepcount;
#endif

/* supported DACs - major types */
enum DACTYPE { SUN8, SUN16, MASS12, ANSI, DOSDAC, XREPLAY, PIPE, LINUXDAC };

/* environment variable look-up table */
struct dac_tab_rec {
	char		*name;		/* name supplied in dac_open */
					/*  or found in DAC environment */
					/*  or in $(SFSBASE)/data/dactable */
	enum DACTYPE	type;		/* major device type */
	int		subtype;	/* minor device type */
};
static struct dac_tab_rec dac_tab[]={
{ "sun",	SUN8,		0 },	/* SPARC-2 with 8-bit ulaw */
{ "sparc2",	SUN8,		0 },	/* SPARC-2 with 8-bit ulaw */
{ "sun8",	SUN8,		1 },	/* SPARC-2 with 8-bit ulaw */
{ "sun8-spkr",	SUN8,		0 },
{ "sun8-phone",	SUN8,		1 },
{ "sun8-line",	SUN8,		2 },
{ "sparc10",	SUN16,		0 },	/* SPARC-10 with dbri */
{ "sun16",	SUN16,		0 },	/* SPARC-10 with dbri */
{ "dbri",	SUN16,		0 },	/* SPARC-10 with dbri */
{ "sun16-spkr",	SUN16,		0 },
{ "sun16-phone",SUN16,		1 },
{ "sun16-line",	SUN16,		2 },
{ "0",		MASS12,		0 },	/* Masscomp DA08 - channel 0 */
{ "1",		MASS12,		1 },	/* --------------- channel 1 */
{ "2",		MASS12,		2 },	/* etc */
{ "3",		MASS12,		3 },
{ "4",		MASS12,		4 },
{ "5",		MASS12,		5 },
{ "6",		MASS12,		6 },
{ "7",		MASS12,		7 },
{ "ansi",	ANSI,		0 },	/* UCL/ANSI stdout transport - 16 bit */
{ "ansi16",	ANSI,		0 },	/* ditto */
{ "ansi12",	ANSI,		1 },	/* UCL/ANSI stdout transport - 12 bit */
{ "pp",		DOSDAC,		1 },	/* PC: Parallel printer - direct drive */
{ "dt2811",	DOSDAC,		2 },	/* PC: Data Translation DT2811 board */
{ "pclx",	DOSDAC,		3 },	/* PC: Laryngograph PC/LX board */
{ "eb12",	DOSDAC,		4 },	/* PC: UCL expansion-bus 12-bit */
{ "sb8",	DOSDAC,		5 },	/* PC: SoundBlaster 8 bit */
{ "sb16",	DOSDAC,		6 },	/* PC: SoundBlaster 16 bit */
{ "extend",	XREPLAY,	0 },	/* Vista Extend, (X-Server/PC) */
{ "pipe",	PIPE,           0 },	/* sends data to bin/dacpipe */
{ "linux",	LINUXDAC,	0 },	/* standard LINUX soundcard */
};
#define NUMDAC (sizeof(dac_tab)/sizeof(struct dac_tab_rec))

/* only one DAC selected at any one time */
static int	dactype= -1;	/* current DAC type */
static int	dacsubtype=0;	/* current DAC subtype */
static int	afd;		/* audio handle */
static short	*prepbuff;	/* prepared buffer */
static int	prepnumf;	/* # sample in prepbuff */
static double	prepsrate;	/* prepared sampling rate */
static short	*lastbuff;	/* last known buffer */
static int	lastnumf;	/* last known # samples */
static double	lastsrate;	/* last known sampling rate */
static int	lastnbits;	/* last known # bits */
static int	lastnchannels;	/* last known # channels */

/* external variables for communication with user */
int		dac_available_channels;	/* # available channels */
double		dac_selected_rate;	/* to hold set sample rate */

void dac_close();

#ifdef DAC_MASSCOMP
int masscomp_traperror()
{
	return(0);
}
#endif

#ifdef DAC_EXTEND

/* open connection to DOS */
static XID XeXtStartDOS(dpy, name)
Display	*dpy;		/* X-windows display */
char	*name;		/* password */
{
	register xStartDOSReq *req;
	xDOSReply rep;
	int tlen;
    
	LockDisplay(dpy);
	GetReq(StartDOS, req);
	req->DOSid = XAllocID(dpy);
	req->minorCode = StartDOS_num;
	tlen = strlen(name) + 1;
	req->length += ((tlen + 3) >> 2);
	_XSend(dpy, (char *)name, tlen);
	_XReply(dpy, &rep, 0, True);
	UnlockDisplay(dpy);
	SyncHandle();
	if (rep.errno) {
		errno = rep.errno;
		return((XID)0);
	}
	maxDOSxfer = rep.size;
	return(req->DOSid);
}


/* open Vista Extend connection */
static int openDisplay(display, password)
char *display, *password;
{

	if ((dpy = XOpenDisplay(display)) == NULL) {
		fprintf(stderr, "%s: Vista Extend: cannot open display %s\n", progname, display);
		return(1);
	}
	if (!XQueryExtension(dpy, "HCL-DOS-Access",
			 &eXt_major_code, &first_event, &first_error) ) {
		fprintf(stderr, "%s: Vista Extend: HCL-DOS-access not present\n", progname);
		return(1);
	}
	if ((dosid = XeXtStartDOS(dpy, password)) == NULL) {
		fprintf(stderr, "%s: Vista Extend: password violation\n", progname);
		return(1);
	}
	doshandle = (getpid()<<16) | ++dosrepcount;
	return(0);
}

/* execute an interrupt using HCL */
static int XeXtPCint(dpy, dosid, func, buf, len, err)
Display	*dpy;		/* X-Windows display */
XID	dosid;		/* DOS handle */
int	func;		/* basic function: 0,1,2 */
char	*buf;		/* outward message */
int 	len;		/* outward message length */
int	*err;		/* error flag */
{
	register xPCintReq *req;
	xPCintReply rep;
	int	useLen;
	char	scrap[XFERBUFFSIZE];
	int 	npad;

	LockDisplay(dpy);
	GetReq(PCint, req);
	req->DOSid = dosid;
	req->func = func;
	req->minorCode = PCint_num;
	req->control = 0xAAAB;
	req->status = 0;
	useLen = len;
	req->dataLen = useLen;
	req->length += ((useLen + 3) >> 2);
	if (useLen) _XSend(dpy, buf, useLen);
	_XReply(dpy, &rep, 0, False);
	if ((func != 0) && (func != 1) && (!rep.errno || rep.errno == ERANGE) && rep.dataLen) {
		/* some bizarre reason, requires us to read something */
		_XRead(dpy, scrap, rep.dataLen);
		npad = 4 - (rep.dataLen & 3);
		if (npad) _XRead(dpy, scrap, npad);
	}
	*err = rep.errno;
	UnlockDisplay(dpy);
	SyncHandle();
	if ((func==0) || (func==1))
		return(rep.dataLen);
	else
		return(rep.control);
}

/* message structure */
static union rmessage msg;

/* open link to Vista Extend/HCL-DOS */
int	extend_open()
{
	char	*p,*getenv();
	int	err;

	if ((p=getenv("DOSPW"))==NULL)
		p = "TurnKey";
	if (openDisplay(NULL,p)!=0)
		return(1);
	maxWRDxfer = XeXtPCint(dpy, dosid, 0, msg, 0, &err);

	if (err) return(1);
	XeXtPCint(dpy, dosid, 1, msg, 0, &err);

	/* now open link through to WRD */
	msg.conn.func = REVERSESHORT(RMSG_CONN);
	msg.conn.flag = 0;
	msg.conn.handle = REVERSELONG(doshandle);
	
	XeXtPCint(dpy, dosid, 2, &msg, sizeof(msg.conn), &err);
	if (err) return(1);
	return(0);
}

/* download a waveform */
int	extend_load(buf,numf,srate,nbit,nchan)
short *buf;	/* sample buffer */
int numf;	/* number of samples in buffer */
double srate;	/* sampling rate */
int nbit;	/* number of bits in waveform (=12 or =16 only) */
int nchan;	/* number of channels of data in buffer */
{
	int		err;
	unsigned short	irate;
	int		headlen;
	int		maxsamp;
	short		*spbuf;
	int		offset;
	int		i,len;

	/* request space for waveform */
	msg.wave.handle = REVERSELONG(doshandle);
	msg.wave.func = REVERSESHORT(RMSG_WAVE);
	msg.wave.flag = 0;
	msg.wave.wsize = numf*sizeof(short);
	msg.wave.wsize = REVERSELONG(msg.wave.wsize);
	irate = srate;
	msg.wave.srate = REVERSESHORT(irate);
	msg.wave.nbyte = REVERSESHORT(sizeof(short));
	msg.wave.nchan = nchan;
	msg.wave.nchan = REVERSESHORT(msg.wave.nchan);
	XeXtPCint(dpy, dosid, 2, &msg, sizeof(msg.wave), &err);
	if (err) return(1);

	/* get download params */
	headlen = (char *)(msg.xfer.data) - (char *)&msg;
	maxsamp = (maxWRDxfer - headlen)/sizeof(short);

	/* download buffer */
	offset = 0;
	while (numf > 1) {
		msg.xfer.func = REVERSESHORT(RMSG_XFER);
		msg.xfer.flag = 0;
		msg.xfer.handle = REVERSELONG(doshandle);
		msg.xfer.offset = offset*sizeof(short);
		msg.xfer.offset = REVERSELONG(msg.xfer.offset);
		len = MIN(numf,maxsamp);
		msg.xfer.xsize = len*sizeof(short);
		msg.xfer.xsize = REVERSESHORT(msg.xfer.xsize);
		spbuf = (short *)msg.xfer.data;
		for (i=0;i<len;i++,spbuf++) {
			if (nbit==12)
				*spbuf = buf[i] << 4;
			else
				*spbuf = buf[i];
			*spbuf = REVERSESHORT(*spbuf);
		}
		len = XeXtPCint(dpy, dosid, 2, &msg, headlen+2*len, &err) - headlen - 2;
		if (err) return(1);
		len = len/sizeof(short);	/* don't care if it's odd - just send again */
		offset += len;
		numf -= len;
		buf += len;
	}
	return(0);
}

/* play the downloaded waveform */
int	extend_play(offset,count)
int	offset;
int	count;
{
	int	err;

	msg.repl.func = REVERSESHORT(RMSG_REPL);
	msg.repl.flag = 0;
	msg.repl.handle = REVERSELONG(doshandle);
	msg.repl.offset = offset*sizeof(short);
	msg.repl.offset = REVERSELONG(msg.repl.offset);
	msg.repl.count = count*sizeof(short);
	msg.repl.count = REVERSELONG(msg.repl.count);
	XeXtPCint(dpy, dosid, 2, &msg, sizeof(msg.repl), &err);
	if (err) return(1);
	return(0);
}

/* close the connection */
int	extend_close()
{
	int	err;

	/* free space for waveform */
	msg.wave.func = REVERSESHORT(RMSG_WAVE);
	msg.wave.flag = REVERSESHORT(1);
	msg.wave.handle = REVERSELONG(doshandle);
	XeXtPCint(dpy, dosid, 2, &msg, sizeof(msg.wave), &err);
	if (err) return(1);

	/* close connection */
	msg.conn.func = REVERSESHORT(RMSG_CONN);
	msg.conn.flag = REVERSESHORT(1);
	msg.conn.handle = REVERSELONG(doshandle);
	XeXtPCint(dpy, dosid, 2, &msg, sizeof(msg.conn), &err);
	if (err) return(1);
	
	XCloseDisplay(dpy);

	return(0);
}
#endif

/* open DAC */
int	dac_open(device)
char	*device;
{
	int	i;
#ifdef DAC_MASSCOMP
	int	err;
#endif

	/* if DAC already open, close it */
	if (dactype >= 0) dac_close(0);

	/* if device is NULL, look for DAC environment variable */
	if (device==NULL) device = getenv("DAC");

	/* if device is NULL, look for terminal in dactable */
#ifndef DOS
	if (device==NULL) {
		char	dacfilename[SFSMAXFILENAME];
		char	*ttynm,*ttyname(),tname[80];
		char	dname[80];
		FILE	*ip;
		
		strcpy(dacfilename,sfsbase());
		strcat(dacfilename,"/data/dacmap");
		if ((ip=fopen(dacfilename,"r"))==NULL) {
			/* no file - abort */
			return(-1);
		}
		/* scan for tty name */
		ttynm=ttyname(0);
		while (fscanf(ip,"%s %s\n",tname,dname)==2) {
			if (strcmp(tname,ttynm)==0) {
				device = dname;
				break;
			}
		}
		fclose(ip);
	}
#endif
	if (!device)
		/* not found - abort */
		return(-1);

	/* by now should have DAC name */
	dactype = -1;
	dacsubtype = 0;
	for (i=0;i<NUMDAC;i++) {
		if (strcmp(device,dac_tab[i].name)==0) {
			dactype = dac_tab[i].type;
			dacsubtype = dac_tab[i].subtype;
			break;
		}
	}
#ifdef IAG
printf("device='%s' dactype=%d dacsubtype=%d\n",device,dactype,dacsubtype);
#endif
	if (dactype < 0)
		/* not found - abort */
		return(-1);

	/* set up default number of available channels */
	dac_available_channels = 1;

	/* OK now open device */
	switch (dactype) {
#ifdef DAC_SUN8
	case SUN8:
		/* SPARC 8-bit ulaw device */
		if ((afd = open(DAC_SUN8_PATH, O_WRONLY)) < 0) {
			fprintf(stderr,"dac_open: could not open '%s'\n",DAC_SUN8_PATH);
			dactype = -1;
			return(-1);
		}

		/* get Sun audio parameters */
		AUDIO_INITINFO(&audinfo);

		/* select output port */
		switch (dacsubtype) {
		case 0:
		default:
			audinfo.play.port = AUDIO_SPEAKER;
			break;
		case 1:
			audinfo.play.port = AUDIO_HEADPHONE;
			break;
		case 2:
			audinfo.play.port = AUDIO_LINE_OUT;
			break;
		}

		if (ioctl(afd,AUDIO_SETINFO,&audinfo) < 0) {
			fprintf(stderr,"dac_open: cannot set audio output parameters\n");
			close(afd);
			dactype = -1;
			return(-1);
		}
		break;
#endif

#ifdef DAC_SUN16
	case SUN16:
		/* SPARC 16-bit dual-basic-rate ISDN */
		if ((afd = open(DAC_SUN16_PATH, O_WRONLY)) < 0) {
			fprintf(stderr,"dac_open: could not open '%s'\n",DAC_SUN16_PATH);
			dactype = -1;
			return(-1);
		}
		dac_available_channels = 2;
		break;
#endif

#ifdef DAC_LINUX
 	case LINUXDAC:
 		/* Linux with Sound Blaster 16 */
		if ((afd = open(DAC_LINUX_PATH, O_WRONLY || O_SYNC)) < 0) {
 			fprintf(stderr,"dac_open: could not open '%s'\n",DAC_LINUX_PATH);
			dactype = -1;
			return(-1);
 		}
 		dac_available_channels = 2;
 		break;
#endif

#ifdef DAC_MASSCOMP
	case MASS12:
		/* Masscomp DA08, 8-channel 12-bit */
		/* set up error handling */
#define ERRDEFAULT 1
		seterrtrap(masscomp_traperror,ERRDEFAULT);

		/* open d-a */
#define EXWRITE 0
		afd = -1;
		if ((err=mropen(&afd,DAC_MASSCOMP_CHAN,EXWRITE)) < 0) {
			if (errno==DA_OPENFAILED) {
				sleep(1);	/* sleep one second and try again */
				if ((err=mropen(&afd,DAC_MASSCOMP_CHAN,EXWRITE)) < 0)
					return(-1);
			}
			else
				return(-1);
		}

		/* open clock */
		cfd = -1;
		if ((err=mropen(&cfd,DAC_MASSCOMP_CLK,EXWRITE)) < 0) {
			if (errno==DA_OPENFAILED) {
				sleep(1);	/* sleep one second and try again */
				if ((err=mropen(&cfd,DAC_MASSCOMP_CLK,EXWRITE)) < 0)
					return(-2);
			}
			else
				return(-2);
		}

		dac_available_channels = 8;
		break;
#endif

#ifdef DAC_ANSI
	case ANSI:
		/* character format, stdout */
		break;
#endif

	case DOSDAC:
		/* DAC attached directly to PC so no need to open */
		break;

#ifdef DAC_EXTEND
	case XREPLAY:
		break;
#endif

#ifdef DAC_PIPE
	case PIPE:
		break;
#endif
	default:
		/* unknown, or unsupported */
		dactype = -1;
		return(-1);
	}
	return(dactype);
}

/* close DAC */
void dac_close(rapid)
int	rapid;		/* set to 1 for rapid close */
{
	/* device specific rapid close */
	if (rapid) switch (dactype) {
#ifdef DAC_SUN16
	case SUN16:
		if (afd >= 0)
			audio_flush_play(afd);
		break;
#endif
	}

	/* do standard close */
	switch (dactype) {
#ifdef DAC_SUN8
	case SUN8:
		if (afd >= 0) close(afd);
		break;
#endif

#ifdef DAC_SUN16
	case SUN16:
		if (afd >= 0) {
			audio_drain(afd,0);
			close(afd);
		}
		break;
#endif

#ifdef DAC_LINUX
	case LINUXDAC:
		if (afd >= 0) close(afd);
		break;
#endif

#ifdef DAC_MASSCOMP
	case MASS12:
		if (afd > 0)
			mrclose(afd);
		if (cfd > 0)
			mrclose(cfd);
		cfd = -1;
		break;
#endif

#ifdef DAC_EXTEND
	case XREPLAY:
		break;
#endif

	}

	/* clear out static area */
	dactype = -1;
	dacsubtype = 0;
	lastbuff = NULL;
	lastnumf = 0;
	if (prepbuff) free(prepbuff);
	prepbuff=NULL;
	prepnumf=0;
	dac_selected_rate = 0;
	dac_available_channels = 0;
	afd = -1;
}

/* select sampling frequency */
#ifdef DAC_SUN16
/* Sun SPARC-10 dbri playback frequency table */
static int	sunfrq[]={8000,9600,11025,16000,18900,
			22050,32000,37800,44100,48000,0};

/* get nearest sample rate */
static int	sunnearfreq(srate)
double	srate;
{
	int	i;
	int	minf,idx;

	idx=0;
	minf=abs(sunfrq[0]-(int)srate);
	for (i=0;sunfrq[i];i++) {
		if (abs(sunfrq[i]-(int)srate)<minf) {
			minf=abs(sunfrq[i]-(int)srate);
			idx=i;
		}
	}
	return(sunfrq[idx]);
}
#endif

#ifdef DAC_SUN8
/* U-law sample conversion from 12 to 8 bits */	
#define	UMAX      ((1 << 13) - 1)

static unsigned char u_law(in)
short	in;
{
	register unsigned char out;

	if (in < 0) {
		out = (unsigned char) 0;
 		in = -in;
    	}
    	else
		out = (unsigned char) 0x80;

	in += 33;

	if (in > UMAX) in = UMAX;

	if (in < 0x200) {
		if (in < 0x080) {
			if (in < 0x040) {
				if (in < 0x20)
					in = 0;
				else
					in = (in >> 1) ^ 0x10;
            		}
			else
				in = (in >> 2);
        	}
	        else {
			if (in < 0x100)
		                in = (in >> 3) ^ 0x30;
            		else
                		in = (in >> 4) ^ 0x20;
        	}
    	}
    	else {
		if (in < 0x800) {
			if (in < 0x400) 
				in = (in >> 5) ^ 0x50;
			else
				in = (in >> 6) ^ 0x40;
		}
		else {
		 	if (in < 0x1000)
				in = (in >> 7) ^ 0x70;
			else
				in = (in >> 8) ^ 0x60;
        	}
    	}
	out |= (unsigned char) in;
	out = ~out;  
	return (out);
}
#endif

#ifdef DAC_SUN16
/* 85 -- waveform sample rate change times 8 divided by 5 */

/* M.A.Huckvale - University College London */

/* version 1.0 - May 1993 */

/* basic sizes */
#define NCOEFF		48	/* 6 multiplies/output sample */
#define CBUFSIZE	8	/* circular buffer memory size
					> max number multiplies/output */
#define CBUFMASK	7	/* bit mask on circular buffer index */

/* static memory */
static float	coeff[NCOEFF];	/* FIR filter = LP @ 0.1 */
static int	pc;		/* position in coefficients */
static short	cbuf[CBUFSIZE];	/* circular buffer memory */
static int	pb;		/* position in circular buffer */

/* initialise */
static void init_resamp_85()
{
	int	i,nc;

	nc = NCOEFF/2;
	for (i=0;i<nc;i++)
		coeff[NCOEFF/2+i] = coeff[NCOEFF/2-i-1] =
			(8.0 * sin((i+0.5)*2.0*3.14159265358/20.0)/
				((i+0.5)*3.14159265358)) *
			(0.54+0.46*cos((i+0.5)*3.14159*2.0/NCOEFF));

	memset(cbuf,0,sizeof(cbuf));
	pc=0;
	pb=CBUFSIZE;
}

/* do resampling */
static int resamp_85(obuf,ibuf,ilen,nbits)	/* returns output count */
short	*obuf;		/* output buffer */
short	*ibuf;		/* input buffer */
int	ilen;		/* input count */
int	nbits;		/* 12 or 16, shift if 12 */
{
	short	*optr=obuf;
	int	ic;
	int	ib;
	float	sum;

	while (ilen-- > 0) {
		/* put input sample into circular buffer */
		cbuf[(pb++ & CBUFMASK)] = *ibuf++;

		/* loop through 1 or 2 output samples */
		do {
			/* find first coefficient */
			ic = pc;

			/* do multiply back through memory */
			ib = pb - 1;
			sum = 0;
			while (ic < NCOEFF) {
				sum += coeff[ic] * cbuf[ib-- & CBUFMASK];
				ic += 8;
			}

			/* save output sample */
			if (nbits==12)
				*optr++ = ((short)sum) << 4;
			else
				*optr++ = (short)sum;

			/* move on to next coefficient start */
			pc += 5;

		} while (pc < 8);

		/* reset coefficient start */
		pc = pc % 8;

	}

	return(optr-obuf);
}
#endif

#ifdef DAC_ANSI

/* spcomp -- speech waveform compression into text stream */

/* M.A.Huckvale - University College London */

/* version 2.0 - February 1993 */

/*
Compression method:
1)	Encode differences between subsequent samples as sequence
	of printable characters using one or two characters
	between x20 and x6F ASCII
2)	If the difference is too large to encode in one or two
	characters, then encode absolute sample value in
	three bytes as 0b0111???? (low 4bits), 0b01?????? (low 6bits),
	0b01?????? (low 6bits).
3)	One and two byte difference coding:
	Terminating codes:
		0x20 -> 0x5F 		0,..,31,-32,..,-1
	Non-terminating codes:
		0x60 -> 0x67		64,128,192,..,512
		0x68 -> 0x6F		-512,..,-192,-128,-64
	Differences between -32 & +31 are encoded as one byte.
	Differences between -512 & -33, and 32 & 511 are encoded as
	a single non-terminating code and a single terminating code.
*/

#include "spcomp.h"

/* static data */
#define SPCBUFSIZE 8192
static short lastsamp;
static unsigned char spcompbuf[SPCBUFSIZE];
static unsigned char *spptr;
static int  spcnt;

#define PUTCODE(f,c)	((--spcnt > 0) ? (*spptr++ = (c)) : flushbuf(f,(c)))

/* write data to channel */
static int flushbuf(fd,v)
int fd;
int v;
{
	*spptr++ = v;
	write(fd,spcompbuf,spptr-spcompbuf);
	spptr = spcompbuf;
	spcnt = SPCBUFSIZE-1;
	return(v);
}

/* initialise a buffer transmission */
static int 	spcomp_init(fd,size,nbit,nchan)
int	fd;		/* file descriptor */
int32	size;		/* # samples */
int	nbit;		/* # bits/sample */
int	nchan;		/* # channels */
{
	/* ANSI commands: y = # bits & # channels; z= # samples to folow */
	sprintf(spcompbuf,"\033[%d;%dy\033[%dz",nbit,nchan,size);
	spptr = spcompbuf + strlen(spcompbuf);
	spcnt = SPCBUFSIZE-strlen(spcompbuf)-1;
	lastsamp = 0;
	return(0);
}

/* actually transmit one buffer section */
static int 	spcomp_buffer(fd,spbuff,count)
int	fd;		/* file descriptor */
short	*spbuff;	/* sample buffer */
int32	count;		/* # samples in buffer */
{
	register int	diff;

	while (count-- > 0) {
		diff = *spbuff - lastsamp;
		lastsamp = *spbuff++;
		if ((diff < SAMP_DIFF_MIN) || (diff > SAMP_DIFF_MAX)) {
			PUTCODE(fd,0x70 + ((lastsamp & 0xF000) >> 12));
			PUTCODE(fd,0x40 + ((lastsamp & 0x0FC0) >> 6));
			PUTCODE(fd,0x40 + (lastsamp & 0x003F));
		}
		else {
			register int	code1,code2;
			code2 = CODE2(diff);
			if ((code1 = CODE1(diff)) != 0) {
				PUTCODE(fd,code1);
				PUTCODE(fd,code2);
			}
			else
				PUTCODE(fd,code2);
		}
	}
	return(0);
}

/* end of buffer transmission */
static int	spcomp_end(fd)
int	fd;
{
	flushbuf(fd,SAMP_EOT);
	return(0);
}

/* request replay of part of buffer */
int	spcomp_replay(fd,offset,numf,srate,ntime)
int	fd;		/* file descriptor */
int	offset;		/* sample offset into buffer */
int	numf;		/* number of samples to replay */
int	srate;		/* sampling rate */
int	ntime;		/* number of times to replay */
{
	int	len;

	/* check buffer transmission completed */
	if (spptr != spcompbuf) flushbuf(SAMP_EOT);

	/* ANSI commands: w= offset, #samples; v= rate, #times */
	sprintf(spcompbuf,"\033[%d;%dw\033[%d;%dv",offset,numf,srate,ntime);
	len=strlen(spcompbuf);
	write(fd,spcompbuf,len);
	spptr = spcompbuf;
	spcnt = SPCBUFSIZE-1;
}
#endif

static char	repbuf[REPBUFSIZE];
#define SREPBUFSIZE	(REPBUFSIZE/2)

/* replay sample buffer */
int32	dac_playback(buff,numf,srate,nbits,nchannels,ntimes)
short	*buff;		/* sample buffer */
int32	numf;		/* number of samples */
double	srate;		/* sample rate */
int	nbits;		/* number of bits
			   (12 or 16 only - always right shifted in word */
int	nchannels;	/* # channels to replay, NOTE: if 2 channels sent,
			   - should 1 channel devices play mono or one channel ?
			   - have subsequent dac_(get/set)config() routines ?
			 */
int	ntimes;		/* number of times to replay */
{
	int	i,rep;
	short	*srepbuf=(short *)repbuf;
#ifdef DAC_SUN16
	int	sun16resamp=0,port;
#endif
#ifdef DAC_MASSCOMP
	double	dtmp;
#endif

#ifdef DAC_LINUX
 	/* linux variables */
 	char *data;
	int stereo, bytesToWrite, bufferSize, sampleRate;
 	int32 noBytes;
#endif

	/* keep record of this */
	lastbuff = buff;
	lastnumf = numf;
	lastsrate = srate;
	lastnbits = nbits;
	lastnchannels = nchannels;

	/* device specific processing */
	switch (dactype) {
#ifdef DAC_SUN8
	case SUN8:
		/* Sun 8-bit ulaw device */

		/* get Sun audio parameters -- IS THIS CODE RIGHT ? */
		AUDIO_INITINFO(&audinfo);
		if (ioctl(afd,AUDIO_SETINFO,&audinfo) < 0) {
			fprintf(stderr,"dac_playback: cannot set audio output parameters\n");
			return(-1);
		}

		/* select sampling rate */
		dac_selected_rate = (double)audinfo.play.sample_rate;

		/* OK do conversion */
		for (rep=0;rep<ntimes;rep++) {
			int32	stepsize;
			int32	offset;
			int	rpos;

			/* calculate step size - 8 bit fixed point */
			stepsize = (int32)(0.5+256.0*srate/dac_selected_rate);

			/* now do conversion on the fly */
			offset=0;
			rpos=0;
			while ((offset >> 8) < numf) {
				/* copy one sample to output */
				if (nbits==16)
					repbuf[rpos] = u_law(buff[offset >> 8] >> 4);
				else
					repbuf[rpos] = u_law(buff[offset >> 8]);

				/* move on to next input position */
				offset += stepsize;

				/* move on to next output position */
				rpos++;
				if (rpos == REPBUFSIZE) {
					/* replay buffer */
					if (write(afd,repbuf,rpos) != rpos) {
						fprintf(stderr,"dac_playback: failed to write to audio\n");
						return(-1);
					}
					rpos=0;
				}
			}

			/* replay anything that's left */
			if ((rpos > 0) && (write(afd,repbuf,rpos) != rpos)) {
				fprintf(stderr,"dac_playback: failed to write to audio\n");
				return(-1);
			}
		}

		break;
#endif

#ifdef DAC_SUN16
	case SUN16:
		/* SPARC 16-bit dual-basic-rate ISDN */

		/* get Sun audio parameters */
		audio_drain(afd,0);
		audio_get_play_config(afd,&devhdr);
		bufhdr = devhdr;

		/* special re-sampling for common 10000 and 20000 rates */
		if (((int)srate+5)/10==1000) {
			sun16resamp=1;
			bufhdr.sample_rate = 16000;
			init_resamp_85();
		}
		else if (((int)srate+5)/10==2000) {
			sun16resamp=1;
			bufhdr.sample_rate = 32000;
			init_resamp_85();
		}
		else
			bufhdr.sample_rate = sunnearfreq(srate);

		/* set up equivalent header for buffer */
		bufhdr.samples_per_unit = 1;
		bufhdr.bytes_per_unit = 2;
		bufhdr.channels = nchannels;
		bufhdr.encoding = AUDIO_ENCODING_LINEAR;
		bufhdr.data_size = (sun16resamp)? 8* numf / 5 : numf;

		/* match the two */
		if (audio_cmp_hdr(&devhdr, &bufhdr) != 0) {
			switch (audio_set_play_config(afd,&bufhdr)) {
			case AUDIO_SUCCESS:
				break;
			default:
				fprintf(stderr,"dac_playback:couldn't set audio param (%d,%d)\n",
					bufhdr.sample_rate,bufhdr.channels);
				return(-1);
			}
		}
		if (sun16resamp)
			dac_selected_rate = (double)bufhdr.sample_rate * 5.0 / 8.0;
		else
			dac_selected_rate = (double)bufhdr.sample_rate;

		/* select output port */
		switch (dacsubtype) {
		case 0:
		default:
			port = AUDIO_SPEAKER;
			break;
		case 1:
			port = AUDIO_HEADPHONE;
			break;
		case 2:
			port = AUDIO_LINE_OUT;
			break;
		}
		audio_set_play_port(afd,&port);
	
		/* replay buffer */
		for (rep=0;rep<ntimes;rep++) {
			int	nsamp=numf;	/* input samples */
			short	*idata=buff;	/* input buffer */
			short	*rdata;		/* output buffer */
			int	cnt,len,plen;
			short	lastsamp;	/* click removal */

			/* replay the buffer */
			while (nsamp > 0) {
				/* get # samples to replay */
				if (sun16resamp)
					cnt = MIN(nsamp,SREPBUFSIZE/2);
				else
					cnt = MIN(nsamp,SREPBUFSIZE);
				nsamp -= cnt;

				/* check for bit shift */
				if (sun16resamp) {
					plen = resamp_85(srepbuf,idata,cnt,nbits);
					idata += cnt;
					rdata = srepbuf;
					cnt = plen;
				}
				else if (nbits==12) {
					register int 	i;
					short		*sdata;
					for (i=0,rdata=srepbuf,sdata=idata;i<cnt;i++)
						*rdata++ = *sdata++ << 4;
					rdata = srepbuf;
					idata += cnt;
				}
				else {
					rdata = idata;
					idata += cnt;
				}

				/* write to device */
				len = cnt * sizeof(short);
				if (write(afd,rdata,len)!=len) {
					fprintf(stderr,"dac_playback: audio write error\n");
					return(-1);
				}
			}

			/* turn off without a click - use slow decay to zero */
			lastsamp = rdata[cnt-1];
			cnt=0;
			while ((lastsamp != 0) && (cnt < SREPBUFSIZE)) {
				srepbuf[cnt++] = lastsamp;
				lastsamp = (2*lastsamp)/3;
			}
			if (cnt > 0)
				write(afd,srepbuf,cnt*2);
		}

		break;
#endif

#ifdef DAC_LINUX
	case LINUXDAC:
		/* set up stereo/mono mode */
		if (nchannels>1) stereo=1; else stereo=0;
		if (ioctl(afd, SNDCTL_DSP_STEREO, &stereo) < 0) {
			fprintf(stderr,"sfsdac: couldn't set mono/stereo mode\n");
			return(-1);
		}

		/* set up the number of bits per sample */
		nbits=16;
		if (ioctl(afd, SNDCTL_DSP_SAMPLESIZE, &nbits) < 0) {
			fprintf(stderr,"sfsdac: couldn't set the sample size\n");
			return(-1);
		}

		/* set up sample rate */
		sampleRate=(int)srate;
		if (ioctl(afd, SNDCTL_DSP_SPEED, &sampleRate) < 0) {
			fprintf(stderr,"sfsdac: couldn't set the sample rate\n");
			return(-1);
		}
		
		/* get buffer size */
		ioctl (afd,SNDCTL_DSP_GETBLKSIZE,&bufferSize);
		
		data=buff;
		noBytes=numf*2;
		while (noBytes > 0) {
			if (noBytes>bufferSize)
				bytesToWrite=bufferSize;
			else 
				bytesToWrite=noBytes;

			if (write(afd, data, bytesToWrite) != bytesToWrite) {
				fprintf(stderr,"write error to audio device\n");
				break;
			}
		    
			noBytes-=bytesToWrite;
			data+=bytesToWrite;
		}
		
		/* ensure that all data are written to the device */
		ioctl(afd,SNDCTL_DSP_SYNC,NULL);
		break;
#endif

#ifdef DAC_MASSCOMP
	case MASS12:
		/* Masscomp DA08, 8-channel 12-bit */
		/* set up clock frequency */
#define NEAREST 0
#define SQUARE 4
#define LOW 0
		if (mrclk1(cfd,NEAREST,srate,&dac_selected_rate,SQUARE,0.0,&dtmp,LOW) < 0)
			return(-1);
#ifdef IAG
fprintf(stderr,"selected rate=%g\n",dac_selected_rate);
#endif
		if (mrclktrig(afd,1,cfd))
			return(-2);

		/* set up d-a modes and channels */
#define BIPOLAR 0
#define TWOS 0
		if (mrdamod(afd,BIPOLAR,TWOS) < 0)
			return(-3);
		if (mrdainc(afd,dacsubtype,nchannels,0,0) < 0)
			return(-4);

		/* do replay */
		for (rep=0;rep < ntimes;rep++) {
			int	buflen;		/* # samples in each buffer */
			int	buftime;	/* buffer duration in ms. */
			int	nsamp	;	/* # input samples */
			short	*idata;		/* input buffer */
			short	*rdata;		/* output buffer */

			/* set up replay buffering */
			mrlock(repbuf,REPBUFSIZE,&i);
			if (mrbufall(afd,srepbuf,MASSCOMP_BUFSPLIT,(REPBUFSIZE/MASSCOMP_BUFSPLIT)) < 0)
				return(-5);

			/* get buffer size and number */
			buflen = SREPBUFSIZE/MASSCOMP_BUFSPLIT;
			buftime = (int) (1000.0 * buflen) / dac_selected_rate;

			/* fill initial set of buffers */
			for (i=0,nsamp=numf,idata=buff,rdata=srepbuf;i<SREPBUFSIZE;i++,nsamp--) {
				if (nsamp > 0)
					*rdata++ = (nbits==12) ? *idata++ : (*idata++) >> 4;
				else
					*rdata++ = 0;
			}
#ifdef IAG
fprintf(stderr,"doing mrxoutq, afd=%d buflen=%d numf=%d cfd=%d\n",afd,buflen,numf,cfd);		
#endif
			/* initiate transfer */
			if (mrxoutq(afd,buflen,numf,NULL) < 0)
				return(-6);

			/* wait for buffers to come free and refill them */
			if (nsamp > 0) do {
#ifdef IAG
fprintf("waiting\n");
#endif
				/* wait a maximum of all bufers played */
				if (mrbufwt(afd,MASSCOMP_BUFSPLIT*buftime) < 0)
					break;	/* timeout */

				/* get freed buffer */
#define SHORTPTR 0
				mrbufget(afd,SHORTPTR,&rdata);

				/* refill it */
				if (nsamp > 0) {
					for (i=0;i<buflen;i++,nsamp--) {
						if (nsamp > 0)
							*rdata++ = (nbits==12) ? *idata++ : (*idata++) >> 4;
						else
							*rdata++ = 0;
					}
					mrbufrel(afd,rdata-buflen);
				}
				else
					break;
			} while (nsamp > 0);
#ifdef IAG
fprintf(stderr,"ends\n");
#endif
			/* wait for it all to stop */
			if (mrevwt(afd,0,MASSCOMP_BUFSPLIT*buftime) < 0)
				return(-7);

			/* reset queued transfer */
			mrreset(afd,-1);

		}

		break;
#endif

#ifdef DAC_ANSI
	case ANSI:
		/* character format, stdout */

		/* send buffer first - once */
		spcomp_init(1,numf,(dacsubtype==0)?16:12,nchannels);
		{
			int	nsamp=numf;	/* input samples */
			short	*idata=buff;	/* input buffer */
			short	*rdata;		/* output buffer */
			int	cnt;

			while (nsamp > 0) {
				/* get # samples to send */
				cnt = MIN(nsamp,SREPBUFSIZE);

				/* check for bit shift */
				if ((dacsubtype==0) && (nbits==12)) {
					/* shift UP */
					register int 	i;
					short		*sdata;
					for (i=0,rdata=srepbuf,sdata=idata;i<cnt;i++)
						*rdata++ = *sdata++ << 4;
					rdata = srepbuf;
				}
				else if ((dacsubtype==1) && (nbits==16)) {
					/* shift DOWN */
					register int 	i;
					short		*sdata;
					for (i=0,rdata=srepbuf,sdata=idata;i<cnt;i++)
						*rdata++ = *sdata++ >> 4;
					rdata = srepbuf;
				}
				else
					rdata = idata;

				/* write to device */
				spcomp_buffer(1,rdata,cnt);
				idata += cnt;
				nsamp -= cnt;
			}
		}
		spcomp_end(1);

		/* now replay required # times */
		if (ntimes > 0)
			spcomp_replay(1,0,numf,(int)srate,ntimes);

		break;
#endif

#ifdef DOS
	case DOSDAC:
		return(dos_dac_playback(dacsubtype,buff,numf,srate,nbits,nchannels,ntimes));
		break;
#endif

#ifdef DAC_EXTEND
	case XREPLAY:
		if (extend_open()) {
			fprintf(stderr,"sfsdac: Could not connect to Vista Extend\n");
			return(-2);
		}
		dac_selected_rate = srate;	/* not actually true .. */
		if (extend_load(buff,numf,srate,nbits,1))
			return(-1);
		for (i=0;i<ntimes;i++)
			if (extend_play(0,numf))
				return(-1);
		extend_close();
		break;
#endif

#ifdef DAC_PIPE
	case PIPE: {
		FILE *op;
	        char pcmd[80];
		sprintf(pcmd,"%s/bin/dacpipe %d %d %d",sfsbase(),(int)srate,(int)srate/1000,(int)numf);
		if ((op=popen(pcmd,"w"))) {
			int len;
			while (numf > 0) {
				len = fwrite(buff,2,numf,op);
				if (len <= 0) break;
				numf -= len;
				buff += len;
			}
			pclose(op);
		}
		dac_selected_rate = srate;	/* not actually true .. */
	      }
		break;
#endif

	default:
		/* unknown, or unsupported */
		return(-1);
	}
	/* 0 is OK */
	return(0);
}	

/* repeat a section of immediately previous buffer */
int32	dac_repeat(offset,numf,ntime)
int32	offset;		/* sample offset into last buffer */
int32	numf;		/* number of frames to replay */
int	ntime;		/* number of times to replay */
{
	double		srate;

	/* check something to replay */
	if (!lastbuff)
		return(-1);

	/* if using prepared buffer, change parameters to match */
	if (lastbuff==prepbuff) {
		offset = (int)((offset*prepsrate)/lastsrate);
		numf = (int)((numf*prepsrate)/lastsrate);
		/* check values */
		if ((offset+numf) > prepnumf)
			return(-1);
		srate = prepsrate;
	}
	else {
		/* check values */
		if ((offset+numf) > lastnumf) 	
			return(-1);
		srate = lastsrate;
	}

	/* OK to do it */
	switch (dactype) {
#ifdef DAC_SUN8
	case SUN8:
		/* SPARC 8-bit ulaw device */
		return(dac_playback(lastbuff+offset,numf,srate,lastnbits,lastnchannels,ntime));
		break;
#endif

#ifdef DAC_SUN16
	case SUN16:
		/* SPARC 16-bit dual-basic-rate ISDN */
		return(dac_playback(lastbuff+offset,numf,srate,lastnbits,lastnchannels,ntime));
		break;
#endif

#ifdef DAC_LINUX
	case LINUXDAC:
		return(dac_playback(lastbuff+offset,numf,srate,lastnbits,lastnchannels,ntime));
		break;
#endif

#ifdef DAC_MASSCOMP
	case MASS12:
		/* Masscomp DA08, 8-channel 12-bit */
		return(dac_playback(lastbuff+offset,numf,srate,lastnbits,lastnchannels,ntime));
		break;
#endif

#ifdef DAC_ANSI
	case ANSI:
		/* character format, stdout */
		spcomp_replay(1,offset,numf,srate,ntime);
		break;
#endif

#ifdef DAC_EXTEND
	case XREPLAY:
		return(dac_playback(lastbuff+offset,numf,srate,lastnbits,lastnchannels,ntime));
		break;
#endif
	default:
		/* unknown, or unsupported */
		return(-1);
	}
	return(0);
}

/* SFS compatibility with old playback routine */
void	playback(buff,numf,srate)
short	*buff;		/* sample buffer */
int32	numf;		/* number of samples */
double	srate;		/* sample rate */
{
	if (dac_open(NULL) > 0) {
		dac_playback(buff,numf,srate,16,1,1);
		dac_close(0);
	}
}

#ifdef IAG
char *progname="sfsdac";
main(argc,argv)
int	argc;
char	*argv[];
{
	short	*sp;
	struct item_header item;
	int	code;

	if (argc!=2)
		error("usage: sfsdac sfsfile");

	getitem(argv[1],SP_TYPE,"0",&item,&sp);
	
	code = dac_open(NULL);
	printf("dac_open returns %d\n",code);

	code=dac_playback(sp,item.numframes,1.0/item.frameduration,16,1,1);
	printf("dac_playback returns %d\n",code);
	printf("sampling rate was %gHz\n",dac_selected_rate);
	
	dac_close(0);

	free(sp);
	exit(0);
}	
#endif
