/* abxplay -- play stimuli in ABX experiment */

/* M.A.Huckvale - University College London */

#define PROGNAME "abxplay"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH ABXPLAY SFS1 UCL
.SH NAME
abxplay -- play out stimuli for perceptual experiments
.SH SYNOPSIS
.B abxplay
(-i item) (-t interval) (-e endpause) file-list
.SH DESCRIPTION
.I abxplay
is a program to concatenate speech items and replay them with fixed timing,
it is used to generate stimuli for perceptual experiments.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number (applies to ALL files).
.TP 11
.BI -t interval
Specify the inter-stimulus pause in seconds.  Default 0.4.
.TP 11
.BI -e endpause
Specify the after-stimulus pause in seconds.  Default 0.0.
.SH INPUT ITEMS
.IP SP
Speech signals
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include "sfs.h"

#define INTERVAL_TIME	0.4

double	interval=INTERVAL_TIME;
double	endpause=0.0;

#define	MAXFILE	10
int			numfile=0;
int			fid[MAXFILE];
struct item_header	spitem[MAXFILE];
short			*spbuff;
int			sprate=0;
int			spcnt;

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char		*ty;		/* item sub type */
	char		*sptype="0";
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		rate;
	int		i;
	short		*p;
	int		nbits=16;
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:t:e:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Play ABX stimuli V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 't' :	/* interval time */
			interval = atof(optarg);
			break;
		case 'e' :	/* end pause */
			endpause = atof(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-t interval) (-e endpause) files",PROGNAME);

	/* get filenames */
	numfile=0;
	while ((optind < argc) && (numfile < MAXFILE)) {
		strcpy(filename,sfsfile(argv[optind]));
		if ((fid[numfile]=sfsopen(filename,"r",NULL))<0)
			error("could not open '%s'",filename);
		if (!sfsitem(fid[numfile],SP_TYPE,sptype,&spitem[numfile]))
			error("no SP in '%s'",filename);
		rate = 10*(int)(0.5+0.1/spitem[numfile].frameduration);
		if (sprate && (sprate!=rate))
			error("sampling rates do not match '%s'",filename);
		else
			sprate = rate;
		nbits = (int)param(spitem[numfile].params,"bits",16.0);
		numfile++;
		optind++;
	}

	if (numfile==0)
		error("no files specified");

	/* allocate memory for signal */
	spcnt = 0;
	spcnt += spitem[0].numframes;
	for (i=1;i<numfile;i++) {
		spcnt += (int)(interval * sprate);
		spcnt += spitem[i].numframes;
	}
	spcnt += endpause * sprate;

	if ((spbuff=(short *)calloc(spcnt,sizeof(short)))==NULL)
		error("could not get buffer for signal");

	/* load in waveforms */
	p = spbuff;
	sfsread(fid[0],0,spitem[0].numframes,p);
	p += spitem[0].numframes;
	sfsclose(fid[0]);
	for (i=1;i<numfile;i++) {
		p += (int)(interval*sprate);
		sfsread(fid[i],0,spitem[i].numframes,p);
		p += spitem[i].numframes;
		sfsclose(fid[i]);
	}

	/* now we can replay ! */
	if (dac_open(0)>=0) {
		dac_playback(spbuff,spcnt,(double)sprate,
			nbits,1,1);
		dac_close(0);
	}

	/* that's all folks ! */
	free(spbuff);
	exit(0);
}
