/* fx -- convert tx data into fx contour */

/* M.A.Huckvale - January 1988 */
/* Peter Davies, 7/88; allow min & max frequencies as options */

/* version 2.0 - April 1991
	- add median smoothing option
*/
/* version 2.1 - Aug 1991
	- put altered frame rate in history */
#define PROGNAME "fx"
#define PROGVERS "2.0s"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH FX SFS1 UCL SFS
.SH NAME
fx -- convert a tx item to an fx item
.SH SYNOPSIS
.B fx
(-i item) (-f sfreq) (-m minfreq) (-M maxfreq) (-s) file
.SH DESCRIPTION
.I fx
is a program to convert a tx item into an fx item.  Output is 
on 5ms frames (200 frames/sec) but may be changed by a command line
option. Frequencies outside the minimum and maximum frequency limits
are stored as zeroes.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item.
.TP 11
.BI -f sfreq
Select sampling frequency of fx item. Default 200.
.TP 11
.BI -m minfreq
Select minimum frequency. Default 40 Hz.
.TP 11
.BI -M maxfreq
Select maximum frequency. Default 800 Hz.
.TP 11
.B -s
Smooth the final contour with a five point median filter.
.SH INPUT ITEMS
.IP TX.xx 11
Any excitation period data item.
.SH OUTPUT ITEMS
.IP FX 11
Fx from Tx.
.SH VERSION/AUTHOR
.IP 1.0 11
Mark Huckvale
.IP 1.1 11
Peter Davies
.IP 2.0 11
Mark Huckvale
.SH BUGS
Tx contour is sampled, not averaged.
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"		/* database structures */

/* manifest constants */
#define FXFRAMEDURATION 0.005	/* fx sample duration */
#define MINFREQ 40		/* 40Hz minimum fx */
#define MAXFREQ 800		/* 800Hz maximum fx */

int minfreq = MINFREQ;
int maxfreq = MAXFREQ;
int	medfilt=0;

/* median filter */
int	filter[16];

int medianfilt(buf,cnt)
int	*buf;
int	cnt;
{
	int	i,j;

	filter[0] = *buf++;

	for (i=1;i<cnt;i++) {
		j = i;
		while ((j > 0) && (filter[j-1] > *buf)) {
			filter[j] = filter[j-1];
			j--;
		}
		filter[j] = *buf++;
	}
	return(filter[cnt/2]);
}

/* output with median filter */
int medmem[5];
int memfil;

void filtwrite(ofid,fxval)
int	ofid;
short	fxval;
{
	medmem[memfil++]=fxval;
	if (memfil==5) {
		fxval=medianfilt(medmem,5);
		if (sfswrite(ofid,1,&fxval)!=1)
			error("write error on output file");
		medmem[0]=medmem[1];
		medmem[1]=medmem[2];
		medmem[2]=medmem[3];
		medmem[3]=medmem[4];
		memfil=4;
	}
}

/* process tx data into fx */
void processtx(fid,ofid,tdur,fdur,medfilt)
int	fid,ofid;		/* input/output file descriptors */
double	tdur,fdur;		/* frame units (s) */
int	medfilt;		/* median filter */
{
	/* processing variables */
	int		i,tx;			/* frame count, tx value */
	double		fxtime=0,txtime=0;	/* fx,tx times (s) */
	double		txval;			/* tx period (s) */
	unsigned short	fxval;			/* fx frequency (Hz) */

	/* init median filter */
	if (medfilt) {
		filtwrite(ofid,0);
		filtwrite(ofid,0);
	}

	/* process tx sample by sample */
	for (i=0; sfsread(fid,i,1,&tx); i++) {

		/* get tx period as time in seconds */
		txval = tx * tdur;
		txtime += txval;

		/* calc fx frequency */
		if (txval > 0.0)
			fxval = 1.0/txval;
		else
			fxval = 0;
		if ((fxval < minfreq) || (fxval > maxfreq)) fxval = 0;

		/* write frequency to file */
		while (fxtime <= txtime) {
			/* write fx frequency to output file */
			if (medfilt) {
				filtwrite(ofid,fxval);
			}
			else {
				if (sfswrite(ofid,1,&fxval) != 1)
					error("write error on temporary file",NULL);
			}
			fxtime += fdur;
		}
	}

	/* flush median filter */
	if (medfilt) {
		filtwrite(ofid,0);
		filtwrite(ofid,0);
	}
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item type selection */
	char		*ty;		/* item match selection */
	char		*txtype="0";
	/* input file variables */
	char		filename[SFSMAXFILENAME]; /* sfs file name */
	int		fid;		/* input file descriptor */
	struct item_header txitem;	/* item header for tx data */
	/* output file variables */
	int		ofid;		/* output file descriptor */
	struct item_header fxitem;	/* item header for fx data */
	double		fdur=FXFRAMEDURATION;
	int		freq=0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:f:m:M:s")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Tx to Fx V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == TX_TYPE) 
					txtype=ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'f' :	/* output sampling freq */
			if ((freq=atoi(optarg)) <= 0)
				error("illegal sampling frequency: %s",optarg);
			fdur = 1.0/(double)freq;
			break;
		case 'm':
			sscanf(optarg,"%d",&minfreq);
			break;
		case 'M':
			sscanf(optarg,"%d",&maxfreq);
			break;
		case 's':
			medfilt++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-f frame rate) (-m minfreq) (-M maxfreq) (-s) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no file specified",NULL);

	/* open data file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("cannot find file: %s",filename);
		else
			error("access error on %s",filename);
	}

	/* locate input tx */
	if (!sfsitem(fid,TX_TYPE,txtype,&txitem))
		error("cannot find TX data in %s",filename);

	/* create output item header */
	sfsheader(&fxitem,FX_TYPE,0,2,1,fdur,txitem.offset,1,0,0);
	if (medfilt) {
	 	char add[100];
		sprintf(fxitem.history,"%s(%d.%02d;m=%d,M=%d,smoothed",
			PROGNAME,txitem.datatype,txitem.subtype,minfreq,maxfreq);
		if (fdur != FXFRAMEDURATION) {
		    sprintf(add,",frame rate = %d",freq);
		    strcat(fxitem.history,add);
	 	}
		strcat(fxitem.history,")");
	}
	else {
	        char add[100];
		sprintf(fxitem.history,"%s(%d.%02d;m=%d,M=%d",
			PROGNAME,txitem.datatype,txitem.subtype,minfreq,maxfreq);
		if (fdur != FXFRAMEDURATION) {
		    sprintf(add,",frame rate = %d",freq);
		    strcat(fxitem.history,add);
	 	}
		strcat(fxitem.history,")");
	}
	/* open output channel */
	if ((ofid=sfschannel(filename,&fxitem)) < 0)
		error("cannot open output channel",NULL);

	/* do processing */
	processtx(fid,ofid,txitem.frameduration,fdur,medfilt);

	/* update data file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks */
	exit(0);
}


