/* **************************************** */
/* * (c) D.M.Howard May 1986 - SPAR GROUP * */
/* * Phonetics & Linguistics, UCL         * */
/* **************************************** */
/* *************************************************** */
/* * pp - digital implementation of the peak-picker  * */
/* *      (Howard, 1985; Howard and Fourcin, 1983)   * */
/* *************************************************** */
/* * version 1.0 by.........                         * */
/* *                     D.M. Howard - Sept 1985     * */
/* *                     pascal version by A. Eaton  * */
/* *************************************************** */
/* * version 2.0 by.........                         * */
/* *                     M.A. Huckvale - Jan 1986    * */
/* *       converted to the new d/base system        * */             
/* *************************************************** */
/* * version 2.1 by..........                        * */
/* *                     D.M. Howard - March 1986    * */
/* *    - writes tx item to file                     * */
/* *    - reads sampling rate from i/p file          * */
/* *    - now split up into subroutines              * */
/* *    - see makefile for details of sources        * */
/* *************************************************** */
/* version 2.2 - November 1986
	1. new item update
*/
/* version 3.0 - March 1991 - Mark Huckvale
	- SFS version
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH PP 1 UCL SFS
.SH NAME
pp - peak picker (Howard D.M. - PhD thesis, London University, 1985)
.SH SYNOPSIS
.B pp
(-I) (-p) (-g gain) (-i item) dbase_file
.SH DESCRIPTION
.I pp
is a speech fundamental frequency estimation algorithm, which operates in the time domain.  It has evolved from patient prostheses developed in the EPI group for the profoundly and the totally deaf.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and its version number and exit.
.TP 11
.B -p
Polarity change of input speech first - optimum operation is with POSITIVE pressure changes POSITIVE (default: no inversion of input speech first).
.TP 11
.B -g gain
Set input gain. Value in fact represents a max voltage at a particular point in the system.
This is equivalent to the gain control setting on the analogue device, which is usually set 
with reference to an output lamp (LED) indicating the presence of output pulses.  
This should be remembered when using pp - thus the gain should be user set with 
reference to the output Tx or Fx.  Default 0.25.
.TP 11
.BI -i item
Select input item number.
.SH INPUT ITEMS
.IP 1.xx 11
Any speech item.
.SH OUTPUT ITEMS
.IP 3 11
Peak-picker Tx.
.SH VERSION/AUTHOR
2.1 - David Howard
.br
3.0 - Mark Huckvale
.SH SEE ALSO
tx, cep, gr, sift (ils).
.SH PROPOSED IMPROVEMENTS
Automatic input speech polarity detection if (-P) not set.  
Automatic setting of an appropriate gain if (-G) not set.
.SH BUGS
When you find them, you know who to call!
*/
/*--------------------------------------------------------------------------*/

#define PROGNAME "pp"
#define PROGVERS "3.0"
char *progname=PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"

/* constants for the peak-picker */
#define  VAL_5_VOLTS 2047		/* value for 5 volts */
#define  CUTOFF 450			/* filter cutoff (Hz) */
#define  GAIN_SET 0.25			/* gain control max output voltage - for (-G) option */
#define  LOG_OFFSET 29			/* log_amp 'offset' */
#define  LOG_SLOPE 91			/* log_amp 'slope' */
#define  LOG_MIN_INPUT 7		/* log_amp minimum input (sample value) which gives output */
#define  LOG_CLIP_LEVEL 2.5		/* log_amp clip voltage */
#define  P_PICK_SETUP_FREQ 70		/* peak-picker decay rate - value per sample */
#define	 COMP_DIV_RATIO 560.0		/* comparator potential divider ratio - (560kX over1kX) */
#define  COMP_HIGH 2047.0		/* comparator 'high' output (+5V) */
#define  COMP_LOW 0.0			/* comparator 'low' output (0v) */ 
#ifndef PI
#define  PI 3.141592654
#endif

/* global data */
char	filename[SFSMAXFILENAME];
struct main_header head;		/* main header data */
struct item_header spitem;		/* speech item header data */
char	*inptype="0";			/* default input subtype = last */
float	systemgain=GAIN_SET;

/* ********************************** */
/* * amplifier with gain parameter  * */
/* * works with pp - version 2.1    * */
/* * DMH Feb 1986                   * */
/* ********************************** */

/* amplifier - multiply array by (gain) */
void amplify(data,length,s_rate,gain)
float	*data;
int	length;
float	s_rate;
float	gain;
{
	register float	*pdata;		/* data array pointer */

	for (pdata=data;pdata<(data+length);pdata++)
		*pdata = *pdata * gain;
}

/* ******************************* */
/* * comparator in peak-picker   * */
/* * works with pp - version 2.1 * */
/* * DMH Feb 1986                * */
/* ******************************* */

/* COMPARATOR */
/* comparator - output is either 'high' - (if input is > 'comp_level') */
/*			      or 'low'  - (if input is < 'comp_level') */

void comp(data,length,s_rate)
float	*data;
int	length;
float	s_rate;
{
	register float	comp_level;
	register float	high, low;	/* 'high', 'low' are clipper ouput values */
	register float	*pdata;		/* data array pointer */

	/* set output values 'high, 'low' and 'clip_level' */
	high = COMP_HIGH;
	low = COMP_LOW;
	comp_level = (1.0 * VAL_5_VOLTS) / COMP_DIV_RATIO;	/* (max value (5v)) over (res ratio) */

	/* main routine */
	for (pdata=data;pdata<(data+length);pdata++)
		*pdata = (*pdata > comp_level) ? high : low;
}

/* ************************************** */
/* * gain control as in analogue device * */
/* * works with pp - version 2.1        * */
/* * DMH Feb 1986                       * */
/* ************************************** */

/* GAIN CONTROL */
/* input gain control */
/* finds max value and makes it equal to user input volts (pos or neg) */

float	gain_control(data,length,s_rate)
float	*data;
int	length;
float	s_rate;
{
	float		max;
	float		maxval();
	float 		temp_var;
	float		gain;
	register float	*pdata;		/* data array pointer */

	/* obtain user setting of gain (about 0.25 V) - if (-G) option not set */
	gain = systemgain;

	max = (maxval(data,length,s_rate));
	if ( max > 0.0) {
		temp_var = (gain * VAL_5_VOLTS)/(5.0 * max);
		for (pdata=data;pdata<(data+length);pdata++)
			*pdata = *pdata * temp_var;
	}
	else
		error("Input data all zero.",NULL);

	return(gain);
}

/* ********************************* */
/* * differentiator in peak-picker * */
/* * works with pp - version 2.1   * */
/* * DMH Feb 1986                  * */
/* ********************************* */

/* DIFFERENTIATOR */
/*    - based on a 1pole high-pass filter */
/* with resistor & capacitor values given */
/* ************************************** */
/* Y(n) = (X(n)-X(n-1)+Y(n-1))/k */
/*     where k = 1+(sam_per/R*C) */
/* ***************************** */

void hpdiff(data,length,s_rate,resistor, capacitor)
float	*data;
int	length;
float	s_rate;
float	resistor, capacitor;

{
	double t[2];
	register double *pt, k_fact;		/* routine vars */
	register float	*pdata;				/* data array pointer */

	k_fact = 1.0+(1.0/(s_rate*resistor*capacitor));

	/* main routine */
	*t = 0;
	*(t+1) = 0;
	for (pdata=(data+2),pt=(t+1);pdata<(data+length);pdata++) { /* two points invalid */
		*t = *pt;
		*pt = *pdata - *(pdata-1);
		*pt = (*pt + *t) / k_fact;
		*(pdata-1) = *t; 
	}

}

/* ******************************* */
/* * waveform (float) inverter   * */
/* * works with pp - version 2.1 * */
/* * DMH Feb 1986                * */
/* ******************************* */

/* INVERTER */
/* invert waveform */
void invert(data,length,s_rate)
float	*data;
int	length;
float	s_rate;
{
	register float	*pdata;

	/* carry out inversion */
	for (pdata=data;pdata<(data+length);pdata++)
		*pdata = - *pdata;
}

/* ********************************************** */
/* * low pass Butterworth filter in peak_picker * */
/* * DMH Feb 1986 - UCL                         * */
/* * works with pp - version 2.1                * */
/* ********************************************** */

/* LOW_PASS FILTER*/
/* low pass filter (butterworth 4-pole) */

/* ******************************************* */
/* Y(n)=(X(n)+4X(n-1)+6X(n-2)+4X(n-3)+X(n-4)-  */
/*      4AY(n-1)-6BY(n-2)-4CY(n-3)-DY(n-4))/E  */
/* ******************************************* */

void l_passbut(data,length,sam_freq,c_freq)
float	*data;		/* data array */
int	length;		/* data length */
float	sam_freq;	/* sampling freq */
int	c_freq;		/* cut_off freq */

{
	double		c1, c2, c3, c4;			/* cut-off constant and its powers */
	register double	a, b, c, d, e;			/* higher level constants from above */
	float  	pi, sam_period;			/* sample period variables */
	float t[5];				/* temporary vars */
	double		sn, cs, cut_off;		/* temporary vars */
	register float	*pt;				/* t pointer */
	register float	*pdata;				/* data array pointer */

	/* set up constants to obtain a, b, c, d, and e */
	sam_period = 1.0/sam_freq;
	cut_off = 1.0*c_freq;			/* change to double */
	pi = 1.0*PI;
	sn = sin(pi*cut_off*sam_period);
	cs = cos(pi*cut_off*sam_period);	/* use sin and cos since tan doesn't work in fpp */

	if (cs == 0)
		error("\n\nDBPP (Peak-picker) ERROR : cos() = 0",NULL);

	c1 = sn/cs;				/* cut off const */
	c2 = c1*c1;				/* cut off const squared */
	c3 = c1*c2;				/* cut off const cubed */
	c4 = c1*c3;				/* cut off const to the fourth */

	a = 1.0+(1.3066*((1.0/c1)-(1.0/c3)))-(1.0/c4);
	b = 1.0-(1.047/c2)+(1.0/c4);
	c = 1.0+(1.3066*((1.0/c3)-(1.0/c1)))-(1.0/c4);
	d = 1.0-(2.6131*((1.0/c3)+(1.0/c1)))+(3.141/c2)+(1.0/c4);
	e = 1.0+(2.6131*((1.0/c3)+(1.0/c1)))+(3.141/c2)+(1.0/c4);


	/* set t[] to 0  */
	for (pt=t;pt<(t+5);pt++)
		*pt = 0.0;

	/* main filtering routine */
	for (pdata=(data+4),pt=(t+4);pdata<(data+length);pdata++) {
			/* start at [4] to give 4 earlier points */
			/* the first 4 points in output are invalid */
			/* temp storage of needed output array spaces */
			/* converted to doubles */
		*t = *(t+1);	/* keep these vars as floats */	
		*(t+1) = *(t+2); /* else int rounding can loose all data */
		*(t+2) = *(t+3);
		*(t+3) = *pt;
		
		/* fill the array */
		*pt = (1.0* *pdata) + (4.0* *(pdata-1)) + (6.0* *(pdata-2));
		*pt = *pt + ((4.0* *(pdata-3)) + (1.0* *(pdata-4)));
		*pt = (*pt - (4.0*a* *(t+3)) - (6.0*b* *(t+2)) - (4.0*c* *(t+1)) - (d* *(t+0))) / e;
		*(pdata-4) = *t;		/* put answer in data[] - back out of the way*/
	}

}

/* ************************************************************************ */
/* LOG_LOOK_UP_TABLE */
/* set up look-up table for logarithmic amplifier */
void log_lookup(o_p)
float	*o_p;
{
	register int	i;
	register float	temp, hold;
	register float	slope, offset;
	int		log_max_input;

	slope = LOG_SLOPE * 1.0;
	offset = LOG_OFFSET * 1.0;
	temp = LOG_CLIP_LEVEL * (VAL_5_VOLTS / 5.0);
	log_max_input = temp;

	for (i=0;i<LOG_MIN_INPUT;i++)
		*(o_p+i) = 0.0;

	for (i=LOG_MIN_INPUT;i<log_max_input;i++) {
		hold = 1.0 * i;
		*(o_p+i) = (log(hold) * slope) - offset;
	}

	temp = *(o_p+log_max_input-1);
	for (i=(log_max_input - 1);i<VAL_5_VOLTS;i++)
		*(o_p+i) = temp;
}

/* ************************************************* */
/* * logarithmic amplifier - as in analogue device * */
/* * works with pp - version 2.1                   * */
/* * DMH Feb 1986                                  * */
/* ************************************************* */

/* LOGARITHMIC AMPLIFIER */
/* implemented using a 3-straight-line approximation to the curve given in Howard 1985 */
/* line 1 - any input < LOG_MIN_INPUT and > -LOG_MIN_INPUT takes an output of 0 */
/* line 2 - Y(t) = (logX(t)*LOG_SLOPE)-LOG_OFFSET) */
/* line 3 - any abs(input) > LOG_MAX_INPUT is clipped to Y(LOG_MAX_INPUT) */
/* output values are set up in a look-up table */ 

void logamp(data,length,s_rate)
float	*data;
int	length;
float	s_rate;
{
	int		j;
	float		log_val[VAL_5_VOLTS + 1];
	register float	*pdata;		/* data array pointer */

	/* set up log look-up table */
	log_lookup(log_val);

	/* main routine */
	for (pdata=data;pdata<(data+length);pdata++) {
		if (*pdata < 0.0) {
			j = - *pdata;		
			*pdata = - *(log_val+j);
		}
		else {
			j = *pdata;
			*pdata = *(log_val+j);
		}
	}
	hpdiff(data,length,s_rate,47000.0, 0.00000022);
}

/* ******************************* */
/* * monostable for peak-picker  * */
/* * works with pp - version 2.1 * */
/* * DMH Feb 1986                * */
/* ******************************* */

/* MONOSTABLE */
/* output monostable - takes clipper output and ensures 'high' lasts for only 1 sample */
/* Find each change to 'high', set all other points 'low'. */
void m_stable(data,length,s_rate,ofid)
float	*data;
int	length;
float	s_rate;
int	ofid;
{
	register int	i;
	register float	high, low;		/* comparator output levels */
	register int	last_pulse_at=0;	/* temp store for last pulse pos */
	register float	*pdata;			/* data array pointer */
	int		txval;

	high = COMP_HIGH;
	low = COMP_LOW;

	/* main routine */
	for (pdata=data,i=0;pdata<(data+length);pdata++,i++) {
		if (*pdata == high) {	/* found a pulse */
			txval = i - last_pulse_at;
			sfswrite(ofid,1,&txval);
			last_pulse_at = i;	/* set marker */
			i++;	/* leave only the first value 'high' */
			pdata++;
			while (*pdata == high) {
				*pdata = low;
				i++;
				pdata++;
			}
		}
	}

	/* send end of waveform marker - if V- at end */
	if ((length-last_pulse_at) != 0) {
		txval=length - last_pulse_at;
		sfswrite(ofid,1,&txval);
	}

}

/* ******************************* */
/* * find max val in array       * */
/* * works with pp - version 2.1 * */
/* * DMH Feb 1986                * */
/* ******************************* */

/* MAX_VAL */
/* find max value in array */
float	maxval(data,length,s_rate)
float	*data;
int	length;
float	s_rate;

{
	float		temp;
	register float	*pdata;
	float	maxval=0;

	for (pdata=data;pdata<(data+length);pdata++) {
		if (*pdata < 0.0)
			temp = (-1.0) * *pdata;
		else
			temp = *pdata;
		if (temp > maxval)
			maxval = temp;
	}
	return(maxval);
}

/* ******************************* */
/* * peak-picker stage in pp     * */
/* * works with pp - version 2.1 * */
/* * DMH Feb 1986                * */
/* ******************************* */

/* PEAK_PICKER */
/* peak-picker */
void p_pick(data,length,s_rate)
float	*data;
int	length;
float	s_rate;
{
	register float	pp_decay_per_sample;
	int		diode_bias;		/* voltage drop across diode */
	register float	*pdata;			/* data array pointer */

	/* set up diode bias */
	diode_bias = ((0.4/5.0)*(VAL_5_VOLTS*1.0));

	/* set up decay rate as a value per sample */
	pp_decay_per_sample = ((3.0 / 5.0) * VAL_5_VOLTS) / ((1.0/P_PICK_SETUP_FREQ) * s_rate);

	/* main peak-picking routine */		
	for (pdata=(data+1);pdata<(data+length);pdata++) {
		  		/* first point needed to start off */
		if (*pdata <  (*(pdata-1) - pp_decay_per_sample))
			*pdata = *(pdata-1) - pp_decay_per_sample;
	}
	
}

/* main processing */
int peak_pick(fid,sam_rate,p_flag)
int	fid;		/* file descriptor located on speech */
float	sam_rate;
int	p_flag;	/* option flags */
{
	short	inpbuf[4096];
	float	*float_data;		/* float arrays for accuracy */
	float	*p;
	float	gain_used;			/* gain factor used */
	int	i;
	struct item_header	txhead;
	int	ofid;
	int	pos,len,cnt;

	/* get float buffer */
	if ((float_data=(float *)calloc(spitem.numframes,sizeof(float)))==NULL)
		error("could not get memory buffer for waveform");

	/* load waveform into float array */
	cnt=spitem.numframes;
	pos=0;
	p = float_data;
	while (cnt > 0) {
		if ((len=sfsread(fid,pos,(cnt>4096)?4096:cnt,inpbuf)) <= 0)
			error("read error on '%s'",filename);
		for (i=0;i<len;i++)
			*p++ = (float)inpbuf[i];
		pos += len;
		cnt -= len;
	}
		
	/* *************************** */
	/* main peak-picking algorithm */
	/* *************************** */

	/* run algorithm */
	if (p_flag)	/* invert if -P(olarity) flag set */
		invert(float_data,spitem.numframes,sam_rate);
	l_passbut(float_data,spitem.numframes,sam_rate,CUTOFF);
	gain_used=(gain_control(float_data,spitem.numframes,sam_rate));
	logamp(float_data,spitem.numframes,sam_rate);
	amplify(float_data,spitem.numframes,sam_rate,2.0);
	p_pick(float_data,spitem.numframes,sam_rate);
	hpdiff(float_data,spitem.numframes,sam_rate,100000.0, 0.000000022);
	amplify(float_data,spitem.numframes,sam_rate,2.2);
	p_pick(float_data,spitem.numframes,sam_rate);
	hpdiff(float_data,spitem.numframes,sam_rate,100000.0, 0.000000022);
	comp(float_data,spitem.numframes,sam_rate);

	/* SEND Tx DATA TO FILE */
	/* write Tx output to file */
	/* create tx header */
	sfsheader(&txhead,TX_TYPE,0,4,1,spitem.frameduration,spitem.offset,0,0,1);
	if (p_flag) 
		sprintf(txhead.history,"%s(%d.%02d;gain=%g,inverted)",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			gain_used);
	else
		sprintf(txhead.history,"%s(%d.%02d;gain=%g)",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			gain_used);

	/* open output channel */
	if ((ofid=sfschannel(filename,&txhead)) < 0)
		error("unable to open output channel");

	/* write out TX */
	m_stable(float_data,spitem.numframes,sam_rate,ofid);

	/* and return to main */
	return(0);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* local variables */
	extern int	optind;			/* option index */
	extern char	*optarg;		/* option argument */
	int		polflg=0;		/* polarity switch */
						/* set up option flags */
	int		errflg=0;		/* option error flag */
	char		c;			/* option char */
 	int		it;
	char		*ty;			/* item specifiers */
	int		fid;
	float		s_rate;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ipg:i:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Pitch epoch picker V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'p' :	/* change P(olarity) of Sp first */
			polflg = 1;
			break;
		case 'g' :	/* use provided G(ain) setting */
			systemgain = atof(optarg);
			if (systemgain <= 0.0)
				error("system gain must be positive");
			break;
		case 'i' :	/* specific input item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					inptype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: pp (-I) (-i item) (-p) (-g gain) file\n",NULL);

	/* get filename */
	if (optind < argc) {
		strcpy(filename,sfsfile(argv[optind]));
	}
	else
		error("no database file specified",NULL);

	/* find input speech item */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on '%s'",filename);
	if (!sfsitem(fid,SP_TYPE,inptype,&spitem))
		error("could not find input item in '%s'",filename);

	/* get sampling rate */
	if ((s_rate = 1.0/spitem.frameduration) < 1 || s_rate > 1000000)
		error("sample rate read error from %s",filename);

	/* now call processing routine */
	peak_pick(fid,s_rate,polflg);		/* item header is in "spitem" */

	/* close input file */
	sfsclose(fid);

	/* and do update */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* ... that's all folks */
	exit(0);
}

