/* respeed -- change speaking rate without pitch change */

/* M.A.Huckvale - University College London */

/* version 0.1 - January 1994 */

#define PROGNAME "respeed"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH RESPEED 1 SFS UCL
.SH NAME
respeed - change speaking rate without change in pitch
.SH SYNOPSIS
.B respeed
(-I) (-i item) (-r rate-change) file
.SH DESCRIPTION
.I respeed
is a program to speed-up or slow-down a piece of speech without changing
the pitch.  It is based on the SOLA algorithm of Roucos and Wilgur
(ICASSP-86).
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -r rate-change
Relative speed up factor. 2.0 = twice as fast, 0.5=half as fast.  Default 1.
.SH INPUT ITEMS
.IP SP 11
Speech item
.SH OUTPUT ITEMS
.IP SP 11
Rate changed speech
.SH HISTORY
.IP rate=
proportional rate-change.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
solout
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <malloc.h>
#include "sfs.h"
#define MAX(x,y) (((x)>(y))?(x):(y))
#define ABS(x) (((x)>0)?(x):-(x))

/* global defines */
#define WINTIME		0.032		/* size of input windows */
#define SYNCTIME	0.010		/* search for synchronisation
						+/- SYNCTIME */

/* global data */
struct item_header	spitem;
struct item_header	opitem;
short			*isp,*osp;	/* input/output signal */
float			*risp,*rosp;	/* real input/output signal */
float			*wisp,*wosp;	/* real input/output window */
int			bufsize;	/* size of internal buffers */
int			winsize;	/* size of an analysis window */
int			stepsize;	/* step size for input frames */
double			newrate=1.0;	/* speed-up rate */
int			verbose=0;	/* dump debugging info */

/* half window signal */
void halfwindow(buf,len,obuf,wbuf)
short	*buf;
int	len;
float	*obuf;
float	*wbuf;
{
	double 	w;
	int	i;

	w = M_PI/(len+1);
	for (i=1;i<=len;i++,buf++,wbuf++,obuf++) {
		*wbuf = 0.5 + 0.5*cos(i*w);
		*obuf = (float)*buf * *wbuf;
	}
}

/* window signal */
void window(buf,len,obuf,wbuf)
short	*buf;
int	len;
float	*obuf;
float	*wbuf;
{
	double 	w;
	int	i;

	w = 2.0*M_PI/(len+1);
	for (i=1;i<=len;i++,buf++,wbuf++,obuf++) {
		*wbuf = 0.5 - 0.5*cos(i*w);
		*obuf = (float)*buf * *wbuf;
	}
}

/* align window with output */
int	align(wbuf,wlen,sbuf,slen,omin,omax,otarg)
short	*wbuf;
int	wlen;
short	*sbuf;
int	slen;
int	omin;
int	omax;
int	otarg;
{
	int	i,j;
	float	sum;
	int	offset;
	float	peakval;
	float	ac[2048];
	int	currdist;

	if (omax > slen) omax=slen;
	if (omax > 2045) omax=2045;	/* to fit ac[], OK for 48kHz/32ms */
	if (omax > wlen) omax=wlen-1;
	if (omin >= wlen)
		return(slen-wlen/2);
	offset = (slen<wlen/2) ? 0 : wlen/2;
	ac[omin-2]=0;
	ac[omin-1]=0;
	for (i=omin;i<=omax;i++) {
		sum = 0;
		for (j=0;j<i;j++)
			sum += (float)wbuf[i-j-1] * (float)sbuf[slen-j-1];
		ac[i] = sum/i;
	}
	ac[omax+1]=0;
	ac[omax+2]=0;

	/* smooth */
	peakval=0;
	for (i=omin;i<=omax;i++) {
		ac[i] = (ac[i-2]+ac[i-1]+ac[i]+ac[i+1]+ac[i+2])/5;
		if (ac[i]>peakval) peakval=ac[i];
	}

	/* search for peak close to target */
	peakval *= 0.75;
	currdist=omax;
	if (verbose==2) printf("peaks at ");
	for (i=omin;i<=omax;i++) if ((ac[i-1] < ac[i]) && (ac[i] > ac[i+1])) {
		if ((verbose==2) && (ac[i]>peakval))
			printf("%d(%g),",i,ac[i]);
		if ((ac[i]>peakval) && (ABS(otarg-i)<currdist)) {
			currdist = ABS(otarg-i);
			offset = i;
		}
	}
	if (verbose==2) printf(" chosen=%d\n",offset);

	return(slen-offset);
}

/* overlap-add two signals */
int	overlapadd(s1buf,w1buf,s1len,s2buf,w2buf,s2off)
float	*s1buf;
float	*w1buf;
int	s1len;
float	*s2buf;
float	*w2buf;
int	s2off;
{
	int	i;

	s2buf += s2off;
	w2buf += s2off;
	for (i=0;i<s1len;i++,s1buf++,w1buf++,s2buf++,w2buf++) {
		*s2buf += *s1buf;
		*w2buf += *w1buf;
	}
	return(s2off+s1len);
}

/* re-synthesize */
void outsynth(obuf,sbuf,wbuf,slen)
short	*obuf;
float	*sbuf;
float	*wbuf;
int	slen;
{
	int	i;

	for (i=0;i<slen;i++,obuf++,sbuf++,wbuf++) {
		if (*wbuf==0.0) {
/*			fprintf(stderr,"zero window\n"); */
			*obuf = 32767;
		}
		else
			*obuf = (short)(*sbuf / *wbuf);
	}
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char		*ty;		/* item sub type */
	char		*sptype="0";
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		ofid;
	double		curtime,endtime;
	double		halfwintime,steptime;
	int		cursamp,newsamp;
	int		frameno,framelen,offset;
	int		overtarg,overmin,overmax;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:r:v")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Speed changing of signal V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype=ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'r' :	/* output rate */
			newrate = atof(optarg);
			break;
		case 'v' :	/* verbose */
			verbose++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-r rate-change) (-v) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL))<0)
		error("access error on '%s'",filename);

	/* locate item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("unable to find input item in '%s'",filename);

	/* get sizes */
	halfwintime = WINTIME/2;
	winsize = WINTIME/spitem.frameduration;
	bufsize = 10*winsize;
	steptime = 0.5*newrate*WINTIME;
	stepsize = steptime/spitem.frameduration;

	/* get buffers */
	if ((isp=(short *)sfsbuffer(&spitem,bufsize))==NULL)
		error("could not get memory buffer");
	if ((osp=(short *)sfsbuffer(&spitem,bufsize))==NULL)
		error("could not get memory buffer");
	if ((risp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");
	if ((rosp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");
	if ((wisp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");
	if ((wosp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");

	/* make output header */
	sfsheader(&opitem,SP_TYPE,0,2,1,spitem.frameduration,spitem.offset,1,0,0);
	sprintf(opitem.history,"%s(%d.%02d;rate=%g)",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			newrate);

	/* get output channel */
	if ((ofid=sfschannel(filename,&opitem))<0)
		error("could not open output channel to '%s'",filename);

	/* pre-charge buffer with half a window */
	memset((char *)osp,0,2*bufsize);
	memset((char *)rosp,0,4*bufsize);
	memset((char *)wosp,0,4*bufsize);
	framelen = sfsread(fid,0,winsize/2,isp);
	halfwindow(isp,winsize/2,rosp,wosp);
	cursamp=winsize/2;
	curtime=halfwintime;
	
	/* create 'dummy' output in osp */
	outsynth(osp,rosp,wosp,cursamp);

	/* calculate min and max overlap alignments */
	overtarg = winsize/2;
	overmin = overtarg-SYNCTIME/spitem.frameduration;
	overmax = overtarg+SYNCTIME/spitem.frameduration;
	if (overmin < winsize/16) overmin = winsize/16;
	if (overmax > 15*winsize/16) overmax = 15*winsize/16;

	/* processing loop */
	endtime=spitem.numframes*spitem.frameduration/newrate;
	while (curtime < endtime) {

		/* calculate next frame to use */
		frameno = (curtime*newrate/steptime - 1)*stepsize;
		if (frameno < 0) frameno=0;

		/* load this frame */
		framelen = sfsread(fid,frameno,winsize,isp);

		/* align with what we have so far */
		offset = align(isp,framelen,osp,cursamp,overmin,overmax,overtarg);

		/* window frame */
		window(isp,framelen,risp,wisp);

		/* add in */
		newsamp = overlapadd(risp,wisp,framelen,rosp,wosp,offset);

		if (verbose)
			printf("Out %6.3f-%.3f\tIn %6.3f-%.3f\tSync at %6.3fs delay.\n",
				curtime+(offset-cursamp)*spitem.frameduration,
				curtime+(newsamp-cursamp)*spitem.frameduration,
				frameno*spitem.frameduration,
				(frameno+framelen)*spitem.frameduration,
				(cursamp-offset)*spitem.frameduration);

		/* create 'dummy' output */
		outsynth(osp,rosp,wosp,newsamp);
		curtime += (newsamp-cursamp)*spitem.frameduration;
		cursamp = newsamp;

		/* purge output from time to time */
		if (cursamp >= (bufsize-winsize)) {
			outsynth(osp,rosp,wosp,bufsize/2);
			/* write out first half */
			sfswrite(ofid,bufsize/2,osp);
			memcpy((char *)osp,(char *)(osp+bufsize/2),bufsize);
			memcpy((char *)rosp,(char *)(rosp+bufsize/2),bufsize*2);
			memcpy((char *)wosp,(char *)(wosp+bufsize/2),bufsize*2);
			memset((char *)(osp+bufsize/2),0,bufsize);
			memset((char *)(rosp+bufsize/2),0,bufsize*2);
			memset((char *)(wosp+bufsize/2),0,bufsize*2);
			cursamp -= bufsize/2;
		}
	}

	/* write last buffer */
	outsynth(osp,rosp,wosp,cursamp);
	sfswrite(ofid,cursamp,osp);

	/* that's all folks */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);
	exit(0);
}	


