 /* spaned - Speech editing controlled by Annotations */

/* M.A.Huckvale - University College London */

/* version 1.0 - May 1995 */
/* version 1.1 - May 1995
	- add -c check flag
*/

#define PROGNAME "spaned"
#define PROGVERS "1.1"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH SPANCAT SFS1 UCL
.SH NAME
spaned - speech editing controlled by annotations
.SH SYNOPSIS
.B spaned
(-w overlapwidth) (-c) control-file outfile
.SH DESCRIPTION
.I spaned
is a program to concatenate a number of speech segments 
from different files into a single speech signal controlled by times
or annotations listed in a control file.
The control file consists of one line per speech section to process
giving (i) the filename, (ii) the speech item, (iii) the annotation item,
(iv) the start annotation or time, (v) the end annotation or time.  If the
item numbers are not given, the last appropriate item in the file
is chosen.  If no start time or annotation is given then it defaults
to the start of the item.  If no end time or annotation is given it defaults
to the end of the annotated region if a start annotation is supplied,
or otherwise to the end of the item.  Items should be separated by commas.
To prevent clicks signals are slightly overlapped and weighted by a
raised cosine window,
Speech items must match in sampling rate.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -w overlap-width
Specify time for overlaps at joins.  Default 0.025s.
.TP 11
.B -c
Check mode.  Access all the files and annotations but don't do the copying.
.SH INPUT ITEMS
.IP SP 11
Speech items
.IP AN 11
Annotation items
.SH OUTPUT ITEMS
.IP SP 11
Concatenated Speech.
.SH HISTORY
.IP file=control
Control file name.
.SH VERSION/AUTHOR
.IP 1.1
Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include "sfs.h"

/* global data */
struct item_header spitem;
struct item_header anitem;
struct an_rec	*an;
struct item_header ospitem;

char	cfilename[SFSMAXFILENAME];
char	fname[SFSMAXFILENAME];
char	*sptype;
char	*antype;
char	astart[80];
char	astop[80];
int	outrate=0;
double	overlaptime=0.025;
int	overlapsamp;
short	*overlap;
int	overlapload=0;
float	*inweight;
float	*outweight;
int	checkonly=0;

#define SKIPSPACE(p) while (isspace(*p)) p++
#define ISFIELD(p) (!((p==',') || isspace(p)))

int	copysp();
int	flushsp();

/* read a line from control file */
int	getcontrol(ip,fname,sptype,antype,start,stop)
FILE	*ip;
char	*fname;
char	**sptype;
char	**antype;
char	*start;
char	*stop;
{
	char		line[256];
	char		*p,*d;
	static char	*zero="0";
	static char	it1[80],it2[80];
	int32		it;

	strcpy(fname,"");
	*sptype=zero;
	*antype=zero;
	strcpy(start,"");
	strcpy(stop,"");

	do {
		if (!fgets(line,256,ip)) return(0);
	} while ((line[0]=='\n') || (line[0]=='#'));

	/* get filename */
	p = line;
	SKIPSPACE(p);
	while (ISFIELD(*p))
		*fname++ = *p++;
	*fname = '\0';
	SKIPSPACE(p);
	if (!*p) return(1);
	p++;

	/* get speech item */
	d = it1;
	SKIPSPACE(p);
	while (ISFIELD(*p))
		*d++ = *p++;
	*d = '\0';
	if (it1[0]) {
		if ((itspec(it1,&it,sptype)!=0) || (it!=SP_TYPE))
			error("bad SP item specification in control file");
	}
	SKIPSPACE(p);
	if (!*p) return(1);
	p++;

	/* get annotation item */
	d = it2;
	SKIPSPACE(p);
	while (ISFIELD(*p))
		*d++ = *p++;
	*d = '\0';
	if (it2[0]) {
		if ((itspec(it2,&it,antype)!=0) || (it!=AN_TYPE))
			error("bad AN item specification in control file");
	}
	SKIPSPACE(p);
	if (!*p) return(1);
	p++;

	/* get start time */
	SKIPSPACE(p);
	while (ISFIELD(*p))
		*start++ = *p++;
	*start = '\0';
	SKIPSPACE(p);
	if (!*p) return(1);
	p++;

	/* get stop time */
	SKIPSPACE(p);
	while (ISFIELD(*p))
		*stop++ = *p++;
	*stop = '\0';

	return(1);
}

/* find time from annotation */
double	findtime(str)
char	*str;
{
	int	i;
	if (!an) return(-1.0);
	for (i=0;i<anitem.numframes;i++)
		if (strcmp(str,an[i].label)==0)
			return(anitem.offset + an[i].posn*anitem.frameduration);
	return(-1.0);
}
double	findend(str)
char	*str;
{
	int	i;
	if (!an) return(-1.0);
	for (i=0;i<anitem.numframes;i++)
		if (strcmp(str,an[i].label)==0)
			return(anitem.offset + (an[i].posn+an[i].size)*anitem.frameduration);
	return(-1.0);
}

/* calculate overlap weighting */
void	calcoverweights()
{
	int	i;
	double	w;

	if (overlapsamp==0) return;

	inweight = (float *)calloc(overlapsamp,sizeof(float));
	outweight = (float *)calloc(overlapsamp,sizeof(float));

	w = 4.0*atan(1.0)/overlapsamp;
	for (i=0;i<overlapsamp;i++) {
		inweight[i] = 0.5-0.5*cos(i*w);
		outweight[i] = 1 - inweight[i];
	}
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		sfid= -1;	/* output file descriptor */
	FILE		*ip;
	int		splen,totsamp=0;
	int		inrate;
	double		stime,etime;
	int		ssamp=0,esamp=0;
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Iw:c")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Speech editing using annotations V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'w' :
			overlaptime = atof(optarg);
			if ((overlaptime < 0.0) || (overlaptime > 1.0))
				error("illegal overlap time");
			break;
		case 'c' :	/* check mode */
			checkonly=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<3))
		error("usage: %s (-I) (-w overlapwidth) (-c) controlfile sfsfile",PROGNAME);

	/* get control file name */
	if (optind < argc)
		strcpy(cfilename,argv[optind]);
	else
		error("no control file specified");
	optind++;
	
	/* get SFS filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open control file */
	if ((ip=fopen(cfilename,"r"))==NULL)
		error("could not open '%s'",cfilename);

	/* get control items */
	while (getcontrol(ip,fname,&sptype,&antype,astart,astop)) {

fprintf(stderr,"fname='%s' sptype='%s' antype='%s' start='%s' stop='%s'\n",
	fname,sptype,antype,astart,astop);

		/* open file */
		if ((fid=sfsopen(fname,"r",NULL)) < 0)
			error("could not open '%s'",fname);

		/* read in annotations */
		if (sfsitem(fid,AN_TYPE,antype,&anitem)) {
			an = (struct an_rec *)sfsbuffer(&anitem,anitem.numframes);
			sfsread(fid,0,anitem.numframes,an);
		}
		else
			an = NULL;

		/* locate speech */
		if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
			error("could not find speech item in '%s'",fname);
		if (outrate) {
			inrate = (int)(0.5 + 0.1/spitem.frameduration);
			if (inrate != outrate)
				error("sample rate doesn't match in '%s'",fname);
		}
		else {
			outrate = (int)(0.5 + 0.1/spitem.frameduration);
			overlapsamp = overlaptime/spitem.frameduration;
			overlap = (short *)sfsbuffer(&spitem,overlapsamp);
			calcoverweights();
		}

		/* if first time through, open output channel */
		if (!checkonly && (sfid < 0)) {
			sfsheader(&ospitem,SP_TYPE,0,2,1,
					spitem.frameduration,0.0,1,0,0);
			sprintf(ospitem.history,"%s(file=%s,overlap=%g)",
				PROGNAME,cfilename,overlaptime);
			if ((sfid=sfschannel(filename,&ospitem)) < 0)
				error("could not open output channel to '%s'",filename);
		}

		/* determine region to copy */
		if (astart[0]=='\0')
			ssamp = 0;
		else if (isdigit(astart[0]))
			ssamp = (atof(astart)-spitem.offset)/spitem.frameduration;
		else if ((stime = findtime(astart)) < 0)
			error("could not find annotation '%s' in '%s'",astart,fname);
		else
			ssamp = (stime-spitem.offset)/spitem.frameduration;
		if (ssamp < 0) ssamp=0;
		if (ssamp > spitem.numframes) ssamp = spitem.numframes;

		if (astop[0]=='\0') {
			if ((astart[0]!='\0') && !isdigit(astart[0])) {
				etime = findend(astart);
				esamp = (etime-spitem.offset)/spitem.frameduration;
			}
			else
				esamp = spitem.numframes;
		}
		else if (isdigit(astop[0]))
			esamp = (atof(astop)-spitem.offset)/spitem.frameduration;
		else if ((etime = findtime(astop)) < 0)
			error("could not find annotation '%s' in '%s'",astop,fname);
		else
			esamp = (etime-spitem.offset)/spitem.frameduration;
		if (esamp < 0) esamp=0;
		if (esamp > spitem.numframes) esamp = spitem.numframes;

fprintf(stderr,"copying samples %d to %d from %s\n",ssamp,esamp,fname);

		/* copy speech */
		splen = copysp(fid,sfid,ssamp,esamp);

		totsamp += splen;

		free(an);
		sfsclose(fid);
	}
	totsamp += flushsp(sfid);

	if (!checkonly) {
		if (!sfsupdate(filename))
			error("update error on '%s'",filename);
	}

fprintf(stderr,"%d samples written to '%s'\n",totsamp,filename);

	/* that's all folks */
	exit(0);
}

/* copy signal */
short	xferbuff[4096];
int	copysp(ifd,ofd,start,stop)
int	ifd,ofd;
int	start,stop;
{
	int	i;
	int	len;
	int	cnt=0;

	if (ofd < 0) return(0);

	if (overlapload) {
		/* process overlap with last segment */
		len = sfsread(ifd,start,overlapsamp,xferbuff);
		start += len;
		for (i=0;i<len;i++)
			xferbuff[i] = (short)(inweight[i]*xferbuff[i]+outweight[i]*overlap[i]);
		for (;i<overlapsamp;i++)
			xferbuff[i] = (short)(outweight[i]*overlap[i]);
		cnt += sfswrite(ofd,overlapsamp,xferbuff);
	}
	while (start < (stop-overlapsamp)) {
		len = sfsread(ifd,start,(((stop-overlapsamp)-start)>4096)?4096:((stop-overlapsamp)-start),xferbuff);
		if (sfswrite(ofd,len,xferbuff)!=len)
			error("output write error");
		start += len;
		cnt += len;
	}
	/* hold over last portion */
	if (overlapsamp > 0) {
		len = sfsread(ifd,start,overlapsamp,overlap);
		overlapload=1;
	}
	return(cnt);
}

/* flush overlap buffer */
int	flushsp(ofd)
int	ofd;
{
	if (ofd < 0)
		return(0);
	if (overlapload)
		return(sfswrite(ofd,overlapsamp,overlap));
	else
		return(0);
}
