/* srusyn -- software implementation of JSRU/Holmes formant synthesizer */

/* M.A.Huckvale - University Colle London */

/* version 2.0 - July 1993 */

/* history:
	- JSRU report 1016 Holmes & Rye
	- "C" version "dbsoft" Bishop & Davies
	- SFS version Huckvale
	- Extended version Breen
	- binary/live version Huckvale
*/

/*-------------------------------------------------------------------------*/
/**MAN
.TH SRUSYN SFS1 UCL
.SH NAME
srusyn - JSRU parallel-formant speech synthesizer with live output
.SH SYNOPSIS
.B srusyn
(-I) (-p parfil) (-g glotfil) (-o outname) (-s) (-t) (controlfile)
.SH DESCRIPTION
.I srunsyn20
is a software implementation of a parallel-formant speech 
synthesizer.  It is a "C"-language implementation of the FORTRAN
software synthesizer of JSRU report 1016.  It is based on the original
SFS 'soft' synthesizer but modified to take parameter tables as input
and output binary or live data.  The default action is to generate
and play the completed waveform using the SFS replay functions.  A switch
'-s' allows for simultaneous synthesis and replay on SPARC systems.
.PP
.I Options
and their meanings are:
.TP
.B -I
Identify program name and version number, then exit.
.TP 11
.BI -p paramfile
Use the parameter file
.I paramfile
instead of the default parameter file.  More information on the
parameter file may be found in
.B parfil(5).
.TP
.BI -g glottalfile
Read the glottal pulse shape and glottal area from the file 
.I glottalfile
instead of the parameter file.  The format for a glottal file is an 
ASCII file containing two lists of integers separated by spaces.  The 
first list has 72 entries in the range -127 to +127 and represents the 
glottal pulse shape.  The second list has 18 entries in the range 
0 to 15 and represents the glottal area.
.TP
.BI -o outname
Selects binary output data format and filename.
.TP
.B -s
Selects SPARC audio output.  This is performed simultaneously with
conversion and so may not keep up on loaded or slow systems.
.TP
.B -t
Selects raw hexadecimal input in format produced by text-to-speech system.
.SH FILES
SFSBASE/data/parfil - default parameter file.
.SH VERSION/AUTHORS
.TP
dbsoft
1.0 - Mark Bishop and Paul Davies
.br
2.1 - Mark Huckvale
.TP
soft
1.0s - Mark Huckvale
.br
1.1s - Andrew Breen
.TP
srusyn20
1.0 - Mark Huckvale
.SH SEE ALSO
soft, nsoft, JSRU report 1016 and "dbsoft" manual (Bishop & Davies)
.SH BUGS
The output sampling frequency (srate) cannot be changed (it is set in
include file at compilation time).
*/
/*------------------------------------------------------------------------*/

/* program name */
#define PROGNAME "srusyn"
#define PROGVERS "1.0"
char	*progname=PROGNAME;

/* header files */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard io library */
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>		/* maths library */
#include <string.h>		/* string library */
#include <malloc.h>
#include <ctype.h>
#include <time.h>
#include "sfs.h"		/* speech filing system headers */

/* set this for real-time replay on Sun SPARC workstations */
#ifdef SUN4
#define SUN_REPLAY
#else
#undef SUN_REPLAY
#endif

/* filter design  */
#include "srusyn11.h"		/* filter coefficients for 11kHz */
/* #include "srusyn20.h"		 filter coefficients for 20kHz */

/* define boolean type */
typedef char boolean;
#define TRUE  1
#define FALSE 0

/* parameter re-ordering for SY item */
#define FN	0
#define F1	1
#define F2	2
#define F3	3
#define ALF	4
#define A1	5
#define A2	6
#define A3	7
#define AHF	8
#define V	9
#define FX	10
#define MS	11

/* macro definitions */
#define	RANGE(lo,val,hi)	(((val)<(lo))?(lo):(((val)>(hi))?(hi):(val)))
#define MIN(x,y)		(((x)<(y))?(x):(y))
#define MAX(x,y)		(((x)>(y))?(x):(y))
#define MOD(x,y)		((x)-(y)*(int)((x)/(y)))

/* boolean variables specifying synthesizer functions 
 * from command line switches 
 * sparcsyn	 TRUE	->	write output to /dev/audio
 */

boolean sparcsyn;

/* filenames */
char	filename[80];		/* data file name */
char	gltname[80];		/* glottal pulse shape file */
char	paramname[80];		/* parameter file */
char	outname[80];		/* output filename */

/* input file descriptors */
FILE	*ip;			/* control parameter file */
FILE	*glt_ptr;		/* glottal pulse file */
FILE	*param_ptr;		/* parameter file */

/* output file descriptors */
int	ofid;			/* speech output */

/* SY item input arrays and pointers to current and last frame */
short	ctrl1[16], ctrl2[16], *old_ptr, *new_ptr;

/* waveform arrays */
float	formant[FSAMP];		/* input and output of formant resonators */
float	uvox[FSAMP];		/* unvoiced excitation */
float	vox[FSAMP];		/* voiced excitation */
float	output[FSAMP];		/* output of synthesizer */

float	glarea[CSAMP]; 		/* 1 frames worth of glottal area function, 
			    		due to voicing component only.
			    	*/
float 	glamix [CSAMP];		/* 1 frames worth of glottal area function,
			    		including equivalent unvoiced component. 
			    	*/
int	ictlst[CSAMP]; 		/* start position of subframes of excitation */
int	ictlfn[CSAMP]; 		/* end position of subframes of excitation   */
float	srate;			/* Sampling rate (hz)	*/
float	frate;			/* Frame rate    (hz)	*/
float	cratef;			/* glottal function sampling rate */
float 	cratex;			/* excitation sampling rate */
int	nc;			/* No. of control values/frame = (lf-1)/lc+1  */
int	lc;			/* No. of samples/control value		 */
int	lf;			/* No. of samples/frame        = srate/frate  */
int	ifrecl;			/* limit of frequency change without
			    		setting the formant control clamping flag.
			    	*/
int	iampcl;			/* limit of amplitude change without
			    		setting the formant control clamping flag.
			    	*/
int	ipocl;			/* fundamental period clamping limit */
float	gouv;			/* glottal area for unvoiced conditions */
int	length;
float	xscale;  		/* maximum value for output,
			    		suitable for output through D/A
			    	*/ 
int	isqthr;  		/* frequency maintenance threshold across frames */
int	lsqthr;  		/* as isqthr for silent formants */
float	ffq[7][CSAMP];		/* Array holding formant frequencies 
			    		in costab position 
			   	*/
float	costab[257];	 	/* 256 values of cos(2*pi*ffq/srate) */
float	exptab[33];		/* 32  values of exp(2*pi*bandw/srate) */
float	gainct [7];		/* formant fixed gain constants	*/
float	flflu  [7];		/* formant l.f. limits in costab positions */
float	finlu  [7];		/* formant frequency steps in costab positions */
float	ffqmod [7] ;  		/* formant frequency modifications in costab positions */
float	forbdw[7];		/* formant fixed bandwidths in exptab positions */
float	fbwmod[7];		/* formant bandwidth modifications in exptab positions */
float	fbdwna[7];		/* formant bandwidth modifications for nasals */
float	b[7];			/* b and c are delay elements in formant generators */
float	c[7];
float	fqzero; 
float	bwzero;
float	dnasal;			/* degree of nasality */ 
float	fotab[64];		/* 63 values of excitation pitch */
float	amptab[64];		/* 63 values of formant amplitude */
float	pltab[64];		/* 63 values of pl lengths in sec. */
float	flflim[6];		/* 6  formant l.f. limits in hertz */
float	fincrm[6];		/* 6  formant frequency steps in hertz */
float	bdwtab[6];		/* 6  formant fixed bandwidths (Hz) */
float	bdwmod[6];		/* 6  formant bandwidth modifications (Hz) */
float	bdwnas[6];		/* 6  formant bandwidth modifications for nasal */ 
float	fqmod[6];		/* 6  formant frequency modifications (Hz) */
float	dvoffs[6];		/* offsets for voiced / unvoiced excitation mixing */
float	fmgain[6];		/* 6 individual gains */
int	ictlof[16];		/* control offsets */
float	grpl; 			/* glottal pulse length gradient */
float	conts; 			/* contant component of glottal pulse length. */
float	foscal, fobase; 	/* values for calculating excitation pitch table */

float	fqlow, fqstep;		/* formant generator cosine table frequencies */
float 	bwlow, bwstep; 		/* formant generator bandwidth table frequencies */
float 	dbstep;			/* number of dB between each control value */
float 	aoffdn;			/* amplitude, frequency and scale for nasality */
float 	foffdn;
float 	scaldn;
float 	expat[72];		/* excitation pattern */
float	gapat[18];		/* glottal area pattern */
float	po[CSAMP];		/* period and pulse length control arrays for excitation */
float 	pl[CSAMP];
float	pov;			/* saved values of po and pl */
float	plv;
float	posn;			/* current excitation sample position */
int  	npatt;			/* current excitation pattern index */
int	lcx;			/* length of subframe of excitation */
int	lc4;			/* length of glottal control subframe */
int	lfx;			/* length of frame of excitation */
int	ncx; 			/* number of subframes in one frame */
float	usrate;
float	gain[6][50];		/* internal controls for amplitude for each channel */
float	voxmix[6];		/* voiced / unvoiced mixing ratio for each channel */


float	pwscl;			/* scale factor for unvoiced excitation */
int	nsil;			/* duration of silence to be replaced by stops */

boolean	uvtov;			/* unvoiced to voiced transition flag */
boolean impexc;			/* single impulse excitation flag (stop) */
boolean clampf;			/* frequency clamping flag */
boolean clampx;			/* excitation clamping flag */
boolean alfcon;			/* enables link between channels ALF channel and FN (params) */
boolean vfr;			/* fully voiced frame */
boolean uvfr;			/* fully unvoiced frame */
boolean uvlfr;			/* last frame unvoiced */

int	ixctl;			/* current control number for excitation */
float	sgap;			/* gap between excitation samples */
float	pgap;			/* gap between excitation pulses */

/* replay buffers */
static short *repbuf=NULL;
int	repcnt=0;
int	reptot=0;
#define REPXTDSIZ	32768

/* overload check on output */
int	overload;		/* =0 none, =1 correctable, =2 severe */

/* input is from TTS */
int	ttsin=0;

/* forward declarations */
#ifdef __STDC__
void openoutputs(void);
void write_flshort(int fid,float *buff);
void read_params(void);
void pritab(void);
void make_tables(void);
void read_float (FILE *fp, int n, float *array);
void read_int(FILE *fp, int n, int *array);
int decode_frame(FILE *fp);
void squeak_removal(void);
void controls(void);
void po_filt(float *arrin, float *arrout);
void pl_filt(float *arrin, float *arrout);
void amp_con_filt(int nchan,float *arrin,float *arrout);
void fre_con_filt(int nchan,float *arrin,float *arrout);
void excite(void);
void gnoise(void);
void gen_excite(float *store);
void adjust(void);
void excite_filt(float *xin,float *xout);
void mix(int nchan);
void arzero(float *array,int len);
void form_gen(int nchan);
void output_filter(float *wave);
void eoutput_filter(float *wave);
void mixing_filt(int nchan, float *arrin, float *arrout);
void make_op_table(char *name);
int maximum(int *stat_array,int po,int dimension);
#else
void openoutputs();
void write_flshort();
void read_params();
void pritab();
void make_tables();
void read_float();
void read_int();
int decode_frame();
void squeak_removal();
void controls();
void po_filt();
void pl_filt();
void amp_con_filt();
void fre_con_filt();
void excite();
void gnoise();
void gen_excite();
void adjust();
void excite_filt();
void mix();
void arzero();
void form_gen();
void output_filter();
void eoutput_filter();
void mixing_filt();
void make_op_table();
int maximum();
#endif

/* main program */
void main (argc,argv)
int	argc;
char 	*argv[];
{
	/* option decoding */
	extern char 	*optarg;
	extern int	optind;
	int		c,errflg=0;
	/* processing control */
	int		nchan;
	int		afid;
	
	/* set up default parameter file name */
	strcpy(paramname,sfsbase());
	strcat(paramname,"/data/parfil");
	if (SRATE==11000) strcat(paramname,".11");

	/* set up default operation modes */
	sparcsyn = FALSE;
	outname[0]='\0';

	/* decode switches */
	while ((c = getopt (argc, argv, "Ip:g:o:st")) != EOF) switch (c) {

		case 'I' : 	/* identify */
			fprintf (stderr, "%s: JSRU/Holmes formant synthesizer V%s\n", PROGNAME, PROGVERS);
			exit (0);
			break;

		case 'g' : 	/* read glottal pulse shape from specified file */
			strcpy (gltname, optarg);
			break;

		case 'p' :	/* specify alternative parameter file */
			strcpy (paramname, optarg);
			break;

		case 'o' :	/* binary output file */
			strcpy(outname, optarg);
			break;

		case 's' :	/* SPARC synthesis */
#ifdef SUN_REPLAY
			sparcsyn=TRUE;
#else
			error("Sun real-time replay not configured");
#endif
			break;

		case 't' :	/* TTS input */
			ttsin=1;
			break;
			
		case '?' : /* unknown flag */
			errflg ++;
		
	}

	/* check for option decoding error or missing filename */
	if (errflg)
		error ("usage : %s (-I) (-p paramfile) (-g glotfile) (-o outfile) (-s) (-t) (controlfile)",PROGNAME);

	/* get filename */
	if (optind < argc) {
		strcpy(filename, sfsfile(argv[optind]));
		if ((ip=fopen(filename,"r"))==NULL)
			error("could not open input file '%s'",filename);
	}
	else
		ip = stdin;

	/* read synthesizer parameter file */
	read_params();

	/* open output channels */
	openoutputs();

	/* make constant data tables */
	make_tables ();

	/* process input data frame by frame */
	while (decode_frame(ip)) {

		/* check slew rates */
		squeak_removal();

		/* set synthesizer controls */
		controls();

		/* generate excitation */
		excite();

		/* clear speech buffer */
		arzero (output, FSAMP);

		/* generate each channel in turn */
		for (nchan = 0; nchan <= 5; nchan ++) {
 
			/* get excitation mixture */
			mix (nchan);

			/* generate formant */
			form_gen (nchan);

			/* add result to signal */
			mixing_filt (nchan, formant, output);
		}

		/* perform output filtering */
		output_filter (output);

		/* write speech to output */
		write_flshort(ofid,output);

	}

	/* check on overload */
	if (overload==1)
		fprintf(stderr,"%s: overload detected on output\n",PROGNAME);
	else if (overload==2)
		fprintf(stderr,"%s: severe overload detected on output\n",PROGNAME);

	/* replay buffer if required */
	if (ofid < 0) {
		if ((afid=dac_open(NULL)) >= 0) {
			dac_playback(repbuf,reptot,(double)SRATE,16,1,1);
			dac_close(0);
		}
	}
	else
		close(ofid);

	/* that's all folks */
	exit (0);
	
}

#ifdef SUN_REPLAY
#include <sun/audioio.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/ioctl.h>
#include "multimedia/libaudio.h"
#include "multimedia/audio_device.h"
static Audio_hdr	devhdr;		/* audio state */
static Audio_hdr	bufhdr;		/* buffer state */
#endif

/* open output channels */
void openoutputs()
{
	if (sparcsyn) {
#ifdef SUN_REPLAY
		int	port;
		char	*s,*getenv();

		/* SPARC 16-bit dual-basic-rate ISDN */
		if ((ofid = open(DAC_SUN16_PATH, O_WRONLY)) < 0)
			error("could not open '%s'\n",DAC_SUN16_PATH);

		/* get Sun audio parameters */
		audio_drain(ofid,0);
		audio_get_play_config(ofid,&devhdr);
		bufhdr = devhdr;

		/* set up equivalent header for buffer */
		if (SRATE==20000)
			bufhdr.sample_rate = 18900;
		else
			bufhdr.sample_rate = 11025;
		bufhdr.samples_per_unit = 1;
		bufhdr.bytes_per_unit = 2;
		bufhdr.channels = 1;
		bufhdr.encoding = AUDIO_ENCODING_LINEAR;
		bufhdr.data_size = -1;

		/* match the two */
		if (audio_cmp_hdr(&devhdr, &bufhdr) != 0) {
			switch (audio_set_play_config(ofid,&bufhdr)) {
			case AUDIO_SUCCESS:
				break;
			default:
				error("couldn't set audio param (%d,%d)\n",
					bufhdr.sample_rate,bufhdr.channels);
			}
		}

		/* select output port */
		port = AUDIO_SPEAKER;
		if (s = getenv("DAC")) {
			if (strcmp(s,"sun16-phone")==0)
				port = AUDIO_HEADPHONE;
			else if (strcmp(s,"sun16-line")==0)
				port = AUDIO_LINE_OUT;
		}
		audio_set_play_port(ofid,&port);
#endif
	}
	else if (outname[0]=='\0') {
		/* write to buffer */
		ofid = -1;
		repbuf = (short *)calloc(REPXTDSIZ,sizeof(short));
		repcnt = REPXTDSIZ;
		reptot=0;
	}
	else {
		/* write to file */
		if ((ofid=open(outname,O_WRONLY|O_CREAT,0644))<0)
			error("could not open output file '%s'",outname);
	}
}

/*
 * write_flshort ()
 * converts a float array to shorts and writes to the file 
 * pointed to by fid
 */

void write_flshort(fid,buff)
int	fid;
float	*buff;
{
	register int	i;
	short 		out[FSAMP];

	/* copy floats to shorts */
	for (i = 0; i < lf; i++) {
		if ((*buff >= -32767.0) && (*buff <= 32767.0)) {
			out[i] = (int)(*buff) << 2;
			if (fabs(*buff) > 16383) overload=1;
		}
		else {
			out[i] = 0;
			overload=2;
		}
		buff++;
	} 

	if (ofid < 0) {
		/* buffer in memory */
		if ((reptot + lf) >= repcnt) {
			repcnt += REPXTDSIZ;
			repbuf = (short *)realloc(repbuf,repcnt*sizeof(short));
		}
		memcpy(repbuf+reptot,out,lf*sizeof(short));
		reptot += lf;
	}
	/* write output record */
	else if (write(fid,out,lf*2) != lf*2)
		error("write error on '%s'",outname);

}

/* read synthesizer parameters from definition file */
void read_params ()
{
	float 	foi, sil, gclf;
	int	i, nfin, nstt, flag;
	int	iexpat[72], igapat [18];
	int	correct_lcx;

	/* open the parameter file */
	if ((param_ptr = fopen(paramname, "r")) == NULL)
		error( "can't open parameter file '%s'",paramname);

	/* read in general parameters */
	if (fscanf(param_ptr,"%f srate%f frate%f cratef%f cratex",
		&srate,&frate,&cratef,&cratex) != 4)
		error("error reading general variables from '%s'",paramname);

	if (fscanf(param_ptr,"%f fqlow%f fqstep%f bwlow%f bwstep%f foscal%f fobase%f dbstep%f aoffdn%f foffdn%f scaldn%f conts%f grpl",
	 	&fqlow,&fqstep,&bwlow,&bwstep,
		&foscal,&fobase,&dbstep,&aoffdn,
		&foffdn,&scaldn,&conts,&grpl) !=12)
		error("error reading general variables from '%s'",paramname);

	if (fscanf(param_ptr,"%f xscale%f foi%f gclf%f sil%f gouv",
		&xscale,&foi,&gclf,&sil,&gouv) !=5)
		error("error reading general variables from '%s'",paramname);

	if (fscanf(param_ptr,"%d isqthr%d lsqthr",&isqthr,&lsqthr) != 2)
		error("error reading general variables from '%s'",paramname);

	/* control parameter offsets */
	fscanf(param_ptr," ictlof ");
	for (i = FN ; i <= MS ; i++ ) {
		if (fscanf(param_ptr,"%d",&ictlof[i]) != 1)
			error("error reading control signal offsets from '%s'",paramname);
	}

	fscanf(param_ptr," flflim ");
	read_float( param_ptr, 6, flflim); /* reads in 6 floats to array flflim */
	fscanf(param_ptr," fincrm ");
	read_float( param_ptr, 6, fincrm);
	fscanf(param_ptr," bdwtab ");
	read_float( param_ptr, 6, bdwtab);
	fscanf(param_ptr," bdwmod ");
	read_float( param_ptr, 6, bdwmod);
	fscanf(param_ptr," bdwnas ");
	read_float( param_ptr, 6, bdwnas);
	fscanf(param_ptr," fqmod ");
	read_float( param_ptr, 6, fqmod );
	fscanf(param_ptr," dvoffs ");
	read_float( param_ptr, 6, dvoffs);
	fscanf(param_ptr," fmgain ");
	read_float( param_ptr, 6, fmgain);

	/* smoothing filter clamping limits */

	if (fscanf(param_ptr,"%d ifrecl%d iampcl%d ipocl",
		&ifrecl,&iampcl,&ipocl) != 3)
		error("error reading smoothing filter clamping limits from '%s'", paramname);

	/* operation control flag */
	if ( fscanf(param_ptr,"%d alfcon",&flag)!= 1)
		error("error reading operation control flag (alfcon) from '%s' ", paramname);

	alfcon = (boolean) flag;

	/* get glottal data from paramfile or user-file */
	if (*gltname != '\0' ) {
		/* open glottal pulse file */
		if ((glt_ptr = fopen (gltname, "r")) == NULL)
			error("cannot open glottal file '%s'",gltname);

		/* read glottal data from glottal file */
		read_int(glt_ptr,72,iexpat);
		read_int(glt_ptr,18,igapat);

		fclose(glt_ptr);
	}
	else {
		/* read glottal data from parameter file */
		fscanf(param_ptr," expat ");
		read_int(param_ptr,72,iexpat);
		fscanf(param_ptr," gapat ");
		read_int(param_ptr,18,igapat);
	}

	/* scale glottal pulse */
	for (i = 0; i < 72; i++)
		expat [i] = iexpat [i] * xscale / 127.0;
	for (i = 0; i < 18; i++)
		gapat [i] = igapat [i] / 15.0;

	/* close parameter file */
	fclose(param_ptr);

	/* check general parameters */
	if ((int)(srate+0.5) != SRATE)
		error("Main sampling rate must be %d", SRATE);
	if (frate <= 0.0)
		error("Frame rate < 0", NULL);
	if (cratef <= 0.0)
		error("Formant smoothed control rate < 0.0", NULL);
	if (cratex <= 0.0)
		error("Excitation smoothed control rate < 0.0", NULL);
	if (cratef < frate)
		error("Formant control rate < frame rate", NULL);
	if (cratex < frate)
		error("Excitation control rate < frame rate", NULL);

	/* Check integer number of samples per frame or frames not too long */	lf = srate / frate + 0.5 ;
	if (MOD(srate,frate) != 0.0 )
		fprintf(stderr,"%s: warning - non integer number of samples per frame\nMain sampling rate adjusted to %f Hz\n",PROGNAME,lf * frate);
	srate = frate * lf;
	if (lf > FSAMP)
		error ("frame rate too low: frame > %d samples", FSAMP);

	/* check integer number of samples per control */
	lc = srate / cratef + 0.5;

	if (MOD(srate,cratef ) != 0.0 )
		fprintf(stderr,"%s: warning - Formant smoothed control rate not submultiple of main rate\nControl rate adjusted to %f Hz\n",PROGNAME,srate/lc);

	correct_lcx = 4.0 * srate / cratex + 0.5;
	if (MOD(4*srate, cratex) != 0.0)
		fprintf(stderr,"%s: warning - Excitation smoothed control rate not a submultiple of higher excitation sampling rate, adjusted to %f Hz\n",PROGNAME,4*srate / correct_lcx);

	cratef = srate /lc;
	cratex = 4.0 * srate / correct_lcx ;

	/* check control rates not too high */
	if ( cratef / frate > 50.0 )
		error (">50 smoothed formant controls per frame", NULL);
	if (cratex / frate > 50.0)
		error (">50 smoothed excitation controls per frame", NULL);

	/* calculate dependent parameters */
	lc = srate  / cratef ;
	lf = srate  / frate + 0.5;
	foi = MAX(0.0 , foi);
	pwscl = xscale * sqrt( foi/srate);
	nsil  = sil * srate / 1000.0;
	nsil  = MIN(lf,nsil);

	if (dbstep != 0.0 )
		aoffdn /= dbstep;

	if (fincrm [0] != 0.0)
		foffdn = (foffdn - flflim [0] ) / fincrm [0];

	conts /= 1000.0;

	nc = (lf - 1)/lc + 1;

	nfin = 0;
	for (i = 0 ; i < nc ; i++ ) {
		nstt = nfin + 1;
		nfin += lc;
		if (nfin > lf) nfin = lf;
		ictlst[i] = nstt-1;
		ictlfn[i] = nfin-1;
	}

	fobase = fabs (fobase);
	foscal = fabs (foscal);
	usrate = 4.0 * srate ;
	lcx    = usrate / cratex;
	lfx    = lf * 4;
	ncx    = (lfx - 1)/ lcx + 1;
	lc4    = lc * 4;
	pov    = 0.01;
	plv    = 0.007;

	gainct [0] = gclf;
	gainct [1] = 1.0;
	gainct [2] = 1.0;
	gainct [3] = 1.0;
	gainct [4] = 1.0;
	gainct [5] = gclf;

}

/* 
 * pritab ()
 *
 * Routine prints out lookup tables from make_tables on standard output.
 *
 */

void pritab ()
{
	int 	i;
	char 	*today;
	time_t	seconds;

	/* get today's date */
	time (&seconds);
	today = ctime (&seconds);

	/* print parameter information */
	printf ("Synthesizer Parameters                  %s",today);
	printf ("----------------------------------------------------------------\n");
	printf ("File                                    = %s\n", paramname);
	printf ("Sampling rate                           = %g\n", srate);
	printf ("Frame rate                              = %g\n", frate);
	printf ("Frame length                            = %d\n", lf);
	printf ("Frame segment                           = %d\n", lc);
	printf ("Number of controls per frame            = %d\n", nc);
	printf ("Excitation sampling rate                = %g\n", usrate);
	printf ("Excitation frame length                 = %d\n", lfx);
	printf ("Excitation frame segment                = %d\n", lcx);
	printf ("Number of excitation controls per frame = %d\n", ncx);
	printf ("Excitation maximum amplification        = %g\n", xscale);
	printf ("Unvoiced / voiced amplification ratio   = %g\n", pwscl / xscale);
	printf ("Constant component of glottal pulse len.= %g\n", conts);
	printf ("Slope component of glottal pulse length = %g\n", grpl);
	printf ("Formant Frequency clamping limit        = %d\n", ifrecl);
	printf ("Formant Amplitude clamping limit        = %d\n", iampcl);
	printf ("Excitation period clamping limit        = %d\n", ipocl);
	printf ("Squeak remover threshold                = %d\n", isqthr);
	printf ("\n");
	printf ("Formant Parameters:\n");
	printf ("-------------------\n");

	printf ("#\tFlflim\tFincrm\tBdwtab\tBdwmod\tBdwnas\tFqmod\tFmgain\tVoffset\n");
	printf ("-----------------------------------------------------------------------\n");

	for (i = 0; i <= 5; i++)
		printf ("%d\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\n",
			i, flflim [i], fincrm [i], bdwtab [i],
			bdwmod [i], bdwnas [i], fqmod [i],
			fmgain [i], dvoffs [i]);

	printf("\n");
}


/* make_tables ()
 * This subroutine produces lookup tables 'costab' , 'exptab' from which
 * feedback constants in the formant generator 'form_gen' are
 * calculated , also lookups to which all the control signal arrays are
 * referred.
 *
 */

#define pi 3.1415927

void make_tables ()
{
	int		i;
	float 		fscale, bscale;

	for (i = 0 ; i <= 255 ; i++)
		costab [i] = cos( (double) (2.0 * pi * (fqlow + fqstep * i) / srate) );

	for (i = 0 ; i<= 31 ; i++)
		exptab [i] = exp( (double) ( -pi * (bwlow + bwstep * i)/ srate ) );

	fscale  = 0.0;
	bscale  = 0.0;
	if ( fqstep != 0.0 )
		fscale = 1.0 / fqstep;
	if ( bwstep != 0.0 )
		bscale = 1.0 / bwstep;

	bwzero = bscale * bwlow - 1.5;
	fqzero = fscale * fqlow - 1.5;

	for (i = 0; i <= 5 ; i++) {
		gainct [i]  = gainct [i] * fmgain[i];
		/* scale formant frequencies and bandwidths to table steps */
		forbdw [i]  = bdwtab [i] * bscale - bwzero;
	 	fbwmod [i]  = bdwmod [i] * bscale ;
		fbdwna [i]  = bdwnas [i] * bscale ;
		flflu  [i]  = flflim [i] * fscale - fqzero;
		finlu  [i]  = fincrm [i] * fscale ;
		ffqmod [i]  = fqmod  [i] * fscale ;
	}

	/* scale fundamental frequency and amplitudes */
	for (i = 0 ; i <= 63 ; i++) {
		fotab [i]   = foscal * pow (fobase,(i+1)/16.0);
		amptab[i]   = pow ( 10.0 , (float)(i-63) * dbstep / 20.0);
	}
	amptab [1] = 0.0;
	amptab [2] = 0.0;

	/* initialise frame pointers and initial "last frame" */
	old_ptr = ctrl1;
	new_ptr = ctrl2;
	for (i = FN; i <= MS; i++)
		old_ptr[i] = -63;

}

/* read in array of floats */
void read_float (fp, n, array)
FILE 		*fp;
int 		n;
float 		*array;
{ 
	while (n-- > 0)
		if (fscanf (fp, "%f", array++) != 1)
			error("error reading '%s'",paramname);
}

/* read in array of integers */
void read_int( fp, n, array)
FILE 		*fp;
int		n;
int  		*array;
{
	while (n-- > 0)
		if (fscanf (fp, "%d", array++) != 1)
			error("error reading file",NULL);
}

/* decode_one_frame ()
 *
 * decodes one frame of synthesizer control data
 */

#define HEXCHR(a)	(((a)<'A')?((a)-'0'):((a)-'A'+10))
#define HEXINT(a,b)	(16*HEXCHR(a) + HEXCHR(b))

int decode_frame(ip)
FILE	*ip;
{
	static char	lbuf[256];
	char		*p;
	float		frame[32];
	int		i;
	short		*temp_ptr;
	double		fxval,log();

	/* get record at frame 'fno' */
	do {
		if (fgets(lbuf,256,ip)==NULL)
			return(0);
	} while (lbuf[0]=='*');

	/* swap current encoded frame with last */	
	temp_ptr = old_ptr;
	old_ptr = new_ptr;
	new_ptr = temp_ptr;

	if (ttsin) {
		*temp_ptr++ = HEXINT(lbuf[ 0],lbuf[ 1]);	/* FN */
		*temp_ptr++ = HEXINT(lbuf[ 4],lbuf[ 5]);	/* F1 */
		*temp_ptr++ = HEXINT(lbuf[ 8],lbuf[ 9]);	/* F2 */
		*temp_ptr++ = HEXINT(lbuf[12],lbuf[13]);	/* F3 */
		*temp_ptr++ = HEXINT(lbuf[ 2],lbuf[ 3]);	/* ALF */
		*temp_ptr++ = HEXINT(lbuf[ 6],lbuf[ 7]);	/* A1 */
		*temp_ptr++ = HEXINT(lbuf[10],lbuf[11]);	/* A2 */
		*temp_ptr++ = HEXINT(lbuf[14],lbuf[15]);	/* A3 */
		*temp_ptr++ = HEXINT(lbuf[16],lbuf[17]);	/* AHF */
		*temp_ptr++ = HEXINT(lbuf[18],lbuf[19]);	/* V */
		*temp_ptr++ = HEXINT(lbuf[20],lbuf[21]);	/* FX */
		*temp_ptr++ = HEXINT(lbuf[22],lbuf[23]);	/* MS */
	}
	else {
		/* encode current frame */
		p = strtok(lbuf," \t\n");
		i=0;
		while (p && isdigit(*p)) {
			frame[i++] = atof(p);
			p = strtok(NULL," \t\n");
		}

		/* expect:  fx, f1, a1, f2, a2, f3, a3, f4, a4, fn, an, v */
		*temp_ptr++ = (frame [9] - 90) / 5;	/* FN */
		*temp_ptr++ = (frame [1] - 100) / 25;	/* F1 */
		*temp_ptr++ = (frame [3] - 500) / 50;	/* F2 */
		*temp_ptr++ = (frame [5] - 1300) / 50;	/* F3 */
		*temp_ptr++ = frame [10];		/* ALF */
		*temp_ptr++ = frame [2];		/* A1 */
		*temp_ptr++ = frame [4];		/* A2 */
		*temp_ptr++ = frame [6];		/* A3 */
		*temp_ptr++ = frame [8];		/* AHF */
		*temp_ptr++ = frame [11] / 4;		/* V */
		fxval = frame[0];			/* FX */
		*temp_ptr++ = fxval == 0.0 ? 0 : (short)(23.09 * log(fxval) - 75.37);
	
		/* Mark-Space ratio (fixed) */
		*temp_ptr = 32;
	
	}
	return(1);
}

/*
 * squeak_removal ()
 *
 * removes rapid jumps from incoming synthesizer control parameters
 */

void squeak_removal ()
{
	int	i;

	for (i = ALF; i <= A3; i++) {
		if ((new_ptr [i] <= (old_ptr [i] - isqthr))
			|| ((new_ptr [i] == 1) && (old_ptr [i] > lsqthr))) {
			new_ptr [i-4] = old_ptr [i-4];
		}
	}
}

/*
 * controls ()
 * Calculates synthesizer controls from control data
 * and sets flags depending on control conditions
 */

void controls ()
{
	float 		povar,plvar,vcvar;
	int 		i, j, channel, ind;
	float 		any[CSAMP];
	float		x, y;
	double		log();

	plvar = conts + grpl / fotab [(new_ptr [MS] - 32) / 2 + new_ptr [FX]];

	povar = 1.0 / fotab[new_ptr[FX]];
	vcvar = (float) (new_ptr [V])/2.0-21.0;

	/* calculate controls for each formant */
	for (channel = FN; channel <= F3; channel++) {
		if (channel == FN)
			ind = 5;
		else
			ind = channel-1;
	
		voxmix [ind] = sqrt (RANGE(0.0, (dvoffs [ind] + vcvar) / 10.0, 1.0));
		for (j = 0; j < nc; j++) {
			ffq [ind][j] = flflu [ind] + (float) (new_ptr [channel] - 1) * finlu [ind];
			gain [ind][j] = amptab [new_ptr [channel+4]];
		}
	}

	/* F4  and unvoiced channel */
	for (ind = 3; ind <= 4; ind ++) {
		if ((ind == 4) && (new_ptr[V] > 32)) {
			for (j = 0; j < nc; j++)
				gain [4][j] = 0.0;
		}
		else {
			for (j = 0; j < nc; j++)
				gain [ind][j] = amptab [new_ptr [AHF]];
		}
		
		for (j = 0; j < nc; j++)
			ffq [ind][j] = flflu [ind];

		voxmix [ind] = sqrt (RANGE(0.0, (dvoffs [ind] + vcvar) / 10.0, 1.0));
	}

	for (j = 0; j < ncx; j++) {
		po [j] = povar;
		pl [j] = plvar;
	}

	/* calculate degree of nasality */
	x = MAX(0.0, ((float) new_ptr [ALF] - (float) new_ptr [A1] - aoffdn));
	y = MAX(0.0, (float) ((float) new_ptr [F1]  - foffdn));
	dnasal = MIN(1.0, x * y / scaldn);

	/* set control flags */
	/* clamp formant controls - due to frequency jumps */
	clampf = FALSE;
	
	for (j = FN; j <= F3; j++)
	       clampf = clampf || (abs (new_ptr [j] - old_ptr [j]) >= ifrecl);
		
	/* clamp formant controls - due to amplitude jumps */
	for (j = ALF; j <= AHF; j++)
		clampf = clampf || (abs (new_ptr [j] - old_ptr [j]) >= iampcl);

	/* excitation clamping */
	clampx = abs (new_ptr [FX] - old_ptr [FX]) >= ipocl;

	/* completely voiced or unvoiced frame */
	uvlfr = uvfr;
	uvfr = TRUE;
	vfr = TRUE;

	for (i=0; i<=5; i++) {
		uvfr = uvfr && (voxmix [i] == 0.0); /* totally unvoiced */
		vfr = vfr && (voxmix [i] == 1.0);   /* totally   voiced */
	}

	/* unvoiced to voiced transition */
	uvtov = uvlfr && (~uvfr);

	/* unvoiced / voiced single impulse */
	impexc = FALSE;
	j = 0;
	for (i = ALF; i <= AHF; i++) {
		if ((new_ptr[i] - old_ptr[i]) >= iampcl)
			j++;
	}
	if (j>=2)
		impexc = TRUE;

	/* smooth synthesizer controls */
	if (clampf) {
		/* clamp formant controls */
		for (i = 1; i <= 20; i++) {
			for (j = 0; j<= 5; j++) {
				amp_con_filt (j, &gain[ind][0], any );
				if (j <= 2)
					fre_con_filt (j, &ffq[j][0], any );
			}
		}
	}

	/* control signal smoothing filters */
	for (j = 0; j<= 5; j++) {
		amp_con_filt (j, &gain[ind][0], &gain[ind][0] );
		if (j <= 2)
			fre_con_filt (j, &ffq[j][0], &ffq[j][0] );
	}

	if (clampx) {
		/* clamp excitation controls */
		for (i=1; i <= 20; i++) {
			po_filt (po, any );
			pl_filt (pl, any );
		}
	}

	po_filt (po, po );
	pl_filt (pl, pl );

}

void po_filt ( arrin, arrout )
float 	*arrin, *arrout;
{
/**FILTER
 Excitation control smoothing filters          
 Both filters contain only a double real pole. 
 Sampling rate 2 kHz				 
 Gain factor   1 at 0 Hz			 
 Poles (-14.0, 0) (-14.0, 0)
*/

	static float	x1 = 0.0, x2 = 0.0;
	register int 	i;

	for ( i = 0; i < ncx; i++ )
		EXCF(*arrout++,*arrin++,x1,x2);
}

void pl_filt ( arrin, arrout )
float 	*arrin, *arrout;
{
/**FILTER
 Excitation control smoothing filters          
 Both filters contain only a double real pole. 
 Sampling rate 2 kHz				 
 Gain factor   1 at 0 Hz			 
 Poles (-14.0, 0) (-14.0, 0)
*/	 

	static float 	x1 = 0.0, x2 = 0.0;
	register int 	i;

	for ( i = 0; i < ncx; i++ )
		EXCF(*arrout++,*arrin++,x1,x2);
}

/**FILTER
 formant control signal smoothing filters 
 all filters contain only a double        	
 real pole 				    	
 sampling rate 5 kHz			    
 gain factor 1.0 at 0 Hz		    
 poles (-47.0, 0 ), (-47.0, 0 )
*/

void amp_con_filt ( nchan, arrin, arrout )
int nchan;
float *arrin, *arrout;
{
	static float 	last1[6];
	static float	last2[6];
	register float 	x1, x2;
	register int 	i;

	x1 = last1 [ nchan ];
	x2 = last2 [ nchan ];

	for ( i = 0; i < nc; i++ )
		FCSF(*arrout++,*arrin++,x1,x2);

	last1 [ nchan ] = x1;
	last2 [ nchan ] = x2;
}

/**FILTER
 formant control signal smoothing filters
 all filters contain only a double 
 real pole 			
 sampling rate 5 kHz		
 gain factor 1.0 at 0 Hz	
 poles (-47.0, 0 ), (-47.0, 0 )	    
*/

void fre_con_filt ( nchan, arrin, arrout )
int	nchan;
float 	*arrin, *arrout;
{
	static float	last1[6];
	static float	last2[6];
	register float	x1, x2;
	register int	i;

	x1 = last1 [ nchan ];
	x2 = last2 [ nchan ];

	for ( i = 0; i <= nc; i++ )
		FCSF(*arrout++,*arrin++,x1,x2);

	last1 [ nchan ] = x1;
	last2 [ nchan ] = x2;
}

/*
 * excite ()
 * 
 * This subroutine prepares one frame of voiced excitation in array 
 * 'store' at a sampling rate of four times the main rate, prior to
 * l.p. filtering and down sampling at the main rate. 
 * A pulse shape read from array 'expat' is distributed throughout 
 * 'store' with period 'po' and pulse length 'pl' (in seconds).
 * Common variables 'npatt' and 'posn' are used to remember 'expat'
 * sample number and actual (real number) associated 'store' position
 * at frame boundaries.
 *
 */

void excite ()
{ 
	int	i, j;
	float 	store [FSAMP*4]; /* temporary array to store pre-downsampled excitation */
	float 	any [FSAMP*4];   /* output of clamping if required */

	/* generate unvoiced excitation if required */
	if (! vfr) {
		arzero (uvox, lf);
		gnoise ();
	}

	arzero ( store, lfx);
	arzero ( glarea, nc);
	if ( !uvfr ) {
		gen_excite ( store );

		/* filter excitation */
		excite_filt ( store, store );

		/* down sample to main rate */
		i = 0;
		for ( j = 4; j <= lfx; j += 4 )
			vox [ i++ ] = store [j-1];	
	}
	else {
		if ( ! uvlfr ) {
			for ( i = 1; i <= 6; i++)
				excite_filt (store, any);
			arzero (vox, lf);
		}
	}

}

/*
 * gnoise ()
 *
 * provides one frame of unvoiced excitation held in array 'uvox' with the
 * same power of the voiced excitation.
 */

void gnoise ()
{
	register int   sum;
	register int   i, j;
	int 	       istart;
	
	istart = 0;
	if (impexc)
		istart = nsil;

	if (istart != lf) {
		for (i = istart; i < lf; i++) {
			sum = 0;
			for ( j = 0; j <= 11; j++)
				sum += rand() & 0x7FFF;
			uvox [i] = (((float) sum) / 32768.0 - 6.0 ) * pwscl;
		}
	}

	if (impexc)
		uvox [9 * nsil / 10] = pwscl * sqrt ((double) nsil);

}


/*
 * gen_excite ()
 * routine generates one frame of excitation in array 'store'
 */

void gen_excite ( store )
float store [];	
{
	float	sample;
	int 	nctl, ip1, ip2, i, j, iga, nseg;

	ixctl = 0;
	if (posn < 0.0)
		error ("From subroutine excite - negative value for sample position", NULL);

	if (uvtov) {
		posn = lfx / 4 ;
		npatt = 0;
	}
	if (posn < 1.0) {
		sgap = usrate * pl [0] / 72.0 ;
		pgap = usrate * ( po [0] - pl [0] );
	}
	else {
		adjust();
	}

	while (TRUE) {
		nctl = ((int) (posn - 0.5 ) / lc4) ;
		for ( i = npatt ; i <= 71 ; i++ ) {
			nseg = ( (int) (posn - 0.5 ) / lc4 );
			if ( nctl < nc ) {
				if (nseg >= nc )
					nseg = nc - 1;
				iga = i / 4 ;
				for ( j = nctl ; j <= nseg ; j++ )
					glarea [j] = gapat [iga];
	
				nctl = nseg ;
			}
			sample = expat [i];
			ip1 =  (int) posn ;
			ip2 = ip1 + 1 ;
			if (ip1 >= lfx + 1) {
				npatt= i;
				posn -= (float) (lfx);
				return;
			}
			if (ip1 > 0 ) {
				store [ip1-1] += (ip2 - posn) * sample ;
			}
			if (ip1 > lfx) {
				npatt = i;
				posn -= (float) (lfx);
				return;
			}
			store [ip2-1] += (posn - ip1) * sample;
			posn += sgap ;
			adjust ();
		}
		npatt = 0;
		posn += pgap;
		adjust ();
	}
}

/* Routine called from excite to adjust current expat sample position
 * in array store and update sample gap (sgap) and pulse gap (pgap),
 * if period (po) and pulse length (pl) changed during frame (at posn= dist)
 */

void adjust()
{
	boolean	space;
	float	dist , ponew , plnew , tgnew , tgv , diff ;
  
	tgv = pov - plv ;
	space = ( npatt == 0 );

	while ( TRUE ) {
		dist = (float) (ixctl * lcx ) + 0.5 ;
 	
		if ( posn <= dist ) {
			return ;
		}
		if ( ixctl >= ncx ) {
			return ;
		}

		ponew = po [ixctl];
	 	plnew = pl [ixctl];
		ixctl++;

		tgnew = (ponew-plnew) ;

		while ( tgnew <= 0.0 ) {
		      	tgnew += ponew ;
		}

		diff = posn - dist ;
	
		if ( space ) {
			posn = dist + diff * tgnew / tgv ;
		}
		else {
			posn = dist + diff * plnew / plv ;
		}	

		plv = plnew ;
		pov = ponew ;
	 	tgv = tgnew ;
	 	sgap = usrate * plv / 72.0;
		pgap = usrate * tgv ;
	}
}      

/* filter excitation waveform */
void excite_filt ( xin, xout )
float	*xin;
float	*xout;
{
/**FILTER
 * Voiced excitation low pass filter.
 * Sixth order LP elliptic, 
 * cut off 4.5 kHz with a partial integrator.
 * 
 * Sampling rate 80 kHz
 * Gain factor   3.9 at 2 kHz
 * poles: (-8,0) (-221.7,4535) (-816.4,3702.4) (-1544,1546.9)
 * zeros: (-640,0) (0,6134.5) (0,7657.2) (0,16820)
 */
	static		float st1, st2, st3, st4, st5, st6, st7;
	register	float x1, x2, x3, x4, x5, x6, x7;
	register	float y;
	register	int i;

	x1 = st1;
	x2 = st2;
	x3 = st3;
	x4 = st4;
	x5 = st5;
	x6 = st6;
	x7 = st7;

	for (i = lfx; i; i--) {

		VELPF1(y,*xin++,x1,x2);

		VELPF2(y,y,x3,x4);

		VELPF3(y,y,x5,x6);

		VELPF4(*xout++,y,x7);

	}

	st1 = x1;
	st2 = x2;
	st3 = x3;
	st4 = x4;
	st5 = x5;
	st6 = x6;
	st7 = x7;
}

/* mix 
 *
 *  A subroutine to mix one frame of voiced excitation held in array 'vox'
 * with one frame of unvoiced excitation held in array 'uvox' according to
 * the degree of voicing control 'voxmix [n]' for the particular formant.
 * The glottal function in array 'glarea' and the constant unvoiced glottal
 * area 'gouv', are mixed in the same way.
 *
 */

void mix(nchan)
int	nchan;
{
	float		v, g, gv, hrel, gouvrl ;
	int		i, j;
	static float	temp [FSAMP];

	v = voxmix [ nchan ];
	if ( v == 1.0 ) { /* fully voiced */
		for ( i = 0 ; i < nc ; i++ ) {
			g = gain [nchan] [i] ; /* get gain */
			glamix [i] = glarea [i]; /* get glottal area due to voicing */
			for ( j = ictlst [i] ; j <= ictlfn [i] ; j++ ) {
				formant [j] = g * vox [j];
			}
		}
	}
	else if (v != 0.0) { /* mixed excitation */
		if ( v > 1.0 ) {
			error("Degree of voicing control > 1.0", NULL);
		}
		hrel = sqrt( (double) (1.0 / (v*v) - 1.0) );
					/* relative unvoiced excitation */
		gouvrl = gouv * hrel;     /* relative unvoiced glottal area */
		for ( i = 0 ; i < nc ; i++ ) {
			gv = gain [nchan] [i] * v;
			glamix [i] = v * (glarea [i] + gouvrl ) ;
						/* glottal area mixing */
			for ( j = ictlst [i] ; j <= ictlfn [i] ; j++ ) {
				/* excitation mixing */
				formant [j] = gv * (vox [j] + hrel * uvox [j]);
			}
		}
	}
	else {
		for ( i = 0; i < nc; i++ ) {
			g = gain [nchan] [i] ;
			glamix [i] = gouv;
			for ( j = ictlst [i] ; j <= ictlfn [i] ; j++ ) {
				formant [j] = g * uvox [j];
			}
		}
	}

	if (alfcon) {
		if (nchan == 0) {
			for ( i = 0 ; i < lf ; i++ ) {
				temp [i] = formant [i];
			}
		}
		else {
			if (nchan == 5) {
				for ( i = 0 ; i < lf ; i++ ) {
					formant [i] -= temp [i];
				}
			}
		}
	}
}

/*
 * arzero (array, len)
 *
 * Routine initialises a real array to zeros.
 */

void arzero (array, len)
float	*array;
int	len;
{
	register int i;
	
	for (i = 0; i < len; i++) *array++ = 0.0;
}

/* This subroutine implements a recursive 2 pole digital filter
 * representing a basic formant . Frequency is controlled by data
 * in array ffq and bandwidth by rule. Both are modified pitch
 * synchronously depending on values in glottal area arrays 'glarea' and 'glamix'
 * i/p , o/p is array work
 */

void form_gen(nchan)
int	nchan;
{ 

	float		bandw, bwmod, freq_mod, gainc, glarbw, glarfq, r;
 	register float	alpha, beta, bn, cn, gf;
	boolean 	condsc;
	register int 	i,j,end;

	/* remember previous state */
	bn = b [ nchan ] ;
	cn = c [ nchan ] ;

	/* set filter parameters   */
	bandw = forbdw [ nchan ] + dnasal * fbdwna [ nchan ] ;
	bwmod  = fbwmod [ nchan ] ;
	freq_mod  = ffqmod [ nchan ] ;
	gainc  = gainct [ nchan ] ;
	gf     = gainc ;
	condsc = (nchan == 0) || (nchan == 5);

	/* set filter coefficients */
	for ( i = 0 ; i < nc ; i++ ) {
		glarbw = glamix [ i ] ;
		glarfq = glarea [ i ] ;
		r      = exptab [ RANGE(1,(int) (bandw + glarbw * bwmod),32) - 1 ];
		alpha  = 2.0 * r * costab [ RANGE(1,(int) (ffq [nchan] [ i ] + glarfq * freq_mod ) , 256 ) - 1];
	  	beta   = - r * r ;

		if ( condsc ) {
			gf = gainc * ( 1.0 - alpha - beta ) ; 
 		}

		/* run filter for duration of constant coefficients */
		end = ictlfn [ i ];
		for ( j = ictlst [ i ] ; j <= end ; j++ ) {
			formant [j] = gf * formant [j] + alpha * bn + beta * cn ;
			cn = bn ;
			bn = formant [j];
		}
	}

	/* retain last state information */
	b[nchan] = bn ;
	c[nchan] = cn ;
}

/*
 * output_filter ()
 * performs the final filtering on the output
 */

void output_filter (wave)
float 	*wave;
{
/**FILTER
 * output weighting filter
 * Eighth order elliptic low pass filter with partial integrator.
 * Sampling rate 20 kHz
 * Gain factor   1 at 0 kHz
 * Poles: (-640,0) (-91.6,4555.1) (-368.0,4351.2) (-998.5,3716.7)
 *        (-2256.9,1837.9)
 * zeros: (0,5030.9) (0,5325.8) (0,6216.1) (0,8381.2)
 */

	static float 	last1, last2, last3, last4, last5;
	static float	last6, last7, last8, last9;
	register float 	x1, x2, x3, x4, x5, x6, x7, x8, x9;
	register float 	y;
	register int 	i;

	x1 = last1;
	x2 = last2;
	x3 = last3;
	x4 = last4;
	x5 = last5;
	x6 = last6;
	x7 = last7;
	x8 = last8;
	x9 = last9;

	for (i = 0; i < lf; i++) {

		OWF1(y,*wave,x1,x2);

		OWF2(y,y,x3,x4);

/*
		OWF3(y,y,x5,x6);

		OWF4(y,y,x7,x8);
*/

		OWF5(*wave++,y,x9);

	}

	last1 = x1;
	last2 = x2;
	last3 = x3;
	last4 = x4;
	last5 = x5;
	last6 = x6;
	last7 = x7;
	last8 = x8;
	last9 = x9;

}

/*
 * mixing_filt ()
 * implements individual spectrum weighting filters for each formant.
 * Mixes the output together.
 * Input array 'arrin' , output array 'arrout'.
 */

void mixing_filt (nchan, arrin, arrout )
int 	nchan;
float 	*arrin, *arrout;
{
	static float f1_last1, f1_last2;
	static float f2_last1;
	static float f3_last1;
	static float f4_last1, f4_last2;
	static float f5_last1, f5_last2, f5_last3, f5_last4, f5_last5, f5_last6;
	register float 	x1, x2;
	register float 	y;
	register int 	i;
	register float	x3, x4, x5, x6;

	switch ( nchan ) {

	case 0:	/*  F1 - a real zero and phase shifting network */
		/*  Sampling rate 20 kHz			*/
		/*  Gain factor   0.141 at 0 Hz			*/
		/*  poles (-270,0)				*/
		/*  zeros (-640,0) (270,0)			*/

		x1 = f1_last1;
		x2 = f1_last2;

		for ( i = 0; i < lf; i++ ) {
			MIXF1(y,*arrin++,x1,x2);
			*arrout++ += y;
		}

		f1_last1 = x1;
		f1_last2 = x2;
		break;
		
	case 1:	/* F2  				*/
		/* Sampling rate 20 Khz		*/
		/* Gain factor   1 at 5 kHz	*/
		/* 1 zero at (0, 0)		*/

		x1 = f2_last1;

		for( i = 0; i < lf; i++) {
			MIXF2(y,*arrin++,x1);
			*arrout++ += y;
		}

		f2_last1 = x1;
	
		break;

	case 2:	/* F3 				*/
		/* Sampling rate 20 Khz		*/
		/* Gain factor   1 at 5 kHz	*/
		/* 1 zero at (0, 0)		*/

		x1 = f3_last1;

		for( i = 0; i < lf; i++) {
			MIXF3(y,*arrin++,x1);
			*arrout++ += y;
		}

		f3_last1 = x1;
	
		break;

	case 3: /* FHFV */
		/* Sampling rate 20 kHz */
		/* Gain factor   1 at 5 kHz */
		/* 1 pole at (-200, 4200) */
		/* 1 zero at (0, 0) */

		x1 = f4_last1;
		x2 = f4_last2;

		for ( i = 0; i < lf; i++ ) {
			MIXF4V(y,*arrin++,x1,x2);
			*arrout++ += y;
		}

		f4_last1 = x1;
		f4_last2 = x2;
	
		break;

	case 4:	/* FHFUV - a real zero, formant pole cancelation
		 * and replace with bandpass.
		 * Sampling rate 20 kHz
		 * Gain factor   1 at 3.5 kHz
		 * poles : (-400,3550) (-150,3250) (-150,3850)
		 * zeros : (0,0) (-155,3500)
		 */
		x1 = f5_last1;
		x2 = f5_last2;
		x3 = f5_last3;
		x4 = f5_last4;
		x5 = f5_last5;
		x6 = f5_last6;

		for ( i = 0; i < lf; i++) {

			MIXF4UV1(y,*arrin++,x1,x2);

			MIXF4UV2(y,y,x3,x4);

			MIXF4UV3(y,y,x5,x6);

			*arrout++ += y;

		}

		f5_last1 = x1;
		f5_last2 = x2;
		f5_last3 = x3;
		f5_last4 = x4;
		f5_last5 = x5;
		f5_last6 = x6;
					
		break;

	case 5: /* FN a null filter with gain .141 at 0 Hz */
		for ( i = 0; i < lf ; i++) {
			*arrout++ += 0.141 *  *arrin++;
		}
		break;
	}

}

