/* trmodsp -- use track item to modify gain of speech waveform */

/* M.A. Huckvale - University College London */

/* version 1.0 - July 1993 */

/*--------------------------------------------------------------------------*/
/**MAN
.TH TRMODSP SFS1 UCL
.SH NAME
trmodsp - modify speech gain according to track item
.SH SYNOPSIS
.B trmodsp
(-d) (-i item) file
.SH DESCRIPTION
.I trmodsp
is a program to modify the amplitude of a speech waveform
according to a supplied track item.  In normal mode, the speech
amplitude is multiplied by the track, an option '-d' converts the
track from dB first.
Interpolation is performed between track samples to remove most
discontinuities from the modified speech.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item.
.TP 11
.B -d
Track item in in dB. 0=multiply by 1, 6dB=multiply by 2, -6dB=multiply by 0.5.
.SH INPUT ITEMS
.IP 1.xx 11
Any speech item.
.IP 16.xx 11
Any track item.
.SH OUTPUT ITEMS
.IP SP 11
Gain modified speech waveform.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/

#define PROGNAME "trmodsp"
#define PROGVERS "1.0"
char	*progname=PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"			/* header structures */

/* process waveform in buffers */
#define BUFSIZE	4096

/* global data */
struct item_header spitem;	/* input item header data */
short	buff[BUFSIZE];		/* speech waveform */
struct item_header tritem;	/* input item header data */
float	*tr;			/* track */
struct item_header opitem;	/* output item header */

/* operations selected */
int	trdb=0;			/* track is in dB */

/* convert track from dB */
void dbconvert(buf,len)
float	*buf;
int	len;
{
	int	i;

	for (i=0;i<len;i++,buf++)
		*buf = pow(10.0,*buf/20.0);
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	int		c;		/* option char */
	int		it;
	char		*ty;
	char		*sptype="0";
	char		*trtype="0";
	char		filename[SFSMAXFILENAME];	
					/* database file name */
	int		fid,ofid;
	int		i,j,len,tridx;
	double		t,ftridx,m1,m2;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:d")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Modify speech by track data V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item spec */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else if (it == TR_TYPE)
					trtype = ty;
				else
					error("unsuitable item specification %s",optarg);
			}
			else
				error("illegal item specification %s",optarg);
			break;
		case 'd' :	/* track is in dB */
			trdb=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-d) file\n",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* load track item */
	getitem(filename,TR_TYPE,trtype,&tritem,(void **)&tr);
	if (trdb) dbconvert(tr,tritem.numframes);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("unable to find file '%s'",filename);
		else
			error("access error on '%s'",filename);
	}

	/* locate input speech item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("unable to find input SP item in '%s'",filename);

	/* create output item header */
	sfsheader(&opitem,spitem.datatype,spitem.floating,
			spitem.datasize,spitem.framesize,
			spitem.frameduration,spitem.offset,
			spitem.windowsize,spitem.overlap,spitem.lxsync);
	if (trdb) {
		sprintf(opitem.history,"%s(%d.%02d,%d.%02d;decibel)",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			tritem.datatype,tritem.subtype);
	}
	else {
		sprintf(opitem.history,"%s(%d.%02d,%d.%02d)",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			tritem.datatype,tritem.subtype);
	}

	/* open output channel */
	if ((ofid=sfschannel(filename,&opitem)) < 0)
		error("unable to open output file",NULL);

	/* process file */
	for (i=0;(len=sfsread(fid,i,BUFSIZE,buff)) > 0;i+=len) {

		for (j=0;j<len;j++) {
			t = spitem.offset+(i+j)*spitem.frameduration;
			ftridx = (t-tritem.offset)/tritem.frameduration;
			if (ftridx < 0) ftridx = 0;
			if (ftridx > tritem.numframes-1)
				ftridx = tritem.numframes-1;
			tridx = (int)ftridx;
			m1 = 1.0 - (ftridx-tridx);
			m2 = (ftridx-tridx);
			buff[j] *= m1*tr[tridx] + m2*tr[tridx+1];
		}

		/* write out result of processing */
		if (sfswrite(ofid,len,buff) != len)
			error("write error on output file",NULL);
	}

	/* update file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* ... that's all folks */
	exit(0);
}

