/* vcalign -- align two vocoder signals and transfer annotations */

/* m.a.huckvale - october 1987 */

/* version 1.0 */
/* version 1.1 - November 1987
	- include annotation history
*/
/* version 1.2 - April 1988
	- optional FX transfer
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH VCALIGN SFS1 UCL
.SH NAME
vcalign - align two vocoder signals to transfer annotations or fx
.SH SYNOPSIS
.B vcalign
(-i item) (-f) reffile file
.SH DESCRIPTION
.I vcalign
performs a non-linear time alignment between a vocoder signal in a 
reference file ("reffile") and a vocoder signal in a destination file
(assumed
to be of the same token) and transfers from reference to destination
either (i) annotations with appropriate changes in positions 
and lengths or (ii) a fundamental frequency contour.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version.
.TP 11
.BI -i item
Select input item numbers.
.TP 11
.B -f
Transfer FX rather than AN.
.SH INPUT ITEMS
.SS REFERENCE FILE
.IP 11.xx 11
19-channel vocoder output.
.IP 4.xx 11
(Optional) Any fundamental frequency item.
.IP 5.xx 11
(Optional) Any annotation item.
.SS DATA FILE
.IP 11.xx 11
19-channel vocoder output.
.SH OUTPUT ITEMS
.IP 4 11
Time aligned fundamental frequency item.
.IP 5 11
Time aligned annotation item in data file.
.SH VERSION/AUTHOR
1.2 - Mark Huckvale
.SH SEE ALSO
voc19(SFS1), voc26(SFS1)
*/
/*--------------------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "vcalign"
#define PROGVERS "1.2s"

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"		/* database filing system structures */

#define FXDURATION 0.01		/* output FX sampled at 100Hz */

/* global data */
struct item_header coitem1;	/* filing system item header for co ref */
struct co_rec		*co1;	/* vocoder data */
struct item_header coitem2;	/* filing system item header for co data */
struct co_rec		*co2;	/* vocoder data */
struct item_header anitem;	/* filing system item header for an */
struct an_rec		*an;	/* annotations to xfer */
struct item_header	fxitem;	/* item header for FX */
short			*fx;	/* buffer for input FX */

/* global data */
#define	DIAG 	1		/* DP path values */
#define UP 	2
#define LEFT	3
unsigned char	*direc;		/* DP directions */
float	*cudist1,*cudist2;	/* DP distance matrix lines */
float	*dist1,*dist2;		/* DP distance matrix lines */
float 	*reg;			/* DP time registration path */
int	*align;			/* DP frame alignment path */
int	xferan=1;		/* transfer options */
int	xferfx=0;

/* get fx value at time t */
int getfx(t)
double	t;
{
	int	idx;

	idx=t/fxitem.frameduration;
	if (idx < 0)
		idx=0;
	else if (idx >= fxitem.numframes)
		idx = fxitem.numframes-1;
	return(fx[idx]);
}

#define EDGE	1E10
#define MINOF(x,y,z) (((x) < (y)) ? (((x) < (z)) ? (x) : (z)) : (((y) < (z)) ? (y) : (z)))
#define MINOFIDX(x,y,z) (((x) < (y)) ? (((x) < (z)) ? 1 : 3) : (((y) < (z)) ? 2 : 3))

/* Dynamic Programming match */
void dpmatch(refnum,newnum)
int		refnum;		/* # horizontal frames */
int		newnum;		/* # vertical frames */
{
	register int	i,j;
	register float	diag,up,left;
	unsigned char	*dptr;
	float		*ftemp;
	float		metric();

	/* initialisation */
	ftemp=cudist1;
	for (i=0;i<refnum;i++) *ftemp++=EDGE;

	/* DP match */
	dptr=direc;
	for (i=0;i<newnum;i++) {
		for (j=0;j<refnum;j++) {
			if ((i==0) && (j==0)) {
				diag=0;
				left=EDGE;
			}
			else if (j==0) {
				diag=EDGE;
				left=EDGE;
			}
			else {
				diag=cudist1[j-1] + dist1[j-1];
				left=cudist2[j-1] + dist2[j-1];
			}
			up = cudist1[j] + dist1[j];
			dist2[j] = metric(&co2[i],&co1[j]);
			cudist2[j] = dist2[j] + MINOF(diag,up,left);
			*dptr++ = MINOFIDX(diag,up,left);
		}
		/* swap over buffers */
		ftemp=cudist1;
		cudist1=cudist2;
		cudist2=ftemp;
		ftemp=dist1;
		dist1=dist2;
		dist2=ftemp;
		/* inform user */
		if (((i%5)==4) && ttytest()) {
			printf("\rFrame %d/%d",i+1,newnum);
			fflush(stdout);
		}
	}
	if (ttytest()) {
		printf("\rFrame %d/%d\n",i,newnum);
		fflush(stdout);
	}

}

/* Dynamic Programming time registration path */
void timereg(refnum,newnum,reg)
int		refnum;
int		newnum;
float		*reg;
{
	register int	i;
	unsigned char	*dptr;

	/* initialise at end of table */
	dptr = direc + refnum*newnum - 1;
	reg[refnum] = (co2[coitem2.numframes-1].posn+co2[coitem2.numframes-1].size)*coitem2.frameduration;
	i = refnum - 1;

	/* pass backwards up registration path */
	while ((dptr > direc) && (i >= 0)) {
		switch (*dptr) {
		case DIAG:
			reg[i--] = co2[(dptr-direc) / refnum].posn*coitem2.frameduration;
			dptr -= refnum + 1;
			break;
		case UP:
			dptr -= refnum;
			break;
		case LEFT:
			reg[i--] = co2[(dptr-direc) / refnum].posn*coitem2.frameduration;
			dptr--;
			break;
		}
	}
}

/* get new time instant from time registration path */
float	newtime(t,reg,refnum)
float		t;		/* time to xfer */
float		*reg;		/* registration path */
int		refnum;		/* length of registration path */
{
	int	p1,p2;
	float	mix,newt;

	/* get pointers into registration path */
	p2=0;
	while ((p2<refnum) && (t >= co1[p2].posn*coitem1.frameduration)) p2++;
	p1= p2 - 1;

	/* check not off end */
	if (p2 >= refnum) return(reg[refnum]);

	/* get addmixture coefficient */
	mix = t - co1[p1].posn*coitem1.frameduration;

	/* interpolate answer */
	newt=(1.0-mix)*reg[p1]+mix*reg[p2];

#ifdef IAG
	printf("t=%g,cot=%g,reg[p1]=%g,reg[p2]=%g,newt=%g\n",
		t,co1[p1].posn*coitem1.frameduration,reg[p1],reg[p2],newt);
#endif

	return(newt);
}

/* getold time instant from time registration path */
float	oldtime(t,reg,refnum)
float		t;		/* time to xfer */
float		*reg;		/* registration path */
int		refnum;		/* length of registration path */
{
	int	p1,p2;
	float	mix,newt;

	/* get pointers into registration path */
	p2=0;
	while ((p2<refnum) && (t >= reg[p2])) p2++;
	p1= p2 - 1;

	/* check not off end */
	if (p2 >= refnum) return(co1[refnum-1].posn*coitem1.frameduration);

	/* get addmixture coefficient */
	mix = t - reg[p1];

	/* interpolate answer */
	newt=(1.0-mix)*co1[p1].posn*coitem1.frameduration+mix*co1[p2].posn*coitem1.frameduration;

#ifdef IAG
	printf("t=%g,cot=%g,reg[p1]=%g,reg[p2]=%g,newt=%g\n",
		t,co1[p1].posn*coitem1.frameduration,reg[p1],reg[p2],newt);
#endif

	return(newt);
}

/* distance metric for vocoder frames */
float metric(a,b)
struct co_rec *a,*b;
{
	register int i;
	register float sumsq=0,val,*ap,*bp;
	double	sqrt();

	ap = &a->data[0];
	bp = &b->data[0];
	for (i=19;i;i--) {
		val = *ap++ - *bp++;
		sumsq += val*val;
	}
	return(sqrt(sumsq));
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	char		*cotype="0";	/* input co-types */
	char		*antype="0";	/* default AN sub-type = last */
	char		*fxtype="0";	/* default FX sub-type = last */
	int		c;		/* option switch */
	int		it;		/* item/sub-type specifiers */
	char		*ty;		/* sub-type match */
	/* file variables */
	char		filename[SFSMAXFILENAME];
	char		refilename[SFSMAXFILENAME];
	int		ofid;
	float		postime,lentime,oldtime(),newtime(),metric();
	int		i,numf;
	double		totime;
	short		fxval;
	struct item_header item;
	struct an_rec	*annot;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:f")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Vocoder-based alignment V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == CO_TYPE)
					cotype = ty;
				else if (it == FX_TYPE) {
					fxtype = ty;
					xferfx++;
					xferan=0;
				}
				else if (it == AN_TYPE) {
					antype = ty;
					xferan++;
					xferfx=0;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'f' :	/* xfer FX */
			xferfx++;
			xferan=0;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check for option decoding error */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-f) ref_file file",PROGNAME);

	/* get reference filename */
	if (optind < argc)
		strcpy(refilename,sfsfile(argv[optind]));
	else
		error("no reference database file specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* read in data items */
	getitem(refilename,CO_TYPE,cotype,&coitem1,(void **)&co1);
	if (xferan)
		getitem(refilename,AN_TYPE,antype,&anitem,(void **)&an);
	else
		getitem(refilename,FX_TYPE,fxtype,&fxitem,(void **)&fx);
	getitem(filename,CO_TYPE,cotype,&coitem2,(void **)&co2);

	/* check can write to data file */
	if ((ofid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on %s",filename);
	close(ofid);

	/* get memory for calculations */
	numf = (coitem1.numframes>coitem2.numframes)?coitem1.numframes:coitem2.numframes;
	dist1=(float *)calloc(numf,sizeof(float));
	dist2=(float *)calloc(numf,sizeof(float));
	cudist1=(float *)calloc(numf,sizeof(float));
	cudist2=(float *)calloc(numf,sizeof(float));
	direc=(unsigned char *)malloc(coitem1.numframes*coitem2.numframes);
	if ((dist1==NULL) || (dist2==NULL) || (cudist1==NULL) || (cudist2==NULL) || (direc==NULL))
		error("unable to get sufficient memory for buffers",NULL);

	/* do DP matching */
	dpmatch(coitem1.numframes,coitem2.numframes);
	free(dist1);
	free(dist2);
	free(cudist1);
	free(cudist2);

#ifdef IAG
	/* print out directions matrix */
	dptr=direc;
	for (i=0;i<coitem2.numframes;i++) {
		for (j=0;j<coitem1.numframes;j++) {
			switch (*dptr++) {
			case 1:		printf("\\"); break;
			case 2:		printf("|"); break;
			case 3:		printf("-"); break;
			default:	printf("?"); break;
			}
		}
		printf("\n");
	}
#endif

	/* create output item header */
	if (xferan) {
		sfsheader(&item,anitem.datatype,
			anitem.floating,
			anitem.datasize,
			anitem.framesize,
			anitem.frameduration,
			coitem2.offset,
			anitem.windowsize,
			anitem.overlap,
			anitem.lxsync);
		sprintf(item.history,"%s/AN(%d.%02d;reffile=%s,items=%d.%02d,%d.%02d,history=%s)",
			PROGNAME,
			coitem2.datatype,coitem2.subtype,
			refilename,
			coitem1.datatype,coitem1.subtype,
			anitem.datatype,anitem.subtype,anitem.history);
	}
	else {
		sfsheader(&item,fxitem.datatype,
			fxitem.floating,
			fxitem.datasize,
			fxitem.framesize,
			FXDURATION,
			coitem2.offset,
			fxitem.windowsize,
			fxitem.overlap,
			fxitem.lxsync);
		sprintf(item.history,"%s/FX(%d.%02d;reffile=%s,items=%d.%02d,%d.%02d,history=%s)",
			PROGNAME,
			coitem2.datatype,coitem2.subtype,
			refilename,
			coitem1.datatype,coitem1.subtype,
			fxitem.datatype,fxitem.subtype,fxitem.history);
	}
	
	/* open output channel */
	if ((ofid=sfschannel(filename,&item)) < 0)
		error("could not open output channel to %s",filename);

	/* calculate time registration path */
	reg = (float *)calloc(coitem1.numframes+1,sizeof(float));
	timereg(coitem1.numframes,coitem2.numframes,reg);

#ifdef IAG
	/* print out alignment */
	for (i=0;i<coitem1.numframes;i++) 
		printf("%d %10.1f ->%10.1f\n",i,
			1000*co1[i].posn*coitem1.frameduration,
			1000*reg[i]);
#endif

	if (xferan) {
		/* transfer annotations */
		for (i=0;i<anitem.numframes;i++) {
			annot = &an[i];

			/* calculate new position and length */
			postime = annot->posn*anitem.frameduration;
			lentime = postime + annot->size*anitem.frameduration;
			annot->posn = (int)(0.5 + newtime(postime,reg,coitem1.numframes)/anitem.frameduration);
			annot->size = (int)(0.5 + newtime(lentime,reg,coitem1.numframes)/anitem.frameduration) - annot->posn;

			/* put annotation */
			if (sfswrite(ofid,1,annot) != 1)
				error("write error on temporary file ",NULL);
#ifdef IAG
		printf("posn=%d,size=%d,text=%s\n",annot->posn,annot->size,annot->label);
#endif
		}
	}
	else {
		/* transfer fx */
		totime = (co2[coitem2.numframes-1].posn+co2[coitem2.numframes-1].size)*coitem2.frameduration;
		numf = totime/FXDURATION;
		for (i=0;i<numf;i++) {
			postime = i*FXDURATION;
			fxval = getfx(oldtime(postime,reg,coitem1.numframes));
			sfswrite(ofid,1,&fxval);
		}
	}

	/* close output file and update */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks */
	exit(0);
}


