/* vtx -- estimate tx from lx by Voiscope method */

/* m.a. huckvale - march 1988 */

/* version 1.1 - December 1993
	- add variable threshold instead of fixed
*/
/* version 1.2 - March 1994
	- add filters for 11025, 22050, 44100 and 48000
*/

#define PROGNAME "vtx"
#define PROGVERS "1.2"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH VTX UCL1 SFS
.SH NAME
vtx - convert lx to tx by Voiscope method
.SH SYNOPSIS
.B vtx
(-i item) file
.SH DESCRIPTION
.I vtx
estimates the location of pitch periods from a Layngograph waveform by the
method employed in the Voiscope.  This consists of: (i) filtering, (ii) agc, and 
(iii) peak following.  This program has the advantage that it copes with any length
input signal.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item number.
.SH INPUT ITEMS
.IP 2.xx 11
Laryngograph waveform sampled at 10000, 11025, 12800, 16000,
20000, 22050, 32000, 44100 or 48000Hz.
.SH OUTPUT ITEMS
.IP 3 11
Voiscope tx
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
.SH BUGS
Works best if Lx is right way up (up = more current flow).
.SH SEE ALSO
HQtx
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"			/* header structures */

/* global data */
struct item_header 	lxitem;		/* item header data */
short			*lx;		/* speech data buffer (1 x 20ms window) */
struct item_header 	txitem;		/* item header data */
int			sampfreq;	/* current sampling frequency */

/* TX estimation control */
int	BUFSIZE=5000;			/* speech buffer size = 0.5 sec */
int	THRESH=100;			/* adapted threshold */
int	maxval=0;

/* definition of 40 Hz high-pass filters */
#define NOCOEFF 3
float	ncoeff10[NOCOEFF]={
 9.823855E-01,
-1.964771E+00,
 9.823855E-01,
};
float	dcoeff10[NOCOEFF]={
 1.000000E+00,   
-1.964461E+00,  
 9.650813E-01,  
};
float	ncoeff11[NOCOEFF]={
   9.840099E-01,
  -1.968020E+00,
   9.840099E-01,
};
float	dcoeff11[NOCOEFF]={
   1.000000E+00,
  -1.967764E+00,
   9.682755E-01,
};
float	ncoeff12[NOCOEFF]={
   9.862121E-01,
  -1.972424E+00,
   9.862121E-01,
};
float	dcoeff12[NOCOEFF]={
 1.000000E+00,  
-1.972234E+00,  
 9.726144E-01,   
};
float	ncoeff16[NOCOEFF]={
  9.889544E-01,
 -1.977909E+00,
  9.889544E-01,
};
float	dcoeff16[NOCOEFF]={
 1.000000E+00,   
-1.977787E+00,  
 9.780309E-01,   
};
float	ncoeff20[NOCOEFF]={
   9.911535E-01,
  -1.982307E+00,
   9.911535E-01,
};
float	dcoeff20[NOCOEFF]={
 1.000000E+00,  
-1.982229E+00,  
 9.823854E-01,
};
float	ncoeff22[NOCOEFF]={
   9.919727E-01,
  -1.983945E+00,
   9.919727E-01,
};
float	dcoeff22[NOCOEFF]={
   1.000000E+00,
  -1.983881E+00,
   9.840100E-01,
};
float	ncoeff32[NOCOEFF]={
   9.944616E-01,
  -1.988923E+00,
   9.944616E-01,
};
float	dcoeff32[NOCOEFF]={
   1.000000E+00,
  -1.988893E+00,
   9.889539E-01,
};
float	ncoeff44[NOCOEFF]={
   9.959784E-01,
  -1.991957E+00,
   9.959784E-01,
};
float	dcoeff44[NOCOEFF]={
   1.000000E+00,
  -1.991941E+00,
   9.919729E-01,
};
float	ncoeff48[NOCOEFF]={
   9.963044E-01,
  -1.992609E+00,
   9.963044E-01,
};
float	dcoeff48[NOCOEFF]={
   1.000000E+00,
  -1.992595E+00,
   9.926224E-01,
};
  
/* highpass filter waveform */
void filter(sp,len)
short			*sp;
int			len;
{
	register int	i,j;
	register float	*ncoeff,*dcoeff;
	register float	out;
	static float	oldout[NOCOEFF];

	/* choose filter */
	if (sampfreq==100) {
		ncoeff=ncoeff10;
		dcoeff=dcoeff10;
	}
	else if (sampfreq==110) {
		ncoeff=ncoeff11;
		dcoeff=dcoeff11;
	}
	else if (sampfreq==128) {
		ncoeff=ncoeff12;
		dcoeff=dcoeff12;
	}
	else if (sampfreq==160) {
		ncoeff=ncoeff16;
		dcoeff=dcoeff16;
	}
	else if (sampfreq==200) {
		ncoeff=ncoeff20;
		dcoeff=dcoeff20;
	}
	else if (sampfreq==220) {
		ncoeff=ncoeff22;
		dcoeff=dcoeff22;
	}
	else if (sampfreq==320) {
		ncoeff=ncoeff32;
		dcoeff=dcoeff32;
	}
	else if (sampfreq==441) {
		ncoeff=ncoeff44;
		dcoeff=dcoeff44;
	}
	else {
		ncoeff=ncoeff48;
		dcoeff=dcoeff48;
	}

	/* process data */
	maxval=0;
	for (i=0;i<len;i++) {
		/* denominator */
		out = *sp;
		for (j=1;j<NOCOEFF;j++) out -= dcoeff[j]*oldout[j];

		/* numerator */
		oldout[0] = out;
		out = out * ncoeff[0];
		for (j=1;j<NOCOEFF;j++) out += ncoeff[j]*oldout[j];

		/* shift old output values */
		for (j=NOCOEFF-1;j>0;j--) oldout[j] = oldout[j-1];

		/* return sample */
		*sp++ = out;
		if (fabs(out) > maxval) maxval = fabs(out);
	}
}

void peak(lx,len)
short	*lx;
int	len;
{
	register int	i;
	static float	val=0;
	static int	follow=0;
	static int	riselen=0;
	float		fraction;

	fraction = exp(log(0.15)/sampfreq);
	THRESH = maxval/15;

	for (i=0;i<len;i++,lx++) {
		if (*lx < val) {	/* not following waveform */
			if ((follow==1) && (val > THRESH)) 
				*lx=riselen;	/* store distance to start of rise */
			else
				*lx=0;
			val *= fraction;
			follow=0;
			riselen=0;		/* start of wait for rise */
		}
		else if (follow) {
			val = *lx;
			*lx = 0;
			riselen++;
		}
		else {
			val = *lx;
			*lx = 0;
			follow=1;
			riselen=1;
		}
	}
}

void writepeak(lx,len,fid)
short	*lx;
int	len;
int	fid;
{
	register int	i;
	static int	count=0;
	int		tx;

	for (i=0;i<len;i++,lx++) {
		count++;
		if ((*lx != 0) && (count > sampfreq/4)) {
			tx = count - *lx;	/* adjust to location of rise */
			sfswrite(fid,1,&tx);
			count = *lx;
		}
	}
	if (len==0) sfswrite(fid,1,&count);
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	int		c;		/* option char */
	int32		it;
	char		*ty;
	char		*lxtype = "0";	/* input sp sub-type = last */
	char		filename[SFSMAXFILENAME];
					/* database file name */
	int		fid,ofid;	/* file descriptors */
	int		i,len;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Tx by Voiscope method V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item spec */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == LX_TYPE)
					lxtype=ty;
				else
					error("unsuitable item specification %s",optarg);
			}
			else
				error("illegal item specification %s",optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) file\n",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("'%s': file not found",filename);
		else
			error("access error on '%s'",filename);
	}

	/* locate input lx item */
	if (!sfsitem(fid,LX_TYPE,lxtype,&lxitem))
		error("cannot find input LX item in '%s'",filename);

	/* get window parameters */
	sampfreq = 0.5 + (0.01/lxitem.frameduration);

	/* check valid sampling frequency */
	if (	(sampfreq!=100) &&
		(sampfreq!=110) &&
		(sampfreq!=128) &&
		(sampfreq!=160) &&
		(sampfreq!=200) &&
		(sampfreq!=220) &&
		(sampfreq!=320) &&
		(sampfreq!=441) &&
		(sampfreq!=480)
	)
		error("speech not sampled at 10000,11025,12800,16000,20000,22050,32000,44100,48000Hz",NULL);

	/* get input buffer */
	BUFSIZE = 20 * 50 * sampfreq;	/* 10 seconds */
	if ((lx=(short *)sfsbuffer(&lxitem,BUFSIZE)) == NULL)
		error("cannot get buffer for lx",NULL);

	/* generate output item header */
	sfsheader(&txitem,TX_TYPE,0,4,1,lxitem.frameduration,lxitem.offset,0,0,1);
	sprintf(txitem.history,"%s(%d.%02d)",
		PROGNAME,lxitem.datatype,lxitem.subtype);

	/* open output channel */
	if ((ofid=sfschannel(filename,&txitem)) < 0)
		error("cannot open temporary file",NULL);

	/* perform processing */
	for (i=0;(len=sfsread(fid,i,BUFSIZE,lx)) > 0;i+=len) {

		/* filter window */
		filter(lx,len);

		/* peakfollow */
		peak(lx,len);

		/* write out tx */
		writepeak(lx,len,ofid);

		/* report progress */
		if (ttytest()) {
			printf("\rFrame %d/%d",i+len,lxitem.numframes);
			fflush(stdout);
		}

	}
	/* flush last tx period */
	writepeak(lx,0,ofid);
	if (ttytest()) printf("\r                       \r");

	/* and update file */
	if (!sfsupdate(filename))
		error("backup error on '%s'",filename);

	/* ... that's all folks */
	exit(0);
}

