/* anload -- load an annotation item from a text description of times and labels */

/* M.A.Huckvale - December 1987 */

/* version 1.0 */
/* version 1.1 - February 1989
	- fix lxsync in output AN
*/
/* version 1.2 - June 1989
	- TIMIT database option, annotations with sample ranges
*/
/* version 1.3 - March 1992
	- add -h HTK format load option
*/

/*-------------------------------------------------------------------*/
/**MAN
.TH ANLOAD SFS1 UCL
.SH NAME
anload -- load an annotation item from text description
.SH SYNOPSIS
.B anload
(-f freq) (-t type) (-s|-h) textfile file
.SH DESCRIPTION
.I anload
constructs a new annotation item in the given file from text
descriptions of annotations in a text file.
The format of the text file is one annotation per line.  In the
normal mode this should have two
entries: firstly the time in seconds, secondly the text label (not
containing spaces).  In the "sample" mode (-s switch), each line
should contain three entries: the starting sample number, the final sample number,
and the annotation.  Ensure that the sampling frequency is correct in the
sample mode.
.PP
.B Options
.TP 11
.B -I
Identify the program version.
.TP 11
.BI -f freq
Use a sampling interval based on 1/freq seconds for sampling the time of each
annotation. Default 10000.
.TP 11
.BI -t type
Place a "type" indicator in the created item header, with the argument
.I type.
.TP 11
.B -s
Sample mode.  Text lines contain sample ranges, not times.  Ensure sampling frequency
is supplied.
.TP 11
.B -h
HTK format load.  Equivalent to switches: -s -f 10000000.
.SH OUTPUT ITEMS
.IP AN 11
Loaded annotation item
.SH VERSION/AUTHOR
1.3 - Mark Huckvale
.SH BUGS
Annotation sizes are calculated automatically, the last annotation
has length zero.
*/
/*---------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "anload"
#define PROGVERS "1.3s"
char	*progname = PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard i-o routines */
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header 	anitem;			/* annotation item header  */
struct an_rec		*an;			/* annotation record */
char			textname[SFSMAXFILENAME];	/* text file name */
char			filename[SFSMAXFILENAME];	/* dbase file name */
double			fdur=1.0E-4;		/* default sample duration */
char			*antype;		/* annotation type */
int			sampmode=0;		/* sample mode */

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;			/* option index */
	extern char	*optarg;		/* option argument ptr */
	int		errflg = 0;		/* option error flag */
	int		c;			/* option switch */

	/* processing variables */
	int		i,first;
	FILE		*ip;			/* input text file */
	int		ofid;			/* output file descriptors */
	double		tim;
	char		line[255],lab[255];
	int		s1,s2;

	/* decode switches */
	while ( (c = getopt(argc,argv,"If:t:sh")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Load AN item from text V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'f' :	/* select sampling frequency */
			i = atoi(optarg);
			if (i<=0)
				error("bad sampling frequency: %s",optarg);
			else
				fdur = 1.0/i;
			break;
		case 't' :	/* annotation type */
			antype=optarg;
			break;
		case 's' :	/* sample mode */
			sampmode++;
			break;
		case 'h' :	/* HTK format load */
			sampmode++;
			fdur = 1.0E-7;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<3))
		error("usage: %s (-I) (-f freq) (-t type) (-s|-h) textfile file",PROGNAME);

	/* get text file name */
	if (optind < argc) 
		strcpy(textname,argv[optind]);
	else
		error("no text file specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open text file */
	if ((ip=fopen(textname,"r"))==NULL)
		error("cannot open %s",textname);

	/* open data file */
	if ((ofid=sfsopen(filename,"w",NULL)) < 0) {
		if (ofid==-1)
			error("cannot find file %s",filename);
		else
			error("access error on %s",filename);
	}
	sfsclose(ofid);

	/* create output item header */
	sfsheader(&anitem,AN_TYPE,-1,1,-1,fdur,0.0,0,0,1);
	if (antype) 
		sprintf(anitem.history,"%s(file=%s,type=%s)",PROGNAME,textname,antype);
	else
		sprintf(anitem.history,"%s(file=%s)",PROGNAME,textname);

	/* open output channel */
	if ((ofid=sfschannel(filename,&anitem)) < 0)
		error("unable to open temporary file",NULL);

	/* get output record buffer */
	if ((an=(struct an_rec *)sfsbuffer(&anitem,1))==NULL)
		error("could not get memory buffer",NULL);

	/* process input file line by line */
	first=1;
	while (fgets(line,255,ip) != NULL) {
		if (sampmode) {
			if (sscanf(line,"%d %d %s\n",&s1,&s2,lab) < 3) break;
			if (!first && (s1 < an->posn))
				error("annotations not in order in '%s'",textname);
			an->posn = s1;
			an->size = s2-s1;
			strcpy(an->label,lab);
			if (sfswrite(ofid,1,an) != 1)
				error("write error on output file",NULL);
		}
		else {
			if (sscanf(line,"%lf %s\n",&tim,lab) < 2) break;
			if (!first) {
				an->size = tim/fdur - an->posn;
				if (sfswrite(ofid,1,an) != 1)
					error("write error on output file",NULL);
			}
			an->posn = tim/fdur;
			strcpy(an->label,lab);
		}
		first=0;
	}
	if (!first && !sampmode) {
		an->size = 0;
		if (sfswrite(ofid,1,an) != 1)
			error("write error on output file",NULL);
	}

	/* update data file */
	if (!sfsupdate(filename))
		error("backup error on %s",filename);

	/* that's all folks */
	exit(0);

}
