/*---------- cnvils - convert dbasefile to/from ilsfile ----------*/

/* M.A. Huckvale -- January 1986 */

/* Version 1.0 - simple sampled data transfer */
/* Version 1.1 - ils analysis to FX item option */
/* Version 1.2 - ils analysis to SY item option */
/* Version 1.3 - will copy over an FX item from SPAR file to ILS file */
/* Version 1.4 - tx item to label file option on sampled data transfer */
/* Version 1.5 - Release 1, new SY item, -r flag for raw formant estimates */
/*	   1.5a - bug in formant count for -t */
/*	   1.5b - bug in getting analysis data from PAN */
/*	        - bug in number of labels generated from tx */
/* Version 1.6  - new item update */
/* Version 1.7  - creates contiguous ILS files */
/* version 1.8  - sets up windowsize and overlap field */
/* version 1.9 - december 1986
	- copies amplitudes unchanged into fm items
	- copies overall gain (ilshead[122]) into fm items
*/
/* version 2.0 - November 1987
	- SFS version
	- bug in TX sampling rate fixed
*/
/*--------------------------------------------------------------------------*/
/**MAN
.TH CNVILS SFS1 UCL
.SH NAME
cnvils - convert data between SFS and ILS formats
.SH SYNOPSIS
.B cnvils
[ (-i item) (-l) ] [ (-o item) (-h history) ] ilsfile dbfile
.SH DESCRIPTION
.I cnvils
copies data from existing SFS files into new ILS files or updates 
existing SFS files with data from whole ILS files.  Default operation 
copies speech item into ILS sampled data file.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.PP
SFS to ILS conversions:
.TP 11
.BI -i sp
Copy speech item from SFS file into new ILS sampled data file.
.TP 11
.BI -i lx
Copy excitation item from SFS file into new ILS sampled data file.
.TP 11
.BI -i fx
Overlay Fx item from SFS file onto existing ILS analysis file.
.TP 11
.B -l
Generate label file from TX data when copying waveforms from SFS to ILS formats.
.TP 11
.BI -i item
Select input item type and subtype.  This switch also actuates appropriate conversion from SFS.
.PP
ILS to SFS conversions:
.TP 11
.BI -o sp
Copy sampled data into SP item.
.TP 11
.BI -o lx
Copy sampled data into LX item.
.TP 11
.BI -o fx
Copy fundamental frequency contour into FX item.
.TP 11
.BI -o fx
Copy fundamental frequency from ILS analysis file to an FX item.
.TP 11
.BI -o sy
Copy processed formant estimates from ILS analysis file to an SY item.
.TP 11
.B -o fm
Copy raw formant estimates from ILS analysis file to an FM item.
.TP 11
.BI -h history
Use the string "history" in the history field of the created item.  This
should include the name of the process that ils has performed and the
item number of the items in the file acting as input.  For example, 
re-synthesized lpc speech may use: -h "sns(1.01)".
.SH INPUT ITEMS
.IP 1.xx 11
Any speech item.
.IP 2.xx 11
Any excitation item.
.IP 3.xx 11
Any period data item (for labels file).
.IP 4.xx 11
Any Fx item.
.SH OUTPUT ITEMS
.IP 1 11
ILS processed speech
.IP 2 11
ILS processed lx
.IP 4 11
ILS processed fundamental frequency.
.IP 7 11
ILS processed synthesizer control parameters
.IP 12 11
ILS processed raw formant estimates
.SH VERSION/AUTHOR
2.0 - Mark Huckvale
.SH BUGS
History fields for created items not updated correctly.  Use "-h hist" switch.
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include "sfs.h"
#define PROGNAME "cnvils"
#define PROGVERS "2.0s"

/* global data */
struct main_header head;	/* main header */
struct item_header item;	/* item header */
struct item_header txitem;	/* tx item header */
char	dbfilename[SFSMAXFILENAME];	/* data file */
char	ilsfilename[SFSMAXFILENAME];	/* ILS file */
char	history[256]=PROGNAME;		/* output history */

/* write integer value to file */
void wrint(op,val)
FILE *op;
int val;
{
	putc((val>>24) & 0xFF,op);
	putc((val>>16) & 0xFF,op);
	putc((val>>8)  & 0xFF,op);
	putc(val       & 0xFF,op);
}

/* write 4-char string as integer value to file */
void wrich(op,s)
FILE *op;
char s[];
{
	putc(s[0],op);
	putc(s[1],op);
	putc(s[2],op);
	putc(s[3],op);
}


/* write ils header */
int ilshead(op)
FILE *op;
{
	int	i;
	int	samplefreq,ms10,ms20;

	samplefreq = 1.0 / item.frameduration;		/* # samples/sec */
	ms10 = 0.010 / item.frameduration + 0.5;	/* # samples/10ms */
	ms20 = 0.020 / item.frameduration + 0.5;	/* # samples/20ms */

	/* set up ils header with suitable default values */
	/* 1. # points in analysis window */
	wrint(op,ms20);
	/* 2. # autoregressive coefficients */
	wrint(op,12);
	/* 3. pre emphasis constant */
	wrint(op,98);
	/* 4. shift interval per frame */
	wrint(op,ms10);
	/* 5. Hamming window */
	wrich(op,"Y   ");
	/* 6. # sampled data blocks */
	i = item.numframes / 256;
	wrint(op,i);
	/* 7. # spectral resonance peaks */
	wrint(op,4);
	/* 8. starting frame for analysis */
	wrint(op,1);
	/* 9. number of frames analysed */
	i = item.numframes / ms10;
	wrint(op,i);
	/* 10. starting sector for analysis */
	wrint(op,0);
	/* 11 */
	wrint(op,0);
	/* 12. # data points in last block */
	i = item.numframes % 256;
	wrint(op,i);
	/* 13 - 16 */
	for (i=13;i<17;i++) wrich(op,"    ");
	/* 17 - 19 */
	for (i=17;i<20;i++) wrint(op,0);
	/* 20. file number of data file */
	wrint(op,1);
	/* 21 - 22 */
	wrint(op,0);
	wrint(op,0);
	/* 23 - 26 */
	for (i=23;i<27;i++) wrich(op,"    ");
	/* 27 - 32 */
	for (i=27;i<33;i++) wrint(op,0);
	/* 33 - 35 */
	for (i=33;i<36;i++) wrich(op,"    ");
	/* 36 - 60 */
	for (i=36;i<61;i++) wrint(op,0);
	/* 61. sample freq (**10) */
	wrint(op,0);
	/* 62. sample freq */
	wrint(op,samplefreq);
	/* 63. sampled data identifier */
	wrint(op,-32000);
	/* 64. initialised flag */
	wrint(op,32149);
	/* 65 - 128 */
	for (i=65;i<129;i++) wrint(op,0);

	return(0);
}

/*========================= FROM ILS ================================*/

/* copy ils sampled data */
void copysamples(ilsfil,outitem,numpadding,numsamples,samplefreq)
FILE	*ilsfil;
int	outitem,numpadding,numsamples;
double	samplefreq;
{
	int 	i,cnt;
	int	fid;
	short	s,buff[256];

	/* create new item header */
	sfsheader(&item,outitem,0,2,1,1.0/samplefreq,0.0,1,0,0);
	strcpy(item.history,history);

	/* open output channel */
	if ((fid=sfschannel(dbfilename,&item)) < 0)
		error("cannot open channel to %s",dbfilename);

	/* write padding */
	s=0;
	for (i=0;i<numpadding;i++) {
		sfswrite(fid,1,(char *)&s);	
		getc(ilsfil);		/* update ILS file pointer */
		getc(ilsfil);
	}
	numsamples -= numpadding;
	/* copy data across */
	while (numsamples > 0) {
		cnt = (numsamples > 256) ? 256 : numsamples;
		fread(buff,2,cnt,ilsfil);
		if (sfswrite(fid,cnt,(char *)buff) != cnt)
			error("write error on temporary file",NULL);
		numsamples -= cnt;
	}
}

/* copy ils fx data */
void copyfx(ilsfil,numpadding,numframes,framdur)
FILE	*ilsfil;
int	numpadding,numframes;
double	framdur;
{
	int	i,fid;
	short	record[128];
	short	s;

	/* create item header */
	sfsheader(&item,FX_TYPE,0,2,1,framdur,0.0,1,0,0);
	strcpy(item.history,history);

	/* open output channel */
	if ((fid=sfschannel(dbfilename,&item)) < 0)
		error("could not open channel to %s",dbfilename);

	/* put in padding */
	s=0;
	for (i=0;i<numpadding;i++) sfswrite(fid,1,(char *)&s);

	/* copy data across */
	for (i=0;i<numframes;i++) {
		if (fread(record,sizeof(record),1,ilsfil) != 1)
			error("read error on ils file",NULL);
		s=record[118];
		sfswrite(fid,1,(char *)&s);
	}
}

/* copy ils formant data */
#define AMPOFF 30
void copysy(ilsfil,numpadding,numframes,framdur)
FILE	*ilsfil;
int	numpadding,numframes;
double	framdur;
{
	int	i,fid;
	short	record[128];
	short	frame[19];
	int	lastf1,lastf2,lastf3,lastf4;

	/* create item header */
	sfsheader(&item,SY_TYPE,0,2,19,framdur,0.0,1,0,0);
	strcpy(item.history,history);

	/* open channel to file */
	if ((fid=sfschannel(dbfilename,&item)) < 0)
		error("could not open channel to %s",dbfilename);

	/* initialise a frame */
	frame[0] = 0;			/* FX */
	frame[1] = 0;
	frame[2] = 0;
	frame[3] = 500;			/* F1 */
	lastf1   = 500;
	frame[4] = 0;			/* A1 */
	frame[5] = 0;
	frame[6] = 1500;		/* F2 */
	lastf2   = 1500;
	frame[7] = 0;			/* A2 */
	frame[8] = 0;
	frame[9] = 2500;		/* F3 */
	lastf3   = 2500;
	frame[10] = 0;			/* A3 */
	frame[11] = 0;
	frame[12] = 4000;		/* F4 */
	lastf4    = 4000;
	frame[13] = 0;			/* AHF */
	frame[14] = 0;
	frame[15] = 250;		/* FN */
	frame[16] = 0;			/* AN */
	frame[17] = 0;
	frame[18] = 0;			/* V */

	/* put in padding */
	for (i=0;i<numpadding;i++) sfswrite(fid,1,(char *)frame);

	/* copy data across */
	for (i=0;i<numframes;i++) {
		/* read ils analysis record */
		if (fread(record,sizeof(record),1,ilsfil) != 1)
			error("read error on ils file",NULL);
		/* copy appropriate data across */
		/* FX */
		frame[0]= record[118];
		/* F1 */
		frame[3] = ((record[54] == 0) || (record[54] < 100) || (record[54] > 1600)) 
				? lastf1 : record[54] ;
		lastf1   = frame[3];
		/* A1 */
		frame[4] = ((record[54] == 0) || (record[56] < AMPOFF)) 
				? 0   : (record[56] - AMPOFF)*10;
		/* B1 */
		frame[5] = ((record[54] == 0) || (record[56] < AMPOFF)) 
				? 0   : record[55] ;
		/* F2 */
		frame[6] = ((record[57] == 0) || (record[57] < 500) || (record[57] > 3500)) 
				? lastf2 : record[57] ;
		lastf2   = frame[6];
		/* A2 */
		frame[7] = ((record[57] == 0) || (record[59] < AMPOFF)) 
				? 0   : (record[59] - AMPOFF)*10;
		/* B2 */
		frame[8] = ((record[57] == 0) || (record[59] < AMPOFF)) 
				? 0   : record[58] ;
		/* F3 */
		frame[9] = ((record[60] == 0) || (record[60] < 1300) || (record[60] > 4300)) 
				? lastf3 : record[60] ;
		lastf3   = frame[9];
		/* A3 */
		frame[10] = ((record[60] == 0) || (record[62] < AMPOFF)) 
				? 0   : (record[62] - AMPOFF)*10;
		/* B3 */
		frame[11] = ((record[60] == 0) || (record[62] < AMPOFF)) 
				? 0   : record[61] ;
		/* F4 */
		frame[12] = ((record[63] == 0) || (record[63] < 2500) || (record[63] > 5000)) 
				? lastf4 : record[63] ;
		lastf4   = frame[12];
		/* A4 */
		frame[13] = ((record[63] == 0) || (record[65] < AMPOFF)) 
				? 0   : (record[65] - AMPOFF)*10;
		/* B4 */
		frame[14] = ((record[63] == 0) || (record[65] < AMPOFF)) 
				? 0   : record[64] ;
		/* AN */
		frame[16] = frame[4];
		/* V */
		frame[18] = (record[118] == 0) ? 0 : 248 ;
		/* write data to db file */
		if (sfswrite(fid,1,(char *)frame) != 1)
			error("write error on temporary file",NULL);
	}
}

/* copy ils formant data */
void copyfm(ilsfil,numpadding,numframes,framdur,samplefreq,npeaks,context,lxsync)
FILE	*ilsfil;
int	numpadding,numframes,npeaks,context,lxsync;
double	framdur,samplefreq;
{
	int		i,j,fid;
	short		record[128];
	struct fm_rec 	*fm;
	int		reclen;

	/* size of output rec */
	reclen = 5 + 3 * npeaks;

	/* create item header */
	sfsheader(&item,FM_TYPE,-1,4,reclen,1.0/samplefreq,0.0,context,0,lxsync);
	strcpy(item.history,history);

	/* open channel to data file */
	if ((fid=sfschannel(dbfilename,&item)) < 0)
		error("could not open channel to %s",dbfilename);

	/* get fm buffer */
	if ((fm=(struct fm_rec *)sfsbuffer(&item,1))==NULL)
		error("could not get memory buffer",NULL);

	/* initialise a frame */
	fm->posn=0;
	fm->size=0;
	fm->flag=0;
	fm->gain=0.0;
	fm->npeaks=0;
	for (i=0;i<npeaks;i++) {
		fm->formant[i].freq=0.0;
		fm->formant[i].amp=0.0;
		fm->formant[i].band=0.0;
	}	

	/* put in padding */
	for (i=0;i<numpadding;i++) {
		sfswrite(fid,1,(char *)fm);
		fm->posn += (framdur * samplefreq) + 0.5;
	}

	/* copy data across */
	for (i=0;i<numframes;i++) {
		/* read ils analysis record */
		if (fread(record,sizeof(record),1,ilsfil) != 1)
			error("read error on ils file",NULL);
		/* copy appropriate data across */
		fm->posn=record[127];			/* msw sample no. */
		fm->posn=(fm->posn << 15) + record[126] - 1;
							/* lsw sample no. */
		fm->size=record[125];			/* window size */
		fm->flag=(record[118] == 0) ? 0 : 1;	/* voicing flag */
		fm->gain=record[121];			/* residual gain */
		fm->npeaks=(record[117] <= npeaks) ? record[117] : npeaks;
							/* number peaks */
		for (j=0;j<fm->npeaks;j++) {
			fm->formant[j].freq = record[30 + 3*j];	/* formant frequencies */
			fm->formant[j].band = record[31 + 3*j];	/* formant bandwidths */
			fm->formant[j].amp  = record[32 + 3*j];	/* formant amplitudes */
		}

		/* write data to db file */
		if (sfswrite(fid,1,(char *)fm) != 1)
			error("write error on temporary file",NULL);
	}

}

/*=========================== TO ILS =====================================*/

/* convert SP or LX to ILS */
void ilsspconv(fid,ilsfil)
int	fid;
FILE	*ilsfil;
{
	int	i,numf,cnt;
	short	record[128];

	i = 0;
	numf = item.numframes;
	while (numf > 0) {
		cnt = (numf > 128) ? 128 : numf;
		sfsread(fid,i,cnt,(char *)record);
		fwrite(record,2,cnt,ilsfil);
		numf -= cnt;
		i += cnt;
	}
}

/* convert Fx item to ILS Fx data */
void ilsfxconv(fid,ilsfil)
int	fid;
FILE 	*ilsfil;
{
	int	i;
	short	record[128];
	short	s;
	int	numpadding,numframes,sampfreq,ilshead[128];

	/* read in ils header */
	if (fread(ilshead,512,1,ilsfil) != 1)
		error("read error on %s",ilsfilename);

	if (ilshead[62] != -29000) 
		error("%s is incorrect ILS file type",ilsfilename);

	/* extract useful information from header */
	numpadding=ilshead[7]-1;			/* ILS HEAD 8 */
	numframes=ilshead[8];				/* ILS HEAD 9 */
	sampfreq = ilshead[61];			/* integer sampling freq. */

	/* copy data across */
	for (i=numpadding;i<numframes+numpadding;i++) {
		if (fread(record,sizeof(record),1,ilsfil) != 1)
			error("read error on ils file",NULL);
		sfsread(fid,i,1,(char *)&s);
		record[118]= s;
		record[119]= (s==0) ? 0 : sampfreq/s;
		fseek(ilsfil,-sizeof(record),1);
		if (fwrite(record,sizeof(record),1,ilsfil) != 1)
			error("write error on ils file %s",ilsfilename);
	}
	
	/* close file */
	fclose(ilsfil);

}

/* convert tx item to an ils label file */
#define MINFREQ 40	/* minimum frequency at which tx counts as voiced */
void ilstxconv(fid,labfil,labfilename)
int	fid;
FILE	*labfil;
char	*labfilename;
{
	int	i,tx,totime=0;
	int	freq;
	double	txtime,lastxtime=10E6;

	freq = (0.01/item.frameduration + 0.005)*100;

	/* create entries in label file */
	fprintf(labfil,"    %-32s\n",labfilename);
	for (i=0;i<txitem.numframes;i++) {
		sfsread(fid,i,1,(char *)&tx);
		txtime = tx * txitem.frameduration;
		if ((1.0/txtime) > MINFREQ) {
			/* valid excitation period */
			fprintf(labfil,"V                   ;  ;        ;     ;                    ;    \n");
			fprintf(labfil,"%6d.;%7d.;%5d;%-32s;         ;    \n",totime,(int)(txtime/item.frameduration),freq,ilsfilename);
		}
		else if ((1.0/lastxtime) > MINFREQ) {
			/* last excitation of voiced region */
			fprintf(labfil,"V                   ;  ;        ;     ;                    ;    \n");
			fprintf(labfil,"%6d.;%7d.;%5d;%-32s;         ;    \n",totime,(int)(lastxtime/item.frameduration),freq,ilsfilename);
		}
		lastxtime = txtime;
		totime += txtime/item.frameduration;
	}
}


/* copy from dbfile to ils file */
void copytoils(inpitem,inptype,txcv,txtype)
int	inpitem,txcv;
char	*inptype,*txtype;
{
	FILE	*ilsfil;
	int	fid,fildes;
	char	labfilename[SFSMAXFILENAME];
	FILE	*labfil;

	/* open datafile */
	if ((fid=sfsopen(dbfilename,"r",NULL)) < 0)
		error("access error on %s",dbfilename);

	/* locate input item */
	if (!sfsitem(fid,inpitem,inptype,&item))
		error("cannot find input item in %s",dbfilename);

	/* open ils file(s) */
	if (inpitem==FX_TYPE) {
		/* file should exist */
		if (access(ilsfilename,0) != 0)
			error("ILS file %s does not exist",ilsfilename);
		if ((ilsfil = fopen(ilsfilename,"r+b")) == NULL)
			error("cannot open %s",ilsfilename);
	}
	else {
		/* create new contiguous file for output */
		if (access(ilsfilename,0) == 0) unlink(ilsfilename);
#ifndef MASSCOMP
#define O_CTG 0
#endif
#ifndef O_BINARY
#define O_BINARY 0
#endif
		if ((fildes = open(ilsfilename,O_WRONLY | O_CTG | O_BINARY | O_CREAT,0664,2*item.numframes+512)) < 0)
			error("cannot open %s",ilsfilename);
		ilsfil = fdopen(fildes,"w");
	}

	/* do conversion */
	if (inpitem==FX_TYPE)		/* convert fx */
		ilsfxconv(fid,ilsfil);
	else {			/* convert speech or lx */
		/* write ils header */
		if (ilshead(ilsfil) != 0) {
			fclose(ilsfil);
			unlink(ilsfilename);
			error("unable to write ils header to %s",ilsfilename);
		}
	
		/* write out data */
		ilsspconv(fid,ilsfil);
	}
	/* close ils file */
	fclose(ilsfil);

	/* write tx data to label file if required */
	if (txcv) {
		strcpy(labfilename,ilsfilename);
		strcat(labfilename,".lab");
		if ((labfil = fopen(labfilename,"wb")) == NULL)
			error("cannot open label file %s",labfilename);
		if (!sfsitem(fid,TX_TYPE,txtype,&txitem))
			error("cannot find input TX item in %s",dbfilename);
		ilstxconv(fid,labfil,labfilename);
		fclose(labfil);
	}
	sfsclose(fid);
}

/* copy from ils file to dbfile */
void copyfromils(outitem)
int	outitem;
{
	FILE	*ilsfil;
	int	fid;
	int	ilshead[128];
	double	samplefreq,framdur,pow();
	int	numsamples,numframes,numpadding,context,numpeaks,lxsync;

	/* opendatafile */
	if ((fid=sfsopen(dbfilename,"w",NULL)) < 0)
		error("access error on %s",dbfilename);
	sfsclose(fid);

	/* open ils file */
	if ((ilsfil = fopen(ilsfilename,"rb")) == NULL)
		error("cannot open %s",ilsfilename);

	/* read in ils header */
	if (fread(ilshead,512,1,ilsfil) != 1)
		error("read error on %s",ilsfilename);

	/* check ils file of right type */
	if ((outitem==FX_TYPE) || (outitem==SY_TYPE) || (outitem==FM_TYPE)) {
		if (ilshead[62] != -29000) error("%s is incorrect ILS file type",ilsfilename);
	}
	else {
		if (ilshead[62] != -32000) error("%s is incorrect ILS file type",ilsfilename);
	}

	/* extract useful information from header */
	samplefreq=ilshead[60];				/* ILS HEAD 61 */
	samplefreq=ilshead[61] * pow(10.0,samplefreq);	/* ILS HEAD 62 */
	numsamples=256*ilshead[5] + ilshead[11];	/* ILS HEAD 6 & 12 */
	framdur=ilshead[3];				/* ILS HEAD 4 */
	context=ilshead[3];
	framdur=framdur / samplefreq;
	numpadding=ilshead[7]-1;			/* ILS HEAD 8 */
	numframes=(ilshead[16]==0) ? ilshead[8] : ilshead[16];
							/* ILS HEAD 9 or 17 */
	lxsync=(ilshead[16] != 0);			/* ILS HEAD 17 */
	numpeaks=(ilshead[1]-4)/2;			/* # LPC coeff */

	/* copy data across */
	if (outitem==FX_TYPE)
		copyfx(ilsfil,numpadding,numframes,framdur);
	else if (outitem==SY_TYPE)
		copysy(ilsfil,numpadding,numframes,framdur);
	else if (outitem==FM_TYPE)
		copyfm(ilsfil,numpadding,numframes,framdur,samplefreq,numpeaks,context,lxsync);
	else
		copysamples(ilsfil,outitem,numpadding*context,numsamples,samplefreq);

	/* close file */
	fclose(ilsfil);

	/* transfer with backup */
	if (!sfsupdate(dbfilename))
		error("update error on %s",dbfilename);
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* command line decoding */
	extern int	optind;
	extern char	*optarg;
	char		c;
	int		errflg=0;
	int		it;
	char		*ty;
	/* operation selection */
	int		inpitem=1;	/* default input = speech */
	char		*inptype="0";	/* default subtype = last */
	int		txcv=0;		/* tx conversion off */
	char		*txtype="0";	/* default tx subtype = last */
	int		reverse=0;	/* default direction -> ils */
	int		outitem=1;	/* default output item = speech */

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:o:lh:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: ILS conversion V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'l' :	/* tx to label file */
			txcv++;
			break;
		case 'i':	/* input item specification */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE) {
					inpitem = SP_TYPE;
					inptype = ty;
				}
				else if (it == LX_TYPE) {
					inpitem = LX_TYPE;
					inptype = ty;
				}
				else if (it == TX_TYPE) {
					txtype = ty;
					txcv++;
				}
				else if (it == FX_TYPE) {
					inpitem = FX_TYPE;
					inptype = ty;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'o':	/* output item specification */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it == SP_TYPE) ||
				    (it == LX_TYPE) ||
				    (it == FX_TYPE) ||
				    (it == SY_TYPE) ||
				    (it == FM_TYPE)) 
					outitem = it;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			reverse++;
			break;
		case 'h' :	/* history field */
			strcpy(history,optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) {(-l) (-i item)} {(-o item) (-h history)} ilsfile dbfile\n",PROGNAME);

	/* get filenames */
	if (optind < argc)
		strcpy(ilsfilename,argv[optind]);
	else
		error("no ils file specified",NULL);
	optind++;
	if (optind < argc) {
		strcpy(dbfilename,sfsfile(argv[optind]));
	}
	else
		error("no database file specified",NULL);

	/* only can do tx label file with SP or LX */
	if (txcv && !((inpitem==SP_TYPE) || (inpitem==LX_TYPE)))
		error("TX conversion only with accompanying SP or LX",NULL);

	/* select copy to/from ils */
	if (reverse) 
		copyfromils(outitem);
	else
		copytoils(inpitem,inptype,txcv,txtype);

	/* that's all folks */
	exit(0);
}

