/* colist -- list CO item in seconds and dB */

/* M.A.Huckvale */

/* version 1.0 - March 1990 */
/* version 1.1 - January 1992
	- add -h option for output in HTK format
*/
/* version 1.2 - July 1992
	- add -e flag to output energy normalised vectors
*/
/* version3 - March 1994
	- add -a flag to list annotations too
*/
/* version 1.3a - February 1995
	- correct annotation time alignment
*/

#define PROGNAME "colist"
#define PROGVERS "1.3a"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH COLIST SFS1 UCL
.SH NAME
colist - list spectral coefficients item
.SH SYNOPSIS
.B colist
(-i item) (-h htkfile) (-e) file
.SH DESCRIPTION
.I colist
lists a CO item in milliseconds and dB to the standard output.  
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -h htkfile
Output data into HTK file format
.TP 11
.B -e
Subtract the mean value from each frame output and add as additional
value to end of frame.
.SH INPUT ITEMS
.IP 11.xx 11
Any CO item.
.SH VERSION/AUTHOR
1.2 - Mark Huckvale
.SH SEE ALSO
sylist, fmlist, coload
.SH BUGS
Only lists position of each frame and spectral contents, not complete
details for frame.
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header coitem;	/* filing system item header for co */
struct item_header anitem;	/* item header for An item */

int	dohtk;
char	htkname[SFSMAXFILENAME];
struct htk_header {
	long	nsamp;
	long	nperiod;
	short	ssize;
	short	skind;
} hhead;

float	submean();
int	doannot=0;
	
/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	char		*cotype="0";	/* default CO sub-type = last */
	char		*antype="0";	/* default AN type = last */
	int		c;		/* option switch */
	int32		it;		/* item/sub-type specifiers */
	char		*ty;
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	int		fid;
	int		afid=0;
	/* processing variables */
	struct co_rec	*corec;		/* input formant record */
	int		ncoeff;
	int		i,j;
	double		time = 0.0;	/* time of current frame */
	FILE		*op;
	double		frlen;
	int		addnrg=0;
	float		meannrg;
	struct an_rec   *anrec=NULL;	/* annotation record */
	double		antime=0;	/* time of annotation */
	int		anidx=0;	/* annotation index */
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:h:ea")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: List spectral coefficients V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == CO_TYPE)
					cotype = ty;
				else if (it == AN_TYPE) {
					antype = ty;
					doannot++;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'h' :	/* HTK output */
			dohtk++;
			strcpy(htkname,optarg);
			break;
		case 'e' :	/* add energy value */
			addnrg=1;
			break;
		case 'a' :	/* list annotations */
			doannot++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check for option decoding error */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-h htkname) (-e) (-a) dbase_file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"r",NULL)) < 0)
		error("access error on %s",filename);

	/* locate CO item */
	if (!sfsitem(fid,CO_TYPE,cotype,&coitem))
		error("unable to find input CO item in %s",filename);
	if ((corec=(struct co_rec *)sfsbuffer(&coitem,1))==NULL)
		error("unable to create memory buffer",NULL);
	ncoeff = SFSRECSIZE(&coitem);

	/* load annotation item if required */
	if (doannot) {
		afid=sfsopen(filename,"r",NULL);
		if (!sfsitem(afid,AN_TYPE,antype,&anitem))
			error("unable to find input AN item in %s",filename);
		if ((anrec=(struct an_rec *)sfsbuffer(&anitem,1))==NULL)
			error("unable to create memory buffer",NULL);
	}

	/* list data */
	if (dohtk) {
		if ((op=fopen(htkname,"wb"))==NULL)
			error("failed to open '%s'",htkname);
		hhead.nsamp = coitem.numframes;
		frlen = coitem.frameduration*(coitem.windowsize-coitem.overlap);
		hhead.nperiod = (long)(0.5+10000000L*frlen);
		hhead.ssize = (ncoeff+addnrg)*sizeof(float);
		hhead.skind = 7;	/* filterbank */
		if (fwrite(&hhead,sizeof(struct htk_header),1,op)!=1)
			error("write error on '%s'",htkname);
		for (i=0;i<coitem.numframes;i++) {
			sfsread(fid,i,1,corec);
			if (addnrg) meannrg = submean(corec->data,ncoeff);
			if (fwrite(corec->data,4,ncoeff,op)!=ncoeff)
				error("write error on '%s'",htkname);
			if (addnrg) {
				if (fwrite((char *)&meannrg,4,1,op)!=1)
					error("write error on '%s'",htkname);
			}
		}
		fclose(op);
	}
	else {
		printf("! file=%s item=%d.%02d history=%s\n",filename,coitem.datatype,coitem.subtype,coitem.history);
		printf("! params=%s\n",coitem.params);
		if (doannot) {
			if (sfsread(afid,0,1,anrec) != 1)
				antime=10E6;
			else
				antime = anitem.offset + anrec->posn * anitem.frameduration;
			anidx=1;
		}
		for (i=0;i<coitem.numframes;i++) {
			sfsread(fid,i,1,corec);
			if (addnrg) meannrg = submean(corec->data,ncoeff);
			time = coitem.offset + corec->posn * coitem.frameduration;
			while ((doannot) && (antime <= time)) {
				printf("! %s\n",anrec->label);
				if (sfsread(afid,anidx++,1,anrec) != 1)
					antime=10E6;
				else
					antime = anitem.offset + anrec->posn * anitem.frameduration;
			}
			printf("%7.1f ",time*1000);
			for (j=0;j<ncoeff;j++) 
				printf("%8.5f ",corec->data[j]);
			if (addnrg)
				printf("%8.5f",meannrg);
			printf("\n");
		}
	}

	/* and exit */
	exit(0);
}

float submean(vec,len)
float	*vec;
int	len;
{
	int	i;
	float	sum=0.0;

	for (i=0;i<len;i++)
		sum += vec[i];
	sum /= (float)len;
	for (i=0;i<len;i++)
		vec[i] -= sum;
	return(sum);
}

