/* dig -- device independent gps filter */

/* m.a. huckvale - april 1987 */

#define PROGNAME "dig"
#define PROGVERS "1.3"
char *progname = PROGNAME;

/* version 1.0 */
/* version 1.1
	- support for new page
*/
/* version 1.2 
	- support for filled circles
	- laser printer in "portrait" mode
	- # copies on laser
*/
/* version 1.3
	- bug fix on h/w text
	- ctrl/c interrupt caught
*/
/*--------------------------------------------------------------------------*/
/**MAN
.TH DIG SFS1 UCL SFS
.SH NAME
dig - device independent graphics metafile filter
.SH SYNOPSIS
.B dig
(-p) (-r) (file)
.SH DESCRIPTION
.I dig
accepts a graphics metafile in a given file or on its standard input and
displays it according to its current operational status.  If the output of the program
is redirected into a file or a pipe, or if the program is executed
on an alphanumeric terminal graphics is produced in metafile format.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program name and version number.
.TP 11
.B -p
Send output directly to printer.
.TP 11
.B -r
Use full display rectangle, rather than square coordinates.
.SH ENVIRONMENT
The environment variable GTERM selects the graphics terminal, GPRINT
selects the printer.
.SH VERSION/AUTHOR
1.3 - Mark Huckvale
.SH SEE ALSO
DIG(SFS3)
.SH BUGS
Does not support pie-slices or shading of filled shapes.  Filled pies are assumed to
be filled circles.  Text is printed in
a roman font that is not identical to the Masscomp gps format.
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#ifdef DOS
#include <io.h>
#endif
#include <fcntl.h>
#include <math.h>
#include <signal.h>

#define	MAX(x,y) ((x) > (y)) ? (x) : (y)
#define MIN(x,y) ((x) < (y)) ? (x) : (y)

/* device independent graphics data */
#include "dig.h"
#include "digdata.h"

/* global data */
int	xmin = 32000, ymin = 32000;
int	xmax = -32000, ymax = -32000;
int	laser = 0;			/* direct laser output */
int	square = 1;			/* square co-ordinates */
int	copies = 1;
#ifdef DOS
char	*tmpfil="dig$$$$$.tmp";
#else
char	tmpfil[64]={"/tmp/dig.XXXXXX"};
#endif

/* metafile instruction */
union	{
	short	command[4096];
	struct	{
		short	type;
		short	rest[4095];
	} instr;
	struct {
		short	type;
		short	coord[4095];
	} line;
	struct {
		short	length;
		short	coord[6];
		unsigned char	bundle;
		unsigned char	zero;
		short	rest[4088];
	} arc;
	struct {
		short	length;
		short	x;
		short	y;
		unsigned char	bundle;
		unsigned char	zero;
		unsigned char	size;
		unsigned char	rotation;
		char	messg[8182];
	} text;
	struct {
		short	length;
		unsigned char	type;
		unsigned char	bundle;
		short	x1;
		short	y1;
		short	x2;
		short	y2;
	} box;
	struct {
		short	length;
		unsigned char	type;
		unsigned char	bundle;
		short	fill;
		short	delta;
		short	x1;
		short	y1;
		short	x2;
		short	y2;
	} fillbox;
	struct {
		short	length;
		unsigned char	type;
		unsigned char	bundle;
		short	fill;
		short	delta;
		short	r;
		short	ang1;
		short	ang2;
		short	x1;
		short	y1;
	} fillpie;
	struct {
		short	length;
		char	command;
		char	comm2;
		short	fill[4094];
	} comment;
} meta;

#ifdef __STDC__
void preproc(int,int);
void view(int);
int bread(int,char *,int);
int brite(int,char *,int);
void digexit(int);
#else
void preproc();
void view();
int bread();
int brite();
void digexit();
#endif

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* option decoding */
	extern int 	optind;
	extern char	*optarg;
	int		c;	
	int		errflg = 0;	/* option error flag */
	int		inp,out;	/* input/output file descriptors */

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ipr")) != EOF ) switch (c) {
		case 'I' :	/* identify */
			fprintf(stderr,"%s: Device-independent graphics display V%s\n",PROGNAME,PROGVERS);
			exit(0);
		case 'p' :	/* laser output */
			laser=1;
			break;
		case 'r' :	/* rectangular coordinates */
			square=0;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg)
		error("usage: %s (-I) (-p) (-r) (gpsfile)",PROGNAME);

	/* make temporary file */
	mktemp(tmpfil);

	/* open output file */
#ifdef DOS
	if ((out = open(tmpfil,O_RDWR | O_CREAT | O_TRUNC | O_BINARY)) < 0)
#else
	if ((out = open(tmpfil,O_RDWR | O_CREAT | O_TRUNC)) < 0)
		error("unable to open temporary file");
#endif

	/* trap errors */
	if (signal(SIGINT,SIG_IGN) != SIG_IGN) signal (SIGINT,digexit);

	/* process file */
	if (optind >=argc) {	/* process standard input */
		/* pre-process to get size */
#ifdef DOS
		setmode(0,O_BINARY);
#endif
		preproc(0,out);
	}
	else {			/* process named file */

	/* open output file */
#ifdef DOS
	if ((inp = open(argv[optind],O_RDONLY | O_BINARY)) < 0)
#else
	if ((inp = open(argv[optind],O_RDONLY)) < 0)
#endif
		error("unable to open %s",argv[optind]);

		/* pre-process to get size */
		preproc(inp,out);
		close(inp);
	}

	/* display pre-processed file */
	lseek(out,0L,0);
	view(out);
	close(out);

	/* delete temporary file and quit */
	unlink(tmpfil);
	exit(0);
}

/* preproc -- pre-process gps file to obtain scaling data */
void preproc(inp,out)
int	inp;
int	out;
{
	/* local variables */
	int 	command,length;
	int	i,ncoord,size;
	short	*sp;

	/* read gps commands from input - copying to output */
	while (bread(inp,(char *)&meta.instr.type,2) == 2) {
		command = (meta.instr.type & 0xF000) >> 12;
		length = 2 * (meta.instr.type & 0xFFF);
		switch (command) {
		case 0 :	/* draw a line */
			bread(inp,(char *)meta.instr.rest,length-2);
			brite(out,(char *)meta.command,length);
			ncoord=(length-4)/2;
			for (i=0;i<ncoord;i=i+2) {
				xmax = MAX(meta.line.coord[i],xmax);
				xmin = MIN(meta.line.coord[i],xmin);
				ymax = MAX(meta.line.coord[i+1],ymax);
				ymin = MIN(meta.line.coord[i+1],ymin);
			}
			break;
		case 2 :	/* text */
			bread(inp,(char *)meta.instr.rest,length-2);
			brite(out,(char *)meta.command,length);
			size = 5 * meta.text.size;
			xmax = MAX(meta.text.x+size,xmax);
			xmin = MIN(meta.text.x-size,xmin);
			ymax = MAX(meta.text.y+size,ymax);
			ymin = MIN(meta.text.y-size,ymin);
			break;
		case 3 :	/* arcs */
			bread(inp,(char *)meta.instr.rest,length-2);
			brite(out,(char *)meta.command,length);
			for (i=0;i<6;i+=2) {
				xmax = MAX(meta.arc.coord[i],xmax);
				xmin = MIN(meta.arc.coord[i],xmin);
				ymax = MAX(meta.arc.coord[i+1],ymax);
				ymin = MIN(meta.arc.coord[i+1],ymin);
			}
			break;
		case 14 :	/* boxes and pies */
			bread(inp,(char *)meta.instr.rest,length-2);
			brite(out,(char *)meta.command,length);
			switch (meta.box.type) {
			case 9:				/* filled rectangle */
				sp = &meta.fillbox.x1;
				for (i=0;i<4;i+=2) {
					xmax = MAX(sp[i],xmax);
					xmin = MIN(sp[i],xmin);
					ymax = MAX(sp[i+1],ymax);
					ymin = MIN(sp[i+1],ymin);
				}
				break;
			case 10:			/* fill pie */
				xmax = MAX(meta.fillpie.x1+meta.fillpie.r,xmax);
				xmin = MIN(meta.fillpie.x1-meta.fillpie.r,xmin);
				ymax = MAX(meta.fillpie.y1+meta.fillpie.r,ymax);
				ymin = MIN(meta.fillpie.y1-meta.fillpie.r,ymin);
				break;
			case 14:			/* edge rectangle */
				sp = &meta.box.x1;
				for (i=0;i<4;i+=2) {
					xmax = MAX(sp[i],xmax);
					xmin = MIN(sp[i],xmin);
					ymax = MAX(sp[i+1],ymax);
					ymin = MIN(sp[i+1],ymin);
				}
				break;
			default:
				fprintf(stderr,"%s: unsupported gps operator: %d,%d\n",PROGNAME,command,meta.box.type);
			}
			break;
		case 15 :	/* comment */
			/* ignore comments unless new-page */
			bread(inp,(char *)meta.instr.rest,length-2);
			if (meta.comment.command=='P')
				brite(out,(char *)meta.command,length);
			break;
		default:	/* unknown */
			bread(inp,(char *)meta.instr.rest,length-2);
			fprintf(stderr,"%s: unsupported gps operator: %d\n",PROGNAME,command);
			break;
		}
	}
}

/* view -- displays file of gps commands using given scaling */
void view(inp)
int	inp;
{
	/* local variables */
	int 	command,length;
	int	i,ncoord,bundle;
	int	firstcommand=1;

	/* startup device independent graphics */
	if (laser) {
		if (isatty(1))
			digstart(DIG_DEFAULT_PRINTER,NULL,0);
		else
			digstart(DIG_DEFAULT_PRINTER,stdout,0);
	}
	else
		digstart(DIG_DEFAULT_TERM,NULL,0);
	digclearscreen();
	digscale((float)(xmax-xmin),(float)(ymax-ymin),square);
	digorigin((float)(0-xmin),(float)(0-ymin));

	/* process gps commands */
	while (bread(inp,(char *)&meta.instr.type,2) == 2) {
		command = (meta.instr.type & 0xF000) >> 12;
		length = 2 * (meta.instr.type & 0xFFF);
		switch (command) {
		case 0 :	/* draw a line */
			bread(inp,(char *)meta.instr.rest,length-2);
			ncoord=(length-4)/2;
			bundle = meta.line.coord[ncoord] >> 8;
			for (i=2;i<ncoord;i=i+2)
				digline(bundle,(float)(meta.line.coord[i-2]),
					(float)(meta.line.coord[i-1]),
					(float)(meta.line.coord[i]),
					(float)(meta.line.coord[i+1]));
			break;
		case 2 :	/* text */
			bread(inp,(char *)meta.instr.rest,length-2);
			i = length - 10;
			meta.text.messg[i]='\0';
			digvtext(meta.text.bundle,
				(float)(meta.text.x)-(2.5*meta.text.size),
				(float)(meta.text.y)-(4.0*meta.text.size),
				meta.text.messg,
				5.0*(float)(meta.text.size),
				360.0*(float)(meta.text.rotation)/256.0,
				1);
			break;
		case 3 :	/* arc */
			bread(inp,(char *)meta.instr.rest,length-2);
			if ((meta.arc.coord[0] != meta.arc.coord[4]) ||
			    (meta.arc.coord[1] != meta.arc.coord[5]))
				digarc(meta.arc.bundle,
					(float)meta.arc.coord[0],
					(float)meta.arc.coord[1],
					(float)meta.arc.coord[2],
					(float)meta.arc.coord[3],
					(float)meta.arc.coord[4],
					(float)meta.arc.coord[5]);
			else
				digcircle(meta.arc.bundle,
					(float)(meta.arc.coord[0]+meta.arc.coord[2])/2.0,
					(float)(meta.arc.coord[1]+meta.arc.coord[3])/2.0,
					sqrt((float)(meta.arc.coord[2]-meta.arc.coord[0])*
					     (float)(meta.arc.coord[2]-meta.arc.coord[0])+
					     (float)(meta.arc.coord[3]-meta.arc.coord[1])*
					     (float)(meta.arc.coord[3]-meta.arc.coord[1]))/2.0);
			break;
		case 14 :	/* boxes and pies */
			bread(inp,(char *)meta.instr.rest,length-2);
			switch (meta.box.type) {
			case 9:				/* filled rectangle */
				digfillbox(meta.fillbox.bundle,
					(float)meta.fillbox.x1,
					(float)meta.fillbox.y1,
					(float)meta.fillbox.x2,
					(float)meta.fillbox.y2);
				break;
			case 10:			/* fill pie */
				digfillcircle(meta.fillpie.bundle,
					(float)meta.fillpie.x1,
					(float)meta.fillpie.y1,
					(float)meta.fillpie.r);
				break;
			case 14:			/* edge rectangle */
				digbox(meta.box.bundle,
					(float)meta.box.x1,
					(float)meta.box.y1,
					(float)meta.box.x2,
					(float)meta.box.y2);
				break;
			default:
				fprintf(stderr,"%s: unsupported gps operator: %d,%d\n",PROGNAME,command,meta.box.type);
			}
			break;
		case 15 :	/* comment */
			/* ignore comments unless new-page */
			bread(inp,(char *)meta.instr.rest,length-2);
			if (!laser && !firstcommand)
				digwait();
			if (meta.comment.command=='P')
				digclearscreen();
			break;
		}
		firstcommand=0;
	}

	/* clean exit */
	digquit((laser)?0:15);
}

int bread(fd,buff,num)
int	fd;
char	*buff;
int	num;
{
	int	cnt=1,tot=0;
	while ((num > 0) && (cnt > 0)) {
		cnt = read(fd,buff,num);
		buff += cnt;
		tot += cnt;
		num -= cnt;
	}
	return(tot);
}
int brite(fd,buff,num)
int	fd;
char	*buff;
int	num;
{
	if (write(fd,buff,num) != num) 
		fprintf(stderr,"write error\n");
	return(num);
}
void digexit(dummy)
int dummy;
{
	unlink(tmpfil);
	digquit(0);
	error("interrupt",NULL);
}
