/* splist -- list speech data in a variety of formats */

/* M.A.Huckvale - University College London */

/* version 1.0 - April 1996 */
	
#define PROGNAME "splist"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH SPLIST 1 UCL
.SH NAME
splist - save speech data in a variety of formats
.SH SYNOPSIS
.B wave
(-i item) (-t format) (-a) (-o outfile) (-s) (-8) sfsfile
.SH DESCRIPTION
.I splist
is a program to extract waveform data from speech and lx items.
The selected data set
is either printed on the standard output as a sequence of numbers or stored in a
foreign format datafile.  An option reverses the byte order in binary files for waveforms.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item.
.TP 11
.BI -t type
Select output file format.  Available are: RAW (binary), WAV or RIFF (Microsoft
RIFF format), VOC (Creative Labs), AU (Sun), AIFF (Audio Interchange), HTK
(Hidden Markov Model Toolkit), ILS (Interactive Laboratory System).
.TP 11
.BI -o datafile
Store the contents of the data set in the file
.I datafile
rather than listing them on the standard output.
.TP 11
.B -a
(Binary file output only) With -t RAW switch, appends data to file.
.TP 11
.B -s
(Binary file output only) swap byte order in binary file.  Default
is native machine order, or as required by foreign file format.
.TP 11
.B -8
(WAV, VOC, AU formats only) Save top 8-bits of samples only.
.SH INPUT ITEMS
.IP SP 11
Any speech item.
.IP LX 11
Any excitation item.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/
/* global structures */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <fcntl.h>
#include <string.h>
#include "sfs.h"

#define MIN(x,y) (((x)<(y))?(x):(y))
#define REVERSESHORT(s)	s = ((s & 0xFF00) >> 8) | ((s & 0x00FF) << 8);
#define REVERSELONG(l)	l = ((l & 0xFF000000L) >> 24) | ((l & 0x00FF0000L) >> 8) | ((l & 0x0000FF00L) << 8) | ((l & 0x000000FFL) << 24);

/* global data */
struct item_header	item;

/* format lookup */
void	binproc(),riffproc(),vocproc(),auproc();
void	aiffproc(),ilsproc(),htkproc();
struct format_rec {
	char	*name;
	void	(*fproc)();
} formattab[]={
	{ "RAW",	binproc },
	{ "RIFF",	riffproc },
	{ "WAV",	riffproc },
	{ "VOC",	vocproc },
	{ "AU",		auproc },
	{ "AIFF",	aiffproc },
	{ "ILS",	ilsproc },
	{ "HTK",	htkproc },
};
#define NUMFORMAT (sizeof(formattab)/sizeof(struct format_rec))

/* copying buffer */
#define COPYBUFSIZE	16384
short	copybuf[COPYBUFSIZE];

/* options */
int	binary=0;
int	format=0;
int	swapbyte=0;
int	append=0;
int	eightbit=0;
int	sampoffset=0;
char	ofilename[SFSMAXFILENAME];

/* swap bytes in buffer */
void swapbuf(buf,count)
char	*buf;
int	count;
{
	int	i;
	char	temp;

	for (i=0;i<count;i+=2) {
		temp=buf[i];
		buf[i]=buf[i+1];
		buf[i+1]=temp;
	}
}

/* compress 16-bit samples to 8-bit in buffer */
void compbuf(obuf,ibuf,count)
char	*obuf;
short	*ibuf;
int	count;
{
	int	i;

	for (i=0;i<count;i++)
		obuf[i] = (char)((unsigned char)((ibuf[i]>>8) + sampoffset));
}

/* copy sample data */
int	copydata(ifid,op,numf)
int	ifid;
FILE	*op;
int	numf;
{
	int	pos;
	int	count;
	int	length=0;
	
	pos=0;
	while (numf > 0) {
		count = sfsread(ifid,pos,MIN(numf,COPYBUFSIZE),copybuf);
		if (count <= 0)
			error("read error on input file");
		if (swapbyte)
			swapbuf((char *)copybuf,2*count);
		if (eightbit) {
			compbuf((char *)copybuf,copybuf,count);
			if (fwrite((char *)copybuf,1,count,op)!=count)
				error("write error on '%s'",ofilename);
			length += count;
		}
		else {
			if (fwrite((char *)copybuf,2,count,op)!=count)
				error("write error on '%s'",ofilename);
			length += 2*count;
		}
		numf -= count;
		pos += count;
	}
	return(length);
}

/* store data in binary file */
void binproc(fid,item,bfilename)
int		fid;		/* input data file */
struct item_header *item;
char		*bfilename;	/* output file name */
{
	FILE	*op;

	/* open output file */
	if ((op=fopen(bfilename,(append)?"ab":"wb")) == NULL)
		error("unable to open output file %s",bfilename);

	/* copy the samples */
	copydata(fid,op,item->numframes);
	
	/* close output file */
	fclose(op);

}

/* store data in RIFF file */
void riffproc(fid,item,bfilename)
int		fid;		/* input data file */
struct item_header *item;
char		*bfilename;	/* output file name */
{
	FILE	*op;
	int	srate;
	struct riff_file_header {
		char		riffid[4];	/* "RIFF" */
		unsigned long	size;		/* length of RIFF block */
	} riffhead;

	struct riff_wave_header {
		char		waveid[4];	/* "WAVE" */
		char		fmtid[4];	/* "fmt " */
		unsigned long	size;		/* 16 */
		unsigned short	format;		/* 1 */
		unsigned short	nchan;		/* 1 or 2 */
		unsigned long	srate;		/* srate */
		unsigned long	brate;		/* srate * #bytes/samp * nchan */
		unsigned short	balign;		/* 1 or 2 or 4 */
		unsigned short	nbits;		/* 8 or 16 */
	} riffwavehead;

	struct riff_data_header {
		char		dataid[4];	/* "data" */
		unsigned long	size;		/* length of signal in bytes */
	} riffdatahead;

	/* open output file */
	if ((op=fopen(bfilename,"wb")) == NULL)
		error("unable to open output file %s",bfilename);

	strcpy(riffhead.riffid,"RIFF");
	if (eightbit)
		riffhead.size = sizeof(struct riff_wave_header) +
				sizeof(struct riff_data_header) +
				item->numframes;
	else
		riffhead.size = sizeof(struct riff_wave_header) +
				sizeof(struct riff_data_header) +
				item->numframes * 2;
#if SFSMACHINE==0
	REVERSELONG(riffhead.size);
#endif
	if (fwrite((char *)&riffhead,sizeof(struct riff_file_header),1,op)!=1)
		error("write error on '%s'",bfilename);

	strcpy(riffwavehead.waveid,"WAVE");
	strcpy(riffwavehead.fmtid,"fmt ");
	riffwavehead.size = 16;
	riffwavehead.format = 1;
	riffwavehead.nchan = 1;
	srate = (int)(0.5+1.0/item->frameduration);
	riffwavehead.srate = srate;
	riffwavehead.brate = srate * ((eightbit)?1:2);
	riffwavehead.balign = ((eightbit) ? 1 : 2);
	riffwavehead.nbits = (eightbit) ? 8 : 16;

#if SFSMACHINE==0
	REVERSELONG(riffwavehead.size);
	REVERSESHORT(riffwavehead.format);
	REVERSESHORT(riffwavehead.nchan);
	REVERSELONG(riffwavehead.srate);
	REVERSELONG(riffwavehead.brate);
	REVERSESHORT(riffwavehead.balign);
	REVERSESHORT(riffwavehead.nbits);
#endif

	if (fwrite((char *)&riffwavehead,sizeof(struct riff_wave_header),1,op)!=1)
		error("write error on '%s'",bfilename);

	strcpy(riffdatahead.dataid,"data");
	if (eightbit)
		riffdatahead.size = item->numframes;
	else
		riffdatahead.size = item->numframes * 2;
#if SFSMACHINE==0
	REVERSELONG(riffdatahead.size);
#endif
	if (fwrite((char *)&riffdatahead,sizeof(struct riff_data_header),1,op)!=1)
		error("write error on '%s'",bfilename);

	/* copy the samples */
#if SFSMACHINE==0
	swapbyte = !swapbyte;
#endif
	if (eightbit) sampoffset = 0x80;
	copydata(fid,op,item->numframes);
	
	/* close output file */
	fclose(op);

}

/* store data in VOC file */
void vocproc(fid,item,bfilename)
int		fid;		/* input data file */
struct item_header *item;
char		*bfilename;	/* output file name */
{
	FILE	*op;
	struct vochead_rec {
		char	magic[20];
		short	offset;
		short	version1;
		short	version2;
	} vochead;
	struct type1_rec {
		unsigned char 	tc1;
		unsigned char 	pack1;
	} voctype1;
	struct type8_rec {
		unsigned short	tc8;
		unsigned char	pack8;
		unsigned char	mode8;
	} voctype8;
	int32	count;

	/* open output file */
	if ((op=fopen(bfilename,"wb")) == NULL)
		error("unable to open output file %s",bfilename);

	/* write headers */
	memcpy(vochead.magic,"Creative Voice File\x1a",20);
	vochead.offset = sizeof(vochead);
	vochead.version1 = 0x010A;
	vochead.version2 = 0x1129;
#if SFSMACHINE==0
	REVERSESHORT(vochead.offset);
	REVERSESHORT(vochead.version1);
	REVERSESHORT(vochead.version2);
#endif
	if (fwrite((char *)&vochead,sizeof(vochead),1,op)!=1)
		error("write error on '%s'",bfilename);

	/* write 16-bit header */
	if (eightbit==0) {
		count = (4 << 8) + 8;
#if SFSMACHINE==0
		REVERSELONG(count);
#endif
		if (fwrite((char *)&count,4,1,op)!=1)
			error("write error on '%s'",bfilename);
		voctype8.tc8 = (unsigned short)(65532-(long)(256000000*item->frameduration));
#if SFSMACHINE==0
		REVERSESHORT(voctype8.tc8);
#endif
		voctype8.pack8 = 4;
		voctype8.mode8 = 0;
		if (fwrite((char *)&voctype8,sizeof(voctype8),1,op)!=1)
			error("write error on '%s'",bfilename);
	}

	/* write standard 8-bit header and data */
	if (eightbit)
		count = ((2+item->numframes) << 8) + 1;
	else
		count = ((2+2*item->numframes) << 8) + 1;
#if SFSMACHINE==0
	REVERSELONG(count);
#endif
	if (fwrite((char *)&count,4,1,op)!=1)
		error("write error on '%s'",bfilename);
	voctype1.tc1 = (unsigned char)(256-(long)(1000000*item->frameduration));
	if (eightbit)
		voctype1.pack1 = 0;
	else
		voctype1.pack1 = 4;
	if (fwrite((char *)&voctype1,sizeof(voctype1),1,op)!=1)
		error("write error on '%s'",bfilename);
		
	/* copy the samples */
#if SFSMACHINE==0
	swapbyte = !swapbyte;
#endif
	if (eightbit) sampoffset = 0x80;
	copydata(fid,op,item->numframes);

	/* put terminating byte */
	putc('\0',op);
	
	/* close output file */
	fclose(op);

}

/* AU file format definitions */
typedef struct {
	unsigned int	magic;		/* magic number */
	unsigned int	hdr_size;	/* size of this header */
	unsigned int	data_size;	/* length of data (optional) */
	unsigned int	encoding;	/* data encoding format */
	unsigned int	sample_rate;	/* samples per second */
	unsigned int	channels;	/* number of interleaved channels */
} Audio_filehdr;

/* Define the magic number */
#define	AUDIO_FILE_MAGIC		((unsigned int)0x2e736e64)

/* Define the encoding fields */
#define	AUDIO_FILE_ENCODING_MULAW_8	(1)	/* 8-bit ISDN u-law */
#define	AUDIO_FILE_ENCODING_LINEAR_8	(2)	/* 8-bit linear PCM */
#define	AUDIO_FILE_ENCODING_LINEAR_16	(3)	/* 16-bit linear PCM */

/* store data in AU file */
void auproc(fid,item,bfilename)
int		fid;		/* input data file */
struct item_header *item;
char		*bfilename;	/* output file name */
{
	FILE	*op;
	Audio_filehdr	head;

	/* open output file */
	if ((op=fopen(bfilename,"wb")) == NULL)
		error("unable to open output file %s",bfilename);

	/* set up AU header */
	head.magic = AUDIO_FILE_MAGIC;
	head.hdr_size = sizeof(Audio_filehdr);
	if (eightbit) {
		head.encoding = AUDIO_FILE_ENCODING_LINEAR_8;
		head.data_size = item->numframes;
	}
	else {
		head.encoding = AUDIO_FILE_ENCODING_LINEAR_16;
		head.data_size = 2*item->numframes;
	}
	head.sample_rate = (unsigned int)(0.5+1.0/item->frameduration);
	head.channels = 1;
#if SFSMACHINE!=0
	REVERSELONG(head.magic);
	REVERSELONG(head.hdr_size);
	REVERSELONG(head.data_size);
	REVERSELONG(head.encoding);
	REVERSELONG(head.sample_rate);
	REVERSELONG(head.channels);
#endif
	if (fwrite((char *)&head,sizeof(Audio_filehdr),1,op)!=1)
		error("write error on '%s'",bfilename);
		
	/* copy the samples */
#if SFSMACHINE!=0
	swapbyte = !swapbyte;
#endif
	copydata(fid,op,item->numframes);
	
	/* close output file */
	fclose(op);

}

static void ConvertToIeeeExtended(double num, char *bytes)
{
	int	sign;
	int 	expon;
	double fMant, fsMant;
	unsigned long hiMant, loMant;

	if (num < 0) {
		sign = 0x8000;
		num *= -1;
	} else {
		sign = 0;
	}

	if (num == 0) {
		expon = 0; hiMant = 0; loMant = 0;
	}
	else {
		fMant = frexp(num, &expon);
		if ((expon > 16384) || !(fMant < 1)) {
			/* Infinity or NaN */
 			expon = sign|0x7FFF;
			hiMant = 0;
			loMant = 0; /* infinity */
 		}
		else {	/* Finite */
			expon += 16382;
			if (expon < 0) {	/* denormalized */
				fMant = ldexp(fMant, expon);
				expon = 0;
			}
			expon |= sign;
			fMant = ldexp(fMant, 32);
			fsMant = floor(fMant);
			hiMant = (unsigned long)(fsMant);
			fMant = ldexp(fMant - fsMant, 32);
			fsMant = floor(fMant);
			loMant = (unsigned long)(fsMant);
 		}
	}
	
	bytes[0] = expon >> 8;
	bytes[1] = expon;
	bytes[2] = hiMant >> 24;
	bytes[3] = hiMant >> 16;
	bytes[4] = hiMant >> 8;
	bytes[5] = hiMant;
	bytes[6] = loMant >> 24;
	bytes[7] = loMant >> 16;
	bytes[8] = loMant >> 8;
	bytes[9] = loMant;
}

/* store data in AIFF file */
void aiffproc(fid,item,bfilename)
int		fid;		/* input data file */
struct item_header *item;
char		*bfilename;	/* output file name */
{
	FILE	*op;
	struct aiff_header {
		char	formid[4];
		int32	size;
		char	aiffid[4];
	} aiffhead;
	struct aiff_comm_rec {
		char	commid[4];
		int32	size;
		short	nchan;
		short	numfhi;
		short	numflo;
		short	nbit;
		unsigned char	srate[10];
	} aiffcomm;
	struct aiff_ssnd_rec {
		char	ssndid[4];
		int32	size;
		int32	junk1;
		int32	junk2;
	} aiffssnd;

	/* open output file */
	if ((op=fopen(bfilename,"wb")) == NULL)
		error("unable to open output file %s",bfilename);

	/* write headers */
	memcpy(aiffhead.formid,"FORM",4);
	aiffhead.size = sizeof(aiffhead)+ 26 /*sizeof(aiffcomm)*/ +sizeof(aiffssnd)+2*item->numframes-8;
	memcpy(aiffhead.aiffid,"AIFF",4);
#if SFSMACHINE!=0
	REVERSELONG(aiffhead.size);
#endif
	if (fwrite((char*)&aiffhead,sizeof(aiffhead),1,op)!=1)
		error("write error on '%s'",bfilename);

	/* common data block */
	memcpy(aiffcomm.commid,"COMM",4);
	aiffcomm.size = 26 /*sizeof(aiffcomm)*/ -8;
	aiffcomm.nchan = 1;
	aiffcomm.numfhi = (item->numframes & 0xFFFF0000)>>16;
	aiffcomm.numflo = (item->numframes & 0x0000FFFF);
	aiffcomm.nbit = 16;
	ConvertToIeeeExtended(1.0/item->frameduration,aiffcomm.srate);
#if SFSMACHINE!=0
	REVERSELONG(aiffcomm.size);
	REVERSESHORT(aiffcomm.nchan);
	REVERSESHORT(aiffcomm.numfhi);
	REVERSESHORT(aiffcomm.numflo);
	REVERSESHORT(aiffcomm.nbit);
#endif
	if (fwrite((char*)&aiffcomm,26 /*sizeof(aiffcomm)*/,1,op)!=1)
		error("write error on '%s'",bfilename);

	/* sound data block */
	memcpy(aiffssnd.ssndid,"SSND",4);
	aiffssnd.size = sizeof(aiffssnd)+2*item->numframes-8;
	aiffssnd.junk1 = 0;
	aiffssnd.junk2 = 0;
#if SFSMACHINE!=0
	REVERSELONG(aiffssnd.size);
	REVERSELONG(aiffssnd.junk1);
	REVERSELONG(aiffssnd.junk2);
#endif
	if (fwrite((char*)&aiffssnd,sizeof(aiffssnd),1,op)!=1)
		error("write error on '%s'",bfilename);

	/* copy the samples */
#if SFSMACHINE!=0
	swapbyte = !swapbyte;
#endif
	copydata(fid,op,item->numframes);
	
	/* close output file */
	fclose(op);
}

/* store data in HTK file */
void htkproc(fid,item,bfilename)
int		fid;		/* input data file */
struct item_header *item;
char		*bfilename;	/* output file name */
{
	FILE	*op;
	struct htk_header {
		long	nsamp;
		long	nperiod;
		short	ssize;
		short	skind;
	} hhead;

	/* open output file */
	if ((op=fopen(bfilename,"wb")) == NULL)
		error("unable to open output file %s",bfilename);

	/* write HTK header */
	hhead.nsamp = item->numframes;
	hhead.nperiod = (long) (10000000.0*item->frameduration);
	hhead.ssize = item->framesize * item->datasize;
	hhead.skind = 0;
#if SFSMACHINE!=0
	REVERSELONG(hhead.nsamp);
	REVERSELONG(hhead.nperiod);
	REVERSESHORT(hhead.ssize);
	REVERSESHORT(hhead.skind);
#endif
	if (fwrite((char *)&hhead,sizeof(struct htk_header),1,op) != 1)
		error("write error on %s",bfilename);

	/* copy the samples */
#if SFSMACHINE!=0
	swapbyte = !swapbyte;
#endif
	copydata(fid,op,item->numframes);
	
	/* close output file */
	fclose(op);
}

/* write integer value to file */
void wrint(op,val)
FILE *op;
int val;
{
	putc((val>>24) & 0xFF,op);
	putc((val>>16) & 0xFF,op);
	putc((val>>8)  & 0xFF,op);
	putc(val       & 0xFF,op);
}

/* write 4-char string as integer value to file */
void wrich(op,s)
FILE *op;
char s[];
{
	putc(s[0],op);
	putc(s[1],op);
	putc(s[2],op);
	putc(s[3],op);
}

/* write ils header */
int ilshead(op,item)
FILE *op;
struct item_header *item;
{
	int	i;
	int	samplefreq,ms10,ms20;

	samplefreq = 1.0 / item->frameduration;		/* # samples/sec */
	ms10 = 0.010 / item->frameduration + 0.5;	/* # samples/10ms */
	ms20 = 0.020 / item->frameduration + 0.5;	/* # samples/20ms */

	/* set up ils header with suitable default values */
	/* 1. # points in analysis window */
	wrint(op,ms20);
	/* 2. # autoregressive coefficients */
	wrint(op,12);
	/* 3. pre emphasis constant */
	wrint(op,98);
	/* 4. shift interval per frame */
	wrint(op,ms10);
	/* 5. Hamming window */
	wrich(op,"Y   ");
	/* 6. # sampled data blocks */
	i = item->numframes / 256;
	wrint(op,i);
	/* 7. # spectral resonance peaks */
	wrint(op,4);
	/* 8. starting frame for analysis */
	wrint(op,1);
	/* 9. number of frames analysed */
	i = item->numframes / ms10;
	wrint(op,i);
	/* 10. starting sector for analysis */
	wrint(op,0);
	/* 11 */
	wrint(op,0);
	/* 12. # data points in last block */
	i = item->numframes % 256;
	wrint(op,i);
	/* 13 - 16 */
	for (i=13;i<17;i++) wrich(op,"    ");
	/* 17 - 19 */
	for (i=17;i<20;i++) wrint(op,0);
	/* 20. file number of data file */
	wrint(op,1);
	/* 21 - 22 */
	wrint(op,0);
	wrint(op,0);
	/* 23 - 26 */
	for (i=23;i<27;i++) wrich(op,"    ");
	/* 27 - 32 */
	for (i=27;i<33;i++) wrint(op,0);
	/* 33 - 35 */
	for (i=33;i<36;i++) wrich(op,"    ");
	/* 36 - 60 */
	for (i=36;i<61;i++) wrint(op,0);
	/* 61. sample freq (**10) */
	wrint(op,0);
	/* 62. sample freq */
	wrint(op,samplefreq);
	/* 63. sampled data identifier */
	wrint(op,-32000);
	/* 64. initialised flag */
	wrint(op,32149);
	/* 65 - 128 */
	for (i=65;i<129;i++) wrint(op,0);

	return(0);
}

/* store data in ILS file */
void ilsproc(fid,item,bfilename)
int		fid;		/* input data file */
struct item_header *item;
char		*bfilename;	/* output file name */
{
	FILE	*op;

	/* open output file */
	if ((op=fopen(bfilename,"wb")) == NULL)
		error("unable to open output file %s",bfilename);

	/* write the header */
	ilshead(op,item);

	/* copy the samples */
#if SFSMACHINE!=0
	swapbyte = !swapbyte;
#endif
	copydata(fid,op,item->numframes);
	
	/* close output file */
	fclose(op);

}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* option decoding */
	extern int	optind;
	extern char	*optarg;
	int		errflg=0;
	int		c;
	int32		it;		/* selected item type */
	char		*ty;		/* item specification */
	int		initem=SP_TYPE;
	char		*intype="0";
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* data file */
	int		fid;
	int		pos,count,numf;
	int		i;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:o:t:as8")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Waveform export V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'o' :	/* binary file output */
			binary++;
			strcpy(ofilename,optarg);
			break;
		case 'a' :	/* append to binary file */
			append++;
			break;
		case 's' :	/* swap byte order */
			swapbyte++;
			eightbit=0;
			break;
		case '8' :	/* 8-bit output */
			eightbit++;
			swapbyte=0;
			break;
		case 'i':	/* input item specification */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it == SP_TYPE) ||
				    (it == LX_TYPE)) {
					initem = it;
					intype = ty;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 't' :	/* file type */
			for (format=0;format<NUMFORMAT;format++)
				if (strcmp(optarg,formattab[format].name)==0)
					break;
			if (format >= NUMFORMAT) {
				fprintf(stderr,"known file format types are:");
				for (i=0;i<NUMFORMAT;i++)
					fprintf(stderr," %s",formattab[i].name);
				fprintf(stderr,"\n");
				error("unknown file format type '%s'",optarg);
			}
			binary++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: wave (-I) (-i item) (-t format) (-o outfile) (-a) (-s|-8) dbase_file\n",NULL);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* check for binary output */
	if (swapbyte && !binary)
		error("byte-swapping on binary files only",NULL);

	/* locate start of data set */
	if ((fid=sfsopen(filename,"r",NULL)) < 0)
		error("access error on '%s'",filename);
	if (!sfsitem(fid,initem,intype,&item))
		error("cannot find input item in '%s'",filename);

	if (binary) {
		(*formattab[format].fproc)(fid,&item,ofilename);
	}
	else {
		pos=0;
		numf=item.numframes;
		while (numf > 0) {
			count = sfsread(fid,pos,MIN(numf,COPYBUFSIZE),copybuf);
			if (count <= 0)
				error("read error on intut item");
			if (eightbit) {
				for (i=0;i<count;i++)
					printf("%d\n",copybuf[i]>>8);
			}
			else {
				for (i=0;i<count;i++)
					printf("%d\n",copybuf[i]);
			}
			pos += count;
			numf -= count;
		}
	}

	/* that's all folks */
	sfsclose(fid);
	exit(0);
}


